//vio_misc.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2011
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

static ssize_t roar_vio_misc_read    (struct roar_vio_calls * vio, void *buf, size_t count) {
 struct roar_vio_misc * self = vio->inst;

 if ( self->read == NULL )
  return -1;

 return self->read(buf, count, vio, self);
}

static ssize_t roar_vio_misc_write   (struct roar_vio_calls * vio, void *buf, size_t count) {
 struct roar_vio_misc * self = vio->inst;

 if ( self->write == NULL )
  return -1;

 return self->write(buf, count, vio, self);
}

static off_t   roar_vio_misc_lseek   (struct roar_vio_calls * vio, off_t offset, int whence) {
 return 0;
}

static int     roar_vio_misc_nonblock(struct roar_vio_calls * vio, int state) {
 struct roar_vio_misc * self = vio->inst;

 if ( self->support_nonblocking )
  return 0;

 if ( state == ROAR_SOCKET_BLOCK )
  return 0;

 return -1;
}

static int     roar_vio_misc_sync    (struct roar_vio_calls * vio) {
 return 0;
}

static int     roar_vio_misc_ctl     (struct roar_vio_calls * vio, int cmd, void * data) {
 struct roar_vio_misc * self = vio->inst;

 if ( vio == NULL || cmd == -1 )
  return -1;

 switch (cmd) {
  case ROAR_VIO_CTL_GET_NAME:
    if ( data == NULL )
     return -1;

    *(const char**)data = self->name;
    return 0;
   break;
  case ROAR_VIO_CTL_SET_NOSYNC:
    vio->sync = NULL;
    return 0;
   break;
 }

 return -1;
}

static int     roar_vio_misc_close   (struct roar_vio_calls * vio) {
 return 0;
}

int     roar_vio_open_misc    (struct roar_vio_calls * calls, const struct roar_vio_misc * callbacks) {
 if ( calls == NULL || callbacks == NULL )
  return -1;

 memset(calls, 0, sizeof(struct roar_vio_calls));
 calls->read     = roar_vio_misc_read;
 calls->write    = roar_vio_misc_write;
 calls->lseek    = roar_vio_misc_lseek;
 calls->nonblock = roar_vio_misc_nonblock;
 calls->sync     = roar_vio_misc_sync;
 calls->ctl      = roar_vio_misc_ctl;
 calls->close    = roar_vio_misc_close;
 calls->inst     = (void*)callbacks;

 return 0;
}

static ssize_t roar_vio_misc_op_return_len (void * buf, size_t len, struct roar_vio_calls * vio, struct roar_vio_misc * callbacks) {
 return len;
}

static ssize_t roar_vio_misc_op_return_zero(void * buf, size_t len, struct roar_vio_calls * vio, struct roar_vio_misc * callbacks) {
 return 0;
}

static ssize_t roar_vio_misc_op_zero(void * buf, size_t len, struct roar_vio_calls * vio, struct roar_vio_misc * callbacks) {
 memset(buf, 0, len);
 return len;
}

static ssize_t roar_vio_misc_op_full(void * buf, size_t len, struct roar_vio_calls * vio, struct roar_vio_misc * callbacks)
{
 roar_err_set(ROAR_ERROR_NOSPC);
 return -1;
}

static ssize_t roar_vio_misc_op_random_nonce(void * buf, size_t len, struct roar_vio_calls * vio, struct roar_vio_misc * callbacks) {
 if ( roar_random_gen_nonce(buf, len) == -1 )
  return -1;

 return len;
}

static ssize_t roar_vio_misc_op_random_salt_nonce(void * buf, size_t len, struct roar_vio_calls * vio, struct roar_vio_misc * callbacks) {
 if ( roar_random_salt_nonce(buf, len) == -1 )
  return -1;

 return len;
}

static const struct roar_vio_misc libroar_vio_miscs[] = {
 {.name = "null", .support_nonblocking = 1, .read = roar_vio_misc_op_return_zero, .write = roar_vio_misc_op_return_len},
 {.name = "full", .support_nonblocking = 1, .read = roar_vio_misc_op_return_zero, .write = roar_vio_misc_op_full},
 {.name = "zero", .support_nonblocking = 1, .read = roar_vio_misc_op_zero,        .write = roar_vio_misc_op_return_len},
 {.name = "nrandom", .support_nonblocking = 1,
  .read = roar_vio_misc_op_random_nonce, .write = roar_vio_misc_op_random_salt_nonce}
};

int     roar_vio_open_misc_by_name(struct roar_vio_calls * calls, const char * name) {
 size_t i;

 if ( calls == NULL || name == NULL )
  return -1;

 for (i = 0; i < (sizeof(libroar_vio_miscs)/sizeof(*libroar_vio_miscs)); i++)
  if ( !strcasecmp(name, libroar_vio_miscs[i].name) )
   return roar_vio_open_misc(calls, &(libroar_vio_miscs[i]));

 return -1;
}

//ll
