"""
Styling for prompt_toolkit applications.
"""
from __future__ import unicode_literals
from abc import ABCMeta, abstractmethod
from collections import namedtuple
from pygments.token import Token
from six import with_metaclass

import pygments.style
import pygments.styles.default

__all__ = (
    'Style',
    'Attrs',
    'DynamicStyle',
    'PygmentsStyle',

    'DEFAULT_STYLE_EXTENSIONS',
    'DEFAULT_STYLE',
)


#: Style attributes.
Attrs = namedtuple('Attrs', 'color bgcolor bold underline italic blink reverse')
"""
:param color: Hexadecimal string. E.g. '000000'
:param bgcolor: Hexadecimal string. E.g. 'ffffff'
:param bold: Boolean
:param underline: Boolean
:param italic: Boolean
:param blink: Boolean
:param reverse: Boolean
"""

_default_attrs = Attrs(color=None, bgcolor=None, bold=False, underline=False,
                       italic=False, blink=False, reverse=False)

#: ``Attrs.bgcolor/fgcolor`` can be in either 'ffffff' format, or can be any of
#: the following in case we want to take colors from the 8/16 color palette.
#: Usually, in that case, the terminal application allows to configure the RGB
#: values for these names.
ANSI_COLOR_NAMES = [
    'black', 'white', 'default',

    # Low intensity.
    'red', 'green', 'yellow', 'blue', 'magenta', 'cyan', 'gray',

    # High intensity. (Not supported everywhere.)
    'dark-gray', 'bright-red', 'bright-green', 'bright-yellow', 'bright-blue',
    'bright-magenta', 'bright-cyan',
]


class Style(with_metaclass(ABCMeta, object)):
    """
    Abstract base class for prompt_toolkit styles.
    """
    @abstractmethod
    def get_attrs_for_token(self, token):
        """
        Return :class:`.Attrs` for the given token.
        """

    @abstractmethod
    def invalidation_hash(self):
        """
        Invalidation hash for the style. When this changes over time, the
        renderer knows that something in the style changed, and that everything
        has to be redrawn.
        """


class DynamicStyle(Style):
    """
    Style class that can dynamically returns an other Style.

    :param get_style: Callable that returns a :class:`.Style` instance.
    """
    def __init__(self, get_style):
        self.get_style = get_style

    def get_attrs_for_token(self, token):
        style = self.get_style()
        assert isinstance(style, Style)

        return style.get_attrs_for_token(token)

    def invalidation_hash(self):
        return self.get_style().invalidation_hash()


class PygmentsStyle(Style):
    """
    Adaptor for using Pygments styles as a :class:`.Style`.

    :param pygments_style_cls: Pygments ``Style`` class.
    """
    def __init__(self, pygments_style_cls):
        assert issubclass(pygments_style_cls, pygments.style.Style)
        self.pygments_style_cls = pygments_style_cls
        self._token_to_attrs_dict = None

    def get_attrs_for_token(self, token):
        try:
            style = self.pygments_style_cls.style_for_token(token)
            return Attrs(color=style['color'],
                         bgcolor=style['bgcolor'],
                         bold=style.get('bold', False),
                         underline=style.get('underline', False),
                         italic=style.get('italic', False),
                         blink=False,
                         reverse=False)

        except KeyError:
            return _default_attrs

    def invalidation_hash(self):
        return id(self.pygments_style_cls)

    @classmethod
    def from_defaults(cls, style_dict=None,
                      pygments_style_cls=pygments.styles.default.DefaultStyle,
                      include_extensions=True):
        """
        Shortcut to create a :class:`.PygmentsStyle` instance from a Pygments
        dictionary and a style class.

        :param style_dict: Dictionary for this style. `{Token: style}`.
        :param pygments_style_cls: Pygments style class to start from.
        :param include_extensions: (`bool`) Include prompt_toolkit extensions.
        """
        assert style_dict is None or isinstance(style_dict, dict)
        assert pygments_style_cls is None or issubclass(pygments_style_cls, pygments.style.Style)

        class _CustomStyle(pygments.styles.default.DefaultStyle):
            background_color = None
            styles = {}

            if pygments_style_cls is not None:
                styles.update(pygments_style_cls.styles)

            if include_extensions:
                styles.update(DEFAULT_STYLE_EXTENSIONS)

            if style_dict is not None:
                styles.update(style_dict)

        return cls(_CustomStyle)


#: Styling of prompt-toolkit specific tokens, that are not know by the default
#: Pygments style.
DEFAULT_STYLE_EXTENSIONS = {
    # Highlighting of search matches in document.
    Token.SearchMatch:                            '#000000 bg:#888888',
    Token.SearchMatch.Current:                    '#ffffff bg:#aa8888 underline',

    # Highlighting of select text in document.
    Token.SelectedText:                           '#ffffff bg:#666666',

    # Highlighting of matching brackets.
    Token.MatchingBracket:                        'bg:#aaaaff #000000',

    # Line numbers.
    Token.LineNumber:                             '#888888',
    Token.LineNumber.Current:                     'bold',

    # Default prompt.
    Token.Prompt:                                 'bold',
    Token.Prompt.Arg:                             'noinherit',
    Token.Prompt.Search:                          'noinherit',
    Token.Prompt.Search.Text:                     'bold',

    # Search toolbar.
    Token.Toolbar.Search:                         'bold',
    Token.Toolbar.Search.Text:                    'nobold',

    # System toolbar
    Token.Toolbar.System:                         'bold',

    # "arg" toolbar.
    Token.Toolbar.Arg:                            'bold',
    Token.Toolbar.Arg.Text:                       'nobold',

    # Validation toolbar.
    Token.Toolbar.Validation:                     'bg:#550000 #ffffff',
    Token.WindowTooSmall:                         'bg:#550000 #ffffff',

    # Completions toolbar.
    Token.Toolbar.Completions:                    'bg:#bbbbbb #000000',
    Token.Toolbar.Completions.Arrow:              'bg:#bbbbbb #000000 bold',
    Token.Toolbar.Completions.Completion:         'bg:#bbbbbb #000000',
    Token.Toolbar.Completions.Completion.Current: 'bg:#444444 #ffffff',

    # Completions menu.
    Token.Menu.Completions.Completion:            'bg:#bbbbbb #000000',
    Token.Menu.Completions.Completion.Current:    'bg:#888888 #ffffff',
    Token.Menu.Completions.Meta:                  'bg:#999999 #000000',
    Token.Menu.Completions.Meta.Current:          'bg:#aaaaaa #000000',
    Token.Menu.Completions.MultiColumnMeta:       'bg:#aaaaaa #000000',
    Token.Menu.Completions.ProgressBar:           'bg:#aaaaaa',
    Token.Menu.Completions.ProgressButton:        'bg:#000000',

    # Scrollbars.
    Token.Scrollbar:                              'bg:#444444',
    Token.Scrollbar.Button:                       'bg:#888888',
    Token.Scrollbar.Arrow:                        'bg:#222222 #ffffff',

    # Auto suggestion text.
    Token.AutoSuggestion:                         '#666666',

    # When Control-C has been pressed. Grayed.
    Token.Aborted:                                '#888888',
}

default_style_extensions = DEFAULT_STYLE_EXTENSIONS  # Old name.


#: The default built-in style.
DEFAULT_STYLE = PygmentsStyle.from_defaults()
