/*	Selector

PIRL CVS ID: Selector.java,v 1.8 2012/04/16 06:14:23 castalia Exp

Copyright (C) 2001-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.PVL;

/**	A <I>Selector</I> is used with a <I>Parameter</I> or <I>Value</I>
	wherever selective comparison of Parameters and/or Values is needed.
<P>
	The criteria are specified as the characteristics of the Parameter
	and/or Value to use in making a selection, and the combinatorial
	boolean logic to apply to the characteristics when comparing one object
	against another. The implementation must provide the definition of the
	criteria and comparison methods declared here. The Selector interface
	defines the criteria symbols:
<P>
	Parameter selection characteristics:
<P>
<BLOCKQUOTE><CODE>
	NAME<BR>
	CLASSIFICATION<BR>
	VALUE<BR>
	COMMENTS<BR>
</CODE></BLOCKQUOTE>
<P>
	Value selection characteristics:
<P>
<BLOCKQUOTE><CODE>
	DATA<BR>
	TYPE<BR>
	BASE<BR>
	UNITS<BR>
</CODE></BLOCKQUOTE>
<P>
	Criteria modifiers:
<P>
<BLOCKQUOTE><CODE>
	ANY<BR>
	ANY_PARAMETER<BR>
	ANY_VALUE<BR>
	SPECIFIC<BR>
	PATTERN_MATCH<BR>
</CODE></BLOCKQUOTE>
<P>
	Logic criteria (as test conditions):
<P>
<BLOCKQUOTE><CODE>
	EQUAL<BR>
	LESS_THAN<BR>
	GREATER_THAN<BR>
	AND<BR>
</CODE></BLOCKQUOTE>
<P>
	How the methods that test for match conditions on pairs of Parameter or
	Value objects can be controlled by combining the selection criteria. A
	Selector criteria code is composed of bit flags stored in an int
	variable.
<P>
	@see	Selection
	@see	Parameter
	@see	Value

	@author		Bradford Castalia, UA/PIRL
	@version	1.8 
*/
public interface Selector
{
//	The selection criteria:
// Java guarantees that an int is 32 bits.

//	Parameter selection criteria (0-7)

/**	Bit mask for the Parameter characteristics criteria.
*/
public static final int
	PARAMETER_SELECTION	= 0xFF;

/**	Criteria to select a Parameter's name.
*/
public static final int
	NAME				= 1 << 0;

/**	Criteria to select a Parameter's classification.
*/
public static final int
	CLASSIFICATION		= 1 << 1;

/**	Criteria to select a Parameter's Value.
*/
public static final int
	VALUE				= 1 << 2;

/**	Criteria to select a Parameter's comments.
*/
public static final int
	COMMENTS			= 1 << 3;


//	Value selection criteria (8-15)

/**	Bit mask for the Value characteristics criteria.
*/
public static final int
	VALUE_SELECTION		= 0xFF00;

/**	Criteria to select the data of a Value.
*/
public static final int
	DATA				= 1 << 8;

/**	Criteria to select a Value's type.
*/
public static final int
	TYPE				= 1 << 9;

/**	Criteria to select a Value's integer number base.
*/
public static final int
	BASE				= 1 << 10;

/**	Criteria to select a Value's units description.
*/
public static final int
	UNITS				= 1 << 11;


//	Modifications to the selection criteria (16-23)

/**	Bit mask for the criteria modifiers.
*/
public static final int
	MODIFIERS			= 0xFF0000;

/**	Criteria to select any characteristic.
*/
public static final int
	ANY					= 0;

/**	Criteria to select any Parameter characteristic.
*/
public static final int
	ANY_PARAMETER		= 1 << 16;

/**	Criteria to select any Value characteristic.
*/
public static final int
	ANY_VALUE			= 1 << 17;

/**	Criteria to use specific criteria matching.
*/
public static final int
	SPECIFIC			= 1 << 18;

/**	Criteria to use pattern matching.
*/
public static final int
	PATTERN_MATCH		= 1 << 19;


//	The combinatorial logic to use (24-31)

/**	Bit mask for the criteria logic.
*/
public static final int
	LOGIC				= 0xFF000000;
	/*
		The ANY_XXX bits are only used in the return values
		for the {Parameter,Value}_Criteria_Match methods to
		flag when the match was due to ANY condition.
	*/

/**	Criteria to use equality logic when comparing values.

	This can be combined with <CODE>LESS_THAN</CODE> for <= logic,
	or <CODE>GREATER_THAN</CODE> for >= logic.
*/
public static final int
	EQUAL				= 1 << 24;

/**	Criteria to use < logic when comparing values.

	This can be combined with <CODE>EQUAL</CODE> for <= logic.
*/
public static final int
	LESS_THAN			= 1 << 25;

/**	Criteria to use > logic when comparing values.

	This can be combined with <CODE>EQUAL</CODE> for >= logic.
*/
public static final int
	GREATER_THAN		= 1 << 26;

/**	Criteria to use logical AND when matching with multiple criteria.

	<B>Note</B>: When this flag is <I>not</I> set, OR logic is used.
*/
public static final int
	AND					= 1 << 27;


//	For exact match.

/**	Criteria to select an exact match of all Value characteristics.

	This is the same the
	"<CODE>DATA | TYPE | BASE | UNITS | AND | SPECIFIC</CODE>"
	criteria combination.
*/
public static final int
	VALUE_MATCH
		= TYPE | BASE | UNITS | DATA | AND | SPECIFIC | EQUAL;

/**	Criteria to select an exact match of all Parameter characteristics.

	This is the same the
	"<CODE>NAME | CLASSIFICATION | VALUE | COMMENTS | VALUE_MATCH</CODE>"
	criteria combination.
*/
public static final int
	PARAMETER_MATCH
		= NAME | CLASSIFICATION | VALUE | COMMENTS | VALUE_MATCH;


/**	Gets a description of the Selector.
*/
public String toString ();


//	Access selection bits directly:

/**	Sets selection criteria to the specified code.
*/
public Selector Criteria (int criteria);

/**	Gets the current selection criteria code.
*/
public int Criteria ();

/**	Gets the current Parameter selection criteria code.
*/
public int Parameter_Criteria ();

/**	Gets the current Value selection criteria code.
*/
public int Value_Criteria ();

/**	Gets the current selection criteria modifications code.
*/
public int Modifiers ();

/**	Gets the current selection logic code.
*/
public int Logic ();


//	Parameter criteria:

/**	Enables or disables the Parameter name criteria.
*/
public Selector Name (boolean mode);

/**	Tests if the Parameter name criteria is enabled.
*/
public boolean Name ();

/**	Enables or disables the Parameter classification criteria.
*/
public Selector Classification (boolean mode);

/**	Tests if the Parameter classification criteria is enabled.
*/
public boolean Classification ();

/**	Enables or disables the Parameter Value criteria.
*/
public Selector Value (boolean mode);

/**	Tests if the Parameter Value criteria is enabled.
*/
public boolean Value ();

/**	Enables or disables the Parameter comments criteria.
*/
public Selector Comments (boolean mode);

/**	Tests if the Parameter comments criteria is enabled.
*/
public boolean Comments ();

/**	Enables or disables using any Parameter characteristic to match.
*/
public Selector Any_Parameter (boolean mode);

/**	Tests if any Parameter characteristic will match.
*/
public boolean Any_Parameter ();


//	Value criteria:

/**	Enables or disables using the data of a Value as a criteria.
*/
public Selector Data (boolean mode);

/**	Tests if the data of a Value is a criteria.
*/
public boolean Data ();

/**	Enables or disables the Value type criteria.
*/
public Selector Type (boolean mode);

/**	Tests if the Value type criteria is enabled.
*/
public boolean Type ();

/**	Enables or disables the Value integer base criteria.
*/
public Selector Base (boolean mode);

/**	Tests if the Value integer base criteria is enabled.
*/
public boolean Base ();

/**	Enables or disables the Value units description criteria.
*/
public Selector Units (boolean mode);

/**	Tests if the Value units description criteria is enabled.
*/
public boolean Units ();

/**	Enables or disables using any Value characteristic to match.
*/
public Selector Any_Value (boolean mode);

/**	Tests if any Value characteristic will match.
*/
public boolean Any_Value ();


//	Criteria modifiers:

/**	Enables or disables using pattern matching for comparisons.
*/
public Selector Pattern_Match (boolean mode);

/**	Tests if pattern matching is to be used for comparisons.
*/
public boolean Pattern_Match ();

/**	Enables of disables specific category matching.
*/
public Selector Specific (boolean mode);

/**	Tests if specific category matching is to be used for comparisons.
*/
public boolean Specific ();


//	Criteria logic:

/**	Enables or disables using equality logic for value comparisons.
*/
public Selector Equal (boolean mode);

/**	Tests if equality logic is to be used for value comparisons.
*/
public boolean Equal ();

/**	Enables or disables using less-than logic for value comparisons.
*/
public Selector Less_Than (boolean mode);

/**	Tests if less-than logic is to be used for value comparisons.
*/
public boolean Less_Than ();

/**	Enables or disables using greater-than logic for value comparisons.
*/
public Selector Greater_Than (boolean mode);

/**	Tests if greater-than logic is to be used for value comparisons.
*/
public boolean Greater_Than ();

/**	Enables or disables using logical AND when matching with multiple criteria.

	Disabling AND logic enables OR logic.
*/
public Selector And (boolean mode);

/**	Tests if AND logic is to be used when matching with multiple criteria.
*/
public boolean And ();


//	Parameter criteria match methods:

/**	Tests if two Parameters match according to the current selection
	criteria.
*/
public boolean Parameters_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	);

/** Gets the criteria that were met, as a selection code, during the
	last use of the <CODE>Parameters_Match</CODE> method.
<P>
	Only those necessary and sufficient criteria that were actually met
	during the comparisons are set. There may be additional criteria
	that could have been met but were not needed to satisfy the match.
*/
public int Parameters_Match ();

/** Gets all the criteria that are met, as a selection code, when
	comparing two Parameters.
<P>
	All current selection criteria are tested and those that are
	met are set in the returned code.
*/
public int Parameters_Criteria_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	);

/**	Tests if the names of two Parameters match according to the current
	selection criteria.
*/
public boolean Names_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	);

/**	Tests if the classifications of two Parameters match according to
	the current selection criteria.
*/
public boolean Classifications_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	);

/**	Tests if the Values of two Parameters match according to the
	current selection criteria.
*/
public boolean Parameter_Values_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	);

/**	Tests if the comments of two Parameters match according to the
	current selection criteria.
*/
public boolean Comments_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	);


//	Value criteria match methods:

/**	Tests if two Values match according to the current selection
	criteria.
*/
public boolean Values_Match
	(
	Value		this_value,
	Value		that_value
	);

/** Gets the criteria that were met, as a selection code, during the
	last use of the <CODE>Values_Match</CODE> method.
<P>
	Only those necessary and sufficient criteria that were actually met
	during the comparisons are set. There may be additional criteria
	that could have been met but were not needed to satisfy the match.
*/
public int Values_Match ();

/** Gets all the criteria that are met, as a selection code, when
	comparing two Values.
<P>
	All current selection criteria are tested and those that are
	met are set in the returned code.
*/
public int Values_Criteria_Match
	(
	Value		this_value,
	Value		that_value
	);

/**	Tests if the data of two Values match according to the current
	selection criteria.
*/
public boolean Data_Match
	(
	Value		this_value,
	Value		that_value
	);

/**	Tests if the types of two Values match according to the current
	selection criteria.
*/
public boolean Types_Match
	(
	Value		this_value,
	Value		that_value
	);

/**	Tests if the integer base numbers of two Values match according to
	the current selection criteria.
*/
public boolean Bases_Match
	(
	Value		this_value,
	Value		that_value
	);

/**	Tests if the units descriptions of two Values match according to
	the current selection criteria.
*/
public boolean Units_Match
	(
	Value		this_value,
	Value		that_value
	);

}
