#!/bin/sh
# EBS Create slave script
#
# Copyright 2011 Canonical
#
# This script will create a new slave instance from a snapshot of the XFS filesystem of the master
# Requires: euca2ools and eucarc config file XFS filesystem


## Required variables
while getops "t:c:m:z:r:l:" flag
do
    case "$flag" in
        t)
            # Define the cloud type
            if [ -z "$OPTARG" ]
            then
                CLOUDTYPE="UEC" # Valid options are UEC or EC2
            else
                CLOUDTYPE="$OPTARG"
            fi
            ;;
        c)
            # Define cloudrc location
            if [ -z "$OPTARG" ]
            then
                CLOUDRC="/etc/cloudconfig/eucarc"
            else
                CLOUDRC="$OPTARG"
            fi
            ;;
        m)
            # Define master IP address
            if [ -z "$OPTARG" ]
            then
                MASTERIP="172.56.125.2"
            else
                MASTERIP="$OPTARG"
            fi
            ;;
        z)
            # Define availability zone
            if [ -z "$OPTARG" ]
            then
                ZONE="canonicloud"
            else
                ZONE="$OPTARG"
            fi
            ;;
        r)
            # Define remote mysql credentials
            if [ -z "$OPTARG" ]
            then
                REMOTE_MYSQL_USER="replicator"
                REMOTE_MYSQL_PASS="myreplica"
            else
                REMOTE_MYSQL_USER=`echo $OPTARG | awk -F":" '{print $1}'`
                REMOTE_MYSQL_PASS=`echo $OPTARG | awk -F":" '{print $2}'`
            fi
            ;;
        l)
            # Define local mysql credentials
            if [ -z "$OPTARG" ]
            then
                LOCAL_MYSQL_USER="root"
                LOCAL_MYSQL_PASS="insecure"
            else
                LOCAL_MYSQL_USER=`echo $OPTARG | awk -F":" '{print $1}'`
                LOCAL_MYSQL_PASS=`echo $OPTARG | awk -F":" '{print $2}'`
            fi
            ;;
        h)
            # Show help
            echo "Usage: $0 -t [cloud_type] -c [cloudrc_location] -m [master_ip_address] -z [availability_zone] -r [remotemysql_credentials] -l [localmysql_credentials]"
            echo "Parameters:"
            echo " t: Cloud type, can be UEC or EC2"
            echo " c: Cloudrc file, is the file where all your credentials can be sourced from"
            echo " m: Master IP address"
            echo " z: EBS Volumes availability zone"
            echo " r: Remote MySQL replication credentials (in user:pass format)"
            echo " l: Local MySQL root credentials (in user:pass format)"
            echo "Example:"
            echo "$0 -t UEC -c /home/ubuntu/.euca/eucarc -m 172.56.33.12 -z canonicloud -r replicator:replica -l root:insecure"
            exit 2
            ;;
    esac
done

## Binaries
case "$CLOUDTYPE" in
    UEC)
        INSTANCE_DESCRIBE="/usr/bin/euca-describe-instances"
        VOLUME_DESCRIBE="/usr/bin/euca-describe-volumes"
        VOLUME_CREATE="/usr/bin/euca-create-volume"
        VOLUME_ATTACH="/usr/bin/euca-attach-volume"
        SNAPSHOT_DESCRIBE="/usr/bin/euca-describe-snapshots"
        TOOLSNAME="euca2ools"
        ;;
    EC2)
        INSTANCE_DESCRIBE="/usr/bin/ec2-describe-instances"
        VOLUME_DESCRIBE="/usr/bin/ec2-describe-volumes"
        VOLUME_CREATE="/usr/bin/ec2-create-volume"
        VOLUME_ATTACH="/usr/bin/ec2-attach-volume"
        SNAPSHOT_DESCRIBE="/usr/bin/ec2-describe-snapshots"
        TOOLSNAME="ec2-api-tools"
        ;;
    *)
        echo "ERROR: No valid cloud type found, please edit the script configuration"
        exit 1
        ;;
esac

## Checks before execution
echo "`date` - Preflight checks"

# Check for euca2ools presence
if [ ! -f "$INSTANCE_DESCRIBE" ]
then
    echo "ERROR: Missing $TOOLSNAME, please make sure they're installed"
    exit 1
fi

# Check that euca2ools works fine, at the same time check that there's a volume attached to our instance

# Source cloud config
echo "`date` - Importing cloud credentials"
. $CLOUDRC
echo "`date` - Recovering Instance ID"
MYIP=`/usr/bin/facter ipaddress`
MYINSTANCE=`$INSTANCE_DESCRIBE | grep $MYIP | awk '{print $2}'`
if [ -z "$MYINSTANCE" ]
then
    echo "ERROR: Missing credentials, please make sure the credentials file is properly referenced in the config"
    exit 1
fi
echo "`date` - Recovering master instance ID"
REMOTEINSTANCE=`$INSTANCE_DESCRIBE | grep $MASTERIP | awk '{print $2}'`
echo "`date` - Recovering master volume ID"
REMOTEVOLUME=`$VOLUME_DESCRIBE | grep ${REMOTEINSTANCE} | awk '{print $2}'`
# Get available snapshots
echo "`date` - Recovering snapshot information"
SNAPSHOT=`$SNAPSHOT_DESCRIBE | grep -i completed | grep ${REMOTEVOLUME} | sort -k5 -r | awk '{print $2}' | head -n1`
echo "`date` - Recovering MySQL replication credentials from master"
MASTERINFO=`/usr/bin/mysql -Bs -u${REMOTE_MYSQL_USER} -p${REMOTE_MYSQL_PASS} -h${MASTERIP} mysql -e "select binlog_file,binlog_position from replication where snapshot='$SNAPSHOT'" | awk '{print $1"|"$2}'`

# Make sure we have all the information we need to proceed forward
if [ -z "$REMOTEINSTANCE" ] || [ -z "$REMOTEVOLUME" ] || [ -z "$SNAPSHOT" ] || [ -z "$MASTERINFO" ]
then
    echo "ERROR: Couldn't recover one or more master parameters, can't properly spawn a slave"
    exit 1
fi

REMOTE_MYSQL_LOGFILE=`echo ${MASTERINFO} | awk -F"|" '{print $1}'`
REMOTE_MYSQL_LOGPOS=`echo ${MASTERINFO} | awk -F"|" '{print $2}'`

# Make sure mysql is stopped
echo "`date` - Stopping MySQL service"
service mysql stop

# Create volume from snapshot
echo "`date` - Creating volume from last snapshot ($SNAPSHOT)"
MYVOL=`$VOLUME_CREATE --snapshot $SNAPSHOT -z $ZONE | awk '{print $2}'`

# Need to wait for volume to be created
echo "`date` - Waiting on volume $MYVOL to be available"
while [ $(${VOLUME_DESCRIBE} | grep $MYVOL | grep -i creating | wc -l) -eq 1 ]
do
    sleep 2
done

# Attach volume
echo "`date` - Attaching volume $MYVOL"
$VOLUME_ATTACH -i $MYINSTANCE -d /dev/sdb $MYVOL

# Wait on Volume to be available
echo "`date` - Waiting on volume $MYVOL to be attached"
while [ $(/sbin/fdisk -l /dev/sdb 2>/dev/null | wc -l) -eq 0 ]
do
    sleep 2
done

# Remove any trace of existance of default mysql install
echo "`date` - Remove previous data on MySQL dir"
rm -rf /var/lib/mysql/*

# Mount volume
echo "`date` - Mounting volume"
/bin/mount /dev/sdb /var/lib/mysql

# Make sure we don't have any master data
echo "`date` - Removing any previous slave config in imported data"
rm -f /var/lib/mysql/master*

# Modify config file
echo "`date` - Generating config file for slave bin"
LOCAL_MYSQL_ID=`perl -e '$random = int( rand(9999)); print "$random\n";' 2>/dev/null`

cat >> /etc/mysql/conf.d/slave-replica.cnf << EOF
# Replication values added by cloud-init
[mysqld]
server-id = $LOCAL_MYSQL_ID
log_bin = /var/log/mysql/mysql-bin.log
EOF

LOCAL_TMP_FILE="/tmp/mysql-promoteslave.sql"
echo "CHANGE MASTER TO MASTER_HOST=\"${MASTERIP}\", MASTER_USER=\"${REMOTE_MYSQL_USER}\", MASTER_PASSWORD=\"${REMOTE_MYSQL_PASS}\", MASTER_LOG_FILE=\"${REMOTE_MYSQL_LOGFILE}\", MASTER_LOG_POS=${REMOTE_MYSQL_LOGPOS};" >> $LOCAL_TMP_FILE
echo "START SLAVE;" >> $LOCAL_TMP_FILE

# Start mysql and recover instance from logs
echo "`date` - Starting MySQL"
service mysql start

echo "`date` - Activating slave"
/usr/bin/mysql -u${LOCAL_MYSQL_USER} -p${LOCAL_MYSQL_PASS} < $LOCAL_TMP_FILE
rm -f ${LOCAL_TMP_FILE}

# All operations okay
exit 0

