/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef _MINOLTA_RAW_H
#define _MINOLTA_RAW_H

/* DISCLAIMER: The following information has been obtained by analysing the format
 * of RAW files generated by Minolta D5, D7, and D7i cameras. It's not based on any
 * official (or otherwise as far as I know) Minolta documentation.
 */

#include <stdio.h>
#include <MRI_Types.h>

/* The Minolta RAW file is divided into three main sections:
 * - a 48 (0x30) byte header
 * - tiff-style data about the image
 * - the raster data.
 */

/* The following definitions describe fields within the PRD block.
 */
#define	MRW_PRD_CAMCODE_OFFSET		0x00
#define	MRW_PRD_CAMCODE_LENGTH		8
#define	MRW_PRD_CCDHEIGHT_OFFSET	0x08
#define MRW_PRD_CCDHEIGHT_LENGTH	2
#define	MRW_PRD_CCDWIDTH_OFFSET		0x0a
#define	MRW_PRD_CCDWIDTH_LENGTH		2
#define	MRW_PRD_IMAGE_HEIGHT		0x0c
#define	MRW_PRD_IMAGE_HEIGHT_LEN	2
#define	MRW_PRD_IMAGE_WIDTH		0x0e
#define	MRW_PRD_IMAGE_WIDTH_LEN		2
#define	MRW_PRD_UNKNOWN2		0x10
#define	MRW_PRD_UNKNOWN2_LEN		2
#define	MRW_PRD_UNKNOWN3		0x12
#define	MRW_PRD_UNKNOWN3_LEN		2
#define	MRW_PRD_UNKNOWN4		0x14
#define	MRW_PRD_UNKNOWN4_LEN		2
#define	MRW_PRD_UNKNOWN5		0x16
#define	MRW_PRD_UNKNOWN5_LEN		2

/* The following definitions describe fields within the TTW block.
 *
 * The data following the initial TTW block header is a standard tiff header.
 *
 * Offsets into the TTW (tiff) data:
 */
#define	MRW_TIFFMAGIC		0x0	/* Always 0x4d4d */
#define MRW_TIFFMAGIC_LENGTH	2
#define	MRW_TIFFVERSION		0x2	/* Always 42. */
#define MRW_TIFFVERSION_LENGTH	2
#define	MRW_TIFFIFD0_OFFSET	0x4	/* Offset from start of tiff data to initial IFD. */
#define MRW_TIFFIFD0_LENGTH	4

#include "mrwtiff.h"

/* The tiff data consists of three known image file directories (IFDs).
 * The following definitions describe tags found in the principal IFD.
 */
#define IFDTAG_WIDTH		0x0100	/* Image width. */
#define IFDTAG_HEIGHT		0x0101	/* Image height. */
#define IFDTAG_COMPRESS		0x0103	/* Compression. */
#define IFDTAG_DCFVER		0x010E	/* DCF version (string). */
#define	IFDTAG_MANUF		0x010F	/* Manufacturer (string). */
#define IFDTAG_CAMERA		0x0110	/* Camera name (string). */
#define IFDTAG_FIRMWARE		0x0131	/* Firmware version (string). */
#define IFDTAG_DATETIME		0x0132	/* Date time (string). */
#define IFDTAG_EXIFOFFSET 	0x8769	/* Offset of EXIF data (long). */
#define IFDTAG_PIMOFFSET	0xC4A5	/* Offset of PIM info (undefined). */

/* The IFDTAG_EXIFOFFSET entry in the principal IFD is a pointer to an
 * EXIF data IFD.
 * The following definitions describe tags found in the EXIF IFD.
 */
#define EXIFTAG_EXPOSURETIME    0x829A
#define EXIFTAG_FNUMBER         0x829D
#define	EXIFTAG_EXPPROG		0x8822		/* Exposure Program. */
#define	EXIFTAG_ISOSPEED 	0x8827		/* ISO Speed. */
#define EXIFTAG_VERSION		0x9000		/* Exif version. */
#define	EXIFTAG_DATETIMEORIG 	0x9003
#define EXIFTAG_DATETIMEDIGITIZED 0x9004
#define EXIFTAG_SHUTTERSPEED 	0x9201
#define EXIFTAG_APERTURE 	0x9202
#define EXIFTAG_EXPOSUREBIAS 	0x9204
#define EXIFTAG_MAXAPERTURE 	0x9205
#define EXIFTAG_METERMODE 	0x9207
#define EXIFTAG_LIGHTSOURCE 	0x9208
#define EXIFTAG_FLASH 		0x9209
#define EXIFTAG_FOCALLEN 	0x920A
#define EXIFTAG_MANUFNOTES 	0x927C
#define EXIFTAG_COLORSPACE 	0xA001

/* The EXIFTAG_MANUFNOTES entry in the EXIF IFD is a pointer to a
 * manufacturer's IFD containing the following (known) entries:
 */
#define	MANUTAG_BLOCKDESC	0x0000
#define	MANUTAG_CAMERASETTINGS	0x0001
#define	MANUTAG_CAMERASETTINGS2	0x0003
#define	MANUTAG_AUTOFOCUS	0x0010
#define	MANUTAG_UNKNOWN		0x0020
#define	MANUTAG_IMAGESIZE	0x0040
#define	MANUTAG_THUMBNAIL	0x0081
#define	MANUTAG_PIMINFO		0x0E00

/* The manufacturers IFD contains a pointer to a block of data describing
 * the camera settings.
 * In the original DiMage 5/7, the pointer is given by tag MANUTAG_CAMERASETTINGS.
 * In the DiMage 7i, the pointer is given by tag MANUTAG_CAMERASETTINGS2.
 */

/* Known offsets into the camera settings info.
 * The following information is known for the original D5/D7.
 * At least the picture number of white balance data seem to be the same
 * for the D7i. The FLASH_OFFSET seems to be wrong for the 7i. I have no
 * information about the other fields.
 */
#define	MRW_MACRO_OFFSET	0x2F
#define	MRW_MACRO_LENGTH	1

#define	MRW_ZOOM_OFFSET		0x48
#define	MRW_ZOOM_LENGTH		4

#define	MRW_FOCUS_OFFSET	0x4c	/* Only if in manual focus mode. */
#define	MRW_FOCUS_LENGTH	4	/* 0 = infinity, otherwise dist in mm */

#define	MRW_FLASH_OFFSET	0x53
#define	MRW_FLASH_LENGTH	1

#define	MRW_PICTNUM_OFFSET	0x6E	/* Value is 0-offset, so need to add one. */
#define	MRW_PICTNUM_LENGTH	2

#define	MRW_BALANCE1_OFFSET	0x72
#define	MRW_BALANCE1_LENGTH	2
#define MRW_BALANCE2_OFFSET	0x76
#define MRW_BALANCE2_LENGTH	2
#define	MRW_BALANCE3_OFFSET	0x7A
#define	MRW_BALANCE3_LENGTH	2

#define MRW_COLOR_OFFSET	0x7F
#define	MRW_COLOR_LENGTH	1

#define	MRW_CONTRAST_OFFSET	0x83
#define	MRW_CONTRAST_LENGTH	1

/* The following definitions describe fields within the WBG block.
 */
#define	MRW_WBG_RSCALE_OFFSET	0x00
#define	MRW_WBG_RSCALE_LENGTH	1
#define	MRW_WBG_G1SCALE_OFFSET	0x01
#define	MRW_WBG_G1SCALE_LENGTH	1
#define	MRW_WBG_G2SCALE_OFFSET	0x02
#define	MRW_WBG_G2SCALE_LENGTH	1
#define	MRW_WBG_BSCALE_OFFSET	0x03
#define	MRW_WBG_BSCALE_LENGTH	1
#define	MRW_WBG_RGAIN_OFFSET	0x04
#define	MRW_WBG_RGAIN_LENGTH	2
#define	MRW_WBG_G1GAIN_OFFSET	0x06
#define	MRW_WBG_G1GAIN_LENGTH	2
#define	MRW_WBG_G2GAIN_OFFSET	0x08
#define	MRW_WBG_G2GAIN_LENGTH	2
#define	MRW_WBG_BGAIN_OFFSET	0x0A
#define	MRW_WBG_BGAIN_LENGTH	2

/**************************************************************************************
 *
 * The following structure stores our (internal) representation of a loaded MRW file.
 *
 */

typedef struct {
	int		type;		/* Block type. */
	int		length;		/* Length of this block's data. */
	unsigned char *	data;		/* Pointer to data for this block. */
} MRW_Block;

typedef struct {
	MRW_Block	mrm;		/* Primary data block at start of file. */
	MRW_Block	prd, ttw, wbg, rif; /* Subdata blocks contained within mrm block. */
	IFD *mrw_ifd;			/* Tiff IFD's contained in ttw block. */
	IFD *exif_ifd;
	IFD *manu_ifd;
	unsigned char *pimdata;
	unsigned char *settings;
} MRW_Header;

typedef struct {
    unsigned char header[12];
    unsigned char *data;
    int data_length;
    IFD *mrw_ifd, *exif_ifd, *manu_ifd;
    unsigned char *pim;
    int pim_length;
    unsigned char *settings;
    int settings_length;
    unsigned char *thumbnail;
    int thumbnail_length;
} MinoltaJPG;


#define mrw_header	MRW_Header	/* Until we clean up all the old mrw_headers */

extern unsigned MRW_Width (const MRW_Header *hdr);

extern unsigned MRW_Height (const MRW_Header *hdr);

extern const char *MRW_DCFVer (const MRW_Header *hdr);

extern const char *MRW_Manufacturer (const MRW_Header *hdr);

extern const char *MRW_CameraName (const MRW_Header *hdr);

extern const char *MRW_Firmware (const MRW_Header *hdr);

extern const char *MRW_DateTime (const MRW_Header *hdr);

extern int MRW_ISO (const MRW_Header *hdr);

extern double MRW_Shutter (const MRW_Header *hdr);

extern double MRW_Aperture (const MRW_Header *hdr);

extern double MRW_FocalLen (const MRW_Header *hdr);

extern double MRW_FocalLen35mm (const MRW_Header *hdr);

extern double MRW_FocusLen (const MRW_Header *hdr);

extern int MRW_Macro (const MRW_Header *hdr);

extern int MRW_Flash (const MRW_Header *hdr);

extern int MRW_Pictnum (const MRW_Header *hdr);

extern MRI_balance MRW_WBalance (const MRW_Header *hdr);

extern int MRW_Color (const MRW_Header *hdr);

extern int MRW_Contrast (const MRW_Header *hdr);

extern int LoadMRW (MRW_Header *hdr, FILE *f, char **errmsgptr);
extern int LoadMinoltaJPG (FILE *f, MinoltaJPG *hdr);

extern int MRW_FocusMode (const MRW_Header *hdr);

extern int MRW_GetPresetBalance (const MRI *mri, const char *presetname, MRI_balance *);
extern MRI_Iterator *MRW_PresetIterator (const MRI *mri);

#endif
