/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include "MRI.h"
#include "vmedian.h"
#include <lcms.h>

struct MRIPPEData {
	struct link *next;
	int	iwidth, owidth;
	int	iheight, oheight;
	int	freedata;
	int	redGreen;
	int	type;
	struct MRI_ScanLine *a, *b;
	struct MRI_ScanLine *bsave;
};

static void
MRIPPEStart (void *private, int width, int height, int freedata)
{
	struct MRIPPEData *wd = private;

	wd->iwidth = width;
	wd->iheight = height;
	switch (wd->type) {
	case MRI_PSEUDO_ORIG:
	    wd->owidth = width-1;
	    wd->oheight = (height-1)*2;
	    break;
	case MRI_PSEUDO_SMOOTH:
	    wd->owidth = width;
	    wd->oheight = (height-1)*2;
	    break;
	default:
	    break;
	}
	(*wd->next->start) (wd->next->private, wd->owidth, wd->oheight, TRUE);
	wd->freedata = freedata;
	wd->a = (struct MRI_ScanLine *)0;
	wd->b = (struct MRI_ScanLine *)0;
	wd->bsave = (struct MRI_ScanLine *)0;
}

static void
MRIPPERow (void *private, void *data)
{
	struct MRIPPEData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;

	if (wd->a != (struct MRI_ScanLine *)0) {
		/* Add pixels to previous pseudo-pixel lines and output. */
		if (wd->redGreen) {
			float *wdbG = wd->b->G;
			float *wdaR = wd->a->R;
			float *wdbR = wd->b->R;
			float *slG = sl->G;
			float *slR = sl->R;
			wdbR[0] = wdaR[0] = slR[0];
			for (x = 1; x < wd->iwidth-2; x+=2) {
				wdaR[x] = wdaR[x+1] = slR[x+1];
				wdbR[x] = wdbR[x+1] = slR[x+1];
			}
			for (x = 0; x < wd->iwidth-2; x+=2)
				wdbG[x] = wdbG[x+1] = slG[x+1];
			wdbG[wd->iwidth-2] = slG[wd->iwidth-1];
		}
		else {
			float *wdbG = wd->b->G;
			float *wdaB = wd->a->B;
			float *wdbB = wd->b->B;
			float *slG = sl->G;
			float *slB = sl->B;
			wdbG[0] = slG[0];
			for (x = 1; x < wd->iwidth-2; x+=2)
				wdbG[x] = wdbG[x+1] = slG[x+1];
			for (x = 0; x < wd->iwidth-2; x+=2) {
				wdaB[x] = wdaB[x+1] = slB[x+1];
				wdbB[x] = wdbB[x+1] = slB[x+1];
			}
			wdbB[wd->iwidth-2] = wdaB[wd->iwidth-2] = slB[wd->iwidth-1];
		}
		(*wd->next->row) (wd->next->private, wd->a);
		(*wd->next->row) (wd->next->private, wd->b);
	}
	/* Add pixels to next two pseudo-pixel rows. */
	wd->a = MRI_NewScanLine (LINETYPE_FLOAT, wd->owidth);
	wd->b = MRI_NewScanLine (LINETYPE_FLOAT, wd->owidth);
	if (wd->redGreen) {
		float *wdaG = wd->a->G;
		float *wdaR = wd->a->R;
		float *wdbR = wd->b->R;
		float *slR = sl->R;
		float *slG = sl->G;
		wdbR[0] = wdaR[0] = slR[0];
		for (x = 1; x < wd->iwidth-2; x+=2) {
			wdaR[x] = wdaR[x+1] = slR[x+1];
			wdbR[x] = wdbR[x+1] = slR[x+1];
		}
		for (x = 0; x < wd->iwidth-2; x+=2)
			wdaG[x] = wdaG[x+1] = slG[x+1];
		wdaG[wd->iwidth-2] = slG[wd->iwidth-1];
	}
	else {
		float *wdaG = wd->a->G;
		float *wdaB = wd->a->B;
		float *wdbB = wd->b->B;
		wdaG[0] = ((float *)(sl->G))[0];
		for (x = 1; x < wd->iwidth-2; x+=2)
			wdaG[x] = wdaG[x+1] = ((float *)(sl->G))[x+1];
		for (x = 0; x < wd->iwidth-2; x+=2) {
			float val = ((float *)(sl->B))[x+1];
			wdaB[x] = wdaB[x+1] = val;
			wdbB[x] = wdbB[x+1] = val;
		}
		wdbB[wd->iwidth-2] = wdaB[wd->iwidth-2] = ((float *)(sl->B))[wd->iwidth-1];
	}
	wd->redGreen = !wd->redGreen;
	if (wd->freedata)
		MRI_FreeScanLine (sl);
}

static void
MRIPPERowSmooth (void *private, void *data)
{
	struct MRIPPEData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;

	if (wd->a != (struct MRI_ScanLine *)0) {
		/* Add pixels to previous pseudo-pixel lines and output. */
		if (wd->redGreen) {
			/* RGRGRG...RGRG */
			float R0, R1, G0, G1;
			float *wdaR = wd->a->R;
			float *wdbR = wd->b->R;
			float *wdbG = wd->b->G;

			R0 = ((float *)(sl->R))[0];
			x = 0;
			while (x < wd->iwidth-2) {
				R1 = ((float *)(sl->R))[x+2];
				wdaR[x] = (wdaR[x] + R0) / 2;
				wdaR[x+1] = ((wdaR[x+1] * 2) + R0 + R1) / 4;
				wdbR[x] = (wdbR[x] + R0*3) / 4;
				wdbR[x+1] = (wdbR[x+1] + R0 + R1 + ((R0 + R1) / 2)) / 4;
				if (wd->bsave != (struct MRI_ScanLine *)0) {
					float *wdbsR = wd->bsave->R;
					wdbsR[x] = (wdbsR[x]*3 + R0) / 4;
					wdbsR[x+1] = (wdbsR[x+1]*3 + ((R0 + R1) / 2)) / 4;
				}
				R0 = R1;
				x += 2;
			}
			wdbR[x] = wdaR[x] = R0;
			wdbR[x+1] = wdaR[x+1] = R0;
			if (wd->bsave != (struct MRI_ScanLine *)0) {
				float *wdbsR = wd->bsave->R;
				wdbsR[x] = (wdbsR[x]*3 + R0) / 4;
				wdbsR[x+1] = (wdbsR[x+1]*3 + R0) / 4;
			}

			G0 = ((float *)(sl->G))[1] / 2;
			wdbG[0] += G0;
			wdbG[1] += G0;
			x = 2;
			while (x < wd->iwidth) {
				float G2;
				G1 = ((float *)(sl->G))[x+1] / 2;
				G2 = wdbG[x] + ((G0 + G1) / 2);
				wdbG[x] = G2 > 65535 ? 65535 : G2;
				G2 = wdbG[x+1] + G1;
				wdbG[x+1] = G2 > 65535 ? 65535 : G2;
				G0 = G1;
				x += 2;
			}
		}
		else {
			/* GBGBGB...GBGB */
			float G0, G1, B0, B1;
			float *wdbG = wd->b->G;
			float *wdaB = wd->a->B;
			float *wdbB = wd->b->B;

			x = 0;
			G0 = ((float *)(sl->G))[0] / 2;
			while (x < wd->iwidth-2) {
				G1 = ((float *)(sl->G))[x+2] / 2;
				wdbG[x] += G0;
				wdbG[x+1] += (G0 + G1) / 2;
				G0 = G1;
				x += 2;
			}
			wdbG[x] = wdbG[x+1] = G0;

			B0 = ((float *)(sl->B))[1];
			wdaB[0] = wdaB[1] = B0;
			wdbB[0] = wdbB[1] = B0;
			if (wd->bsave != (struct MRI_ScanLine *)0) {
				float *wdbsB = wd->bsave->B;
				wdbsB[0] = (wdbsB[0]*3 + B0) / 4;
				wdbsB[1] = (wdbsB[1]*3 + B0) / 4;
			}
			x = 2;
			while (x < wd->iwidth) {
				B1 = ((float *)(sl->B))[x+1];
				wdaB[x] = (B0 + B1) / 2;
				wdaB[x+1] = B1;
				wdbB[x] = (B0 + B1) / 2;
				wdbB[x+1] = B1;
				if (wd->bsave != (struct MRI_ScanLine *)0) {
					float *wdbsB = wd->bsave->B;
					wdbsB[x] = (wdbsB[x]*3 + ((B0 + B1) / 2)) / 4;
					wdbsB[x+1] = (wdbsB[x+1]*3 + B1) / 4;
				}
				B0 = B1;
				x += 2;
			}
		}
		if (wd->bsave != (struct MRI_ScanLine *)0)
			(*wd->next->row) (wd->next->private, wd->bsave);
		(*wd->next->row) (wd->next->private, wd->a);
		wd->bsave = wd->b;
	}
	/* Add pixels to next two pseudo-pixel rows. */
	wd->a = MRI_NewScanLine (LINETYPE_FLOAT, wd->owidth);
	wd->b = MRI_NewScanLine (LINETYPE_FLOAT, wd->owidth);
	if (wd->redGreen) {
		float R0, R1, G0, G1;
		float *wdaR = wd->a->R;
		float *wdbR = wd->b->R;
		float *wdaG = wd->a->G;
		float *wdbG = wd->b->G;

		if (wd->bsave != (struct MRI_ScanLine *)0) {
			float *wdbsB = wd->bsave->B;
			float *wdaB = wd->a->B;
			float *wdbB = wd->b->B;
			x = 0;
			while (x < wd->iwidth) {
				wdaB[x] = wdbB[x] = wdbsB[x];
				x++;
			}
		}

		R0 = ((float *)(sl->R))[0];
		x = 0;
		while (x < wd->iwidth-2) {
			R1 = ((float *)(sl->R))[x+2];
			wdaR[x] = R0;
			wdaR[x+1] = (R0 + R1) / 2;
			wdbR[x] = R0;
			wdbR[x+1] = (R0 + R1) / 2;
			R0 = R1;
			x += 2;
		}
		wdaR[x] = wdaR[x+1] = R0;
		wdbR[x] = wdbR[x+1] = R0;

		G0 = ((float *)(sl->G))[1];
		wdaG[0] = wdaG[1] = G0;
		wdbG[0] = wdbG[1] = G0 / 2;
		x = 2;
		while (x < wd->iwidth) {
			G1 = ((float *)(sl->G))[x+1];
			wdaG[x] = (G0 + G1) / 2;
			wdaG[x+1] = G1;
			wdbG[x] = (G0 + G1) / 4;
			wdbG[x+1] = G1 / 2;
			G0 = G1;
			G0 = G1;
			x+=2;
		}
	}
	else {
		/* GBGBGB...GBGBGB */
		float G0, G1, B0, B1;
		float *wdaB = wd->a->B;
		float *wdbB = wd->b->B;
		float *wdaG = wd->a->G;
		float *wdbG = wd->b->G;

		if (wd->bsave != (struct MRI_ScanLine *)0) {
			float *wdbsR = wd->bsave->R;
			float *wdaR = wd->a->R;
			float *wdbR = wd->b->R;
			x = 0;
			while (x < wd->iwidth) {
				wdaR[x] = wdbR[x] = wdbsR[x];
				x++;
			}
		}

		G0 = ((float *)(sl->G))[0];
		x = 0;
		while (x < wd->iwidth-2) {
			G1 = ((float *)(sl->G))[x+2];
			wdaG[x] = G0;
			wdaG[x+1] = (G0 + G1) / 2;
			wdbG[x] = G0 / 2;
			wdbG[x+1] = (G0 + G1) / 4;
			G0 = G1;
			x+=2;
		}
		wdaG[x] = wdaG[x+1] = G0;
		wdbG[x] = wdbG[x+1] = G0 / 2;

		B0 = ((float *)(sl->B))[1];
		wdaB[0] = wdaB[1] = B0;
		wdbB[0] = wdbB[1] = B0;
		x = 2;
		while (x < wd->iwidth) {
			B1 = ((float *)(sl->B))[x+1];
			wdaB[x] = (B0 + B1) / 2;
			wdaB[x+1] = B1;
			wdbB[x] = (B0 + B1) / 2;
			wdbB[x+1] = B1;
			B0 = B1;
			x += 2;
		}
	}
	wd->redGreen = !wd->redGreen;
	if (wd->freedata)
		MRI_FreeScanLine (sl);
}

static void
MRIPPERowDouble (void *private, void *data)
{
	struct MRIPPEData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;

	if (wd->a != (struct MRI_ScanLine *)0) {
		/* Add pixels to previous pseudo-pixel lines and output. */
		if (wd->redGreen) {
			/* RGRGRG...RGRG */
			float R0, R1, G0, G1;
			float *wdaR = wd->a->R;
			float *wdbR = wd->b->R;
			float *wdbG = wd->b->G;

			R0 = ((float *)(sl->R))[0];
			x = 0;
			while (x < wd->iwidth-2) {
				R1 = ((float *)(sl->R))[x+2];
				wdaR[x] = (wdaR[x] + R0) / 2;
				wdaR[x+1] = ((wdaR[x+1] * 2) + R0 + R1) / 4;
				wdbR[x] = (wdbR[x] + R0*3) / 4;
				wdbR[x+1] = (wdbR[x+1] + R0 + R1 + ((R0 + R1) / 2)) / 4;
				if (wd->bsave != (struct MRI_ScanLine *)0) {
					float *wdbsR = wd->bsave->R;
					wdbsR[x] = (wdbsR[x]*3 + R0) / 4;
					wdbsR[x+1] = (wdbsR[x+1]*3 + ((R0 + R1) / 2)) / 4;
				}
				R0 = R1;
				x += 2;
			}
			wdbR[x] = wdaR[x] = R0;
			wdbR[x+1] = wdaR[x+1] = R0;
			if (wd->bsave != (struct MRI_ScanLine *)0) {
				float *wdbsR = wd->bsave->R;
				wdbsR[x] = (wdbsR[x]*3 + R0) / 4;
				wdbsR[x+1] = (wdbsR[x+1]*3 + R0) / 4;
			}

			G0 = ((float *)(sl->G))[1] / 2;
			wdbG[0] += G0;
			wdbG[1] += G0;
			x = 2;
			while (x < wd->iwidth) {
				G1 = ((float *)(sl->G))[x+1] / 2;
				wdbG[x] += (G0 + G1) / 2;
				wdbG[x+1] += G1;
				G0 = G1;
				x += 2;
			}
		}
		else {
			/* GBGBGB...GBGB */
			float G0, G1, B0, B1;
			float *wdbG = wd->b->G;
			float *wdaB = wd->a->B;
			float *wdbB = wd->b->B;

			x = 0;
			G0 = ((float *)(sl->G))[0] / 2;
			while (x < wd->iwidth-2) {
				G1 = ((float *)(sl->G))[x+2] / 2;
				wdbG[x] += G0;
				wdbG[x+1] += (G0 + G1) / 2;
				G0 = G1;
				x += 2;
			}
			wdbG[x] = wdbG[x+1] = G0;

			B0 = ((float *)(sl->B))[1];
			wdaB[0] = wdaB[1] = B0;
			wdbB[0] = wdbB[1] = B0;
			if (wd->bsave != (struct MRI_ScanLine *)0) {
				float *wdbsB = wd->bsave->B;
				wdbsB[0] = (wdbsB[0]*3 + B0) / 4;
				wdbsB[1] = (wdbsB[1]*3 + B0) / 4;
			}
			x = 2;
			while (x < wd->iwidth) {
				B1 = ((float *)(sl->B))[x+1];
				wdaB[x] = (B0 + B1) / 2;
				wdaB[x+1] = B1;
				wdbB[x] = (B0 + B1) / 2;
				wdbB[x+1] = B1;
				if (wd->bsave != (struct MRI_ScanLine *)0) {
					float *wdbsB = wd->bsave->B;
					wdbsB[x] = (wdbsB[x]*3 + ((B0 + B1) / 2)) / 4;
					wdbsB[x+1] = (wdbsB[x+1]*3 + B1) / 4;
				}
				B0 = B1;
				x += 2;
			}
		}
		if (wd->bsave != (struct MRI_ScanLine *)0)
			(*wd->next->row) (wd->next->private, wd->bsave);
		(*wd->next->row) (wd->next->private, wd->a);
		wd->bsave = wd->b;
	}
	/* Add pixels to next two pseudo-pixel rows. */
	wd->a = MRI_NewScanLine (LINETYPE_FLOAT, wd->owidth);
	wd->b = MRI_NewScanLine (LINETYPE_FLOAT, wd->owidth);
	if (wd->redGreen) {
		float R0, R1, G0, G1;
		float *wdaR = wd->a->R;
		float *wdbR = wd->b->R;
		float *wdaG = wd->a->G;
		float *wdbG = wd->b->G;

		if (wd->bsave != (struct MRI_ScanLine *)0) {
			float *wdbsB = wd->bsave->B;
			float *wdaB = wd->a->B;
			float *wdbB = wd->b->B;
			x = 0;
			while (x < wd->iwidth) {
				wdaB[x] = wdbB[x] = wdbsB[x];
				x++;
			}
		}

		R0 = ((float *)(sl->R))[0];
		x = 0;
		while (x < wd->iwidth-2) {
			R1 = ((float *)(sl->R))[x+2];
			wdaR[x] = R0;
			wdaR[x+1] = (R0 + R1) / 2;
			wdbR[x] = R0;
			wdbR[x+1] = (R0 + R1) / 2;
			R0 = R1;
			x += 2;
		}
		wdaR[x] = wdaR[x+1] = R0;
		wdbR[x] = wdbR[x+1] = R0;

		G0 = ((float *)(sl->G))[1];
		wdaG[0] = wdaG[1] = G0;
		wdbG[0] = wdbG[1] = G0 / 2;
		x = 2;
		while (x < wd->iwidth) {
			G1 = ((float *)(sl->G))[x+1];
			wdaG[x] = (G0 + G1) / 2;
			wdaG[x+1] = G1;
			wdbG[x] = (G0 + G1) / 4;
			wdbG[x+1] = G1 / 2;
			G0 = G1;
			G0 = G1;
			x+=2;
		}
	}
	else {
		/* GBGBGB...GBGBGB */
		float G0, G1, B0, B1;
		float *wdaB = wd->a->B;
		float *wdbB = wd->b->B;
		float *wdaG = wd->a->G;
		float *wdbG = wd->b->G;

		if (wd->bsave != (struct MRI_ScanLine *)0) {
			float *wdbsR = wd->bsave->R;
			float *wdaR = wd->a->R;
			float *wdbR = wd->b->R;
			x = 0;
			while (x < wd->iwidth) {
				wdaR[x] = wdbR[x] = wdbsR[x];
				x++;
			}
		}

		G0 = ((float *)(sl->G))[0];
		x = 0;
		while (x < wd->iwidth-2) {
			G1 = ((float *)(sl->G))[x+2];
			wdaG[x] = G0;
			wdaG[x+1] = (G0 + G1) / 2;
			wdbG[x] = G0 / 2;
			wdbG[x+1] = (G0 + G1) / 4;
			G0 = G1;
			x+=2;
		}
		wdaG[x] = wdaG[x+1] = G0;
		wdbG[x] = wdbG[x+1] = G0 / 2;

		B0 = ((float *)(sl->B))[1];
		wdaB[0] = wdaB[1] = B0;
		wdbB[0] = wdbB[1] = B0;
		x = 2;
		while (x < wd->iwidth) {
			B1 = ((float *)(sl->B))[x+1];
			wdaB[x] = (B0 + B1) / 2;
			wdaB[x+1] = B1;
			wdbB[x] = (B0 + B1) / 2;
			wdbB[x+1] = B1;
			B0 = B1;
			x += 2;
		}
	}
	wd->redGreen = !wd->redGreen;
	if (wd->freedata)
		MRI_FreeScanLine (sl);
}

static void
MRIPPEClose (void *private)
{
	struct MRIPPEData *wd = private;

	if (wd->bsave != (struct MRI_ScanLine *)0)
		(*wd->next->row) (wd->next->private, wd->bsave);
	MRI_FreeScanLine (wd->a);
	MRI_FreeScanLine (wd->b);
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}


struct link *
MRI_GenPseudoPixelExpanderFloat (struct link *next, int RGFirst, int type)
{
	struct MRIPPEData *wd = malloc (sizeof (struct MRIPPEData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct MRIPPEData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	wd->type = type;
	ep->start = MRIPPEStart;
	switch (type) {
	case MRI_PSEUDO_ORIG:
	    ep->row = MRIPPERow;
	    break;
	case MRI_PSEUDO_SMOOTH:
	    ep->row = MRIPPERowSmooth;
	    break;
	default:
	    break;
	}
	ep->close = MRIPPEClose;
	ep->private = wd;
	wd->next = next;
	wd->redGreen = RGFirst;

	return ep;
}
