#include <iostream>
#include <string>
#include <sstream>

#include "Dictionary.h"
#include "StringConvert.h"
#include "ReadStream.h"
#include "DicomFile.h"

using namespace jcs;
using namespace std;

DicomFile::DicomFile(const char* filename) 
: mVrSyntax(EXPLICIT_VR), mByteOrder(dLITTLE_ENDIAN), 
  mFileName(filename), mBeginning(0),
  mInputFile(filename, std::ios::binary)
{
	mErrors.reset();
	mDicom = Dicom_Dictionary::Instance();
	mInitializeMap();
	
	if (!mSkipPreamble()) {
		if (mErrors.test(NO_DICM)) {
			// attempt to read as dataset
			DicomElement e1, e2;
			mReadNextElement(e1, mInputFile);
			mReadNextElement(e2, mInputFile);
			mInputFile.seekg(std::ios::beg);
			if ((e1.tag.group == 8) && (e2.tag.group == 8)) {
				mErrors.reset();
			} else {
				mSetTransferSyntax(IMPLICIT_VR, dLITTLE_ENDIAN);
				mReadNextElement(e1, mInputFile);
				mReadNextElement(e2, mInputFile);
				mInputFile.seekg(std::ios::beg);
				if ((e1.tag.group == 8) && (e2.tag.group == 8)) {
					mErrors.reset();
				}
			}
		}
	} else {
		mReadMetafile();
	}
	mCheckOk();

	string manufacturer;
	Find("Manufacturer", manufacturer);

	if (manufacturer == "SIEMENS") {
		string software;
		Find("SoftwareVersion", software);
		if (software == ("VA13F"))
			mSpecial = Numaris_Dictionary::Instance();
		else mSpecial = mDicom;
	} 
	else if (manufacturer == "GE MEDICAL SYSTEMS") {
		string product_id;
		Find(DicomTag(0x0009, 0x1004), product_id);
		if (product_id.find("SIGNA") != string::npos) 
			mSpecial = Excite_Dictionary::Instance();
		else 
			mSpecial = mDicom;
	} 
	else if (manufacturer.find("Philips") != string::npos){
		string modality;
		Find("Modality", modality);
		if (modality == "MR")
			mSpecial = PhilipsMr_Dictionary::Instance();
		else
			mSpecial = mDicom;
	}
	else {
		mSpecial = mDicom;
	
	}

	foundPixelDataStart = false;
}

bool
DicomFile::IsOk() 
{
        return mErrors.none(); 
}

// This is where we put reasons to reject files
void
DicomFile::mCheckOk()
{
	if (mErrors.none()) {

		string s;

		if(!Find("SopInstance", s))
			mErrors.set(NO_UID);

		if(!Find("SeriesInstanceUid", s)) 
			mErrors.set(NO_SERIES_UID);

		if (!Find("InstanceNumber", s)) 
			mErrors.set(NO_INSTANCE);

		if (!Find("PhotometricInterpretation", s)) 
			mErrors.set(NO_IMAGE);

	}

}

const char*
DicomFile::ErrorMessage() const
{
	if (mErrors.none()) 
		return "No errors.";

	if (mErrors.test(NO_BYTES))
		return "File is <128 bytes. Not a valid DICOM file.";

	if (mErrors.test(NO_DICM))
		return "Unable to read file, possibly not DICOM format.";
	
	if (mErrors.test(METAFILE_ERROR))
		return "Problem reading metafile header.";

	if (mErrors.test(TRANSFER_SYNTAX))
		return "Unsupported transfer syntax.";
	
	if (mErrors.test(NO_SERIES_UID))
		return "Unable to find series UID.";

	if (mErrors.test(NO_UID))
		return "Unable to find instance UID.";

	if (mErrors.test(NO_INSTANCE))
		return "Unable to find instance number.";

	if (mErrors.test(NO_IMAGE))
		return "File does not contain image data.";

	// should never get here, but...
	return "Error reading DICOM file.";
}

// todo: change to GetSeriesUid (or move to DicomSeries)
// Does anybody even call this anyway?
//std::string
//DicomFile::GetUid()
//{
//	std::string uid; 
//	if(!Find("SeriesInstanceUid", uid))
//		uid = "none";
//	return uid;
//}


// Map indices refer to VR types, defined in PS 3.5.
void
DicomFile::mInitializeMap() {
	ReadMap["FL"] = ReadStream<float>::doit;
	ReadMap["FD"] = ReadStream<double>::doit;
	ReadMap["OW"] = ReadStream<wxUint16>::doit;
	ReadMap["AT"] = ReadStream<wxUint16>::doit;
	ReadMap["SL"] = ReadStream<wxInt32>::doit;
	ReadMap["SS"] = ReadStream<wxInt16>::doit;
	ReadMap["UL"] = ReadStream<wxUint32>::doit;
	ReadMap["US"] = ReadStream<wxUint16>::doit;
	ReadMap["UN"] = ReadStream<string>::doit;	// Default
}


DicomFile::~DicomFile()
{
	mInputFile.close();
}

int
DicomFile::FindElement(DicomElement& d, istream& input) 
{
	if (!input.good()) {
		return 0;
	}

	DicomElement e;
	if (!mFindElement(d.tag, e, input)) {
		return 0;
	}
	if (mVrSyntax == EXPLICIT_VR && e.vr != "OB" && e.vr != "UN"  )
		d.vr = e.vr;
	d.value_length = e.value_length;
	
	return 1;

}

int
DicomFile::GetCSAImageHeader(string& header)
{
	string software;
	Find("SoftwareVersion", software);

	if (software.find("syngo") == string::npos) 
		return 0; // not a syngo file

	wxUint16 group, element;
	group = 0x0029;
	bool found = 0;
	for (element = 0x0010; element < 0x0100; ++element) {
		DicomElement creator(DicomTag(group, element));
		string creator_name;
		if (Find(creator, creator_name)) {
			if (creator_name == "SIEMENS CSA HEADER") {
				found = 1;
				break;
			}
		}
	}
	
	if (!found) 
		return -1; // Siemens CSA Header not found in private creator list

	element = (element << 8) | 0x0010;
	DicomTag csa_header_info_tag(group, element);
	int retval;
	if(!Find(csa_header_info_tag, header)) 
		retval = -2; //header not found
	else 
		retval = 1;
	return retval;
}

bool
DicomFile::HasSiemensMrHeader()
{
	string software;
	Find("SoftwareVersion", software);

	if (software.find("syngo") == string::npos) 
		return false; // not a syngo file

	wxUint16 group, element;
	group = 0x0019;
	for (element = 0x0010; element < 0x0100; ++element) {
		DicomElement creator(DicomTag(group, element));
		string creator_name;
		if (Find(creator, creator_name)) {
			if (creator_name == "SIEMENS MR HEADER") {
				return true;
			}
		}
	}
	
	return false; // Siemens MR Header not found in private creator list
}

int
DicomFile::GetCSASeriesHeader(string& header)
{
	string software;
	Find("SoftwareVersion", software);

	if (software.find("syngo") == string::npos) 
		return 0; // not a syngo file

	wxUint16 group, element;
	group = 0x0029;
	bool found = 0;
	for (element = 0x0010; element < 0x0100; ++element) {
		DicomElement creator(DicomTag(group, element));
		string creator_name;
		if (Find(creator, creator_name)) {
			if (creator_name == "SIEMENS CSA HEADER") {
				found = 1;
				break;
			}
		}
	}
	
	if (!found) return -1; // Siemens CSA Header not found in private creator list

	element = (element << 8) | 0x0020;
	DicomTag csa_header_info_tag(group, element);
	if(!Find(csa_header_info_tag, header)) 
		return -2; //header not found
	else 
		return 1;
}


int
DicomFile::ReadCSAImageHeader(const string& search, string& value)
{
	value = string(); // set value to empty string

	string csa_header_info;
	if (GetCSAImageHeader(csa_header_info) < 1) 
		return 0;

	return ReadCSAHeader(csa_header_info, search, value);
}

int
DicomFile::ReadCSAImageHeader(const string& search, vector<string>& value)
{

	value.clear(); 

	string csa_header_info;
	if (GetCSAImageHeader(csa_header_info) < 1) 
		return 0;

	return ReadCSAHeader(csa_header_info, search, value);
}


int
DicomFile::ReadCSASeriesHeader(const string& search, string& value)
{
	value = string(); // set value to empty string

	string csa_header_info;

	if (GetCSASeriesHeader(csa_header_info) < 1) 
		return 0;

	return ReadCSAHeader(csa_header_info, search, value);
}

int
DicomFile::ReadCSASeriesHeader(const string& search, vector<string>& value)
{
	value.clear();

	string csa_header_info;

	if (GetCSASeriesHeader(csa_header_info) < 1) 
		return 0;

	return ReadCSAHeader(csa_header_info, search, value);
}


int
DicomFile::ReadCSAHeader(const string& header, const string& search, string& value)
{

	string::size_type pos = 0;

	wxUint32 intval2;
	wxUint32 intval3;
	
	for (;;) {
		pos = header.find(search, pos);
		if (pos == string::npos) return 0; // search string not found

		if (search.compare(header.substr(pos, 64).c_str())) { // wrong hit
			pos += 1;
			continue;
		}

		pos += 64;

		wxInt32 n_values;
		n_values = *reinterpret_cast<wxInt32*> (&header.substr(pos)[0]);

		if (n_values > 10000) { // crude way to look for another wrong hit
			pos -= 63;
			continue;
		}

		pos += 3*4;
		intval2 = *reinterpret_cast<wxUint32*>(&header.substr(pos)[0]);

		pos += 4;
		intval3 = *reinterpret_cast<wxUint32*>(&header.substr(pos)[0]);

		if (intval3 == 205) 
			return -1; // search string found, but no value given

		if ((intval2 % 6) == 0 && intval3 == 77) break;

	}

	pos += 4*4;

	wxUint32 value_length = *reinterpret_cast<wxUint32*>(&header.substr(pos)[0]);
	pos += 4;
	value = header.substr(pos, value_length);

	return 1;
}


int
DicomFile::ReadCSAHeader(const string& header, const string& search, vector<string>& value)
{

	string::size_type pos = 0;

	wxUint32 intval2;
	wxUint32 intval3;
	wxInt32 n_values;
	
	for(;;) {

		pos = header.find(search, pos);
		if (pos == string::npos) return 0; // search string not found

		if (search.compare(header.substr(pos, 64).c_str())) { // wrong hit
			pos += 1;
			continue;
		}

		pos += 64;

		n_values = *reinterpret_cast<wxInt32*> (&header.substr(pos)[0]);

		if (n_values > 10000) { // crude way to look for another wrong hit
			pos -= 63;
			continue;
		}

		pos += 3*4;
		intval2 = *reinterpret_cast<wxUint32*>(&header.substr(pos)[0]);

		pos += 4;
		intval3 = *reinterpret_cast<wxUint32*>(&header.substr(pos)[0]);

		if (intval3 == 205) 
			return -1; // search string found, but no value given

		if ((intval2 % 6) == 0 && intval3 == 77) break;


	}

	pos += 4;
	for (;;) {
		pos += 4;
		intval2 = *reinterpret_cast<wxUint32*>(&header.substr(pos)[0]);
		pos += 4;
		intval3 = *reinterpret_cast<wxUint32*>(&header.substr(pos)[0]);

		if (intval3 != 77 || pos == string::npos) break; 

		pos += 4;
		wxUint32 value_length = *reinterpret_cast<wxUint32*>(&header.substr(pos)[0]);
		pos += 4;
		value.push_back(header.substr(pos, value_length));
		while (value_length % 4) {
			++value_length;
		}
		pos += value_length;
	}

	return 1;
}

std::vector<DicomElement>
DicomFile::DataDump()
{
	std::vector<DicomElement> dump;

	mInputFile.clear();
	mInputFile.seekg(mBeginning);

	std::vector<DicomElement> temp;
	temp = mReadNextElementAll();
	
	while(mInputFile.good()) {
		dump.insert(dump.end(), temp.begin(), temp.end());
		temp = mReadNextElementAll();
	};
/*
	vector<DicomElement>::iterator it = dump.begin();
	vector<DicomElement>::iterator it_end = dump.end();

	while (it != it_end) {
		
		DicomElement dict_entry;

		if (it->tag.group & 1)  // private tags
			dict_entry = mSpecial->Lookup(it->tag);
		else
			dict_entry = mDicom->Lookup(it->tag);
					
		it->description = dict_entry.description;

		// 2nd cond fixes philips problems with explicit VR = OB
//		if ((mVrSyntax != EXPLICIT_VR) || (dict_entry.vr != "UN"))
//			it->vr = dict_entry.vr;

		++it;

	}
*/
	return dump;
}

void
DicomFile::DataDump(const char *filename)
{
	std::ofstream output;
	output.open(filename);
	DicomElement e;
	mReadNextElement(e, mInputFile);
	
	while(e.tag != Tag("PixelData")) {

		output << e.tag;
		std::cout << e.tag;

		if (e.value_length != UNDEFINED) {
			output << e.value_length << ' ';
			std::cout << e.value_length << ' ';

			std::string s;
			mInputFile.seekg(-static_cast<std::streamoff>(e.value_length), 
				std::ios::cur);
			for (unsigned int i = 0; i < e.value_length; ++i) {
				s += mInputFile.get();
			}

			output << s << std::endl;
			std::cout << s << std::endl;
		}

		else {
			output << "Undefined length item" << std::endl;
			std::cout << "Undefined length item" << std::endl;
		}


		mReadNextElement(e, mInputFile);

	}
	output.close();
}

int
DicomFile::GetHeaderSize()
{
	mInputFile.seekg(mBeginning);

	DicomElement element;
	mReadNextElement(element, mInputFile);
	while (mInputFile.good() && (element.tag != Tag("PixelData"))) {
		mReadNextElement(element, mInputFile);
	}
	return (static_cast<int>(mInputFile.tellg()) - element.value_length);
}

int
DicomFile::GetFileSize()
{
	mInputFile.seekg(0, std::ios::end);
	return static_cast<int>(mInputFile.tellg());
}


int 
DicomFile::mSkipPreamble() 
{
	if (!mInputFile.is_open()) 
		return 0;
	if (!IsOk()) 
		return 0;

	mInputFile.ignore(128);
	
	if (!mInputFile.good()) {
		mErrors.set(NO_BYTES);
		return 0;
	}

	std::string s;

	for (int i = 0; i < 4; ++i) {
		s += mInputFile.get();
	}

	if (s != "DICM") {
		mErrors.set(NO_DICM);
		mInputFile.seekg(std::ios::beg);
		return 0;
	}
	else return 1;

}

int
DicomFile::mReadMetafile() 
{
	if (!mInputFile.is_open()) 
		return 0;
	if (!IsOk()) 
		return 0;

	mBeginning = mInputFile.tellg();
	streamsize group_length;
	int Findresult = Find("MfGroupLength", group_length);
	if (!Findresult) {
		mErrors.set(METAFILE_ERROR);
		return 0;
	}

	streampos meta_file_end = mInputFile.tellg() + 
		static_cast<streampos>(group_length);

	std::string s;
	if(!Find("MfTransferSyntax", s)) {
		mErrors.set(TRANSFER_SYNTAX);
		return 0;
	}
//	else {
//		std::cout << "--Transfer Syntax: " << s << std::endl;
//	}


	// Can only read little endian
	if (strcmp(s.c_str(), "1.2.840.10008.1.2") == 0) {
		mSetTransferSyntax(IMPLICIT_VR, dLITTLE_ENDIAN);
	}
	else if(strcmp(s.c_str(), "1.2.840.10008.1.2.1") == 0) {
		mSetTransferSyntax(EXPLICIT_VR, dLITTLE_ENDIAN);
	}	
	else {
		mErrors.set(TRANSFER_SYNTAX);
		return 0;
	}

	mInputFile.seekg(meta_file_end);
	mBeginning = mInputFile.tellg();
	return 1;
}



void 
DicomFile::mSetTransferSyntax(VrSyntaxType v, dicomEndianType b) 
{
	mByteOrder = b;
	mVrSyntax = v;
}

// 20070511cdt - byte-swap on big-endian machines
void
DicomFile::mReadElementTag(DicomElement& e, istream& input) 
{
	input.read(reinterpret_cast<char*> (&e.tag.group), 2);
	input.read(reinterpret_cast<char*> (&e.tag.element), 2);

	wxUint16 group_raw = e.tag.group;
	wxUint16 element_raw = e.tag.element;

	// Assumes group and element values are unsigned 16-bit ints.
//	e.tag.group = ByteSwap (e.tag.group);
//	e.tag.element = ByteSwap (e.tag.element);

}

void
DicomFile::mReadValueRepresentation(DicomElement& e, istream& input) 
{
	// Check VR Syntax, but if group is < 0x0008, we are in the 
	//  header, which is always EXPLICIT_VR. 20070524cdt
	if ((mVrSyntax == IMPLICIT_VR) && (e.tag.group >= 0x0008))
		return;
	e.vr = input.get();
	e.vr += input.get();
}

void
DicomFile::mReadValueLength(DicomElement& e, istream& input) 
{

	wxUint32 value_length_raw = e.value_length;

	// Check VR Syntax, but if group is < 0x0008, we are in the 
	//  header, which is always EXPLICIT_VR. 20070524cdt
	if ((mVrSyntax == IMPLICIT_VR) && (e.tag.group >= 0x0008)) {

		input.read(reinterpret_cast<char*> (&e.value_length), 
			sizeof(e.value_length));
//		e.value_length = ByteSwap (e.value_length);
	}
	// These VR types have a 32-bit value length, see Table 7.1-1
	//  of PS 3.5.
	else if ( e.vr == "OB" ||	
		  e.vr == "OW" || 
		  e.vr == "OF" ||   // 20070516cdt
		  e.vr == "UT" ||   // 20070516cdt
		  e.vr == "SQ" ||
		  e.vr == "UN" ) {

			input.ignore(2);	// 2 bytes are reserved
				
			input.read(reinterpret_cast<char*> (&e.value_length), 
				sizeof(e.value_length));
//			e.value_length = ByteSwap (e.value_length);
		} else {
		
			wxUint16 vl_16;
			input.read(reinterpret_cast<char*> (&vl_16), 
				sizeof(vl_16));
//			vl_16 = ByteSwap (vl_16);
			e.value_length = static_cast<wxUint32> (vl_16);
		}

}

void
DicomFile::mReadNextElement(DicomElement &rElement, istream& input) 
{

	mReadElementTag(rElement, input);
	mReadValueRepresentation(rElement, input);
	mReadValueLength(rElement, input);

	if (rElement.value_length != UNDEFINED) {
		input.ignore(rElement.value_length);
	}

	// Undefined length item
	else {
		istream::pos_type start_pos = input.tellg();
		DicomElement item;
		mReadElementTag(item, input);
		input.read(reinterpret_cast<char*> (&item.value_length), 
			sizeof(item.value_length));

		while (item.tag != Tag("SqDelimiter")) { 

			if (item.value_length != UNDEFINED) {
				input.ignore(item.value_length);
			}

			else {
				DicomElement e;
				mReadElementTag(e, input);
				while (e.tag != Tag("ItemDelimiter")) {
					input.seekg(-4, std::ios::cur);
					mReadNextElement(e, input);
					mReadElementTag(e, input);
				}

				input.ignore(4);

			}

			mReadElementTag(item, input);
			input.read(reinterpret_cast<char*> (&item.value_length), 
				sizeof(item.value_length));

		}
		rElement.value_length = input.tellg() - start_pos;
	}
}

vector<DicomElement> 
DicomFile::mReadNextElementAll() 
{
	DicomElement e;
	vector<DicomElement> elementVector;

	mReadElementTag(e, mInputFile);
	mReadValueRepresentation(e, mInputFile);
	mReadValueLength(e, mInputFile);

	if (e.tag == Tag("PixelData")) {
		mInputFile.ignore(e.value_length);
		elementVector.push_back(e);

	} else if (e.value_length != UNDEFINED) {
		
		DicomElement dict_entry;
		if (e.tag.group & 1)  // private tags
			dict_entry = mSpecial->Lookup(e.tag);
		else
			dict_entry = mDicom->Lookup(e.tag);
			
		if ((mVrSyntax != EXPLICIT_VR) || (dict_entry.vr != "UN"))
			e.vr = dict_entry.vr;

		e.description = dict_entry.description;

		mReadValue(e, e.value, mInputFile);
		elementVector.push_back(e);

	} else {  // Undefined length item
		elementVector.push_back(e);
		
		DicomElement item;
		mReadElementTag(item, mInputFile);
		mInputFile.read(reinterpret_cast<char*> (&item.value_length), 
			sizeof(item.value_length));


		while (item.tag != Tag("SqDelimiter")) {

			if (item.value_length != UNDEFINED) {

				mReadValue(item, item.value, mInputFile);
				elementVector.push_back(item);

			} else {
				DicomElement ee;
				mReadElementTag(ee, mInputFile);
				while (ee.tag != Tag("ItemDelimiter")) {
					mInputFile.seekg(-4, std::ios::cur);
					vector<DicomElement> nv = mReadNextElementAll();
					elementVector.insert(elementVector.end(), nv.begin(), nv.end());
					mReadElementTag(ee, mInputFile);
				}

				mInputFile.ignore(4);

			}

			mReadElementTag(item, mInputFile);
			mInputFile.read(reinterpret_cast<char*> (&item.value_length), 
				sizeof(item.value_length));

		}
	}

	return elementVector;

}

int
DicomFile::mFindElement(const DicomTag& tag, DicomElement& element,
						istream& input) 
{
	mReadNextElement(element, input);

	if (element.tag == tag){
		return 1;
	}

	if ((element.tag > tag) || !input.good() || element.tag == DicomTag(0,0)){
		input.clear();
		if (input == mInputFile) 
			input.seekg(mBeginning);
		else 
			input.seekg(0);
	} 	

	while (input.good()) {
		mReadNextElement(element, input);
		if (element.tag == tag){
			return 1;
		}

		if (element.tag > tag){
			return 0;
		} 
	}

	return 0;	
}

// specialization of readvalue for strings
int
DicomFile::mReadValue(DicomElement d, std::string& sValue, istream& input)
{
	sValue.clear();
	if (!mInputFile.good()) 
		return 0;
	vector<string> v;

	if (ReadMap.count(d.vr))
		v = ReadMap[d.vr](input, d.value_length);
	else
		v = ReadMap["UN"](input, d.value_length);

	if (!v.empty()){
		sValue = v.front();
		vector<string>::iterator it = v.begin();
		while(++it != v.end()) {
			sValue += "/";
			sValue += *it;
		}
	}

	//  remove leading/trailing spaces
	sValue.erase(0, sValue.find_first_not_of(' '));
	sValue.resize(sValue.find_last_not_of(' ') + 1);

	return 1;
}



//todo: return 0 when not found!
int
DicomFile::FindInSequence(const string& seq, const char* str, 
						  std::vector<string>& v)
{ 
	DicomElement de = DicomElement(Tag(str));
	return FindInSequence(seq, de, v);
}

int 
DicomFile::FindInSequence(const string& seq, DicomElement& d, 
						  std::vector<string>& v)
{
	int found = 0;
	stringstream ss(seq);

	DicomElement item;

	mReadElementTag(item, ss);
	ss.read(reinterpret_cast<char*> (&item.value_length), sizeof(item.value_length));

	int counter = 0;
	while (ss.good()) { 
		++counter;
		wxLogStatus(_T("%d"), counter);

		if (item.value_length != UNDEFINED) {
			istream::pos_type pos = ss.tellg();
			istream::pos_type end_pos = pos + static_cast<std::streamoff>(item.value_length);
			DicomElement e;
			while (ss.good() && (pos < end_pos)) {
				mReadNextElement(e, ss);
				if (ss.good() && (e.tag == d.tag)) {
					string str;
					ss.seekg(- static_cast<std::streamoff>(e.value_length), 
						std::ios::cur);

					mReadValue(e, str, ss);
					v.push_back(str);
					found = 1;
				}
				pos = ss.tellg();
			}
		}

		else {
			DicomElement e;
			mReadElementTag(e, ss);
			while (e.tag != Tag("ItemDelimiter")) {
				ss.seekg(-4, std::ios::cur);
				mReadNextElement(e, ss);
				if (e.tag == d.tag) {
					string str;
					ss.seekg(- static_cast<std::streamoff>(e.value_length), 
						std::ios::cur);

					mReadValue(e, str, ss);
					v.push_back(str);
					found = 1;
				}
				mReadElementTag(e, ss);
			}

			ss.ignore(4);

		}

		mReadElementTag(item, ss);
		ss.read(reinterpret_cast<char*> (&item.value_length), 
			sizeof(item.value_length));

	}

	return found;
}


std::iostream::pos_type
DicomFile::GetPixelDataStart()
{
	if (!foundPixelDataStart) {
		if (!mInputFile.is_open()) return 0;
		mInputFile.clear();

		// this is more correct, but it will be slower...
		mInputFile.seekg(mBeginning);

		DicomElement element;
		mReadNextElement(element, mInputFile);
		while ((mInputFile.good()) && (element.tag != Tag("PixelData"))) {
			mReadNextElement(element, mInputFile);
		}
	
		std::streamoff offset = static_cast<std::streamoff>(element.value_length);
		// assume if input file not good, data is at end
		if (!mInputFile.good()) {
			mInputFile.clear();
			mInputFile.seekg(-offset, std::ios::end);
		}
		else mInputFile.seekg(-offset, std::ios::cur);

		pixelDataStart = mInputFile.tellg();
		foundPixelDataStart = true;
	}
	return pixelDataStart;
}

wxUint16
DicomFile::ByteSwap(wxUint16 value_to_swap) 
{
     return wxUINT16_SWAP_ON_BE(value_to_swap);
}
wxInt16
DicomFile::ByteSwap(wxInt16 value_to_swap) 
{
	return wxINT16_SWAP_ON_BE(value_to_swap);
}
wxUint32
DicomFile::ByteSwap(wxUint32 value_to_swap) 
{
	return wxUINT32_SWAP_ON_BE(value_to_swap);
}
wxInt32
DicomFile::ByteSwap(wxInt32 value_to_swap) 
{
	return wxINT32_SWAP_ON_BE(value_to_swap);
}

