# -*- coding: utf-8 -*-

# Copyright (c) 2009, 2010 Jack Kaliko <efrim@azylum.org> {{{
# Copyright (c) 2009 J. Alexander Treuman (Tag collapse method)
# Copyright (c) 2008 Rick van Hattem
#
#  This file is part of MPD_sima
#
#  MPD_sima is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  MPD_sima is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with MPD_sima.  If not, see <http://www.gnu.org/licenses/>.
#
#
#  }}}

import time


class Track(object):
    """
    Track object.
    Instanciate with mpd replies.
    """

    def __init__(self, file=None, time=0, **kwargs):
        self.title = self.artist = self.album = self.albumartist = ''
        self.pos = None
        self.empty = False
        self._file = file
        if not kwargs:
            self.empty = True
        self.time = time
        self.__dict__.update(**kwargs)
        self.tags_to_collapse = list(['artist', 'album', 'title', 'date',
            'genre', 'albumartist'])
        #  have tags been collapsed?
        self.collapse_tags_bool = False
        self.collapsed_tags = list()
        # Needed for multipule tags which returns a list instead of a string
        self.collapse_tags()

    def collapse_tags(self):
        """
        Necessary to deal with tags defined multiple times.
        These entries are set as lists instead of strings.
        """
        for tag, value in self.__dict__.iteritems():
            if tag not in self.tags_to_collapse:
                continue
            if isinstance(value, list):
                #self.__dict__[tag] = ", ".join(set(value))
                self.collapse_tags_bool = True
                self.collapsed_tags.append(tag)
                self.__dict__.update({tag: ', '.join(set(value))})

    def get_fuzzy(self, what):
        """
        Get fuzzy artist name|title
        """
        if what not in self.__dict__.keys():
            raise KeyError
        return True

    def get_artist(self):
        """return lowercase UNICODE name of artist"""
        return unicode(self.artist, 'utf-8')

    def get_albumartist(self):
        """return lowercase UNICODE name of artist"""
        return unicode(self.albumartist, 'utf-8')

    def get_title(self):
        """return lowercase UNICODE title of song"""
        return unicode(self.title, 'utf-8')

    def get_album(self):
        """return lowercase UNICODE album of song"""
        return unicode(self.album, 'utf-8')

    def get_filename(self):
        """return filename"""
        if not self.file:
            return None
        return unicode(self.file, 'utf-8')

    def get_position(self):
        """return position of track in the playlist"""
        return int(self.pos)

    def __repr__(self):
        return '<%s: %s - %s - %s (%s)>' % (
            self.__class__.__name__,
            self.album,
            self.artist,
            self.title,
            self.duration,
        )

    def __int__(self):
        return self.time

    def __add__(self, other):
        return Track(time=self.time + other.time)

    def __sub__(self, other):
        return Track(time=self.time - other.time)

    def __hash__(self):
        if self.file:
            return hash(self.file)
        else:
            return id(self)

    def __eq__(self, other):
        return hash(self) == hash(other)

    def __ne__(self, other):
        return hash(self) != hash(other)

    def __nonzero__(self):
        if self.empty:
            return False
        return True

    @property
    def file(self):
        """file is an immutable attribute that's used for the hash method"""
        return self._file

    def get_time(self):
        """get time property"""
        return self._time

    def set_time(self, value):
        """set time property"""
        self._time = int(value)

    time = property(get_time, set_time, doc='song duration in seconds')

    @property
    def duration(self):
        """Compute fancy duration"""
        temps = time.gmtime(int(self.time))
        if temps.tm_hour:
            fmt = '%H:%M:%S'
        else:
            fmt = '%M:%S'
        return time.strftime(fmt, temps)


def main():
    pass

# Script starts here
if __name__ == '__main__':
    main()

# VIM MODLINE
# vim: ai ts=4 sw=4 sts=4 expandtab
