//MountManager - the program for easy mounting of storage devices in Linux
//Copyright (C) 2007-2008 Tikhonov Sergey
//
//This file is part of MountManager Gui
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#include <QtGui/QVBoxLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QPushButton>
#include <QtGui/QStackedWidget>
#include <QtGui/QScrollArea>
#include <QtGui/QCheckBox>
#include <QtGui/QAction>
#include <QtGui/QComboBox>
#include <QtGui/QLabel>
#include <QtGui/QToolButton>
#include <QtGui/QLineEdit>
#include <QtGui/QTextBrowser>
#include <QtCore/QSettings>
#include <QtCore/QTimer>
#include "const.h"
#include "treewidget.h"
#include "../core/diskcore.h"
#include "../core/diskdevice.h"
#include "centralwidget.h"
#include "messagebar.h"
#include "volumewidget.h"
#include "swapwidget.h"
#include "diskwidget.h"
#include "infobutton.h"
#include "widgetformount.h"


CentralWidget::CentralWidget(DiskCore *diskCore,PopupWindow *popup) {
	core = diskCore;
	popupWindow = popup;
	mountWidget = new WidgetForMount(core);
	mountWidget->hide();

	timer = new QTimer;
	timer->setInterval(10);
	
	// Left widget
	leftWidget = new QWidget;

	treeWidget = new TreeWidget(true);
	treeWidget->setColumnCount(3);
	treeWidget->setHeaderLabels(QStringList() << tr("Device") << tr("Size") << tr("File system"));
	treeWidget->setContextMenuIcon(QIcon(PROGRAM_ICON_PATH));
	treeWidget->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

	mountAction = new QAction(treeWidget);
	mountAction->setText(tr("Mount"));
	mountAction->setIcon(QIcon(ICONS_PATH"mount.png"));
	
	unmountAction = new QAction(treeWidget);
	unmountAction->setText(tr("Unmount"));
	unmountAction->setIcon(QIcon(ICONS_PATH"umount.png"));
	
	setAutomountAction = new QAction(treeWidget);
	setAutomountAction->setText(tr("Set auto-mounting"));
	setAutomountAction->setIcon(QIcon(ICONS_PATH"set_auto_mount.png"));

	treeWidget->addContextMenuAction(mountAction);
	treeWidget->addContextMenuAction(unmountAction);
	treeWidget->addContextMenuSeparator();
	treeWidget->addContextMenuAction(setAutomountAction);
	
	settingsWidget = new QWidget;
	settingsWidget->setMaximumHeight(0);
	rollToUp = false;
	settingsWidget->show();

	settingsWidgetLayout = new QVBoxLayout;
	settingsWidgetLayout->addWidget(new QLabel("<b>" + tr("Settings") + "</b>"));

	showTrayMessages = new QCheckBox(tr("Show tray messages"));
	settingsWidgetLayout->addWidget(showTrayMessages);
	
	format = new QComboBox;
	format->addItem(tr("As UUID"),"uuid");
	format->addItem(tr("As label"),"label");
	format->addItem(tr("As name"),"name");

	formatInfoButton = new InfoButton(popupWindow);
	formatInfoButton->setPopupText(tr("There are some formats of writing in configuration file. \"Name\" format is the worst variant, for devices names can be changed,  \"label\" format is also not very good, because not every device has its label, but \"UUID\" format is the best, because every device has a unique UUID."));
	formatInfoButton->setPopupHeaderText(tr("Writing format"));

	QHBoxLayout *formatLayout = new QHBoxLayout;
	formatLayout->addWidget(formatInfoButton);
	formatLayout->addWidget(new QLabel(tr("Format") + ":"));
	formatLayout->addWidget(format);
	formatLayout->addStretch();

	devicesFilter = new QLineEdit;
	connect(devicesFilter,SIGNAL(textChanged(const QString&)),this,SLOT(setFilter(const QString&)));
	
	filterButton = new QToolButton;
	filterButton->setIcon(QIcon(ICONS_PATH"clear_left.png"));
	filterButton->setAutoRaise(true);
	filterButton->setEnabled(false);
	connect(filterButton,SIGNAL(clicked()),devicesFilter,SLOT(clear()));

	QHBoxLayout *filterLayout = new QHBoxLayout;
	filterLayout->addWidget(devicesFilter);
	filterLayout->addWidget(filterButton);

	settingsWidgetLayout->addLayout(formatLayout);
	
	settingsWidget->setLayout(settingsWidgetLayout);
	
	settingsButton = new QPushButton(tr("Settings"));
	settingsButton->setToolTip(tr("Show advanced settings"));
	
	settingsButtonLayout = new QHBoxLayout;
	settingsButtonLayout->addWidget(settingsButton);
	settingsButtonLayout->addStretch();
	
	leftWidgetLayout = new QVBoxLayout;
	leftWidgetLayout->addWidget(treeWidget);
	leftWidgetLayout->addLayout(filterLayout);
	leftWidgetLayout->addWidget(settingsWidget);
	leftWidgetLayout->addLayout(settingsButtonLayout);
		
	leftWidget->setLayout(leftWidgetLayout);

	// Right widget
	rightWidget = new QWidget;

	rightSplitter = new QSplitter;
	
	stackedWidget = new QStackedWidget;
	messageBarsArea = new QScrollArea;
	
	donationBrowser = new QTextBrowser;
	donationBrowser->setSource(QUrl(IMAGES_PATH"support_page.html"));
	donationBrowser->setMaximumHeight(50);
	supportLabel = new QLabel;
	supportLabel->setPixmap(QPixmap(IMAGES_PATH"support_image.png"));
	
	hideButton = new QPushButton(tr("Hide"));
	connect(hideButton,SIGNAL(clicked()),donationBrowser,SLOT(hide()));
	
	supportButton = new QPushButton(tr("Support Page"));
	connect(supportButton,SIGNAL(clicked()),this,SIGNAL(goToSupportPage()));
	
	QHBoxLayout *browserLayout = new QHBoxLayout;
	browserLayout->addWidget(supportLabel);
	browserLayout->addStretch();
	browserLayout->addWidget(supportButton);
	browserLayout->addWidget(hideButton);
	donationBrowser->setLayout(browserLayout);

	rightSplitter->addWidget(stackedWidget);
	rightSplitter->addWidget(messageBarsArea);
	rightSplitter->setStretchFactor(0,1);
	rightSplitter->setOrientation(Qt::Vertical);
	rightSplitter->setCollapsible(0,false);
	rightSplitter->setCollapsible(1,false);
		
	rightWidgetLayout = new QVBoxLayout;
	rightWidgetLayout->addWidget(mountWidget);
	rightWidgetLayout->addWidget(rightSplitter,1);
	rightWidgetLayout->addWidget(donationBrowser);

	rightWidget->setLayout(rightWidgetLayout);

	areaWidget = new QWidget;
	areaWidgetLayout = new QVBoxLayout;
	areaWidget->setLayout(areaWidgetLayout);
	messageBarsArea->setWidget(areaWidget);
	messageBarsArea->setWidgetResizable(true);
	
	addWidget(leftWidget);
	addWidget(rightWidget);
	
	setStretchFactor(1,1);
	setCollapsible(0,false);
	setCollapsible(1,false);

	for (int i = core->devices().count()-1; i >= 0; i--)
		addDevice(core->devices()[i]);
	
	treeWidget->expandAll();
	treeWidget->resizeColumnToContents(0);
	treeWidget->resizeColumnToContents(1);
	treeWidget->resizeColumnToContents(2);
	checkMessageBars();

	connect(timer,SIGNAL(timeout()),this,SLOT(updateSettingsWidgetSize()));
	connect(settingsButton,SIGNAL(clicked()),this,SLOT(showOrHideSettings()));
	connect(treeWidget,SIGNAL(contextMenuEventSignal()),this,SLOT(showMenu()));
	connect(treeWidget,SIGNAL(currentItemChanged(QTreeWidgetItem*,QTreeWidgetItem*)),this,SLOT(itemChanged(QTreeWidgetItem*,QTreeWidgetItem*)));
	connect(core,SIGNAL(deviceAdded(DiskDevice *)),this,SLOT(deviceAdded(DiskDevice *)));
	connect(core,SIGNAL(deviceRemoved(DiskDevice *)),this,SLOT(removeDevice(DiskDevice *)));
	connect(core,SIGNAL(errorAppeared(const QString &)),this,SLOT(showError(const QString &)));
	connect(mountAction,SIGNAL(triggered()),this,SLOT(mount()));
	connect(unmountAction,SIGNAL(triggered()),this,SLOT(unmount()));
	connect(setAutomountAction,SIGNAL(triggered()),this,SLOT(setUsbDeviceAutomount()));
}

CentralWidget::~CentralWidget() {
	delete timer;
	
	delete mountAction;
	delete unmountAction;
	delete setAutomountAction;
	
	delete devicesFilter;
	delete filterButton;
	delete treeWidget;
	delete showTrayMessages;
	delete format;
	delete formatInfoButton;
	delete settingsWidgetLayout;
	delete settingsWidget;
	delete settingsButton;
	delete settingsButtonLayout;
	delete leftWidgetLayout;
	delete leftWidget;

	delete hideButton;
	delete supportButton;
	delete supportLabel;
	delete donationBrowser;
	
	delete stackedWidget;
	delete messageBarsArea;
	delete mountWidget;
	delete rightSplitter;
	delete rightWidgetLayout;
	delete rightWidget;
}

QList<QAction *> CentralWidget::actions() const {
	return QList<QAction *>() << mountAction << unmountAction << setAutomountAction;
}

void CentralWidget::addDevice(DiskDevice *addedDevice) {
	QTreeWidgetItem *parentItem = parentItemForDevice(addedDevice);
	DiskDevice *parentDevice = parentForDevice(addedDevice);
	QTreeWidgetItem *addedDeviceItem = 0;
	VolumeWidget *volumeWidget = 0;
	DiskWidget *diskWidget = 0;
	SwapWidget *swapWidget = 0;
	if (addedDevice->isCdOrFloppy()) {
		/*
			Cd and floppy devices have another behaviour as other devices.
			When cd or floppy disk is inserted, then child item is shown
			and the child item has similar widget as from physical disks.
		*/
		if (addedDevice->isDisk()) {
			addedDeviceItem = new QTreeWidgetItem(treeWidget);
			if (addedDevice->isSupportDvd())
				addedDeviceItem->setText(2,tr("Changable"));
			else if (addedDevice->type() == "cdrom")
				addedDeviceItem->setText(2,"iso9660");
			else
				addedDeviceItem->setText(2,addedDevice->fileSystem());
			
			volumeWidget = new VolumeWidget(core,addedDevice,popupWindow);
			stackedWidget->addWidget(volumeWidget);
		} else {
			parentDevice->setCdOrFloppyChild(addedDevice);
			parentItem->setText(1,addedDevice->size(DiskDevice::Truncation));
			parentItem->setText(2,addedDevice->fileSystem());
			return;
		}
	} else if (addedDevice->isDisk()) {
		addedDeviceItem = new QTreeWidgetItem(treeWidget);
		addedDeviceItem->setText(2,tr("Disk"));

		diskWidget = new DiskWidget(addedDevice);
		stackedWidget->addWidget(diskWidget);
	} else {
		addedDeviceItem = new QTreeWidgetItem(parentItem);
		// Maybe this device is extended and has unknown file system
		if (addedDevice->fileSystem().isEmpty()) {
			delete addedDeviceItem;
			return;
		}
		addedDeviceItem->setText(2,addedDevice->fileSystem());

		if (addedDevice->fileSystem() == "swap") {
			swapWidget = new SwapWidget(addedDevice);
			stackedWidget->addWidget(swapWidget);
		} else {
			volumeWidget = new VolumeWidget(core,addedDevice,popupWindow);
			stackedWidget->addWidget(volumeWidget);
		}
		Device *parent = deviceByItem(parentItem);
		if (parent && parent->diskWidget)
			parent->diskWidget->addVolume(addedDevice);
	}

	if (parentItem) treeWidget->expandItem(parentItem);

	if (addedDeviceItem) {
		if (addedDevice->fileSystem().isEmpty() && showTrayMessages->isChecked())
			emit(showTrayMessage(tr("Device added"),tr("New storage device with unknown file system was detected: %1").arg(addedDevice->blockName())));
		else if (showTrayMessages->isChecked())
			emit(showTrayMessage(tr("Device added"),tr("New storage device was detected: %1").arg(addedDevice->blockName())));
		if (addedDevice->type() == "cdrom")
			addedDeviceItem->setIcon(0,QIcon(ICONS_PATH"cdrom_disk.png"));
		else if (addedDevice->type() == "floppy")
			addedDeviceItem->setIcon(0,QIcon(ICONS_PATH"floppy_disk.png"));
		else if (addedDevice->type() == "memory_stick")
			addedDeviceItem->setIcon(0,QIcon(ICONS_PATH"memory_stick_disk.png"));
		else if (addedDevice->isRemovable())
			addedDeviceItem->setIcon(0,QIcon(ICONS_PATH"usb_disk.png"));
		else if (addedDevice->isDisk())
			addedDeviceItem->setIcon(0,QIcon(ICONS_PATH"physical_disk.png"));
		else
			addedDeviceItem->setIcon(0,QIcon(ICONS_PATH"logical_disk.png"));

		if (volumeWidget)
			connect(volumeWidget,SIGNAL(optionsChanged()),this,SLOT(updateCurrentFstabContent()));

		QString shortBlockName = addedDevice->blockName();
		shortBlockName.remove("/dev/");
		addedDeviceItem->setText(0,shortBlockName);
		addedDeviceItem->setText(1,addedDevice->size(DiskDevice::Truncation));

		Device *device = new Device;
		device->treeWidgetItem = addedDeviceItem;
		device->diskDevice = addedDevice;
		device->volumeWidget = volumeWidget;
		device->diskWidget = diskWidget;
		device->swapWidget = swapWidget;
		device->index = devices.count();
		devices.append(device);
	}
}

void CentralWidget::deviceAdded(DiskDevice *addedDevice) {
	if (!addedDevice->isDisk()) {
		MessageBar *messageBar = new MessageBar;
		areaWidgetLayout->insertWidget(0,messageBar);
		messageBar->setDevice(addedDevice);

		connect(messageBar,SIGNAL(activated()),this,SLOT(checkMessageBars()));
		messageBar->setMessage(MessageBar::Information,tr("New storage device was detected: %1").arg(addedDevice->blockName()));
		messageBar->addAction(MessageBar::Mount);
		messageBar->addAction(MessageBar::UnMount);

		connect(messageBar,SIGNAL(mountSignal(DiskDevice *)),this,SLOT(mount(DiskDevice *)));
		connect(messageBar,SIGNAL(unmountSignal(DiskDevice *)),this,SLOT(unmount(DiskDevice *)));
		messageBars.append(messageBar);
		checkMessageBars();
	}
	addDevice(addedDevice);
	emit (deviceWasAdded(addedDevice));
}

void CentralWidget::removeDevice(DiskDevice *removedDevice) {
	if (removedDevice->isCdOrFloppy()) {
		removedDevice->parent()->removeCdOrFloppyChild();
		QTreeWidgetItem *parentItem = parentItemForDevice(removedDevice);
		DiskDevice *parentDevice = removedDevice->parent();
		parentItem->setText(1,parentDevice->size(DiskDevice::Truncation));
		parentItem->setText(2,parentDevice->fileSystem());
		if (parentDevice->isSupportDvd())
			parentItem->setText(2,tr("Changable"));
		else if (parentDevice->type() == "cdrom")
			parentItem->setText(2,"iso9660");
		else
			parentItem->setText(2,parentDevice->fileSystem());
	}
	if (showTrayMessages->isChecked())
		emit(showTrayMessage(tr("Device removed"),tr("Storage device was removed: %1").arg(removedDevice->blockName())));
	foreach (Device *device,devices) {
		if (device->diskDevice == removedDevice) {
			delete device->treeWidgetItem;
			// Hide all messages bars which have information about this device
			foreach (MessageBar *messageBar,messageBars)
				if (messageBar->device() == removedDevice)
					messageBar->hide();
			devices.removeAll(device);
			emit (deviceWasRemoved(device->diskDevice));
			return;
		}
	}
}

QTreeWidgetItem *CentralWidget::parentItemForDevice(DiskDevice *childDevice) {
	foreach (Device *device,devices)
		if (device->diskDevice == childDevice->parent())
			return device->treeWidgetItem;
	return 0;
}

DiskDevice *CentralWidget::parentForDevice(DiskDevice *childDevice) {
	foreach (Device *device,devices)
		if (device->diskDevice == childDevice->parent())
			return device->diskDevice;
	return 0;
}

QList<Device *> CentralWidget::childsForDevice(DiskDevice *parentDevice) {
	QList<Device *> childDevices;
	foreach (Device *device,devices)
		if (device->diskDevice->parent() == parentDevice)
			childDevices.append(device);
	return childDevices;
}

DiskDevice *CentralWidget::currentDiskDevice() {
	foreach (Device *device,devices)
		if (device->treeWidgetItem == treeWidget->currentItem())
			return device->diskDevice;
	return 0;
}

Device *CentralWidget::deviceByItem(QTreeWidgetItem *item) {
	foreach (Device *device,devices)
		if (device->treeWidgetItem == item)
			return device;
	return 0;
}

Device *CentralWidget::currentDevice() {
	foreach (Device *device,devices)
		if (device->diskDevice == currentDiskDevice())
			return device;
	return 0;
}

void CentralWidget::showInformation(const QString &informationContent) {
	showMessage(MessageBar::Information,informationContent);
}

void CentralWidget::showWarning(const QString &warningContent) {
	showMessage(MessageBar::Warning,warningContent);
}

void CentralWidget::showError(const QString &errorContent) {
	showMessage(MessageBar::Error,errorContent);
}

MessageBar* CentralWidget::showMessage(int messageType,const QString &messageContent) {
	MessageBar *messageBar = new MessageBar;
	messageBar->setMessage(messageType,messageContent);
	areaWidgetLayout->insertWidget(0,messageBar);
	messageBars.append(messageBar);
	connect(messageBar,SIGNAL(activated()),this,SLOT(checkMessageBars()));
	checkMessageBars();
	
	return messageBar;
}

void CentralWidget::showMessageWithActions(const QString& message,const QList<QAction *>& actions) {
	MessageBar *messageBar = showMessage(0,message);
	foreach (QAction *action,actions)
		messageBar->addAction(action);
}

void CentralWidget::checkMessageBars() {
	bool showMessageBarsArea = false;
	foreach (MessageBar *bar,messageBars)
		if (!bar->isHidden()) {
			showMessageBarsArea = true;
			break;
		}
	messageBarsArea->setVisible(showMessageBarsArea);
}

void CentralWidget::saveSettings() {
	QSettings settings(ORGANIZATION,PROGRAM_NAME);
	settings.setValue("CentralWidget/State",saveState());
	settings.setValue("CentralWidget/ShowTrayMessages",showTrayMessages->isChecked());
	settings.setValue("CentralWidget/Format",format->currentIndex());
	settings.setValue("CentralWidget/RightSplitterState",rightSplitter->saveState());
	settings.setValue("CentralWidget/ShowDonationPanel",donationBrowser->isVisible());
	emit (finished());
}

void CentralWidget::loadSettings() {
	QSettings settings(ORGANIZATION,PROGRAM_NAME);
	restoreState(settings.value("CentralWidget/State").toByteArray());
	showTrayMessages->setChecked(settings.value("CentralWidget/ShowTrayMessages",true).toBool());
	rightSplitter->restoreState(settings.value("CentralWidget/RightSplitterState").toByteArray());
	format->setCurrentIndex(settings.value("CentralWidget/Format").toInt());
	donationBrowser->setVisible(settings.value("CentralWidget/ShowDonationPanel",true).toBool());
}

void CentralWidget::showMenu() {
	if (!treeWidget->currentItem())
		return;
	DiskDevice *current = currentDiskDevice();
	treeWidget->setContextMenuHeader(current->blockName());
	setAutomountAction->setEnabled(current->isUsb());
	if (current->isCdOrFloppy() && !current->isCdOrFloppyInserted()) {
		mountAction->setEnabled(false);
		unmountAction->setEnabled(false);
		return;
	}
	if (current->isDisk() && !current->isCdOrFloppy()) {
		mountAction->setEnabled(false);
		unmountAction->setEnabled(false);
		return;
	}
	mountAction->setEnabled(!current->isMounted());
	unmountAction->setEnabled(current->isMounted());
}

void CentralWidget::itemChanged(QTreeWidgetItem* current,QTreeWidgetItem* previous) {
	if (current == 0)
		current = previous;
	if (current == 0)
		return;
	Device *currentDevice = deviceByItem(current);
	stackedWidget->setCurrentIndex(currentDevice->index);
	showMenu();
	emit (currentDeviceFsChanged(currentDevice->diskDevice->fileSystem()));
}

void CentralWidget::mount(DiskDevice *diskDevice) {
	if (diskDevice == 0) {
		Device *device = currentDevice();
		if (device->volumeWidget)
			mountWidget->setOptionsForMount(device->diskDevice->blockName(),device->volumeWidget->mountPoint(),
									device->diskDevice->fileSystem(),device->volumeWidget->options());
		else
			mountWidget->setOptionsForMount(device->diskDevice->blockName(),device->diskDevice->fstabMountPoint(),
									device->diskDevice->fileSystem(),device->diskDevice->fstabOptions());
	} else
		mountWidget->setOptionsForMount(diskDevice->blockName(),diskDevice->fstabMountPoint(),
								diskDevice->fileSystem(),diskDevice->fstabOptions());
	mountWidget->show();
}

void CentralWidget::unmount(DiskDevice *device) {
	if (device == 0)
		device = currentDiskDevice();
	core->unmount(device);
}

QString CentralWidget::currentFstabText(bool warn) {
	QString text;
	foreach (Device *device,devices) {
		if (device->diskDevice->isUsb())
			continue;
		if (device->volumeWidget)
			text += device->volumeWidget->fstabLine(warn,format->itemData(format->currentIndex()).toString()) + '\n';
		if (device->swapWidget != 0)
			text += device->swapWidget->fstabLine(format->itemData(format->currentIndex()).toString()) + '\n';
	}
	return text;
}

void CentralWidget::setUsbDeviceAutomount() {
	Device *device = currentDevice();
	DiskDevice *deviceToAutomount = 0;
	if (device->treeWidgetItem->childCount() > 0) {
		Device *childDevice = childsForDevice(device->diskDevice).first();
		deviceToAutomount = childDevice->diskDevice;
	} else
			deviceToAutomount = device->diskDevice;
	emit (setUsbDeviceAutomount(deviceToAutomount));
}

void CentralWidget::updateCurrentFstabContent() {
	currentFstabContent = currentFstabText(false);
	emit (currentFstabContentChanged(currentFstabContent));
}

void CentralWidget::emitSignalsForPlugins() {
	if (currentFstabContent.isEmpty())
		updateCurrentFstabContent();
	else
		emit (currentFstabContentChanged(currentFstabContent));
	if (currentDiskDevice())
		emit (currentDeviceFsChanged(currentDiskDevice()->fileSystem()));
}

void CentralWidget::setFilter(const QString& filter) {
	filterButton->setEnabled(!filter.isEmpty());
	foreach (Device *device,devices)
		if (device->treeWidgetItem && device->diskDevice->isDisk())
			device->treeWidgetItem->setHidden(true);
	foreach (QTreeWidgetItem *item, treeWidget->findItems(filter,Qt::MatchContains,0))
		item->setHidden(false);
	foreach (QTreeWidgetItem *item, treeWidget->findItems(filter,Qt::MatchContains,1))
		item->setHidden(false);
	foreach (QTreeWidgetItem *item, treeWidget->findItems(filter,Qt::MatchContains,2))
		item->setHidden(false);
}

QList<DiskDevice *> CentralWidget::usbDevices() const {
	QList<DiskDevice *> usb_devices;
	foreach (Device *device,devices)
		if (device->diskDevice->isUsb())
			usb_devices << device->diskDevice;
	return usb_devices;
}

QString CentralWidget::mountOptionsOfDevice(DiskDevice *diskDevice) const {
	foreach (Device *device,devices)
		if (device->diskDevice == diskDevice && device->volumeWidget != 0)
			return device->volumeWidget->options();
	return QString();
}

QString CentralWidget::mountPointOfDevice(DiskDevice *diskDevice) const {
	foreach (Device *device,devices)
		if (device->diskDevice == diskDevice && device->volumeWidget != 0)
			return device->volumeWidget->mountPoint();
	return QString();
}

void CentralWidget::showOrHideSettings() {
	rollToUp = !rollToUp;
	timer->start();
}

void CentralWidget::updateSettingsWidgetSize() {
	if (rollToUp) {
		if (settingsWidget->height() + 10 >= settingsWidget->sizeHint().height())
			settingsWidget->setMaximumHeight(settingsWidget->sizeHint().height());
		else
			settingsWidget->setMaximumHeight(settingsWidget->height() + 10);
		if (settingsWidget->height() != settingsWidget->sizeHint().height())
			timer->start();
		else
			timer->stop();
	} else {
		if (settingsWidget->height() - 10 <= 0)
			settingsWidget->setMaximumHeight(0);
		else
			settingsWidget->setMaximumHeight(settingsWidget->height() - 10);
		if (settingsWidget->height() != 0)
			timer->start();
		else
			timer->stop();
	}
	settingsWidget->resize(settingsWidget->width(),settingsWidget->sizeHint().height());
}
