# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.

from elisa.core.utils.i18n import install_translation
from elisa.plugins.poblesec import modal_popup

_ = install_translation('poblesec')

def error_message_treatment(main_controller, message, sender):
    """
    Uses C{main_controller} to inform the user of an error that occured for
    which C{message} has been sent over the bus by C{sender}.

    @param main_controller: poblesec's controller
    @type main_controller:  L{elisa.plugins.poblesec.main.PoblesecController}
    @param message: error message to display
    @type message: L{elisa.plugins.base.messages.error_message.ErrorMessage}
    @param sender: the sender of the error message
    @type sender: C{object}
    """
    from twisted.internet.error import DNSLookupError
    from elisa.plugins.poblesec.player_video import PlaybackError
    from elisa.plugins.poblesec.history import EnteringControllerError

    exception = message.exception
    details = str(exception)

    if isinstance(exception, EnteringControllerError):
        original_failure = exception.failure
        if original_failure.type is DNSLookupError:
            title = _("CONNECTION ERROR")
            subtitle = _("There is a problem with your Internet connection.")
        else:
            title = _("ERROR ENTERING A MENU")
            subtitle = _("It was impossible to enter the menu \'%s\'.") \
                         % exception.display_name
            details = exception.failure.getErrorMessage()

    elif isinstance(exception, PlaybackError):
        title = _("PLAYBACK ERROR")
        if exception.partly_playable:
            subtitle = _("Unfortunately Moovida can only partly play the " \
                         "selected media.")
        else:
            subtitle = _("Unfortunately Moovida cannot play the selected " \
                         "media.")
            if sender is main_controller.current_player:
                # Cannot even partly play the media, exit the player
                sender.exit()

        if exception.missing_decoders:
            details = _("Installing the following codecs should allow " \
                        "playing back the media:\n")
            # Localization of the names of the missing decoders is
            # automatically taken care of by gstreamer.
            details += _(', ').join(exception.missing_decoders) + _('.')
        else:
            details = exception.gstreamer_message

    else:
        title = _("UNKNOWN ERROR")
        subtitle = _("An error occured that Moovida was not able to" \
                     " identify.")

    if details:
        text = _("Moovida was able to extract the following technical" \
                 " details that may help troubleshoot the issue:\n\n%s") \
                 % details
    else:
        text = ""
    buttons = [(_("Close"), main_controller.hide_popup)]

    main_controller.enqueue_popup(title, subtitle, text, buttons, \
                                  modal_popup.ErrorPopup)
