/*
 *
 * mga_vid.c
 *
 * Copyright (C) 1999 Aaron Holtzman
 * 
 * Module skeleton based on gutted agpgart module by Jeff Hartmann 
 * <slicer@ionet.net>
 *
 * Matrox MGA G200/G400 YUV Video Interface module Version 0.1.0
 * 
 * BES == Back End Scaler
 * 
 * This software has been released under the terms of the GNU Public
 * license. See http://www.gnu.org/copyleft/gpl.html for details.
 */

// YUY2 support (see config.format) added by A'rpi/ESP-team
// double buffering added by A'rpi/ESP-team
// brightness/contrast introduced by eyck
// multiple card support by Attila Kinali <attila@kinali.ch>
// ported to the 2.6 series kernels by F.O. Tempel
//  thankfully using the ground work done by Ed Sweetman (for the devfs work)
//  and Gergely Nagy for pushing into the right direction with 
//  his patch for 2.6.0-test1

#define MODULENAME "mga_vid"

// we are only allowed to register 16 cards
// more might collide with other drivers
#define MGA_MAX_CARDS 16
#define MGA_MAX_CARDS_INIT_ARRAY {0}

#define MGA_VSYNC_POS 2

// Set this value, if autodetection fails! (video ram size in megabytes)
// #define MGA_MEMORY_SIZE 16

#define DEFAULT_MGA_VID_MAJOR 83

//use the 'major' parameter to override the default major number (83)
/* mknod /dev/mga_vid c 83 0 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kobject.h>
#include <linux/kobj_map.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/errno.h>

#include <linux/device.h>

#include <linux/pci.h>
#include <linux/ioport.h>
#include <linux/init.h>

#include <linux/spinlock.h>

#include "mga_vid.h"

#ifdef CONFIG_MTRR
#include <asm/mtrr.h>
#endif

#include <asm/uaccess.h>
#include <asm/system.h>
#include <asm/io.h>

#ifndef PCI_DEVICE_ID_MATROX_G200_PCI
#define PCI_DEVICE_ID_MATROX_G200_PCI 0x0520
#endif

#ifndef PCI_DEVICE_ID_MATROX_G200_AGP
#define PCI_DEVICE_ID_MATROX_G200_AGP 0x0521
#endif

#ifndef PCI_DEVICE_ID_MATROX_G400
#define PCI_DEVICE_ID_MATROX_G400 0x0525
#endif

#ifndef PCI_DEVICE_ID_MATROX_G550
#define PCI_DEVICE_ID_MATROX_G550 0x2527
#endif

#ifndef PCI_SUBSYSTEM_ID_MATROX_G400_DH_16MB
#define PCI_SUBSYSTEM_ID_MATROX_G400_DH_16MB 0x2159
#endif

#ifndef PCI_SUBSYSTEM_ID_MATROX_G400_16MB_SGRAM
#define PCI_SUBSYSTEM_ID_MATROX_G400_16MB_SGRAM 0x19d8
#endif

#ifndef PCI_SUBSYSTEM_ID_MATROX_G400_16MB_SDRAM
#define PCI_SUBSYSTEM_ID_MATROX_G400_16MB_SDRAM 0x0328
#endif

MODULE_AUTHOR("Aaron Holtzman <aholtzma@engr.uvic.ca>");
MODULE_LICENSE("GPL");

#define PARAM_BRIGHTNESS "brightness="
#define PARAM_CONTRAST "contrast="
#define PARAM_BLACKIE "blackie="

// set PARAM_BUFF_SIZE to just below 4k because some kernel versions
// store additional information in the memory page which leads to
// the allocation of an additional page if exactly 4k is used
#define PARAM_BUFF_SIZE 4000

#ifndef min
#define min(x,y) (((x)<(y))?(x):(y))
#endif

typedef struct bes_registers_s {
	//BES Control
	uint32_t besctl;
	//BES Global control
	uint32_t besglobctl;
	//Luma control (brightness and contrast)
	uint32_t beslumactl;
	//Line pitch
	uint32_t bespitch;

	//Buffer A-1 Chroma 3 plane org
	uint32_t besa1c3org;
	//Buffer A-1 Chroma org
	uint32_t besa1corg;
	//Buffer A-1 Luma org
	uint32_t besa1org;

	//Buffer A-2 Chroma 3 plane org
	uint32_t besa2c3org;
	//Buffer A-2 Chroma org
	uint32_t besa2corg;
	//Buffer A-2 Luma org
	uint32_t besa2org;

	//Buffer B-1 Chroma 3 plane org
	uint32_t besb1c3org;
	//Buffer B-1 Chroma org
	uint32_t besb1corg;
	//Buffer B-1 Luma org
	uint32_t besb1org;

	//Buffer B-2 Chroma 3 plane org
	uint32_t besb2c3org;
	//Buffer B-2 Chroma org
	uint32_t besb2corg;
	//Buffer B-2 Luma org
	uint32_t besb2org;

	//BES Horizontal coord
	uint32_t beshcoord;
	//BES Horizontal inverse scaling [5.14]
	uint32_t beshiscal;
	//BES Horizontal source start [10.14] (for scaling)
	uint32_t beshsrcst;
	//BES Horizontal source ending [10.14] (for scaling) 
	uint32_t beshsrcend;
	//BES Horizontal source last 
	uint32_t beshsrclst;

	//BES Vertical coord
	uint32_t besvcoord;
	//BES Vertical inverse scaling [5.14]
	uint32_t besviscal;
	//BES Field 1 vertical source last position
	uint32_t besv1srclst;
	//BES Field 1 weight start
	uint32_t besv1wght;
	//BES Field 2 vertical source last position
	uint32_t besv2srclst;
	//BES Field 2 weight start
	uint32_t besv2wght;

	//configurable stuff
	int blackie;

} bes_registers_t;

//All register offsets are converted to word aligned offsets (32 bit)
//because we want all our register accesses to be 32 bits
#define VCOUNT      0x1e20

#define PALWTADD      0x3c00	// Index register for X_DATAREG port
#define X_DATAREG     0x3c0a

#define XMULCTRL      0x19
#define BPP_8         0x00
#define BPP_15        0x01
#define BPP_16        0x02
#define BPP_24        0x03
#define BPP_32_DIR    0x04
#define BPP_32_PAL    0x07

#define XCOLMSK       0x40
#define X_COLKEY      0x42
#define XKEYOPMODE    0x51
#define XCOLMSK0RED   0x52
#define XCOLMSK0GREEN 0x53
#define XCOLMSK0BLUE  0x54
#define XCOLKEY0RED   0x55
#define XCOLKEY0GREEN 0x56
#define XCOLKEY0BLUE  0x57

// Backend Scaler registers
#define BESCTL      0x3d20
#define BESGLOBCTL  0x3dc0
#define BESLUMACTL  0x3d40
#define BESPITCH    0x3d24

#define BESA1C3ORG  0x3d60
#define BESA1CORG   0x3d10
#define BESA1ORG    0x3d00

#define BESA2C3ORG  0x3d64
#define BESA2CORG   0x3d14
#define BESA2ORG    0x3d04

#define BESB1C3ORG  0x3d68
#define BESB1CORG   0x3d18
#define BESB1ORG    0x3d08

#define BESB2C3ORG  0x3d6C
#define BESB2CORG   0x3d1C
#define BESB2ORG    0x3d0C

#define BESHCOORD   0x3d28
#define BESHISCAL   0x3d30
#define BESHSRCEND  0x3d3C
#define BESHSRCLST  0x3d50
#define BESHSRCST   0x3d38
#define BESV1WGHT   0x3d48
#define BESV2WGHT   0x3d4c
#define BESV1SRCLST 0x3d54
#define BESV2SRCLST 0x3d58
#define BESVISCAL   0x3d34
#define BESVCOORD   0x3d2c
#define BESSTATUS   0x3dc4

#define CRTCX	    0x1fd4
#define CRTCD	    0x1fd5
#define	IEN	    0x1e1c
#define ICLEAR	    0x1e18
#define STATUS      0x1e14

struct class *mga_vid_class;
int mga_vid_major;

/* Global handles for cdev */
static struct cdev *mga_vid_cdev;
static dev_t mga_cdev_handle;

// card local config
typedef struct mga_card_s {
	uint8_t *param_buff;	// buffer for read()
	size_t param_buff_size;
	size_t param_buff_len;
	bes_registers_t regs;
	uint32_t vid_in_use;
	uint32_t card_type;
	uint32_t vid_src_ready;
	uint32_t vid_overlay_on;

	uint8_t *mmio_base;
	uint32_t mem_base;
	int src_base;		// YUV buffer position in video memory
	uint32_t ram_size;	// how many megabytes videoram we have
	uint32_t top_reserved;	// reserved space for console font (matroxfb + fastfont)

	int brightness;		// initial brightness
	int contrast;		// initial contrast

	struct pci_dev *pci_dev;

	mga_vid_config_t config;
	int configured;		// set to 1 when the card is configured over ioctl

	int colkey_saved;
	int colkey_on;
	unsigned char colkey_color[4];
	unsigned char colkey_mask[4];

	int next_frame;

	spinlock_t lock;
} mga_card_t;

static unsigned int mga_cards_num = 0;
static mga_card_t *mga_cards[MGA_MAX_CARDS] = MGA_MAX_CARDS_INIT_ARRAY;

// module parameters
static int major = DEFAULT_MGA_VID_MAJOR;
static int mga_ram_size[MGA_MAX_CARDS] = MGA_MAX_CARDS_INIT_ARRAY;
static int mga_brightness[MGA_MAX_CARDS] = MGA_MAX_CARDS_INIT_ARRAY;
static int mga_contrast[MGA_MAX_CARDS] = MGA_MAX_CARDS_INIT_ARRAY;
static int mga_top_reserved[MGA_MAX_CARDS] = MGA_MAX_CARDS_INIT_ARRAY;

module_param_array(mga_ram_size, int, NULL, 0);
module_param_array(mga_top_reserved, int, NULL, 0);
module_param_array(mga_brightness, int, NULL, 0);
module_param_array(mga_contrast, int, NULL, 0);
module_param(major, int, 0);

static void mga_vid_frame_sel(mga_card_t * card, int frame)
{
	//we don't need the vcount protection as we're only hitting
	//one register (and it doesn't seem to be double buffered)
	card->regs.besctl = (card->regs.besctl & ~0x07000000) + (frame << 25);
	writel(card->regs.besctl, card->mmio_base + BESCTL);

	writel(card->regs.besglobctl + (MGA_VSYNC_POS << 16),
	       card->mmio_base + BESGLOBCTL);
}

static void mga_vid_write_regs(mga_card_t * card, int restore)
{
	//Make sure internal registers don't get updated until we're done
	writel((readl(card->mmio_base + VCOUNT) - 1) << 16,
	       card->mmio_base + BESGLOBCTL);

	// color or coordinate keying

	if (restore && card->colkey_saved) {
		// restore it
		card->colkey_saved = 0;

#ifdef MP_DEBUG
		printk("mga_vid: Restoring colorkey (ON: %d  %02X:%02X:%02X)\n",
		       card->colkey_on, card->colkey_color[0],
		       card->colkey_color[1], card->colkey_color[2]);
#endif

		// Set color key registers:
		writeb(XKEYOPMODE, card->mmio_base + PALWTADD);
		writeb(card->colkey_on, card->mmio_base + X_DATAREG);

		writeb(XCOLKEY0RED, card->mmio_base + PALWTADD);
		writeb(card->colkey_color[0], card->mmio_base + X_DATAREG);
		writeb(XCOLKEY0GREEN, card->mmio_base + PALWTADD);
		writeb(card->colkey_color[1], card->mmio_base + X_DATAREG);
		writeb(XCOLKEY0BLUE, card->mmio_base + PALWTADD);
		writeb(card->colkey_color[2], card->mmio_base + X_DATAREG);
		writeb(X_COLKEY, card->mmio_base + PALWTADD);
		writeb(card->colkey_color[3], card->mmio_base + X_DATAREG);

		writeb(XCOLMSK0RED, card->mmio_base + PALWTADD);
		writeb(card->colkey_mask[0], card->mmio_base + X_DATAREG);
		writeb(XCOLMSK0GREEN, card->mmio_base + PALWTADD);
		writeb(card->colkey_mask[1], card->mmio_base + X_DATAREG);
		writeb(XCOLMSK0BLUE, card->mmio_base + PALWTADD);
		writeb(card->colkey_mask[2], card->mmio_base + X_DATAREG);
		writeb(XCOLMSK, card->mmio_base + PALWTADD);
		writeb(card->colkey_mask[3], card->mmio_base + X_DATAREG);

	} else if (!card->colkey_saved) {
		// save it
		card->colkey_saved = 1;
		// Get color key registers:
		writeb(XKEYOPMODE, card->mmio_base + PALWTADD);
		card->colkey_on = (unsigned char)readb(card->mmio_base + X_DATAREG) & 1;

		writeb(XCOLKEY0RED, card->mmio_base + PALWTADD);
		card->colkey_color[0] = (unsigned char)readb(card->mmio_base + X_DATAREG);
		writeb(XCOLKEY0GREEN, card->mmio_base + PALWTADD);
		card->colkey_color[1] = (unsigned char)readb(card->mmio_base + X_DATAREG);
		writeb(XCOLKEY0BLUE, card->mmio_base + PALWTADD);
		card->colkey_color[2] = (unsigned char)readb(card->mmio_base + X_DATAREG);
		writeb(X_COLKEY, card->mmio_base + PALWTADD);
		card->colkey_color[3] = (unsigned char)readb(card->mmio_base + X_DATAREG);

		writeb(XCOLMSK0RED, card->mmio_base + PALWTADD);
		card->colkey_mask[0] = (unsigned char)readb(card->mmio_base + X_DATAREG);
		writeb(XCOLMSK0GREEN, card->mmio_base + PALWTADD);
		card->colkey_mask[1] = (unsigned char)readb(card->mmio_base + X_DATAREG);
		writeb(XCOLMSK0BLUE, card->mmio_base + PALWTADD);
		card->colkey_mask[2] = (unsigned char)readb(card->mmio_base + X_DATAREG);
		writeb(XCOLMSK, card->mmio_base + PALWTADD);
		card->colkey_mask[3] = (unsigned char)readb(card->mmio_base + X_DATAREG);

#ifdef MP_DEBUG
		printk("mga_vid: Saved colorkey (ON: %d  %02X:%02X:%02X)\n",
		       card->colkey_on, card->colkey_color[0],
		       card->colkey_color[1], card->colkey_color[2]);
#endif

	}

	if (!restore) {
		writeb(XKEYOPMODE, card->mmio_base + PALWTADD);
		writeb(card->config.colkey_on, card->mmio_base + X_DATAREG);
		if (card->config.colkey_on) {
			uint32_t r = 0, g = 0, b = 0;

			writeb(XMULCTRL, card->mmio_base + PALWTADD);
			switch (readb(card->mmio_base + X_DATAREG)) {
			case BPP_8:
				/* Need to look up the color index, just using color 0 for now. */
				break;

			case BPP_15:
				r = card->config.colkey_red >> 3;
				g = card->config.colkey_green >> 3;
				b = card->config.colkey_blue >> 3;
				break;

			case BPP_16:
				r = card->config.colkey_red >> 3;
				g = card->config.colkey_green >> 2;
				b = card->config.colkey_blue >> 3;
				break;

			case BPP_24:
			case BPP_32_DIR:
			case BPP_32_PAL:
				r = card->config.colkey_red;
				g = card->config.colkey_green;
				b = card->config.colkey_blue;
				break;
			}

			// Disable color keying on alpha channel 
			writeb(XCOLMSK, card->mmio_base + PALWTADD);
			writeb(0x00, card->mmio_base + X_DATAREG);
			writeb(X_COLKEY, card->mmio_base + PALWTADD);
			writeb(0x00, card->mmio_base + X_DATAREG);

			// Set up color key registers
			writeb(XCOLKEY0RED, card->mmio_base + PALWTADD);
			writeb(r, card->mmio_base + X_DATAREG);
			writeb(XCOLKEY0GREEN, card->mmio_base + PALWTADD);
			writeb(g, card->mmio_base + X_DATAREG);
			writeb(XCOLKEY0BLUE, card->mmio_base + PALWTADD);
			writeb(b, card->mmio_base + X_DATAREG);

			// Set up color key mask registers
			writeb(XCOLMSK0RED, card->mmio_base + PALWTADD);
			writeb(0xff, card->mmio_base + X_DATAREG);
			writeb(XCOLMSK0GREEN, card->mmio_base + PALWTADD);
			writeb(0xff, card->mmio_base + X_DATAREG);
			writeb(XCOLMSK0BLUE, card->mmio_base + PALWTADD);
			writeb(0xff, card->mmio_base + X_DATAREG);
		}

	}
	// Backend Scaler
	writel(card->regs.besctl, card->mmio_base + BESCTL);
	if (card->card_type != MGA_G200)
		writel(card->regs.beslumactl, card->mmio_base + BESLUMACTL);
	writel(card->regs.bespitch, card->mmio_base + BESPITCH);

	writel(card->regs.besa1org, card->mmio_base + BESA1ORG);
	writel(card->regs.besa1corg, card->mmio_base + BESA1CORG);
	writel(card->regs.besa2org, card->mmio_base + BESA2ORG);
	writel(card->regs.besa2corg, card->mmio_base + BESA2CORG);
	writel(card->regs.besb1org, card->mmio_base + BESB1ORG);
	writel(card->regs.besb1corg, card->mmio_base + BESB1CORG);
	writel(card->regs.besb2org, card->mmio_base + BESB2ORG);
	writel(card->regs.besb2corg, card->mmio_base + BESB2CORG);
	if (card->card_type != MGA_G200) {
		writel(card->regs.besa1c3org, card->mmio_base + BESA1C3ORG);
		writel(card->regs.besa2c3org, card->mmio_base + BESA2C3ORG);
		writel(card->regs.besb1c3org, card->mmio_base + BESB1C3ORG);
		writel(card->regs.besb2c3org, card->mmio_base + BESB2C3ORG);
	}

	writel(card->regs.beshcoord, card->mmio_base + BESHCOORD);
	writel(card->regs.beshiscal, card->mmio_base + BESHISCAL);
	writel(card->regs.beshsrcst, card->mmio_base + BESHSRCST);
	writel(card->regs.beshsrcend, card->mmio_base + BESHSRCEND);
	writel(card->regs.beshsrclst, card->mmio_base + BESHSRCLST);

	writel(card->regs.besvcoord, card->mmio_base + BESVCOORD);
	writel(card->regs.besviscal, card->mmio_base + BESVISCAL);

	writel(card->regs.besv1srclst, card->mmio_base + BESV1SRCLST);
	writel(card->regs.besv1wght, card->mmio_base + BESV1WGHT);
	writel(card->regs.besv2srclst, card->mmio_base + BESV2SRCLST);
	writel(card->regs.besv2wght, card->mmio_base + BESV2WGHT);

	//update the registers somewhere between 1 and 2 frames from now.
	writel(card->regs.besglobctl + ((readl(card->mmio_base + VCOUNT) + 2) << 16),
	       card->mmio_base + BESGLOBCTL);

#ifdef MP_DEBUG
	printk(KERN_DEBUG "mga_vid: wrote BES registers\n");
	printk(KERN_DEBUG "mga_vid: BESCTL = 0x%08x\n",
	       readl(card->mmio_base + BESCTL));
	printk(KERN_DEBUG "mga_vid: BESGLOBCTL = 0x%08x\n",
	       readl(card->mmio_base + BESGLOBCTL));
	printk(KERN_DEBUG "mga_vid: BESSTATUS= 0x%08x\n",
	       readl(card->mmio_base + BESSTATUS));
#endif

}

static int mga_vid_set_config(mga_card_t * card)
{
	int x, y, sw, sh, dw, dh;
	int besleft, bestop, ifactor, ofsleft, ofstop, baseadrofs, weight, weights;
	mga_vid_config_t *config = &card->config;
	int frame_size = card->config.frame_size;

	x = config->x_org;
	y = config->y_org;
	sw = config->src_width;
	sh = config->src_height;
	dw = config->dest_width;
	dh = config->dest_height;

#ifdef MP_DEBUG
	printk(KERN_DEBUG "mga_vid: Setting up a %dx%d+%d+%d video window (src %dx%d) format %X\n",
	       dw, dh, x, y, sw, sh, config->format);
#endif

	if (sw < 4 || sh < 4 || dw < 4 || dh < 4) {
		printk(KERN_ERR "mga_vid: Invalid src/dest dimenstions\n");
		return -1;
	}
	//FIXME check that window is valid and inside desktop

	//Setup the BES registers for a three plane 4:2:0 video source 

	card->regs.besglobctl = 0;

	switch (config->format) {
	case MGA_VID_FORMAT_YV12:
	case MGA_VID_FORMAT_I420:
	case MGA_VID_FORMAT_IYUV:
		card->regs.besctl = 1	// BES enabled
		    + (0 << 6)	// even start polarity
		    + (1 << 10)	// x filtering enabled
		    + (1 << 11)	// y filtering enabled
		    + (1 << 16)	// chroma upsampling
		    + (1 << 17)	// 4:2:0 mode
		    + (1 << 18);	// dither enabled
		break;

	case MGA_VID_FORMAT_YUY2:
		card->regs.besctl = 1	// BES enabled
		    + (0 << 6)	// even start polarity
		    + (1 << 10)	// x filtering enabled
		    + (1 << 11)	// y filtering enabled
		    + (1 << 16)	// chroma upsampling
		    + (0 << 17)	// 4:2:2 mode
		    + (1 << 18);	// dither enabled

		card->regs.besglobctl = 0;	// YUY2 format selected
		break;

	case MGA_VID_FORMAT_UYVY:
		card->regs.besctl = 1	// BES enabled
		    + (0 << 6)	// even start polarity
		    + (1 << 10)	// x filtering enabled
		    + (1 << 11)	// y filtering enabled
		    + (1 << 16)	// chroma upsampling
		    + (0 << 17)	// 4:2:2 mode
		    + (1 << 18);	// dither enabled

		card->regs.besglobctl = 1 << 6;	// UYVY format selected
		break;

	default:
		printk(KERN_ERR "mga_vid: Unsupported pixel format: 0x%X\n", config->format);
		return -1;
	}

	// disable BES vertical filter if width is greater than 1024
	// and if we have a G550
	if (sw > 1024 && card->card_type == MGA_G550)
		card->regs.besctl &= ~(1 << 11);

	// setting black&white mode 
	card->regs.besctl |= (card->regs.blackie << 20);

	//Enable contrast and brightness control
	card->regs.besglobctl |= (1 << 5) + (1 << 7);

	// brightness (-128..127) && contrast (0..255)
	card->regs.beslumactl =
	    (card->brightness << 16) | ((card->contrast + 0x80) & 0xFFFF);

	//Setup destination window boundaries
	besleft = x > 0 ? x : 0;
	bestop = y > 0 ? y : 0;
	card->regs.beshcoord = (besleft << 16) + (x + dw - 1);
	card->regs.besvcoord = (bestop << 16) + (y + dh - 1);

	//Setup source dimensions
	card->regs.beshsrclst = (sw - 1) << 16;
	card->regs.bespitch = (sw + 31) & ~31;

	//Setup horizontal scaling
	ifactor = ((sw - 1) << 14) / (dw - 1);
	ofsleft = besleft - x;

	card->regs.beshiscal = ifactor << 2;
	card->regs.beshsrcst = (ofsleft * ifactor) << 2;
	card->regs.beshsrcend = card->regs.beshsrcst + (((dw - ofsleft - 1) * ifactor) << 2);

	//Setup vertical scaling
	ifactor = ((sh - 1) << 14) / (dh - 1);
	ofstop = bestop - y;

	card->regs.besviscal = ifactor << 2;

	baseadrofs = ((ofstop * card->regs.besviscal) >> 16) * card->regs.bespitch;
	//frame_size = ((sw + 31) & ~31) * sh + (((sw + 31) & ~31) * sh) / 2;
	card->regs.besa1org = (uint32_t) card->src_base + baseadrofs;
	card->regs.besa2org = (uint32_t) card->src_base + baseadrofs + 1 * frame_size;
	card->regs.besb1org = (uint32_t) card->src_base + baseadrofs + 2 * frame_size;
	card->regs.besb2org = (uint32_t) card->src_base + baseadrofs + 3 * frame_size;

	if (config->format == MGA_VID_FORMAT_YV12
	    || config->format == MGA_VID_FORMAT_IYUV
	    || config->format == MGA_VID_FORMAT_I420) {
		// planar YUV frames:
		if (card->card_type != MGA_G200)
			baseadrofs = (((ofstop * card->regs.besviscal) / 4) >> 16) * card->regs.bespitch;
		else
			baseadrofs = (((ofstop * card->regs.besviscal) / 2) >> 16) * card->regs.bespitch;

		if (config->format == MGA_VID_FORMAT_YV12 || (card->card_type == MGA_G200)) {
			card->regs.besa1corg = (uint32_t) card->src_base + baseadrofs + card->regs.bespitch * sh;
			card->regs.besa2corg = (uint32_t) card->src_base + baseadrofs + 1 * frame_size + card->regs.bespitch * sh;
			card->regs.besb1corg = (uint32_t) card->src_base + baseadrofs + 2 * frame_size + card->regs.bespitch * sh;
			card->regs.besb2corg = (uint32_t) card->src_base + baseadrofs + 3 * frame_size + card->regs.bespitch * sh;
			card->regs.besa1c3org = card->regs.besa1corg + ((card->regs.bespitch * sh) / 4);
			card->regs.besa2c3org = card->regs.besa2corg + ((card->regs.bespitch * sh) / 4);
			card->regs.besb1c3org = card->regs.besb1corg + ((card->regs.bespitch * sh) / 4);
			card->regs.besb2c3org = card->regs.besb2corg + ((card->regs.bespitch * sh) / 4);
		} else {
			card->regs.besa1c3org = (uint32_t) card->src_base + baseadrofs + card->regs.bespitch * sh;
			card->regs.besa2c3org = (uint32_t) card->src_base + baseadrofs + 1 * frame_size + card->regs.bespitch * sh;
			card->regs.besb1c3org = (uint32_t) card->src_base + baseadrofs + 2 * frame_size + card->regs.bespitch * sh;
			card->regs.besb2c3org = (uint32_t) card->src_base + baseadrofs + 3 * frame_size + card->regs.bespitch * sh;
			card->regs.besa1corg = card->regs.besa1c3org + ((card->regs.bespitch * sh) / 4);
			card->regs.besa2corg = card->regs.besa2c3org + ((card->regs.bespitch * sh) / 4);
			card->regs.besb1corg = card->regs.besb1c3org + ((card->regs.bespitch * sh) / 4);
			card->regs.besb2corg = card->regs.besb2c3org + ((card->regs.bespitch * sh) / 4);
		}

	}

	weight = ofstop * (card->regs.besviscal >> 2);
	weights = weight < 0 ? 1 : 0;
	card->regs.besv2wght = card->regs.besv1wght = (weights << 16) + ((weight & 0x3FFF) << 2);
	card->regs.besv2srclst = card->regs.besv1srclst = sh - 1 - (((ofstop * card->regs.besviscal) >> 16) & 0x03FF);

	mga_vid_write_regs(card, 0);
	return 0;
}

static long mga_vid_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int frame, result = 0;
	uint32_t tmp;
	mga_card_t *card = (mga_card_t *) file->private_data;

	spin_lock(&card->lock);

	switch (cmd) {
	case MGA_VID_GET_VERSION:
		tmp = MGA_VID_VERSION;
		if (copy_to_user((uint32_t *) arg, &tmp, sizeof(uint32_t))) {
			printk(KERN_ERR "mga_vid: failed copy %p to userspace %p\n", &tmp, (uint32_t *) arg);
			result = -EFAULT;
			goto ioctl_out;
		}
		break;

	case MGA_VID_CONFIG:
		if (copy_from_user (&card->config, (mga_vid_config_t *) arg, sizeof(mga_vid_config_t))) {
			printk(KERN_ERR "mga_vid: failed copy from userspace\n");
			result = -EFAULT;
			goto ioctl_out;
		}
		// return some information back to user space in case we hit an error
		card->config.card_type = card->card_type;
		card->config.ram_size = card->ram_size;
		card->config.capabilities = 0;
		if (copy_to_user ((mga_vid_config_t *) arg, &card->config, sizeof(mga_vid_config_t))) {
			printk(KERN_ERR "mga_vid: failed copy to userspace\n");
			result = -EFAULT;
			goto ioctl_out;
		}

		if (card->config.version != MGA_VID_VERSION) {
			printk(KERN_ERR "mga_vid: incompatible version! driver: %X  requested: %X\n", MGA_VID_VERSION, card->config.version);
			result = -EFAULT;
			goto ioctl_out;
		}

		if (card->config.frame_size == 0 || card->config.frame_size > 1920 * 1080 * 2) {
			printk(KERN_ERR "mga_vid: illegal frame_size: %d\n", card->config.frame_size);
			result = -EFAULT;
			goto ioctl_out;
		}

		if (card->config.num_frames < 1 || card->config.num_frames > 4) {
			printk(KERN_ERR "mga_vid: illegal num_frames: %d\n", card->config.num_frames);
			result = -EFAULT;
			goto ioctl_out;
		}

		if ((card->config.src_width > 1024 || card->config.src_height > 1024)
		    && (card->card_type != MGA_G550 && card->config.src_height > 1024)) {
			printk(KERN_ERR "mga_vid: overlay sizes bigger than 1024x1024 are not supported with this card type, rescale the picture in software\n");

			result = -EFAULT;
			goto ioctl_out;
		}

		card->src_base = (card->ram_size * 0x100000 - card->config.num_frames * card->config.frame_size - card->top_reserved);
		if (card->src_base < 0) {
			printk(KERN_ERR "mga_vid: not enough memory for frames!\n");
			result = -EFAULT;
			goto ioctl_out;
		}
		card->src_base &= (~0xFFFF);	// 64k boundary
#ifdef MP_DEBUG
		printk(KERN_DEBUG "mga YUV buffer base: 0x%X\n", card->src_base);
#endif
		if (copy_to_user ((mga_vid_config_t *) arg, &card->config, sizeof(mga_vid_config_t))) {
			printk(KERN_ERR "mga_vid: failed copy to userspace\n");
			result = -EFAULT;
			goto ioctl_out;
		}

		result = mga_vid_set_config(card);
		if (!result)
			card->configured = 1;
		goto ioctl_out;
		break;

	case MGA_VID_ON:
#ifdef MP_DEBUG
		printk(KERN_DEBUG "mga_vid: Video ON\n");
#endif
		card->vid_src_ready = 1;
		if (card->vid_overlay_on) {
			card->regs.besctl |= 1;
			mga_vid_write_regs(card, 0);
		}
		card->next_frame = 0;
		break;

	case MGA_VID_OFF:
#ifdef MP_DEBUG
		printk(KERN_DEBUG "mga_vid: Video OFF (ioctl)\n");
#endif
		card->vid_src_ready = 0;
		card->regs.besctl &= ~1;
		card->regs.besglobctl &= ~(1 << 6);	// UYVY format selected
		mga_vid_write_regs(card, 0);
		break;

	case MGA_VID_FSEL:
		if (copy_from_user(&frame, (int *)arg, sizeof(uint32_t))) {
			printk(KERN_ERR "mga_vid: FSEL failed copy from userspace\n");
			result = -EFAULT;
			goto ioctl_out;
		}

		mga_vid_frame_sel(card, frame);
		break;

	case MGA_VID_GET_LUMA:
		//tmp = card->regs.beslumactl;
		//tmp = (tmp&0xFFFF0000) | (((tmp&0xFFFF) - 0x80)&0xFFFF);
		tmp = (card->brightness << 16) | (card->contrast & 0xFFFF);

		if (copy_to_user((uint32_t *) arg, &tmp, sizeof(uint32_t))) {
			printk(KERN_ERR "mga_vid: failed copy %p to userspace %p\n", &tmp, (uint32_t *) arg);
			result = -EFAULT;
			goto ioctl_out;
		}
		break;

	case MGA_VID_SET_LUMA:
		tmp = arg;
		card->brightness = tmp >> 16;
		card->contrast = tmp & 0xFFFF;
		//card->regs.beslumactl = (tmp&0xFFFF0000) | ((tmp + 0x80)&0xFFFF);
		card->regs.beslumactl = (card-> brightness << 16) | ((card->contrast + 0x80) & 0xFFFF);
		mga_vid_write_regs(card, 0);
		break;

	default:
		printk(KERN_ERR "mga_vid: Invalid ioctl\n");
		result = -EINVAL;
		goto ioctl_out;
	}

	ioctl_out:
	spin_unlock(&card->lock);
	return result;
}

static void cards_init(mga_card_t * card, struct pci_dev *dev, int card_number, int card_type);

// returns the number of found cards
static int mga_vid_find_card(void)
{
	struct pci_dev *dev = NULL;
	char *mga_dev_name;
	mga_card_t *card;

	while ((dev = pci_get_device(PCI_VENDOR_ID_MATROX, PCI_ANY_ID, dev))) {
		mga_dev_name = "";
		mga_cards_num++;
		if (mga_cards_num == MGA_MAX_CARDS) {
			printk(KERN_WARNING "mga_vid: Trying to initialize more than %d cards\n", MGA_MAX_CARDS);
			mga_cards_num--;
			break;
		}

		card = kzalloc(sizeof(mga_card_t), GFP_KERNEL);
		if (!card) {
			printk(KERN_ERR "mga_vid: memory allocation failed\n");
			mga_cards_num--;
			break;
		}

		mga_cards[mga_cards_num - 1] = card;

		switch (dev->device) {
		case PCI_DEVICE_ID_MATROX_G550:
			mga_dev_name = "MGA G550";
			printk(KERN_INFO "mga_vid: Found %s at %s\n", mga_dev_name, pci_name(dev));
			cards_init(card, dev, mga_cards_num - 1, MGA_G550);
			break;
		case PCI_DEVICE_ID_MATROX_G400:
			mga_dev_name = "MGA G400/G450";
			printk(KERN_INFO "mga_vid: Found %s at %s\n", mga_dev_name, pci_name(dev));
			cards_init(card, dev, mga_cards_num - 1, MGA_G400);
			break;
		case PCI_DEVICE_ID_MATROX_G200_AGP:
			mga_dev_name = "MGA G200 AGP";
			printk(KERN_INFO "mga_vid: Found %s at %s\n", mga_dev_name, pci_name(dev));
			cards_init(card, dev, mga_cards_num - 1, MGA_G200);
			break;
		case PCI_DEVICE_ID_MATROX_G200_PCI:
			mga_dev_name = "MGA G200";
			printk(KERN_INFO "mga_vid: Found %s at %s\n", mga_dev_name, pci_name(dev));
			cards_init(card, dev, mga_cards_num - 1, MGA_G200);
			break;
		default:
			mga_cards_num--;
			printk(KERN_INFO "mga_vid: ignoring unsupported matrox device (%d) at %s\n", dev->device, pci_name(dev));
			break;
		}
	}

	if (!mga_cards_num) {
		printk(KERN_ERR "mga_vid: No supported cards found\n");
	} else {
		printk(KERN_INFO "mga_vid: %d supported cards found\n", mga_cards_num);
	}

	return mga_cards_num;
}

static void mga_param_buff_fill(mga_card_t * card)
{
	size_t len;
	size_t size = card->param_buff_size;
	char *buf = card->param_buff;
	len = 0;
	len += snprintf(&buf[len], size - len, "Interface version: %04X\n", MGA_VID_VERSION);
	len += snprintf(&buf[len], size - len, "Memory: %x:%dM\n", card->mem_base, (unsigned int)card->ram_size);
	len += snprintf(&buf[len], size - len, "MMIO: %p\n", card->mmio_base);
	len += snprintf(&buf[len], size - len, "Configurable stuff:\n");
	len += snprintf(&buf[len], size - len, "~~~~~~~~~~~~~~~~~~~\n");
	len += snprintf(&buf[len], size - len, PARAM_BRIGHTNESS "%d\n", card->brightness);
	len += snprintf(&buf[len], size - len, PARAM_CONTRAST "%d\n", card->contrast);
	len += snprintf(&buf[len], size - len, PARAM_BLACKIE "%s\n", card->regs.blackie ? "on" : "off");
	card->param_buff_len = len;
}

static ssize_t mga_vid_read(struct file *file, char *buf, size_t count, loff_t * ppos)
{
	size_t size;
	mga_card_t *card = (mga_card_t *) file->private_data;

	if (!card->param_buff)
		return -ESPIPE;
	if (!(*ppos))
		mga_param_buff_fill(card);
	if (*ppos >= card->param_buff_len)
		return 0;
	size = min(count, card->param_buff_len - (size_t) (*ppos));
	copy_to_user(buf, card->param_buff, size);
	*ppos += size;
	return size;
}

static ssize_t mga_vid_write(struct file *file, const char *buf, size_t count, loff_t * ppos)
{
	mga_card_t *card = (mga_card_t *) file->private_data;
	char * buffer;

	if(count >= PAGE_SIZE)
	{
		// only one parameter is handled at a time,
		// so writing more than one page does not
		// make sense anyways
                printk(KERN_ERR "mga_vid: writing more than %lu bytes to the device file\n",PAGE_SIZE);
		return -EFBIG;
	}

	// allocate memory and copy data into our own buffer
	buffer = kmalloc(count, GFP_KERNEL);
	
	if(!buffer)
	{
                printk(KERN_ERR "mga_vid: couln't allocate buffer\n");
		return -EIO;
	}

	memset(buffer, 0, count);
	// copy_from_user returns the amount of data _still_ to be copied
	if(copy_from_user(buffer, buf, count))
	{
		// something strange happend, bail out
                printk(KERN_ERR "mga_vid: couldn't copy data from user space\n");
		kfree(buffer);
		return -EIO;
	}

	if (memcmp(buffer, PARAM_BRIGHTNESS, min(count, strlen(PARAM_BRIGHTNESS))) == 0) {
		short brightness;
		brightness = simple_strtol(&buffer[strlen(PARAM_BRIGHTNESS)], NULL, 10);
		if (brightness > 127 || brightness < -128) {
			brightness = 0;
		}
//              printk(KERN_DEBUG "mga_vid: brightness modified ( %d ) \n",brightness);
		card->brightness = brightness;
	} else
	    if (memcmp(buffer, PARAM_CONTRAST, min(count, strlen(PARAM_CONTRAST))) == 0) {
		short contrast;
		contrast = simple_strtol(&buffer[strlen(PARAM_CONTRAST)], NULL, 10);
		if (contrast > 127 || contrast < -128) {
			contrast = 0;
		}
//              printk(KERN_DEBUG "mga_vid: contrast modified ( %d ) \n",contrast);
		card->contrast = contrast;
	} else
	 if (memcmp(buffer, PARAM_BLACKIE, min(count, strlen(PARAM_BLACKIE))) == 0) {
		short blackie;
		blackie = simple_strtol(&buffer[strlen(PARAM_BLACKIE)], NULL, 10);
//              printk(KERN_DEBUG "mga_vid: shadow mode: ( %d ) \n",blackie);
		card->regs.blackie = (blackie > 0) ? 1 : 0;
	} else
		count = -EIO;
	// TODO: reset settings
	
	kfree(buffer);
	return count;
}

static int mga_vid_mmap(struct file *file, struct vm_area_struct *vma)
{
	mga_card_t *card = (mga_card_t *) file->private_data;
	spin_lock(&card->lock);

#ifdef MP_DEBUG
	printk(KERN_DEBUG "mga_vid: mapping video memory into userspace\n");
#endif

	if (!card->configured) {
		printk(KERN_ERR "mga_vid: card is not configured, cannot mmap\n");
		spin_unlock(&card->lock);
		return (-EAGAIN);
	}
	//FIXME: should we set VM_IO too ? remap_pfn_range does it for us,
	//but this may change in future
	vma->vm_flags |= VM_RESERVED;	// don't swap the page out

	if (remap_pfn_range(vma, vma->vm_start,
			    (card->mem_base + card->src_base) >> PAGE_SHIFT,
			    vma->vm_end - vma->vm_start, vma->vm_page_prot)) {
		printk(KERN_ERR "mga_vid: error mapping video memory\n");
		spin_unlock(&card->lock);
		return (-EAGAIN);
	}

	spin_unlock(&card->lock);
	return 0;
}

static int mga_vid_release(struct inode *inode, struct file *file)
{
	mga_card_t *card;

	//Close the window just in case
#ifdef MP_DEBUG
	printk(KERN_DEBUG "mga_vid: Video OFF (release)\n");
#endif

	card = (mga_card_t *) file->private_data;
	spin_lock(&card->lock);

	card->vid_src_ready = 0;
	card->regs.besctl &= ~1;
	card->regs.besglobctl &= ~(1 << 6);	// UYVY format selected
//      card->config.colkey_on=0; //!!!
	mga_vid_write_regs(card, 1);
	card->vid_in_use = 0;
	spin_unlock(&card->lock);
	return 0;
}

static long long mga_vid_lseek(struct file *file, long long offset, int origin)
{
	return -ESPIPE;
}

static int mga_vid_open(struct inode *inode, struct file *file)
{
	mga_card_t *card;

	int minor = iminor(inode);

	// we don't have that many cards
	if (minor >= mga_cards_num)
		return (-ENXIO);

	file->private_data = mga_cards[minor];

	card = (mga_card_t *) file->private_data;

	spin_lock(&card->lock);
	if (card->vid_in_use == 1)
	{
		spin_unlock(&card->lock);
		return (-EBUSY);
	}

	card->vid_in_use = 1;
	card->configured = 0;
	spin_unlock(&card->lock);
	return 0;
}

static struct file_operations mga_vid_fops = {
      llseek:mga_vid_lseek,
      read:mga_vid_read,
      write:mga_vid_write,
      unlocked_ioctl:mga_vid_ioctl,
      compat_ioctl:mga_vid_ioctl,
      mmap:mga_vid_mmap,
      open:mga_vid_open,
      release:mga_vid_release
};

static void cards_init(mga_card_t * card, struct pci_dev *dev, int card_number, int card_type)
{
	unsigned int card_option;

	spin_lock_init(&card->lock);
	card->pci_dev = dev;

	card->card_type = card_type;

	card->param_buff = kmalloc(PARAM_BUFF_SIZE, GFP_KERNEL);
	if (card->param_buff)
		card->param_buff_size = PARAM_BUFF_SIZE;

	card->brightness = mga_brightness[card_number];
	card->contrast = mga_contrast[card_number];
	card->top_reserved = mga_top_reserved[card_number];

	card->mmio_base = ioremap_nocache(dev->resource[1].start, 0x4000);
	card->mem_base = dev->resource[0].start;

	printk(KERN_INFO "mga_vid: MMIO at 0x%p framebuffer: 0x%08X\n", card->mmio_base, card->mem_base);

	pci_read_config_dword(dev, 0x40, &card_option);
	printk(KERN_INFO "mga_vid: OPTION word: 0x%08X  mem: 0x%02X  %s\n",
	       card_option, (card_option >> 10) & 0x17,
	       ((card_option >> 14) & 1) ? "SGRAM" : "SDRAM");

	if (mga_ram_size[card_number]) {
		printk(KERN_INFO "mga_vid: RAMSIZE forced to %d MB\n", mga_ram_size[card_number]);
		card->ram_size = mga_ram_size[card_number];
	} else {

#ifdef MGA_MEMORY_SIZE
		card->ram_size = MGA_MEMORY_SIZE;
		printk(KERN_INFO "mga_vid: hard-coded RAMSIZE is %d MB\n", (unsigned int)card->ram_size);

#else

		if (card->card_type != MGA_G200) {
			switch ((card_option >> 10) & 0x17) {
				// SDRAM:
			case 0x00:
			case 0x04:
				card->ram_size = 16;
				break;
			case 0x03:
			case 0x05:
				card->ram_size = 32;
				break;
				// SGRAM:
			case 0x10:
			case 0x14:
				card->ram_size = 32;
				break;
			case 0x11:
			case 0x12:
				card->ram_size = 16;
				break;
			default:
				card->ram_size = 16;
				printk(KERN_INFO "mga_vid: Couldn't detect RAMSIZE, assuming 16MB!");
			}
			/* Check for buggy 16MB cards reporting 32 MB */
			if (card->ram_size != 16 &&
			    (dev->subsystem_device == PCI_SUBSYSTEM_ID_MATROX_G400_16MB_SDRAM
			     || dev->subsystem_device == PCI_SUBSYSTEM_ID_MATROX_G400_16MB_SGRAM
			     || dev->subsystem_device == PCI_SUBSYSTEM_ID_MATROX_G400_DH_16MB)) {
				printk(KERN_INFO "mga_vid: Detected 16MB card reporting %d MB RAMSIZE, overriding\n", card->ram_size);
				card->ram_size = 16;
			}
		} else {
			/* 
			 * Don't do any autodetection on G200 cards
			 * and force them to 8MB 
			 */
			card->ram_size = 8;
		}

		printk(KERN_INFO "mga_vid: detected RAMSIZE is %d MB\n", (unsigned int)card->ram_size);
#endif
	}
}

/* 
 * Main Initialization Function 
 */

static int mga_vid_initialize(void)
{
	int i;
	printk(KERN_INFO "Matrox MGA G200/G400/G450/G550 YUV Video interface v2.01 (c) Aaron Holtzman & A'rpi\n");

	if (mga_vid_find_card()) {
		/* Have the kernel generate a major device number */
		if (major == 0) {
			if (!alloc_chrdev_region (&mga_cdev_handle, 0, mga_cards_num, "mga_vid")) {
				major = MAJOR(mga_cdev_handle);
				printk(KERN_INFO "mga_vid: using major: %d (dynamically alocated!)\n", major);
			}
		} else {
			mga_cdev_handle = MKDEV(major, 0);
			if (!register_chrdev_region (mga_cdev_handle, mga_cards_num, "mga_vid")) {
				mga_vid_major = major;

				mga_vid_class = class_create(THIS_MODULE, "mga_vid");
				for (i = 0; i < mga_cards_num; i++) {
					device_create(mga_vid_class, NULL, MKDEV(mga_vid_major, i), NULL, "mga_vid%d", i);
				}

				printk(KERN_INFO "mga_vid: using major: %d (assigned or default!)\n", major);
			}
		}
		/* Allocate a cdev for this character device, and fill in some parameters it needs */
		mga_vid_cdev = cdev_alloc();
		mga_vid_cdev->owner = THIS_MODULE;
		kobject_set_name(&mga_vid_cdev->kobj, MODULENAME);
		mga_vid_cdev->ops = &mga_vid_fops;
		/* Add this character device to the system */
		cdev_add(mga_vid_cdev, mga_cdev_handle, mga_cards_num);
		for (i = 0; i < mga_cards_num; i++) {
			if (mga_ram_size[i]) {
				if (mga_ram_size[i] < 4 || mga_ram_size[i] > 64) {
					printk(KERN_ERR "mga_vid: invalid RAMSIZE: %d MB\n", mga_ram_size[i]);
					return -EINVAL;
				}
			}
		}

		return 0;
	} else {
		return -EFAULT;
	}
}

module_init(mga_vid_initialize);

static void mga_cleanup_module(void)
{
	int i;
	mga_card_t *card;

	printk(KERN_INFO "mga_vid: Cleaning up module\n");
	for (i = 0; i < MGA_MAX_CARDS; i++) {
		card = mga_cards[i];
		if (card) {
			if (card->mmio_base)
				iounmap(card->mmio_base);
			if (card->param_buff)
				kfree(card->param_buff);
			kfree(card);
			mga_cards[i] = NULL;
		}
	}

	for (i = 0; i < mga_cards_num; i++) {
		device_destroy(mga_vid_class, MKDEV(mga_vid_major, i));
	}
	class_destroy(mga_vid_class);

	//FIXME turn off BES
	cdev_del(mga_vid_cdev);
	unregister_chrdev_region(mga_cdev_handle, mga_cards_num);
}

module_exit(mga_cleanup_module);
