/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "dbus_monitor.h"

#include <core/posix/exec.h>

#include <iostream>
#include <map>

namespace
{
std::map<std::string, std::string> copy_env()
{
    std::map<std::string, std::string> env;
    core::posix::this_process::env::for_each([&env](const std::string& k, const std::string& v)
                                             {
                                                 env.insert(std::make_pair(k, v));
                                             });
    return env;
}
}

testing::DBusMonitor::DBusMonitor(const std::string& bus_address)
    : cp{core::posix::exec(dm, {"--address", bus_address}, copy_env(), core::posix::StandardStream::stdout)}
{
}

testing::DBusMonitor::~DBusMonitor()
{
    try
    {
        // Shutting down the monitor, which results in the pipe
        // established for stdout redirection being closed, too.
        cp.send_signal_or_throw(core::posix::Signal::sig_kill);
        cp.wait_for(core::posix::wait::Flags::untraced);
    }
    catch (...)
    {
        // Silently drop all issues here. We can hardly do anything
        // about them and we just try to be as clean as possible.
    }
}

std::istream& testing::DBusMonitor::cout()
{
    return cp.cout();
}
