/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/qt/runtime.h>

#include <glog/logging.h>

#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_generators.hpp>
#include <boost/uuid/uuid_io.hpp>

#include <atomic>

namespace
{
namespace log
{
// Initializer bundles setup/teardown of the google::log infrastructure.
struct Initializer
{
    Initializer()
    {
        FLAGS_logtostderr = true;
        FLAGS_colorlogtostderr = true;
        google::InitGoogleLogging("messaging::framework");
    }

    ~Initializer()
    {
        google::ShutdownGoogleLogging();
    }

    bool has_been_called() const
    {
        return true;
    }
};

const Initializer& initializer()
{
    static Initializer instance;
    return instance;
}
}

// Atomic flag tracking whether a messaging::qt::Runtime instance
// is currently in flight.
std::atomic<bool> is_initialized{false};

// Throws messaging::qt::Runtime::AlreadyCreated if there is a
// messaging::qt::Runtime instance in flight.
void throw_if_already_created()
{
    if (is_initialized.exchange(true))
        throw messaging::qt::Runtime::AlreadyCreated{};
}

// Returns a custom deleter that resets the global initialized flag to false again.
// With that, giving up on a messaging::qt::Runtime allows for creating a new one.
std::function<void(messaging::qt::Runtime*)> make_resetting_deleter()
{
    return [](messaging::qt::Runtime* runtime)
    {
        delete runtime;
        is_initialized.exchange(false);
    };
}

// message_handler maps Qt's logging subsystem to our own internal one. Right now,
// this relies on Google log and is hidden from the public face of the library.
// This should change however for testing purposes. For now, this is good enough to
// debug test cases.
void message_handler(QtMsgType type, const QMessageLogContext& context, const QString& msg)
{
    // We make sure that Google Log has been initialized once.
    // We avoid any sort of static init fiascos and make sure that logging works flawlessly
    // even prior to main().
    static const bool glog_has_been_initialized{log::initializer().has_been_called()};
    Q_UNUSED(glog_has_been_initialized)

    google::LogSeverity severity{google::GLOG_INFO};

    switch (type)
    {
#if (QT_VERSION >= QT_VERSION_CHECK(5, 5, 0))
        case QtMsgType::QtInfoMsg:
#endif
        case QtMsgType::QtDebugMsg:
            severity = google::GLOG_INFO;
            break;
        case QtMsgType::QtWarningMsg:
            severity = google::GLOG_WARNING;
            break;
        case QtMsgType::QtCriticalMsg:
            severity = google::GLOG_ERROR;
            break;
        case QtMsgType::QtFatalMsg:
            severity = google::GLOG_FATAL;
            break;
    }

    google::LogMessage lm((context.file ? context.file : "messaging::fw"), context.line, severity);

    lm.stream() << msg.toStdString();
    // google::LogMessage's dtor takes care of flushing the stream.
}

struct TaskEvent : public QEvent
{
    static QEvent::Type type;

    TaskEvent(const std::function<void()>& task)
        : QEvent{TaskEvent::type}
        , task{task}
    {
    }

    void run()
    {
        task();
    }

    std::function<void()> task;
};

QEvent::Type TaskEvent::type{static_cast<QEvent::Type>(QEvent::registerEventType())};

struct TaskEventFilter : public QObject
{
    TaskEventFilter(QObject* parent)
        : QObject{parent}
    {
    }

    bool eventFilter(QObject* obj, QEvent* event)
    {
        if (event->type() != TaskEvent::type)
        {
            return QObject::eventFilter(obj, event);
        }

        auto te = static_cast<TaskEvent*>(event);
        te->run();

        return true;
    }
};
}

messaging::qt::Runtime::AlreadyCreated::AlreadyCreated()
    : std::logic_error{"messaging::qt::Runtime can only be instantiated once."}
{
}

std::shared_ptr<messaging::qt::Runtime> messaging::qt::Runtime::create_once_or_throw(QDBusConnection::BusType bus_type)
{
    throw_if_already_created();
    return std::shared_ptr<messaging::qt::Runtime>{new messaging::qt::Runtime{bus_type}, make_resetting_deleter()};
}

std::shared_ptr<messaging::qt::Runtime> messaging::qt::Runtime::create_once_or_throw(const QString& dbus_address)
{
    throw_if_already_created();
    return std::shared_ptr<messaging::qt::Runtime>{new messaging::qt::Runtime{dbus_address}, make_resetting_deleter()};
}

void messaging::qt::Runtime::enter_with_task(const std::function<void()>& task)
{
    app_.postEvent(&app_, new TaskEvent{task});
}

int messaging::qt::Runtime::run()
{
    return app_.exec();
}

void messaging::qt::Runtime::stop(int code)
{
    app_.exit(code);
}

const QDBusConnection& messaging::qt::Runtime::dbus_connection() const
{
    return dbus_connection_;
}

namespace
{
static int argc = 0;

QString unique_dbus_connection_name()
{
    return QString::fromStdString(boost::uuids::to_string(boost::uuids::random_generator()()));
}
}

messaging::qt::Runtime::Runtime(QDBusConnection::BusType bus_type)
    : app_{argc, nullptr}
    , dbus_connection_{QDBusConnection::connectToBus(bus_type, unique_dbus_connection_name())}
{
    qInstallMessageHandler(message_handler);
    app_.installEventFilter(new TaskEventFilter(&app_));
}

messaging::qt::Runtime::Runtime(const QString& dbus_address)
    : app_{argc, nullptr}
    , dbus_connection_{QDBusConnection::connectToBus(dbus_address, unique_dbus_connection_name())}
{
    qInstallMessageHandler(message_handler);
    app_.installEventFilter(new TaskEventFilter(&app_));
}
