/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_QT_TP_TEXT_CHANNEL_H_
#define MESSAGING_QT_TP_TEXT_CHANNEL_H_

#include <messaging/chat.h>
#include <messaging/messenger.h>
#include <messaging/qt/runtime.h>

#include <messaging/qt/tp/interfaces/base_channel_destroyable.h>
#include <messaging/qt/tp/interfaces/base_channel_roles.h>
#include <messaging/qt/tp/interfaces/base_channel_subject.h>
#include <messaging/qt/tp/interfaces/types.h>
#include <messaging/qt/tp/interfaces/constants.h>

#include <TelepathyQt/BaseChannel>
#include <TelepathyQt/Types>

#include <memory>
#include <set>

namespace messaging
{
// A handy fwd decl.
class Connection;
class Recipient;
namespace qt
{
namespace tp
{
// A handy fwd decl.
class Connection;

class TextChannel
        : public Tp::BaseChannel
        , public messaging::Chat::Observer
        , public messaging::GroupManager::Observer
{
    Q_OBJECT
public:
    /// @brief Safes us some typing.
    typedef Tp::SharedPtr<TextChannel> Ptr;

    /// @brief Creates a new managed TextChannel instance.
    static Ptr create(Connection* tp_connection,
                      uint th,
                      uint tht,
                      const std::shared_ptr<qt::Runtime>& runtime,
                      const std::shared_ptr<Messenger>& messenger,
                      const std::shared_ptr<Recipient>& recipient,
                      const std::shared_ptr<GroupManager>& group_manager);

    /// @brief on_message_delivery_report informs about a change in the delivery
    /// status of an outgoing message
    void on_message_delivery_report(const std::string& id, DeliveryStatus status) override;

    /// @brief on_message_received is invoked for every new incoming message.
    void on_message_received(const Message& message) override;

    /// @brief on_message_id_changed is invoked when a message temporal id is updated to the
    /// definitive one
    void on_message_id_changed(const std::string& old_id, const std::string& new_id) override;

    /// @brief on_group_created when the group has been created ok in the server
    void on_group_created() override;

    /// @brief on_group_cancelled when the group is no longer valid
    void on_group_cancelled(CancelGroupReason reason, const std::string &message) override;

    /// @brief on_group_title_changed when group title has changed in the server
    void on_group_title_changed(const std::string& new_title) override;

    /// @brief on_group_subject_changed when group title has changed in the server
    void on_group_subject_changed(const std::string& new_subject,
                                const std::shared_ptr<Member>& actor,
                                const std::chrono::system_clock::time_point& when) override;

    /// @brief on_members_updated when group members have changed (added/removed) in server
    void on_members_updated(const Members& members) override;

    /// @brief on_group_permissions_changed when some permission has changed.
    void on_group_permissions_changed(const Flags<GroupPermissions> &permissions) override;

    /// @brief on_file_receiving is invoked when a file is downloading.
    void on_file_receiving(const std::string&, uint, uint) override;

    /// @brief on_file_received is invoked when a file has finished downloading.
    void on_file_received(const std::string&, uint, uint) override;

    /// @brief on_file_sending is invoked when a file is being uploaded.
    void on_file_sending(const std::string&, uint, uint) override;

    /// @brief on_participant_starts_typing is invoked when a participant in this chat has started composing a message
    void on_participant_starts_typing(const std::shared_ptr<User> &user) override;

    /// @brief on_participant_ends_typing is invoked when a participant in this chat ends composing a message
    void on_participant_ends_typing(const std::shared_ptr<User> &user) override;

public Q_SLOTS:
    void delay_close();

private:
    TextChannel(Connection* tp_connection,
                uint th,
                uint tht,
                const std::shared_ptr<qt::Runtime>& runtime,
                const std::shared_ptr<Messenger>& messenger,
                const std::shared_ptr<Recipient>& recipient,
                const std::shared_ptr<GroupManager>& group_manager);

    /// @brief Observer provides means to monitor the state of a Chat instance.
    ///
    /// Dispatches calls over to the qt::Runtime and makes sure that any invocations on
    /// tp::TextChannel happen on the qt main thread.
    class Observer
            : public messaging::Chat::Observer
            , public messaging::GroupManager::Observer
            , public std::enable_shared_from_this<Observer>
    {
      public:
        Observer(const std::shared_ptr<qt::Runtime>& runtime, const Ptr& tc);

        /// @brief on_message_delivery_report informs about a change in the delivery
        /// status of an outgoing message
        void on_message_delivery_report(const std::string& id, DeliveryStatus status) override;

        /// @brief on_message_received is invoked for every new incoming message.
        void on_message_received(const Message& message) override;

        /// @brief on_message_id_changed is invoked when a message temporal id is updated to the
        /// definitive one
        void on_message_id_changed(const std::string& old_id, const std::string& new_id) override;

        /// @brief on_group_created when the group has been created ok in the server
        void on_group_created() override;

        /// @brief on_group_cancelled when the group is no longer valid
        void on_group_cancelled(CancelGroupReason reason, const std::string &message) override;

        /// @brief on_group_title_changed when group title has changed in the server
        void on_group_title_changed(const std::string& new_title) override;

        /// @brief on_group_subject_changed when group title has changed in the server
        void on_group_subject_changed(const std::string& new_subject,
                                      const std::shared_ptr<Member>& actor,
                                      const std::chrono::system_clock::time_point& when) override;

        /// @brief on_members_updated when group members have changed (added/removed) in server
        void on_members_updated(const Members& members) override;

        /// @brief on_group_permissions_changed when some permission has changed.
        void on_group_permissions_changed(const Flags<GroupPermissions> &permissions) override;

        /// @brief on_file_receiving is invoked when a file is downloading.
        void on_file_receiving(const std::string&, uint, uint) override;

        /// @brief on_file_received is invoked when a file has finished downloading.
        void on_file_received(const std::string&, uint, uint) override;

        /// @brief on_file_sending is invoked when a file is being uploaded.
        void on_file_sending(const std::string&, uint, uint) override;

        /// @brief on_participant_starts_typing is invoked when a participant in this chat has started composing a message
        void on_participant_starts_typing(const std::shared_ptr<User> &user) override;

        /// @brief on_participant_ends_typing is invoked when a participant in this chat ends composing a message
        void on_participant_ends_typing(const std::shared_ptr<User> &user) override;

      private:
        std::shared_ptr<qt::Runtime> runtime;
        Ptr tc;
    };

    // Handler acting on send message requests. Never throws, reports issues via error.
    QString send_message(const Tp::MessagePartList& message, uint flags, Tp::DBusError* error);
    // Handler acting on "message acknowldedged" events.
    void message_acknowledged(const QString& id);
    // Handler acting on add members request
    void add_members(const Tp::UIntList& handles, const QString& message, Tp::DBusError* error);
    // Handler action on remove members request
    void remove_members(const Tp::UIntList& handles, const QString& message, uint reason, Tp::DBusError* error);
    // Handler action to change the state of a chat participant
    void set_chat_state(uint state, Tp::DBusError *error);
    // Handler action to update the group configuration
    void update_group_configuration(const QVariantMap &properties, Tp::DBusError *error);
    // Handler action to update roles for current channel group members
    void update_roles(const interfaces::HandleRolesMap &contact_roles, Tp::DBusError *error);
    // Handle action to update current channel subject
    void set_subject(const QString &subject, Tp::DBusError *error);

    // channel destruction request. only supported in group chats
    void destroy(Tp::DBusError* error);
    // internal method to close the channel when requested from server side
    void close_channel_with_reason(uint reason, const QString &message);

    // Helper functions encapsulating setup tasks.
    void register_callbacks_once();
    void plug_interfaces_once();
    void plug_interface_if_available(const Tp::AbstractChannelInterfacePtr &interface);
    bool self_is_admin();

    Tp::BaseChannelTextTypePtr text_type_interface;
    Tp::BaseChannelMessagesInterfacePtr messages_interface;
    Tp::BaseChannelRoomInterfacePtr room_interface;
    Tp::BaseChannelRoomConfigInterfacePtr room_config_interface;
    Tp::BaseChannelGroupInterfacePtr group_interface;
    Tp::BaseChannelChatStateInterfacePtr chat_state_interface;
    interfaces::BaseChannelDestroyableInterfacePtr destroyable_interface;
    interfaces::BaseChannelRolesInterfacePtr roles_interface;
    interfaces::BaseChannelSubjectInterfacePtr subject_interface;

    // NOTE(rmescandon): this is a temporal member in the class not needed from Telepathy Qt >= 0.9.7.0 which
    // includes Tp::BaseChannel::connection() method to obtain it whenever is needed.
    Connection* tp_connection;
    std::shared_ptr<Observer> observer;
    std::shared_ptr<messaging::Chat> chat;
    QMap<std::string, Tp::MessagePartList> pending_messages;
    // NOTE(rmescandon): this is needed to map sent messages with certain temp id to the original message, in
    // case a definitive id is received. In that case, the original message is got from this map and updated their
    // headers to be substituted in client.
    QMap<std::string, Tp::MessagePartList> sent_messages;
};
}
}
}

#endif  // MESSAGING_QT_TP_TEXT_CHANNEL_H_
