/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_CHAT_H
#define MESSAGING_CHAT_H

#include <messaging/non_copyable.h>
#include <messaging/non_movable.h>
#include <messaging/message.h>
#include <messaging/group_manager.h>
#include <messaging/utils.h>
#include <messaging/has_interfaces.h>

namespace messaging
{

/// @brief Chat models a textual conversation between one or more participants.
class MESSAGING_FW_PUBLIC Chat : public HasInterfaces, NonCopyable, NonMovable
{
  public:
    /// @cond
    DEFINE_PTRS(Chat);
    /// @endcond

    /// @brief Observer provides means to monitor the state of a Chat instance.
    class Observer : NonCopyable, NonMovable
    {
      public:
        /// @brief on_message_delivery_report informs about a change in the delivery
        /// status of an outgoing message
        virtual void on_message_delivery_report(const std::string& id, DeliveryStatus status) = 0;

        /// @brief on_message_received is invoked for every new incoming message.
        virtual void on_message_received(const Message& message) = 0;

        /// @brief on_message_id_changed is invoked when a message temporal id is updated to the
        /// definitive one
        virtual void on_message_id_changed(const std::string& old_id, const std::string& new_id) = 0;

        /// @brief on_file_receiving is invoked when a file is downloading.
        virtual void on_file_receiving(const std::string&, uint, uint) = 0;

        /// @brief on_file_received is invoked when a file has finished downloading.
        virtual void on_file_received(const std::string&, uint, uint) = 0;

        /// @brief on_file_sending is invoked when a file is uploading.
        virtual void on_file_sending(const std::string&, uint, uint) = 0;

        /// @brief on_participant_starts_typing is invoked when a participant in this chat has started composing a message
        virtual void on_participant_starts_typing(const std::shared_ptr<User> &user) = 0;

        /// @brief on_participant_ends_typing is invoked when a participant in this chat ends composing a message
        virtual void on_participant_ends_typing(const std::shared_ptr<User> &user) = 0;

      protected:
        Observer() = default;        
    };

    /// @brief send_message enqueues the given message for delivery.
    /// @returns a numeric identifier referencing back to the delivery request.
    /// @throws std::runtime_error in case of issues.
    virtual std::string send_message(const Message& message) = 0;

    /// @brief mark_message_as_read marks received message as read for the sender
    virtual void mark_message_as_read(const std::string& message_id) = 0;

    /// @brief request to start downloading a file
    virtual std::string download_file(std::string) = 0;

    /// @brief starts composing a message notification
    virtual void start_typing() = 0;

    /// @brief stop composing a message notification signal
    virtual void end_typing() = 0;

  protected:
    /// @brief Chat constructs a new instance, installing the given observer instance.
    Chat(const std::shared_ptr<Observer>& observer);

    /// @brief announce_message_delivery_report informs about a change in the delivery
    /// status of an outgoing message
    void announce_message_delivery_report(const std::string& id, DeliveryStatus status);

    /// @brief announce_message_received triggers the installed observer
    /// with on_message_received.
    void announce_message_received(const Message&);

    /// @brief announce_message_id_changed triggers the installed observer with
    /// on_message_id_changed
    void announce_message_id_changed(const std::string& old_id, const std::string& new_id);

    /// @brief announce_file_receiving triggers the installed observer
    /// with on_file_receiving
    void announce_file_receiving(const std::string&, uint, uint);

    /// @brief announce_file_received triggers the installed observer
    /// with on_file_received
    void announce_file_received(const std::string&, uint, uint);

    /// @brief announce_file_sending triggers the installed observer
    /// with on_file_sending
    void announce_file_sending(const std::string&, uint, uint);

    /// @brief announce_participant_starts_typing triggers the installed observer with on_participant_starts_typing
    void announce_participant_starts_typing(const std::shared_ptr<User>& user);

    /// @brief announce_participant_ends_typing triggers the installed observer with on_participant_ends_typing
    void announce_participant_ends_typing(const std::shared_ptr<User> &user);

  private:
    /// @cond
    struct Private;
    std::shared_ptr<Private> impl;
    /// @endcond
};
}

#endif // MESSAGING_CHAT_H
