/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_BOOST_VARIANT_H_
#define MESSAGING_BOOST_VARIANT_H_

#include <messaging/variant.h>
#include <messaging/visibility.h>

#include <boost/variant.hpp>

#include <map>
#include <vector>

namespace messaging
{
/// @brief BoostVariant implemeents Variant relying on boost::variant.
class MESSAGING_FW_PUBLIC BoostVariant : public Variant
{
public:
    /// @brief Value marks our underlying Value type.
    typedef typename boost::make_recursive_variant<bool,
                                                   std::int64_t,
                                                   double,
                                                   std::string,
                                                   std::vector<char>,
                                                   std::vector<boost::recursive_variant_>,
                                                   std::map<boost::recursive_variant_, boost::recursive_variant_>>::type
        Value;

    explicit BoostVariant(const Value& v);
    explicit BoostVariant(bool b);
    explicit BoostVariant(std::int64_t i);
    explicit BoostVariant(double d);
    explicit BoostVariant(const std::string& s);
    explicit BoostVariant(const char* data, std::size_t size);
    explicit BoostVariant(const std::vector<char>& d);
    explicit BoostVariant(const std::vector<Value>& v);
    explicit BoostVariant(const std::map<Value, Value>& m);

    BoostVariant(const BoostVariant& rhs);
    BoostVariant& operator=(const BoostVariant& rhs);

    /// type returns the type of the variant.
    Type type() const override;

    /// @brief as_bool tries to extract a boolean value from this instance.
    /// @throw TypeMismatch in case of issues.
    bool as_bool() const override;

    /// @brief as_int tries to extract an integer value from this instance.
    /// @throw TypeMismatch in case of issues.
    std::int64_t as_int() const override;

    /// @brief as_double tries to extract a double value from this instance.
    /// @throw TypeMismatch in case of issues.
    double as_double() const override;

    /// @brief as_string tries to extract a string value from this instance.
    /// @throw TypeMismatch in case of issues.
    std::string as_string() const override;

    /// @brief as_date tries to extract a an array of bytes from this instance.
    const char *as_data() const override;

    /// @brief size returns the size of the array returned by as_data().
    std::size_t data_size() const override;

    /// @brief size returns the number of children of this variant.
    std::size_t size() const override;

    /// @brief keys returns a deque of the keys of the Variant if it's a map
    /// a logic error will be thrown if not a map type
    std::deque<std::string> keys() const override;

    /// @brief value returns the value related with the received key. Returns default_value if not found
    /// This operation only has sense with maps
    std::shared_ptr<Variant> at(const std::string& key) const override;

    /// @brief at returns the variant instance at the given idx.
    /// @throw std::out_of_range if the idx >= size().
    std::shared_ptr<Variant> at(std::size_t idx) const override;

private:
    Value value;
};
}

#endif  // MESSAGING_BOOST_VARIANT_H_
