#
# Copyright (c) 2010 Canonical
#
# Written by Gustavo Niemeyer <gustavo@niemeyer.net>
#
# This file is part of the Xpresser GUI automation library.
#
# Xpresser is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3,
# as published by the Free Software Foundation.
#
# Xpresser is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import time

import gtk

from xpresser import xutils
from xpresser.image import Image
from xpresser.opencvfinder import OpenCVFinder

from xpresser.tests.images import get_image_path
from xpresser.lib.testing import TestCase


class XUtilsTestBase(TestCase):

    def setUp(self):
        xutils.hover(0, 0)

    def flush_gtk(self):
        # Why do we need to do so much to get the button in place before
        # actually trying to click on it? :-(  If we just run until there
        # are no more events, and without sleep, the button will simply
        # return (0, 0) as its position.
        while gtk.events_pending():
            gtk.main_iteration()
        time.sleep(0.1) # Why oh why? :-(
        while gtk.events_pending():
            gtk.main_iteration()
        time.sleep(0.1)
        while gtk.events_pending():
            gtk.main_iteration()

    def create_window(self, child):
        window = gtk.Window(gtk.WINDOW_TOPLEVEL)
        window.connect("delete_event", lambda widget, event: False)
        window.add(child)
        child.show()
        window.show()
        return window

    def create_button_window(self, image_path=None):
        button = gtk.Button()
        if image_path is None:
            image_path = get_image_path("red-square.png")
        button.set_image(gtk.image_new_from_file(image_path))
        return self.create_window(button)

    def create_image_window(self, image_path):
        image = gtk.image_new_from_file(image_path)
        return self.create_window(image)


class XUtilsTest(XUtilsTestBase):

    def test_type(self):
        entry = gtk.Entry()
        window = self.create_window(entry)
        try:
            window.present()
            entry.grab_focus()
            self.flush_gtk()
            xutils.type("Hello there!")
            self.flush_gtk()
            self.assertEquals(entry.get_text(), "Hello there!")
        finally:
            window.destroy()

    def test_take_screenshot(self):
        """Verify that take_screenshot works, with a real screenshot.

        For that, we will put a known image in the screen, and will then
        try to find it in the screenshot.
        """
        red_square = Image("red-square",
                           filename=get_image_path("red-square.png"))
        window = self.create_image_window(red_square.filename)

        self.flush_gtk()

        resolution = gtk.gdk.get_default_root_window().get_size()

        window_x, window_y = window.get_child().window.get_position()
        window_width, window_height = window.get_child().window.get_size()

        big_screenshot = xutils.take_screenshot()
        small_screenshot = xutils.take_screenshot(window_x, window_y,
                                                  window_width, window_height)

        window.destroy()
        self.flush_gtk()

        # Check the basic attributes set
        self.assertEquals(big_screenshot.name, "screenshot")
        self.assertEquals(big_screenshot.width, resolution[0])
        self.assertEquals(big_screenshot.height, resolution[1])

        self.assertEquals(small_screenshot.name, "screenshot")
        self.assertEquals(small_screenshot.width, window_width)
        self.assertEquals(small_screenshot.height, window_height)

        # Now verify the actual images taken.
        finder = OpenCVFinder()

        big_match = finder.find(big_screenshot, red_square)
        small_match = finder.find(small_screenshot, red_square)

        self.assertEquals(big_match.image, red_square)
        self.assertTrue(big_match.similarity > 0.95, big_match.similarity)

        self.assertEquals(small_match.image, red_square)
        self.assertTrue(small_match.similarity > 0.95, small_match.similarity)

        # The match we found in the big screenshot should be in the same
        # position as the window we created.  Note that this may fail if
        # you have the image opened elsewhere. ;-)
        self.assertEquals(big_match.x, window_x)
        self.assertEquals(big_match.y, window_y)

        # With the small match, it should be in the origin, since the
        # screenshot was taken on the precise area.
        self.assertEquals(small_match.x, 0)
        self.assertEquals(small_match.y, 0)



class XUtilsButtonTest(XUtilsTestBase):

    def setUp(self):
        super(XUtilsButtonTest, self).setUp()
        self.window = self.create_button_window()
        self.button = self.window.get_child()
        self.button_clicked = False
        self.button_rclicked = False
        self.button_hovered = False

        def clicked(widget, event):
            if event.button == 1:
                self.button_clicked = True
            elif event.button == 3:
                self.button_rclicked = True
            self.window.destroy()

        def entered(widget):
            self.button_hovered = True

        self.button.connect("button_press_event", clicked)
        self.button.connect("enter", entered)

        self.flush_gtk()

    def tearDown(self):
        self.window.destroy()

    def get_button_center(self):
        button_x, button_y = self.button.window.get_position()
        button_width, button_height = self.button.window.get_size()
        return (button_x + button_width//2, button_y + button_height//2)

    def test_click(self):
        xutils.click(*self.get_button_center())
        self.flush_gtk()
        self.assertTrue(self.button_clicked)

    def test_right_click(self):
        xutils.right_click(*self.get_button_center())
        self.flush_gtk()
        self.assertTrue(self.button_rclicked)

    def test_hover(self):
        xutils.hover(*self.get_button_center())
        self.flush_gtk()
        self.assertTrue(self.button_hovered)
        self.assertFalse(self.button_clicked)
