/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "lifeograph.hpp"
#include "app_window.hpp"
#include "view_entry.hpp"
#include "printing.hpp"


using namespace LIFEO;


EntryView::EntryView()
:   m_tag_operation_cur( TO_NONE ), m_entry_sync( NULL ),
    m_flag_entrychanged( false )
{
    Entry::shower = this;

    Gtk::Button* button_print;

    Gtk::Button* button_hide( NULL );
    Gtk::Button* button_hide_before( NULL );
    Gtk::Button* button_hide_after( NULL );

    try
    {
        Lifeograph::builder->get_widget( "Bx_entry", m_vbox );
        Lifeograph::builder->get_widget( "G_entry_popover", m_grid_popover );
        Lifeograph::builder->get_widget( "RB_entry_not_todo", m_button_not_todo );
        Lifeograph::builder->get_widget( "RB_entry_todo", m_button_todo );
        Lifeograph::builder->get_widget( "RB_entry_progressed", m_button_progressed );
        Lifeograph::builder->get_widget( "RB_entry_done", m_button_done );
        Lifeograph::builder->get_widget( "RB_entry_canceled", m_button_canceled );
        Lifeograph::builder->get_widget( "L_entry_info", m_label_info );
        Lifeograph::builder->get_widget( "TB_entry_favorite", m_button_toggle_favorite );
        Lifeograph::builder->get_widget( "B_entry_print", button_print );
        Lifeograph::builder->get_widget( "B_entry_filter", button_hide );
        Lifeograph::builder->get_widget( "B_entry_filter_before", button_hide_before );
        Lifeograph::builder->get_widget( "B_entry_filter_after", button_hide_after );
        Lifeograph::builder->get_widget( "B_entry_trash", m_button_trash );
        Lifeograph::builder->get_widget( "B_entry_restore", m_button_restore );
        Lifeograph::builder->get_widget( "B_entry_dismiss", m_button_dismiss );


        Lifeograph::builder->get_widget( "hbox_entry_toolbar_text", m_hbox_editing_toolbar );
        Lifeograph::builder->get_widget( "button_entry_bold", m_button_bold );
        Lifeograph::builder->get_widget( "button_entry_italic", m_button_italic );
        Lifeograph::builder->get_widget( "button_entry_strikethrough", m_button_strikethrough );
        Lifeograph::builder->get_widget( "button_entry_highlight", m_button_highlight );
        Lifeograph::builder->get_widget( "label_entry_highlight", m_label_highlight );
        Lifeograph::builder->get_widget( "evntb_entry_highlight", m_eventbox_highlight );
        Lifeograph::builder->get_widget( "button_entry_add_bullet", m_button_add_bullet );
        Lifeograph::builder->get_widget( "button_entry_add_checkbox", m_button_add_checkbox );
        Lifeograph::builder->get_widget( "tgglbutton_entry_comments", m_button_toggle_comments );
        Lifeograph::builder->get_widget_derived( "textview_entry", m_textview );

        Lifeograph::builder->get_widget( "Bx_entry_tag_edit", m_box_tag_edit );
        Lifeograph::builder->get_widget( "Bx_entry_tag_linked", m_Bx_tag_linked );
        Lifeograph::builder->get_widget_derived( "E_tag_filter", m_tag_widget );
        Lifeograph::builder->get_widget( "B_entry_tag", m_button_tag_operation );
        Lifeograph::builder->get_widget( "B_entry_tag_theme", m_button_set_theme );
        Lifeograph::builder->get_widget_derived( "drawingarea_entry_tags", m_widgettaglist );
    }
    catch( ... )
    {
        throw LIFEO::Error( "Failed to create the entry view" );
    }

    // TEXT VIEW
    m_textview->drag_dest_get_target_list()->add( Lifeograph::p->drag_targets_tag );
    m_textview->drag_dest_get_target_list()->add( Lifeograph::p->drag_targets_entry );

    // TAGS WIDGET
    m_widgettaglist->drag_dest_set( Lifeograph::p->drag_targets_tag );

    // SIGNALS
    m_textview->m_buffer->signal_changed().connect(
            sigc::mem_fun( this, &LIFEO::EntryView::handle_textview_changed ) );

    m_button_not_todo->signal_toggled().connect(
            sigc::bind( sigc::mem_fun( this, &EntryView::set_todo_status ), ES::NOT_TODO ) );
    m_button_todo->signal_toggled().connect(
            sigc::bind( sigc::mem_fun( this, &EntryView::set_todo_status ), ES::TODO ) );
    m_button_progressed->signal_toggled().connect(
            sigc::bind( sigc::mem_fun( this, &EntryView::set_todo_status ), ES::PROGRESSED ) );
    m_button_done->signal_toggled().connect(
            sigc::bind( sigc::mem_fun( this, &EntryView::set_todo_status ), ES::DONE ) );
    m_button_canceled->signal_toggled().connect(
            sigc::bind( sigc::mem_fun( this, &EntryView::set_todo_status ), ES::CANCELED ) );

    m_button_toggle_favorite->signal_toggled().connect(
            sigc::mem_fun( this, &LIFEO::EntryView::toggle_favoredness ) );
    m_button_trash->signal_clicked().connect(
            sigc::mem_fun( this, &LIFEO::EntryView::trash_entry ) );
    m_button_restore->signal_clicked().connect(
            sigc::mem_fun( this, &LIFEO::EntryView::restore_entry ) );
    m_button_dismiss->signal_clicked().connect(
            sigc::mem_fun( this, &LIFEO::EntryView::dismiss_entry ) );
    button_print->signal_clicked().connect(
            sigc::mem_fun( this, &EntryView::print ) );
    button_hide->signal_clicked().connect( sigc::mem_fun( this, &EntryView::hide ) );
    button_hide_before->signal_clicked().connect(
            sigc::mem_fun( this, &EntryView::hide_before ) );
    button_hide_after->signal_clicked().connect(
            sigc::mem_fun( this, &EntryView::hide_after ) );

    m_button_add_bullet->signal_clicked().connect(
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::add_bullet ) );
    m_button_add_checkbox->signal_clicked().connect(
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::add_checkbox ) );

    m_textview->signal_drag_data_received().connect(
            sigc::mem_fun( this, &EntryView::handle_drag_data_received ) );

    m_tag_widget->signal_changed().connect(
            sigc::mem_fun( this, &EntryView::handle_entry_tag_changed ) );
    m_textview->signal_focus_in_event().connect_notify(
            sigc::mem_fun( this, &EntryView::handle_tag_entry_focus_out ) );
    m_tag_widget->signal_activate().connect(
            sigc::mem_fun( this, &EntryView::handle_button_tag_clicked ) );
    m_button_tag_operation->signal_clicked().connect(
            sigc::mem_fun( this, &EntryView::handle_button_tag_clicked ) );
    m_button_set_theme->signal_clicked().connect(
            sigc::mem_fun( this, &EntryView::handle_set_theme ) );
    m_widgettaglist->signal_drag_data_received().connect(
            sigc::mem_fun( this, &EntryView::handle_drag_data_received ) );
    m_widgettaglist->signal_add_tag().connect(
            sigc::mem_fun( this, &EntryView::focus_tag_entry ) );
    m_widgettaglist->signal_tag_selected().connect(
            sigc::mem_fun( this, &EntryView::edit_tag_in_entry ) );

    // EDITOR ACTIONS
    Lifeograph::create_action(
            true, m_action_bold, "Bold", "", "",
            Gtk::AccelKey( GDK_KEY_b, Gdk::CONTROL_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::toggle_bold ),
            m_button_bold );
    Lifeograph::create_action(
            true, m_action_italic, "Italic", "", "",
            Gtk::AccelKey( GDK_KEY_i, Gdk::CONTROL_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::toggle_italic ),
            m_button_italic );
    Lifeograph::create_action(
            true, m_action_strikethrough, "Strikethrough", "", "",
            Gtk::AccelKey( GDK_KEY_s, Gdk::CONTROL_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::toggle_strikethrough ),
            m_button_strikethrough );
    Lifeograph::create_action(
            true, m_action_highlight, "Highlight", "", "",
            Gtk::AccelKey( GDK_KEY_h, Gdk::CONTROL_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::toggle_highlight ),
            m_button_highlight );

    Lifeograph::create_action(
            true, m_action_focus_tag, "FocusTag", "", "",
            Gtk::AccelKey( GDK_KEY_t, Gdk::CONTROL_MASK ),
            sigc::mem_fun( this, &EntryView::focus_tag_entry ) );

    Lifeograph::create_toggle_action(
            false, m_action_toggle_comments, "ToggleComments", "", "",
            Gtk::AccelKey( GDK_KEY_c, Gdk::MOD1_MASK ),
            sigc::mem_fun( this, &EntryView::toggle_comments ),
            m_button_toggle_comments );

    Lifeograph::create_action(
            true, m_action_indent, "Indent", "", "",
            Gtk::AccelKey( GDK_KEY_i, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::handle_indent ) );
    Lifeograph::create_action(
            true, m_action_unindent, "Unindent", "", "",
            Gtk::AccelKey( GDK_KEY_u, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::handle_unindent ) );

    Lifeograph::create_action(
            true, m_action_add_empty_line_above, "AddEmptyLineAbove", "", "",
            Gtk::AccelKey( GDK_KEY_l, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::add_empty_line_above ) );
    Lifeograph::create_action(
            true, m_action_remove_empty_line_above, "RemoveEmptyLineAbove", "", "",
            Gtk::AccelKey( GDK_KEY_l, Gdk::MOD1_MASK|Gdk::SHIFT_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::remove_empty_line_above ) );

    Lifeograph::create_action(
            true, m_action_move_line_up, "MoveLineUp", "", "",
            Gtk::AccelKey( GDK_KEY_Up, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::move_line_up ) );
    Lifeograph::create_action(
            true, m_action_move_line_down, "MoveLineDown", "", "",
            Gtk::AccelKey( GDK_KEY_Down, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::move_line_down ) );

    Lifeograph::create_action(
            true, m_action_time_stamp, "TimeStamp", "", "",
            Gtk::AccelKey( GDK_KEY_t, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::insert_time_stamp ) );
}

void
EntryView::sync()
{
    if( m_flag_entrychanged )
    {
        m_entry_sync->set_date_changed( time( NULL ) );
        m_entry_sync->set_text( m_textview->m_buffer->get_text() );
        m_flag_entrychanged = false;
        PRINT_DEBUG( "entry synced" );
    }
}

void
EntryView::remove_selected_tags()
{
    /*std::vector< Gtk::TreeModel::Path > selected_items =
            m_iconview_tags->get_selected_items();
    for( std::vector< Gtk::TreeModel::Path >::iterator iter = selected_items.begin();
         iter != selected_items.end();
         ++iter )
    {
        Gtk::TreeRow row = *m_liststore_tags->get_iter( *iter );
        DiaryElement* element = row[ PanelExtra::colrec->ptr ];
        Tag* tag = dynamic_cast< Tag* >( element );

        m_ptr2elem->remove_tag( tag );
        m_liststore_tags->erase( row );
    }*/
}

void
EntryView::handle_login()
{
    m_textview->m_buffer->handle_login();

    // READ ONLY
    bool editable( ! Diary::d->is_read_only() );
    m_textview->set_editable( editable );
    m_widgettaglist->set_editable( editable );
    m_button_not_todo->set_visible( editable );
    m_button_todo->set_visible( editable );
    m_button_progressed->set_visible( editable );
    m_button_done->set_visible( editable );
    m_button_canceled->set_visible( editable );
    //m_menuseparator_4->set_visible( editable );
    m_button_toggle_favorite->set_visible( editable );
    //m_menuseparator_1->set_visible( editable );
    //m_menuseparator_2->set_visible( editable );
    m_button_trash->set_visible( editable );
    m_button_restore->set_visible( editable );
    m_button_dismiss->set_visible( editable );
    m_hbox_editing_toolbar->set_visible( Lifeograph::settings.show_formatting_toolbar &&
                                         editable );
}

void
EntryView::handle_textview_changed()
{
    if( Lifeograph::m_internaloperation ) return;

    m_flag_entrychanged = true;

    PRINT_DEBUG( "entry changed" );

    // TODO:
    //if( m_hbox_tagtools->is_visible() )
        //m_hbox_tagtools->hide();
}

void
EntryView::handle_drag_data_received(
                const Glib::RefPtr< Gdk::DragContext >& context,
                int x, int y,
                const Gtk::SelectionData& selection_data,
                uint info,
                guint time )
{
    if( Diary::d->is_read_only() )
        return;

    context->drag_finish( true, false, time );

    if( Lifeograph::s_elem_dragged == NULL )
        return;

    switch( Lifeograph::s_elem_dragged->get_type() )
    {
        case DiaryElement::ET_ENTRY:
            m_textview->m_buffer->insert_link( Lifeograph::s_elem_dragged );
            break;
        case DiaryElement::ET_TAG:
        {
            Tag* tag = dynamic_cast< Tag* >( Lifeograph::s_elem_dragged );
            if( m_ptr2elem->add_tag( tag ) ) // if does not already have the tag
            {
                m_widgettaglist->update();
                if( tag->get_has_own_theme() )
                    update_theme();
                // note: even if tag has its own theme it is not applied if entry has already
                // a theme tag before
            }
            break;
        }
        case DiaryElement::ET_UNTAGGED:
            if( m_ptr2elem->clear_tags() )
            {
                m_widgettaglist->update();
                update_theme();
            }
            break;
        default:
            break;
    }
}

void
EntryView::toggle_favoredness()
{
    if( Lifeograph::m_internaloperation )
        return;

    m_ptr2elem->toggle_favored();
    AppWindow::p->panel_main->update_overlay_icon();
    Gtk::TreeRow row = AppWindow::p->get_element_row( m_ptr2elem );
    row[ ListData::colrec->icon ] = m_ptr2elem->get_icon();
}

void
EntryView::set_todo_status( ElemStatus status )
{
    if( Lifeograph::m_internaloperation )
        return;

    m_ptr2elem->set_todo_status( status );
    AppWindow::p->panel_main->set_icon( m_ptr2elem->get_icon32() );
    Gtk::TreeRow row = AppWindow::p->get_element_row( m_ptr2elem );
    row[ ListData::colrec->icon ] = m_ptr2elem->get_icon();
    row[ ListData::colrec->info ] = m_ptr2elem->get_list_str();
}

void
EntryView::print()
{
    AppWindow::p->panel_main->hide_popover();

    AppWindow::p->m_entry_view->sync();

    Glib::RefPtr< PrintOpr > print = PrintOpr::create();

    try
    {
        AppWindow::p->freeze_auto_logout();
        print->set_hide_comments( m_button_toggle_comments->get_active() );
        print->run( Gtk::PRINT_OPERATION_ACTION_PRINT_DIALOG, *AppWindow::p );
        AppWindow::p->unfreeze_auto_logout();
    }
    catch( const Gtk::PrintError &ex )
    {
        // See documentation for exact Gtk::PrintError error codes:
        print_error( "An error occurred while trying to run a print operation:" +
                ex.what() );
    }
}

void
EntryView::hide()
{
    AppWindow::p->panel_main->hide_popover();

    Diary::d->get_filter()->add_entry( m_ptr2elem );
    AppWindow::p->panel_diary->update_calendar();
    AppWindow::p->panel_diary->update_entry_list();
}

void
EntryView::hide_before()
{
    Diary::d->set_filter_date_begin( m_ptr2elem->get_date_t() );
    AppWindow::p->panel_diary->update_calendar();
    AppWindow::p->panel_diary->update_entry_list();
}

void
EntryView::hide_after()
{
    Diary::d->set_filter_date_end( m_ptr2elem->get_date_t() );
    AppWindow::p->panel_diary->update_calendar();
    AppWindow::p->panel_diary->update_entry_list();
}

void
EntryView::update_theme()
{
    m_textview->m_buffer->set_theme( m_ptr2elem->get_theme() );
    update_highlight_button();
}

void
EntryView::trash_entry()
{
    AppWindow::p->panel_main->hide_popover();

    Diary::d->show();

    AppWindow::p->panel_main->remove_element_from_history( m_ptr2elem );
    m_ptr2elem->set_trashed( true );
    if( Diary::d->get_startup_elem() == m_ptr2elem )
    {
        Diary::d->set_startup_elem( Diary::d->get_id() );
        AppWindow::p->m_diary_view->update_startup_elem();
    }
    Diary::d->get_filter()->set_status_outstanding();

    Lifeograph::m_internaloperation++;

    AppWindow::p->panel_diary->update_calendar();
    AppWindow::p->panel_diary->update_entry_list();

    Lifeograph::m_internaloperation--;
}

void
EntryView::restore_entry()
{
    AppWindow::p->panel_main->hide_popover();

    Diary::d->show();

    m_ptr2elem->set_trashed( false );
    Diary::d->get_filter()->set_status_outstanding();

    Lifeograph::m_internaloperation++;

    AppWindow::p->panel_diary->update_calendar();
    AppWindow::p->panel_diary->update_entry_list();

    Lifeograph::m_internaloperation--;
}

void
EntryView::dismiss_entry()
{
    AppWindow::p->panel_main->hide_popover();

    if( ! AppWindow::p->confirm_dismiss_element( m_ptr2elem ) )
        return;

    // first make sure that entry to be deleted is completely out of business...
    // ...else dark forces may take action and cause SEGFAULT:
    Diary::d->show();

    AppWindow::p->panel_main->remove_element_from_history( m_ptr2elem );
    Diary::d->dismiss_entry( m_ptr2elem );

    Lifeograph::m_internaloperation++;

    AppWindow::p->panel_diary->update_calendar();
    AppWindow::p->panel_diary->update_entry_list();

    Lifeograph::m_internaloperation--;
}

void
EntryView::toggle_comments()
{
    if( AppWindow::p->panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
        m_textview->m_buffer->set_comment_visibility(
                ! m_textview->m_buffer->get_comment_visibility() );
}

bool
EntryView::check_title_applicable( const Glib::ustring& str ) const
{
    return( Date::parse_string( NULL, str ) == OK );
    // TODO communicate the nature of the error to the user (possibly via a tooltip)
}

bool
EntryView::apply_title( const Glib::ustring& str )
{
    Date date( str );
    if( !date.is_set() )
        return false;

    if( !date.is_ordinal() )
        date.reset_order_1();

    Diary::d->set_entry_date( m_ptr2elem, date );
    AppWindow::p->panel_diary->update_entry_list();
    AppWindow::p->panel_diary->select_date_in_calendar( m_ptr2elem->get_date() );

    return true;
}

void
EntryView::handle_entry_tag_changed()
{
    Glib::ustring filter = m_tag_widget->get_text();

    m_button_set_theme->hide();

    if( filter.empty() )
    {
        m_Bx_tag_linked->get_style_context()->remove_class( "linked" );
        m_button_tag_operation->hide();
        m_tag_operation_cur = TO_NONE;

    }
    else
    {
        m_Bx_tag_linked->get_style_context()->add_class( "linked" );
        m_button_tag_operation->show();
        Tag* tag = Diary::d->get_tags()->get_tag( filter );
        if( tag == NULL )
        {
            m_button_tag_operation->set_label( _( "Create Tag" ) );
            m_tag_operation_cur = TO_CREATE_AND_ADD;
        }
        else
        {
            if( m_ptr2elem->get_tags().checkfor_member( tag ) )
            {
                m_button_tag_operation->set_label( _( "Remove Tag" ) );
                m_tag_operation_cur = TO_REMOVE;

                if( tag->get_has_own_theme() && m_ptr2elem->get_theme_tag() != tag )
                    m_button_set_theme->show();
            }
            else
            {
                m_button_tag_operation->set_label( _( "Add Tag" ) );
                m_tag_operation_cur = TO_ADD;
            }
        }
    }
}

void
EntryView::handle_button_tag_clicked()
{
    Tag* tag;
    Glib::ustring filter = m_tag_widget->get_text();

    switch( m_tag_operation_cur )
    {
        case TO_NONE:
            return; // don't even clear
        case TO_REMOVE:
            tag = Diary::d->get_tags()->get_tag( filter );
            m_ptr2elem->remove_tag( tag );
            break;
        case TO_CREATE_AND_ADD:
            tag = Diary::d->create_tag( filter );
            m_ptr2elem->add_tag( tag );
            AppWindow::p->panel_extra->populate();
            m_tag_widget->populate();
            break;
        case TO_ADD:
            tag = Diary::d->get_tags()->get_tag( filter );
            m_ptr2elem->add_tag( tag );
            break;
    }


    if( tag->get_has_own_theme() )
        update_theme();
    m_widgettaglist->update();
    m_tag_widget->set_text( "" );
}

void
EntryView::handle_set_theme()
{
    if( m_tag_operation_cur == TO_REMOVE ) // only tags in entry can be set as the theme
    {
        Tag* tag = Diary::d->get_tags()->get_tag( m_tag_widget->get_text() );

        if( tag->get_has_own_theme() )
        {
            m_ptr2elem->set_theme_tag( tag );
            update_theme();
            m_button_set_theme->set_visible( false );
        }
    }
}

void
EntryView::focus_tag_entry()
{
    if( AppWindow::p->panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
    {
        m_box_tag_edit->show();
        m_tag_widget->set_text( "" );
        m_tag_widget->grab_focus();
    }
}

void
EntryView::handle_tag_entry_focus_out( GdkEventFocus* )
{
    m_box_tag_edit->set_visible( false );
}

void
EntryView::edit_tag_in_entry( const Tag* tag )
{
    if( AppWindow::p->panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY &&
        !Diary::d->is_read_only() )
    {
        m_box_tag_edit->show();
        m_tag_widget->set_text( tag->get_name() );
        m_tag_widget->grab_focus();
    }
}

void
EntryView::show( Entry& entry )
{
    // do nothing if entry is already the current element:
    if( AppWindow::p->panel_main->is_cur_elem( &entry ) )
        return;

    m_ptr2elem = &entry; // must be first thing

    m_box_tag_edit->set_visible( false );
    AppWindow::p->panel_main->show( this );

    m_label_info->set_markup( get_extended_info_str() );

    Lifeograph::m_internaloperation++;

    if( ! Diary::d->is_read_only() )
    {
        m_button_toggle_favorite->set_active( entry.is_favored() );
        m_button_trash->set_visible( !entry.is_trashed() );
        m_button_restore->set_visible( entry.is_trashed() );
        m_button_dismiss->set_visible( entry.is_trashed() );

        switch( entry.get_todo_status() )
        {
            case ES::NOT_TODO:
                m_button_not_todo->set_active();
                break;
            case ES::TODO:
                m_button_todo->set_active();
                break;
            case ES::PROGRESSED:
                m_button_progressed->set_active();
                break;
            case ES::DONE:
                m_button_done->set_active();
                break;
            case ES::CANCELED:
                m_button_canceled->set_active();
                break;
        }
    }

    // TOOLBAR
    update_highlight_button();

    // BODY
    m_textview->set_richtext( &entry );

    Lifeograph::m_internaloperation--;

    m_widgettaglist->set_entry( &entry );
    m_tag_widget->populate();

    m_entry_sync = m_ptr2elem;

    PRINT_DEBUG( "entry shown" );
}

void
EntryView::prepare_for_hiding( Entry& entry )
{
    sync();
    m_tag_widget->set_text( "" );
}

void
EntryView::update_formatting_toolbar()
{
    m_hbox_editing_toolbar->set_visible( Lifeograph::settings.show_formatting_toolbar &&
                                         Diary::d->is_read_only() == false );
}

void
EntryView::update_highlight_button()
{
    const Theme* theme = m_ptr2elem->get_theme();
    m_label_highlight->override_color( theme->color_text, Gtk::STATE_FLAG_NORMAL );
    m_label_highlight->override_color( theme->color_text, Gtk::STATE_FLAG_PRELIGHT );
    m_label_highlight->override_color( theme->color_text, Gtk::STATE_FLAG_ACTIVE );
    m_eventbox_highlight->override_background_color(
            theme->color_highlight, Gtk::STATE_FLAG_NORMAL );
    m_eventbox_highlight->override_background_color(
            theme->color_highlight, Gtk::STATE_FLAG_PRELIGHT );
    m_eventbox_highlight->override_background_color(
            theme->color_highlight, Gtk::STATE_FLAG_ACTIVE );
}
