#!/usr/bin/python
# -*- coding: utf-8 -*-
### BEGIN LICENSE
# Copyright (C) 2009 Jono Bacon <jono@ubuntu.com>
# Copyright (C) 2010 Michael Budde <mbudde@gmail.com>
# Copyright (c) 2011 John S Gruber <johnsgruber@gmail.com>
#
#This program is free software: you can redistribute it and/or modify it
#under the terms of the GNU General Public License version 3, as published
#by the Free Software Foundation.
#
#This program is distributed in the hope that it will be useful, but
#WITHOUT ANY WARRANTY; without even the implied warranties of
#MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
#PURPOSE.  See the GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License along
#with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import glib
import gobject
import gtk
import os
import time
import urlparse
import random
import pynotify
import logging
from datetime import timedelta

import lernid.DateTime as dt
from lernid.widgets.Widget import Widget
from lernid.lernidconfig import get_data_path
from lernid.Sessions import Session, parse_ical, read_ical, start_read_ical


class Schedule(Widget):

    __gtype_name__ = 'LernidSchedule'

    __gsignals__ = {
        'session-changed': (
            gobject.SIGNAL_RUN_LAST, None, (object,)
        ),
        'session-ended': (
            gobject.SIGNAL_RUN_LAST, None, ()
        ),
    }

    COL_ICON, COL_DATE, COL_START, COL_END, COL_TITLE, COL_SESSION = range(6)

    def __init__(self):
        Widget.__init__(self, 'schedule')
        viewport = gtk.Viewport()
        scroll = gtk.ScrolledWindow()
        scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        viewport.set_shadow_type(gtk.SHADOW_NONE)
        viewport.add(scroll)
        self.add(viewport)

        self._treeview = gtk.TreeView()
        scroll.add(self._treeview)
        self._model = gtk.ListStore(str, str, str, str, str, object)
        self._treeview.set_model(self._model)
        self._treeview.set_search_column(self.COL_TITLE)
        def search_equal_func(model, column, key, iter):
            if key.upper() in model.get_value(iter, column).upper():
                return False
            return True
        self._treeview.set_search_equal_func(search_equal_func)
        self._treeview.set_headers_visible(False)

        column = gtk.TreeViewColumn()
        cell = gtk.CellRendererPixbuf()
        cell.set_property('xalign', 0.5)
        column.pack_start(cell, True)
        column.set_attributes(cell, stock_id=self.COL_ICON)
        column.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
        column.set_fixed_width(30)
        self._treeview.append_column(column)

        column = gtk.TreeViewColumn(_('Date'))
        cell = gtk.CellRendererText()
        cell.set_property('xpad', 6)
        column.pack_start(cell, False)
        column.set_attributes(cell, text=self.COL_DATE)
        column.set_alignment(0.5)
        self._treeview.append_column(column)

        column = gtk.TreeViewColumn(_('Starts'))
        cell = gtk.CellRendererText()
        cell.set_property('xpad', 6)
        column.pack_start(cell, False)
        column.set_attributes(cell, text=self.COL_START)
        column.set_alignment(0.5)
        self._treeview.append_column(column)

        column = gtk.TreeViewColumn(_('Ends'))
        cell = gtk.CellRendererText()
        cell.set_property('xpad', 6)
        column.pack_start(cell, False)
        column.set_attributes(cell, text=self.COL_END)
        column.set_alignment(0.5)
        self._treeview.append_column(column)

        column = gtk.TreeViewColumn(_('Title'))
        cell = gtk.CellRendererText()
        cell.set_property('xpad', 6)
        column.pack_start(cell, True)
        column.set_attributes(cell, markup=self.COL_TITLE)
        label = gtk.Label(_('Title'))
        label.set_padding(6, 0)
        label.show()
        column.set_widget(label)
        self._treeview.append_column(column)

        self._schedule = None
        self._current_session = None
        self._update_handle = None
        self._ical = None
        self._icalurl = None
        self._connected = False

        self.show_all()

    def do_event_connect(self, event_man, event):
        self._event = event_man.current_event()
        self._connected = True
        def finished_initial_ical_load(gfile, result):
            self._ical = read_ical(gfile, result, (event,self._ical))
            logging.debug('Finished initial ical load')
            if not (gfile.get_uri() == self._icalurl):
                logging.debug('Old I/O completion ignored')
                return
            self._update(event, self._ical)
            secs = time.localtime().tm_sec
            self._update_handle = glib.timeout_add_seconds(65-secs, set_timeout)
        def scramble_case(string):
            random.seed()
            newstring = ''
            for s in string:
                if random.random() >= .5:
                   newstring += s.upper()
                else:
                   newstring += s.lower()
            return newstring
        # scramble the case of the url for this event connection.
        # Avoids occasional problem with gvfsd-http wedging 
        # particular url's after loss of connectivity
        parsed = [j for j in urlparse.urlparse(event.icalurl)]
        parsed[1]= scramble_case(parsed[1])
        self._icalurl = urlparse.urlunparse(parsed)
        start_read_ical(self._icalurl, finished_initial_ical_load)
        logging.debug('Started initial ical load %s' % self._icalurl)

        def set_timeout():
            if self._connected:
                self._update_handle = glib.timeout_add_seconds(60, self._calendar_refresh, event, self._ical)
            # Wait at least one minute before updating again,
            # to avoid duplicate notifications.
            logging.debug("In set_timeout")
            return False

    def _calendar_refresh(self, event, ical):
        def finished_refresh_ical_load(gfile, result):
            self._ical = read_ical(gfile, result, (event, self._ical))
            if not (gfile.get_uri() == self._icalurl):
                logging.debug('Old I/O completion ignored')
                return
            logging.debug('Finished ical load')
            if not self._connected:
                return
            self._update(event, self._ical)
            secs = time.localtime().tm_sec
            self._update_handle = glib.timeout_add_seconds(65-secs, self._calendar_refresh, event, self._ical) 
            return
        start_read_ical(self._icalurl, finished_refresh_ical_load)
        logging.debug('Started ical load')
        return False

    def _update(self, event, ical):
        def quote(text):
            quote_table = {
                 '<' : '&lt;',
                 '>' : '&gt;',
                 "'" : '&apos;',
                 '"' : '&quot;',
                 '&' : '&amp;'
                }
            return_value = ''
            for l in text:
                return_value= return_value + quote_table.get(l,l)
            return return_value

        self._schedule = parse_ical(event, ical)
        self._model.clear()
        self._scroll_to = None
        for session in self._schedule:
            sessionrow = []
            sessionrow.append('')
            sessionrow.append(session.start_local_date)
            sessionrow.append(session.start_local_time)
            sessionrow.append(session.end_local_time)
            instructors = reduce(lambda x, y: x + y + ', ', session.instructors, '  ')[:-2]
            title = quote(session.title)
            if session.event:
                sessionrow.append(title + '<small>' + instructors + '</small> <i>' + quote(session.event) +'</i>')
            else:
                sessionrow.append(title + '<small>' + instructors + '</small>')
            sessionrow.append(session)
            current_row = self._model.append(sessionrow)
            if not self._scroll_to and session.state in (session.FUTURE, session.NOW):
                self._scroll_to = self._model.get_path(current_row)
        self._update_currency()

        if self._scroll_to:
            self._treeview.scroll_to_cell(self._scroll_to, use_align=True, row_align=.2)


        self._treeview.set_headers_visible(True)

    def do_event_disconnect(self, event_man, event):
        self._connected = False
        self._schedule = []
        self._current_session = None
        self._model.clear()
        if self._update_handle:
            glib.source_remove(self._update_handle)
        self._treeview.set_headers_visible(False)

    def _update_currency(self):
        ended = False
        if self._current_session and self._current_session.state == Session.PAST:
            ended = True
            self._current_session = None
        for i, row in enumerate(self._model):
            session = row[self.COL_SESSION]
            if self._current_session and self._current_session.uid == session.uid:
                self._current_session = session  
            if session.state == Session.NOW:
                row[self.COL_ICON] = gtk.STOCK_GO_FORWARD
                if session != self._current_session:
                    if self._current_session == None:
                        session_message = _("Please keep the Session "
                        "tab selected to see session"
                        " learning materials.")
                        self.show_notification(
                            _('Session started'),
                            _('The session "{0}" has started. ').format(session.title) +
                            session_message
                        )
                    self._current_session = session
                    logging.debug('session changed')
                    self.emit('session-changed', session)
            if session.state == Session.FUTURE:
                row[self.COL_ICON] = None
                if timedelta(minutes=9) < session.start_datetime - dt.now_local() < timedelta(minutes=10):
                    self.show_notification(
                        _('Session upcoming'),
                        _('The session "{0}" will begin in 10 minutes.').format(session.title)
                    )
            if session.state == Session.PAST:
                row[self.COL_ICON] = gtk.STOCK_NO
        if not self._current_session and ended:
            logging.debug('session ended')
            self.emit('session-ended')
        return True

    def get_current_session(self):
        return self._current_session

    def on_faculty(self,name):
        return self._current_session and name.lower() in \
	[i.lower() for i in self._current_session.instructors + self._current_session.helpers]

    def get_question_token(self):
        if self._current_session:
            if self._current_session.question_token:
                return self._current_session.question_token
            if self._current_session.locale:
                if self._current_session.locale.startswith('es'):
                    return 'PREGUNTA:'
                else:
                    return 'QUESTIION:'
        try:
            event = self._event
        except:
            event = None # Called when no event is opened
        if event and event.question_token:
            return event.question_token
        if event and event.locale:
            if event.locale.startswith('es'):
                return 'PREGUNTA:'
            return 'QUESTION:'
        return 'QUESTION:'

    def show_notification(self, summary, body):
        n = pynotify.Notification(summary, body, 'lernid')
        try:
            n.show()
        except:
            logging.debug("Notification failed from Schedule Widget")
