# +==========================================================================+
# || CipUX::RBAC::Simple                                                    ||
# ||                                                                        ||
# || Somple CipUX RBAC Role Based Access Control                            ||
# ||                                                                        ||
# || Copyright (C) 2009 by Christian Kuelker. All rights reserved!          ||
# ||                                                                        ||
# || License: GNU General Public License - GNU GPL version 2                ||
# ||          or (at your opinion) any later version                        ||
# ||                                                                        ||
# +==========================================================================+
# ID:       $Id$
# Revision: $Revision$
# Head URL: $HeadURL$
# Date:     $Date$
# Source:   $Source$

package CipUX::RBAC::Simple;

use 5.008001;
use strict;
use warnings;
use Carp;
use CipUX::Task;
use Class::Std;
use Data::Dumper;
use English qw( -no_match_vars);
use Graph;
use Graph::Directed;
use Log::Log4perl qw(get_logger :levels);
use Readonly;

use base qw(CipUX);

{    # BEGIN CLASS

    use version; our $VERSION = qv('3.4.0.0');
    use re 'taint';    # Keep data captured by parens tainted
    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};    # Make %ENV safe

    # +======================================================================+
    # || CONST                                                              ||
    # +======================================================================+
    Readonly::Scalar my $EMPTY_STRING => q{};
    Readonly::Scalar my $SCRIPT       => 'CipUX::RBAC::Simple';

    # +======================================================================+
    # || OBJECT                                                             ||
    # +======================================================================+

    # +======================================================================+
    # || GLOBAL                                                             ||
    # +======================================================================+
    my $cipux = undef;

    my %cat_module = ();
    my %rpc_intern = ();
    my %task       = ();

    # +======================================================================+
    # || DESTRUCTOR                                                         ||
    # +======================================================================+
    sub DEMOLISH {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $ident ) = @_;

        # +------------------------------------------------------------------+
        # | main
        undef $cipux;

        undef %cat_module;
        undef %rpc_intern;
        undef %task;

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    # +======================================================================+
    # || open module features                                               ||
    # +======================================================================+

    # +======================================================================+
    # || access_to_rpc_intern                                               ||
    # +======================================================================+
    sub access_to_rpc_intern {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $user
            = exists $arg_r->{user}
            ? $self->l( $arg_r->{user} )
            : $self->perr('user');

        my $role
            = exists $arg_r->{role}
            ? $self->l( $arg_r->{role} )
            : $self->perr('role');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $key = "$user $role";
        $logger->debug(" key [$key]");

        # +------------------------------------------------------------------+
        # | API
        return $rpc_intern{$key} if exists $rpc_intern{$key};

        $logger->debug(" key [$key] do not exist");
        my $g  = Graph::Directed->new;
        my $t3 = 'cipux_task_list_members_of_role_account';
        my $a3 = 'memberUid';

        $g->add_vertex($role);

        #my @user = $self->query( {
        #task => $t3, attr => $a3, object => $group
        #} );
        my @user = query( $t3, $a3, $role );
        foreach my $uid (@user) {
            $logger->debug("RBAC: user [$user]");
            $g->add_vertex($uid);
            $g->add_edge( $role, $uid );
        }

        my $tcg = $g->TransitiveClosure_Floyd_Warshall;

        #                        destination, source
        if ( $tcg->is_reachable( $role, $user ) ) {
            $logger->debug('1 rpc_intern access TRUE (1)');
            $logger->debug('END');
            $rpc_intern{$key} = 1;
            return 1;
        }
        else {
            $logger->debug('rpc_intern access FALSE (0)');
            $logger->debug('END');
        }
        $rpc_intern{$key} = 0;

        # +------------------------------------------------------------------+
        # | API
        return 0;

    }

    # +======================================================================+
    # || access_to_task                                                     ||
    # +======================================================================+
    sub access_to_task {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $task
            = exists $arg_r->{task}
            ? $self->l( $arg_r->{task} )
            : $self->perr('task');

        my $user
            = exists $arg_r->{user}
            ? $self->l( $arg_r->{user} )
            : $self->perr('user');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $key = "$user $task";
        $logger->debug(" key [$key]");

        # +------------------------------------------------------------------+
        # | API
        return $task{$key} if exists $task{$key};

        $logger->debug(" key [$key] do not exist");

        my $g  = Graph::Directed->new;
        my $t1 = 'cipux_task_list_members_of_task';
        my $t2 = 'cipux_task_list_members_of_cat_module';
        my $t3 = 'cipux_task_list_members_of_role_account';
        my $a1 = 'cipuxMemberPid';
        my $a2 = 'cipuxMemberRid';
        my $a3 = 'memberUid';

        # my @cat = $self->query( {
        # task => $t1, attr => $a1, object => $task
        # } );
        my @cat = query( $t1, $a1, $task );
        my @group = ();

        $g->add_vertex($task);
        foreach my $module (@cat) {
            $logger->debug("RBAC: module [$module]");
            $g->add_vertex($module);
            $g->add_edge( $task, $module );

            #my @role = $self->query( {
            # task => $t2, attr => $a2, object => $module
            #} );
            my @role = query( $t2, $a2, $module );
            foreach my $group (@role) {
                $logger->debug("RBAC: group [$group]");
                $g->add_vertex($group);
                $g->add_edge( $module, $group );
                $g->add_vertex($group);

                #my @user = $self->query( {
                #task => $t3, attr => $a3, object => $group
                #} );
                my @user = query( $t3, $a3, $group );
                foreach my $uid (@user) {
                    $logger->debug("RBAC: user [$user]");
                    $g->add_vertex($uid);
                    $g->add_edge( $group, $uid );
                }

            }
        }

        my $tcg = $g->TransitiveClosure_Floyd_Warshall;

        #                        destination, source
        if ( $tcg->is_reachable( $task, $user ) ) {
            $logger->debug('1 access TRUE (1)');
            $logger->debug('END');
            $task{$key} = 1;
            return 1;
        }
        else {
            $logger->debug('access FALSE (0)');
            $logger->debug('END');
        }

        $task{$key} = 0;

        # +------------------------------------------------------------------+
        # | API
        return 0;
    }

    # +======================================================================+
    # || access_to_cat_module                                               ||
    # +======================================================================+
    sub access_to_cat_module {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $cat_module
            = exists $arg_r->{cat_module}
            ? $self->l( $arg_r->{cat_module} )
            : $self->perr('cat_module');

        my $user
            = exists $arg_r->{user}
            ? $self->l( $arg_r->{user} )
            : $self->perr('user');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $key = "$user $cat_module";
        $logger->debug(" key [$key]");

        # +------------------------------------------------------------------+
        # | API
        return $cat_module{$key} if exists $cat_module{$key};

        $logger->debug(" key [$key] do not exist");

        my $g  = Graph::Directed->new;
        my $t2 = 'cipux_task_list_members_of_cat_module';
        my $t3 = 'cipux_task_list_members_of_role_account';
        my $a2 = 'cipuxMemberRid';
        my $a3 = 'memberUid';

        $logger->debug("RBAC: module [$cat_module]");
        $g->add_vertex($cat_module);

        my @role = query( $t2, $a2, $cat_module );
        foreach my $group (@role) {
            $logger->debug("RBAC: group [$group]");
            $g->add_vertex($group);
            $g->add_edge( $cat_module, $group );
            $g->add_vertex($group);

            my @user = query( $t3, $a3, $group );
            foreach my $uid (@user) {
                $logger->debug("RBAC: user [$user]");
                $g->add_vertex($uid);
                $g->add_edge( $group, $uid );
            }

        }

        my $tcg = $g->TransitiveClosure_Floyd_Warshall;

        #                        destination, source
        if ( $tcg->is_reachable( $cat_module, $user ) ) {
            $logger->debug('1 access cat_module TRUE (1)');
            $logger->debug('END');
            $cat_module{$key} = 1;
            return 1;
        }
        else {
            $logger->debug('access cat_module FALSE (0)');
            $logger->debug('END');
        }

        $cat_module{$key} = 0;

        # +------------------------------------------------------------------+
        # | API
        return 0;
    }

    # +======================================================================+
    # || query                                                              ||
    # +======================================================================+
    sub query {

        # +------------------------------------------------------------------+
        # | API
        my $task   = shift;
        my $attr   = shift;
        my $object = shift;

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        $logger->debug('using task command');

        # everything is packed we will start the journey
        my %mattrvalue = ();
        if ( defined $object ) {
            $logger->debug("exec task: $task -o $object");
        }
        else {
            $logger->debug("exec task: $task");
        }

        $cipux = CipUX::Task->new();
        $logger->debug("cipux [$cipux]");
        my $return_hr = $cipux->task(

            {
                script  => $SCRIPT,
                task    => $task,
                mode    => 'cmd',
                object  => $object,
                attr_hr => \%mattrvalue,

            }
        );

        $logger->debug( 'TASK return_hr:',
            { filter => \&Dumper, value => $return_hr } );

        my @answer = keys %{ $return_hr->{taskres_r} };

        my @list = ();
        foreach my $a (@answer) {
            if (    exists $return_hr->{taskres_r}
                and defined $a
                and exists $return_hr->{taskres_r}->{$a}
                and defined $return_hr->{taskres_r}->{$a}
                and defined $attr
                and exists $return_hr->{taskres_r}->{$a}->{$attr}
                and defined $return_hr->{taskres_r}->{$a}->{$attr}
                and ref $return_hr->{taskres_r}->{$a}->{$attr} eq 'ARRAY' )
            {
                push @list, @{ $return_hr->{taskres_r}->{$a}->{$attr} };
            }
        }

        # +------------------------------------------------------------------+
        # | API
        return @list;
    }

    # +======================================================================+
    # || flush                                                              ||
    # +======================================================================+
    sub flush {

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        %cat_module = ();
        %rpc_intern = ();
        %task       = ();
        $logger->debug('cache flushed');
        $logger->debug('END');

        my @cat_module_size = ();
        @cat_module_size = keys %cat_module;
        return 0 if scalar @cat_module_size > 0;
        undef @cat_module_size;

        my @rpc_intern_size = ();
        @rpc_intern_size = keys %rpc_intern;
        return 0 if scalar @rpc_intern_size > 0;
        undef @rpc_intern_size;

        my @task_size = ();
        @task_size = keys %task;
        return 0 if scalar @task_size > 0;
        undef @task_size;

        # +------------------------------------------------------------------+
        # | API
        return 1;
    }

    # +======================================================================+
    # || cat_module_cache_size                                              ||
    # +======================================================================+
    sub cat_module_cache_size {

        my @size = keys %cat_module;
        my $size = scalar @size;

        undef @size;

        # +------------------------------------------------------------------+
        # | API
        return $size;
    }

    # +======================================================================+
    # || rpc_intern_cache_size                                              ||
    # +======================================================================+
    sub rpc_intern_cache_size {

        my @size = keys %rpc_intern;
        my $size = scalar @size;

        undef @size;

        # +------------------------------------------------------------------+
        # | API
        return $size;
    }

    # +======================================================================+
    # || task_cache_size                                                    ||
    # +======================================================================+
    sub task_cache_size {

        my @size = keys %task;
        my $size = scalar @size;

        undef @size;

        # +------------------------------------------------------------------+
        # | API
        return $size;
    }

}    # END INSIDE-OUT CLASS

1;

__END__

=pod

=for stopwords CipUX::RBAC::Simple RBAC CipUX RPC XML-RPC CipUX::RPC Kuelker destructor

=head1 NAME

CipUX::RBAC::Simple - RBAC class for CipUX

=head1 VERSION

version 3.4.0.0

=head1 SYNOPSIS

  use CipUX::RBAC::Simple;

=head1 DESCRIPTION

Provides functions for Role Based Access Control.

=head1 ABSTRACT

The CipUX RBAC Simple class provides services to CipUX XML-RPC (CipUX::RPC)
server.

=head1 CONFIGURATION AND ENVIRONMENT

Not needed.

=head1 DEPENDENCIES

 Carp
 CipUX
 CipUX::Task
 Class::Std
 Data::Dumper
 English
 Graph
 Graph::Directed
 Log::Log4perl
 Readonly
 version


=head1 SUBROUTINES/METHODS

The following functions will be exported by CipUX::RBAC::Simple.

=head2 DEMOLISH

destructor

=head2 access_to_rpc_intern( { user=>$user, role=>$role } )

Return 1 on access 0 otherwise.

=head2 access_to_task( { user=>$user, task=>$task } )

Return 1 on access 0 otherwise.

=head2 access_to_cat_module( { user=>$user, cat_module=>$cat_module } )

Return 1 on access 0 otherwise.

=head2 query( $task, $attr, $object )

Queries the storage layer and returns a Perl list as the answer from the task
command.

=head2 flush

Flush the cat_module, rpc_intern, task cache. Returns 1 on success 0 if
flushing of one cache fails.

=head2 cat_module_cache_size

Returns number of cache entries.

=head2 rpc_intern_cache_size

Returns number of cache entries.

=head2 task_cache_size

Returns number of cache entries.


=head1 DIAGNOSTICS

Do not have specific messages. It uses perr() from CipUX. See Perl Module CipUX
for more information about perr().

=head2 perr():

I<access_to_rpc_intern>: user, role

I<access_to_task>: user, task

I<access_to_cat_module>: user, cat_module


=head1 INCOMPATIBILITIES

Not known.

=head1 BUGS AND LIMITATIONS

No bugs have been reported.

=head1 SEE ALSO

See the CipUX web page and the manual at L<http://www.cipux.org>

See the mailing list L<http://sympa.cipworx.org/wws/info/cipux-devel>

=head1 AUTHOR

Christian Kuelker  E<lt>christian.kuelker@cipworx.orgE<gt>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2009 by Christian Kuelker

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA

=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE IS WITH
YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL
NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE SOFTWARE AS PERMITTED BY THE ABOVE LICENSE, BE
LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE
THE SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.


=cut

