/*
 * autotrust.c - implements automated updates of DNSSEC Trust Anchors (RFC5011).
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#include <config.h>

#include "src/autotrust.h"
#include "src/options.h"
#include "src/util.h"

/* Our environment in which we can perform automated updates of trust anchors */
autotrust_t* env = NULL;

/* Close autotrust environment and exit program:
 * @param status: return status
 */
static void
free_environment(int status)
{
	if (status != STATUS_OK)
		fprintf(stderr, "[%s] exit status: %s\n",
					AUTOTRUST_NAME, status_by_id(status));
	if (env)
	{
		free_trustpoints(env->trustpoints);
		free_options(env->options);
		free(env);
	}
	close_logfile();
	exit(status);
}

/* Initialize autotrust environment:
 * @return: environment on success, NULL if allocation of environment failed.
 */
static autotrust_t*
initialize_environment(void)
{
	autotrust_t* new_env = (autotrust_t*) malloc(sizeof(autotrust_t));
	if (!new_env)
		return NULL;
	/* options */
	new_env->configfile = DEFAULT_CONFIGFILE;
	new_env->options = initialize_options();
	if (!new_env->options)
		return NULL;
	/* trust points */
	new_env->trustpoints = create_trustpoint_set();
	if (new_env->trustpoints == NULL)
		return NULL;
	/* nothing changed (yet) */
	new_env->changed = 0;
	/* no signal received (yet) */
	new_env->sig_quit = 0;
	/* default log to stderr, verbosity 1 */
	open_logfile(NULL, 1);
	/* done */
	return new_env;
}

/** Print usage */
static void
usage(void)
{
	fprintf(stderr, "Usage: %s [OPTIONS]\n", AUTOTRUST_NAME);
	fprintf(stderr, "Automated updates of trust anchors (RFC 5011).\n\n");
	fprintf(stderr,
		"Supported options:\n"
		"  -c configfile        Read specified <configfile> instead of "
		"the default.\n"
		"  -d                   Run %s as daemon.\n", AUTOTRUST_NAME);
	fprintf(stderr,
		"  -h                   Print this help information.\n"
		"  -v verbosity-level   Set the verbosity level to "
		"<verbosity-level>.\n"
	);

	fprintf(stderr, "Version %s, Report bugs to <%s>.\n",
		PACKAGE_VERSION, PACKAGE_BUGREPORT);
}

/** Write pidfile */
static int
writepid(char* pidfile, pid_t pid)
{
	FILE * fd;
	char pidbuf[32];
	size_t result = 0, size = 0;

	snprintf(pidbuf, sizeof(pidbuf), "%lu\n", (unsigned long) pid);

	if ((fd = fopen(pidfile, "w")) ==  NULL )
	{
		error("cannot open pidfile %s: %s", pidfile, strerror(errno));
		return -1;
	}

	size = strlen(pidbuf);
	if (size == 0)
		result = 1;
	result = fwrite((const void*) pidbuf, 1, size, fd);
	if (result == 0)
	{
		error("write to pidfile failed: %s", strerror(errno));
		result = 0;
    }
	else if (result < size)
	{
		error("short write to pidfile (disk full?)");
        result = 0;
	}
	else
		result = 1;

	if (!result)
	{
		error("cannot write pidfile %s: %s", pidfile, strerror(errno));
		fclose(fd);
		return -1;
	}

	fclose(fd);
	/* chown? */

    return 0;
}

/** Signal handling. */
void
sig_handler(int sig)
{
	switch (sig)
	{
		case SIGTERM:
			debug(("sigterm received"));
			env->sig_quit = 1;
			break;
		case SIGHUP:
			debug(("sighup received"));
			env->sig_quit = 1;
			break;
		default:
			break;
	}
	return;
}

/** Daemonize setup. */
static pid_t
setup_daemon(void)
{
	pid_t pid = -1;
	struct sigaction action;

	switch ((pid = fork()))
	{
		case 0: /* child */
			break;
		case -1: /* error */
			error("fork() failed: %s", strerror(errno));
			free_environment(1);
		default: /* parent is done */
			debug(("fork() ok, parent quit"));
			free_environment(0);
	}
	if (setsid() == -1)
	{
		error("setsid() failed: %s", strerror(errno));
		free_environment(1);
	}

	/* setup signal handing */
	action.sa_handler = sig_handler;
	sigfillset(&action.sa_mask);
	action.sa_flags = 0;
	sigaction(SIGHUP, &action, NULL);

	pid = getpid();
	if (writepid(env->options->pidfile, pid) == -1)
		free_environment(1);

	return pid;
}

/** Poll the authoritative servers for DNSKEYs */
static int
active_refresh(tp_t* tp)
{
	struct ub_ctx* ctx = NULL;
	int return_status = STATUS_OK;

	verbos(2, "active refresh for zone %s", tp->name);
	ctx = ub_ctx_create();
	if (!ctx)
	{
		error("failed to create unbound context, aborting "
		      "active refresh for %s", tp->name);
		return STATUS_ERR_QUERY;
	}

	/* configuration settings */
	return_status = set_resolver_options(env->options, tp, ctx);
	if (return_status != STATUS_OK)
		warning("could not set libunbound resolver options: %s, continuing "
				"with default settings", status_by_id(return_status));
	/* resolve and auto update */
	return_status = query_dnskeys(ctx, tp);
	if (return_status != STATUS_OK)
		error("active refresh for zone %s failed: %s", tp->name,
			status_by_id(return_status));
	else /* perform RFC 5011 statetable */
		return_status = do_statetable(tp, env->options);

	if (return_status == STATUS_CHANGED)
	{
		env->changed = 1;
		return_status = STATUS_OK;
	}
	if (return_status != STATUS_OK)
		error("failed to update state for %s: %s", tp->name,
			status_by_id(return_status));
	/* delete context */
	ub_ctx_delete(ctx);

	return return_status;
}

/** Update the files on disk. */
static int
update_files(void)
{
	int return_status = STATUS_OK;
	int return_status2 = STATUS_OK;

	debug(("update files..."));
	return_status = update_state(env->trustpoints, env->options);
	if (return_status != STATUS_OK)
		error("could not update state file: %s", status_by_id(return_status));
	if (return_status == STATUS_OK && env->changed)
	{
		return_status = update_trustanchor_files(env->trustpoints,
			env->options);
		if (return_status != STATUS_OK)
			error("could not update (all) trust anchor files: %s",
						status_by_id(return_status));
		else
		{
			return_status2 = update_trustedkeys_files(env->trustpoints,
				env->options->trustedkeysfiles);
			if (return_status2 != STATUS_OK)
			{
				error("could not update (all) trusted keys files");
				return_status = return_status2;
			}
		}
	}
	return return_status;
}

/** Do a single run over all trust points. */
static int
do_single_run(time_t* wait)
{
	int return_status = STATUS_OK;
	tp_t* tp = NULL;
	rbnode_t* node = NULL;
	time_t now = time(NULL), refresh = 0;
	strlist_t* list = NULL;

	/* setting: state-file: load state from this file */
	return_status = load_state(env->trustpoints, env->options->statefile);
	if (return_status != STATUS_OK)
		error("could not load state: %s", status_by_id(return_status));
	debug(("trust anchors loaded"));

#ifdef TDEBUG
	debug_print_trustpoints(env->trustpoints);
#endif /* TDEBUG */

	/* resolve and update */
	debug(("start updating"));
	if (env->trustpoints)
		node = rbt_first(env->trustpoints->sep);
	while (node != NULL)
	{
		/* trust point */
		tp = (tp_t*) node->key;
		if (!tp)
		{
			node = rbt_successor(node);
			continue;
		}
		/* calculate first next refresh */
		refresh = next_refresh(tp);
		if (now < refresh)
		{
			*wait = min(*wait, refresh);
			node = rbt_successor(node);
			continue;
		}

		/* query for new DNSKEYs */
		return_status = active_refresh(tp);

		/* update wait period */
		refresh = next_refresh(tp);
		if (now < refresh)
			*wait = min(*wait, refresh);

		/* next zone */
		node = rbt_successor(node);
	}

	/* update trust anchor files */
	return_status = update_files();

	/* setting: resolver-pidfile: signal resolver if trust
	 * anchors were updated.
	 */
	if (return_status == STATUS_OK && env->options)
	{
		list = env->options->resolvers;
		while (list)
		{
			signal_resolver(list->str);
			list = list->next;
		}

		list = env->options->reloadcmd;
		while (list)
		{
			reload_resolver(list->str);
			list = list->next;
		}
    }

	return return_status;
}

/*
 * Main routine:
 * Set configuration settings
 * Change to the working directory
 * - working directory is current directory by default
 * - configuration file is relative to the current directory
 * - configuration settings are relative to the working directory
 * Open logfile
 * - by default, log messages are outputted to stderr
 * - log messages before the logfile is openend are outputted to stderr
 * Daemonize?
 * Load trust anchors from files
 *
 * For each zone:
 *	Query for new DNSKEYs
 *	Execute update algorithm
 * Update trust anchor files
 * Signal the resolver in case of any changes in the configured trust anchors
 *
 * @param argc: number of commandline arguments.
 * @param argv: array of commandline arguments.
 * @return: exit status of the program.
 */
int
main(int argc, char* argv[])
{
	/* some necessary variables */
	int c, r, running;
	int return_status = STATUS_OK;
	time_t wait = 0, now = time(NULL);
	pid_t pid;

	/* initialize the autotrust environment */
	env = initialize_environment();
	if (!env)
		free_environment(STATUS_ERR_MALLOC);
	debug(("autotrust environment initialized"));

	/* read command line options */
	while ((c = getopt(argc, argv, "c:dhv:")) != -1)
	{
		switch (c)
		{
			case 'c': /* config file */
				env->configfile = optarg;
				break;
			case 'd': /* daemonize */
				r = set_cfg_option_int(env->options,
								"daemonize", 1);
				break;
			case 'v': /* verbosity */
				r = set_cfg_option_int(env->options,
						"verbosity", atoi(optarg));
				break;
			case 'h': /* help, print usage */
			case '?':
			default:
				usage();
				free_environment(STATUS_USAGE);
 				break;
		}
	}
	argc -= optind;
	argv += optind;
	if (argc != 0)
	{
		usage();
		free_environment(STATUS_ERR_GETOPT);
	}
	debug(("commandline options parsed"));

	/* read options */
	return_status = load_cfg_settings(env->options, env->configfile);
	if (return_status != STATUS_OK)
		free_environment(return_status);
	debug(("configuration settings loaded"));
#ifdef TDEBUG
	debug_print_options(env->options);
#endif /* TDEBUG */

	/* apply options */
	if (env->options)
	{
		if (env->options->use_syslog)
		{
#ifdef HAVE_SYSLOG_H
			openlog("autotrust", LOG_NDELAY, LOG_DAEMON);
			set_logfunction(log_syslog);
#else
			warning("syslog not supported, logging to stderr");
#endif /* HAVE_SYSLOG_H */
		}
		else
			open_logfile(env->options->logfile, env->options->verbosity);

		debug(("logfile %s opened with verbosity %i",
			env->options->use_syslog?"syslog":(env->options->logfile?env->options->logfile:"stderr"),
			env->options->verbosity));
	}

	if (env->options && env->options->workingdir &&
		env->options->workingdir[0])
	{
		if (chdir(env->options->workingdir) != 0)
		{
			verbos(3, "cannot chdir to %s: %s", env->options->workingdir,
				strerror(errno));
			free_environment(STATUS_ERR_CHDIR);
		}
		debug(("working directory changed to %s", env->options->workingdir));
	}

	/* trust anchor files */
	return_status = load_trustanchor_files(env->trustpoints,
		env->options->trustanchorfiles);
	if (return_status != STATUS_OK)
		error("could not load trust anchor files: %s",
			status_by_id(return_status));

	return_status = load_trustedkeys_files(env->trustpoints,
		env->options->trustedkeysfiles);
	if (return_status != STATUS_OK)
		error("could not load trusted keys: %s",
			status_by_id(return_status));

#if 0
	return_status = load_trustanchor_options(env->trustpoints,
		env->options->trustanchors);
	if (return_status != STATUS_OK)
		error("could not load trust anchor options: %s",
			status_by_id(return_status));
#endif

	/* if we run as daemon, some extra work is needed */
	running = env->options->daemonize;
	if (running)
		pid = setup_daemon();

	/* do a single or multiple runs */
	do
	{
		/* if a sigterm is received, the party is over */
		if (env->sig_quit) { running = 0; break; }

		/* have we waited long enough? */
		now = time(NULL);
		if (now < wait && !env->sig_quit)
		{
			sleep(wait-now);
			continue;
		}

		/* now we have, set a new wait period */
		wait = now + FIFTEEN_DAYS;
		/* do an active refresh for all trust points */
		return_status = do_single_run(&wait);
		if (running)
		{
			debug(("waiting %i seconds to next refresh", (wait-now)));
			flush();
		}
	}
	while (running);

	/* done */
	debug(("done"));
#ifdef HAVE_SYSLOG_H
	if (env->options->use_syslog)
		closelog();
#endif /* HAVE_SYSLOG_H */

	free_environment(return_status);

	/* not reached */
	return 1;
}
