/** \file src/type-dialog.hh */
/*
 * This file is part of assoGiate,
 * an editor of the file types database for GNOME.
 *
 * Copyright (C) 2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef ASSOGIATE__TYPE_DIALOG_HH
#define ASSOGIATE__TYPE_DIALOG_HH

#include <libassogiate/misc/throttle.hh>
#include "views.hh"

#include <gtkmm/buttonbox.h>
#include <gtkmm/celllayout.h>
#include <gtkmm/checkbutton.h>
#include <gtkmm/entry.h>
#include <gtkmm/image.h>
#include <gtkmm/notebook.h>
#include <gtkmm/treestore.h>
#include <gtkmm/window.h>

/******************************************************************************/
namespace assoGiate {
/******************************************************************************/

/******************************************************************************/
class TypeDialog : public Gtk::Window {
/******************************************************************************/

public:

	virtual ~TypeDialog();

	ustring get_full_name();

	virtual void set_type(MimeType& type);
	void reset_type();
	
	sigc::signal<void> signal_changed();

protected:

	TypeDialog(const RefPtr<MimeDatabase>& database, MimeType* initial_type,
		Location target, bool renameable);

	int get_notebook_page() const;

	void set_indirect_sensitive(bool sensitive);

	void update();
	void update_icon();
	int m_updating;

	Location m_locations, m_target;
	MimeType *m_fixed_type, *m_target_type;
	
	RefPtr<MimeDatabase> m_database;

	Gtk::HButtonBox m_buttons;
	
	virtual void on_help();

private:
	
	sigc::signal<void> s_changed;

	void accumulate_lists(MimeType& type, bool removable);
	void append_magic(const Gtk::TreeIter& iter, const MagicMatch& match,
		const Magic& group, bool removable);
	
	void on_category_changed();
	void on_name_changed();
	void on_description_changed();
	
	void on_select_default_icon();
	
	void one_ustring_common(const Gtk::TreeIter& iter, const ustring& title,
		const ustring& label, std::list<ustring>& list,
		const RefPtr<Gtk::ListStore>& store);
	void on_alias_changed(const Gtk::TreeIter& iter);
	void on_superclass_changed(const Gtk::TreeIter& iter);
	void on_glob_changed(const Gtk::TreeIter& iter);
	void on_magic_changed(const Gtk::TreeIter& iter);
	void on_xml_changed(const Gtk::TreeIter& iter);
	
	static void update_sensitivity(const Gtk::ToggleButton* source,
		Gtk::Widget* target);
		
	Gtk::Notebook m_notebook;

	Gtk::CellLayout *m_category_widget;
	Gtk::Label m_category_id;
	Gtk::Widget *m_name_widget;
	Gtk::Entry m_description_entry;
	Gtk::Label m_location;

	Gtk::Image m_icon;
	Gtk::Label m_default_icon;
	Gtk::Button m_default_icon_select;

	RefPtr<Gtk::ListStore> m_alias_store, m_superclass_store,
		m_glob_store;
	RefPtr<Gtk::TreeStore> m_magic_store;
	RefPtr<Gtk::ListStore> m_xml_store;
	EditableListView m_alias_view, m_superclass_view,
		m_glob_view, m_magic_view, m_xml_view;

}; /* class TypeDialog */

/******************************************************************************/
class NewTypeDialog : public TypeDialog {
/******************************************************************************/

public:

	explicit NewTypeDialog(const RefPtr<MimeDatabase>& database);

protected:
	
	virtual void on_help();

private:

	void on_changed();
	
	void on_new_type();
	
	void show_error(const ustring& explanation);

	Gtk::Button m_new;

}; /* class NewTypeDialog */

/******************************************************************************/
class EditTypeDialog : public TypeDialog {
/******************************************************************************/

public:

	EditTypeDialog(const RefPtr<MimeDatabase>& database, MimeType& type,
		Location target);
	virtual ~EditTypeDialog();

	virtual void set_type(MimeType& type);

private:

	void on_changed();
	void real_on_changed();
	misc::Throttle m_throttle_changed;
	
	ustring m_old_name;

}; /* class EditTypeDialog */

} /* namespace assoGiate */

#endif /* ASSOGIATE__TYPE_DIALOG_HH */
