/** \file throttle.cc
 * Throttle regulates the rate at which functions are called.
 */
/* This file is part of libmisc, an assortment of code for reuse.
 *
 * Copyright (C) 2006-2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "throttle.hh"

#include <sigc++/adaptors/bind.h>
#include <glibmm/main.h>

/******************************************************************************/
namespace misc {
/******************************************************************************/

/******************************************************************************/
/* class Throttle                                                             */
/******************************************************************************/

Throttle::Throttle(Mode mode, double threshold, bool immediate)
	throw(std::invalid_argument)
:	m_mode(OPEN), m_threshold(0.0),
	m_timer(), m_running(false),
	m_queue(), m_update()
{	set(mode, threshold, immediate); }

Throttle::~Throttle()
{	cancel_update(); }

Throttle::Mode
Throttle::get_mode() const throw()
{	return m_mode; }

bool
Throttle::is_running() const throw()
{	return m_running; }

void
Throttle::set(Mode mode, bool immediate) throw(std::invalid_argument)
{
	do_set(mode, immediate);
	update();
}

void
Throttle::set(Mode mode, double threshold, bool immediate)
	throw(std::invalid_argument)
{
	do_set(mode, immediate);
	set_threshold(threshold);
}

void
Throttle::reset(bool immediate) throw()
{
	do_set(m_mode, immediate);
	update();
}

double
Throttle::get_threshold() const throw()
{	return m_threshold; }

void
Throttle::set_threshold(double threshold) throw(std::invalid_argument)
{
	if (threshold < 0.0)
		throw std::invalid_argument("negative threshold");
	m_threshold = threshold;
	update();
}

void
Throttle::queue(const Action& action, bool idempotent) throw()
{
	if (idempotent) clear();
	m_queue.push(action);
	update(true);
}

bool
Throttle::empty() const throw()
{	return m_queue.empty(); }

void
Throttle::clear() throw()
{
	while (!m_queue.empty())
		m_queue.pop();
}

void
Throttle::do_set(Mode mode, bool immediate) throw(std::invalid_argument)
{
	switch (mode) {
	case OPEN:
	case CLOSED:
	case LIMIT:
	case DELAY:
		stop();
		break;
	case PAUSE:
	case UNTIL:
		if (immediate) start();
		else stop();
		break;
	default:
		throw std::invalid_argument("invalid Throttle::Mode");
	}

	cancel_update();
	m_mode = mode;
}

void
Throttle::start() throw()
{
	m_timer.start();
	m_running = true;
}

void
Throttle::stop() throw()
{
	m_timer.stop();
	m_running = false;
}

bool
Throttle::update(bool from_queue) throw()
{
	cancel_update();

	switch (m_mode) {
	case OPEN:
		run_all();
		return false;
	case CLOSED:
		return false;
	case PAUSE:
		if (m_running && m_timer.elapsed() >= m_threshold) {
			do_set(OPEN, false);
			run_all();
			return false;
		}
		if (!m_running && from_queue) start();
		break;
	case UNTIL:
		if (m_running && m_timer.elapsed() >= m_threshold) {
			do_set(CLOSED, false);
			return false;
		}
		if (!m_running && from_queue) start();
		run_all();
		return false;
	case LIMIT:
		if (!m_running || m_timer.elapsed() >= m_threshold) {
			if (!m_queue.empty()) start();
			run_top();
		}
		break;
	case DELAY:
		if (m_running && m_timer.elapsed() >= m_threshold) {
			if (!m_queue.empty()) start();
			run_top();
		} else if (!m_running || from_queue) start();
		break;
	default:
		return false;
	}

	if (m_queue.empty())
		return false;
	else {
		request_update();
		return true;
	}
}

void
Throttle::request_update(unsigned int interval) throw()
{
	if (interval == 0) {
		double until = m_threshold - m_timer.elapsed();
		interval = (until > 0.0) ? (unsigned int)(until * 1000.0) : 0u;
	}
	if (!m_update)
		m_update = Glib::signal_timeout().connect(sigc::bind(sigc::mem_fun
			(*this, &Throttle::update), false), ++interval);
}

void
Throttle::cancel_update() throw()
{
	if (m_update) m_update.disconnect();
	m_update = sigc::connection();
}

void
Throttle::run_top() throw()
{
	if (!m_queue.empty()) {
		try { m_queue.front()(); } catch (...) {}
		m_queue.pop();
	}
}

void
Throttle::run_all() throw()
{
	while (!m_queue.empty()) {
		try { m_queue.front()(); } catch (...) {}
		m_queue.pop();
	}
}

} /* namespace misc */
