/** \file libassogiate/mime-package.hh */
/*
 * This file is part of assoGiate,
 * an editor of the file types database for GNOME.
 *
 * Copyright (C) 2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef LIBASSOGIATE__MIME_PACKAGE_HH
#define LIBASSOGIATE__MIME_PACKAGE_HH

#include <libassogiate/mime-type.hh>

#include <libxml++/document.h>
#include <sigc++/signal.h>
#include <libgnomevfsmm/monitor-handle.h>

/******************************************************************************/
namespace assoGiate {
/******************************************************************************/

class MimeDirectory; /* defined in mime-directory.hh */

/******************************************************************************/
class MimeDatabaseError : public std::runtime_error {
/******************************************************************************/

protected:

	explicit MimeDatabaseError(const ustring& description) throw();

}; /* class MimeDatabaseError */

/******************************************************************************/
class MimeDatabaseLoadError : public MimeDatabaseError {
/******************************************************************************/

public:

	explicit MimeDatabaseLoadError(const ustring& description) throw();

	void unhandled() throw();
	static sigc::signal<void, MimeDatabaseLoadError> signal_unhandled() throw();

}; /* class MimeDatabaseLoadError */

/******************************************************************************/
class MimeDatabaseWriteError : public MimeDatabaseError {
/******************************************************************************/

public:

	explicit MimeDatabaseWriteError(const ustring& description) throw();

}; /* class MimeDatabaseWriteError */

/******************************************************************************/
class MimeDatabaseUpdateError : public MimeDatabaseError {
/******************************************************************************/

public:

	explicit MimeDatabaseUpdateError(const ustring& description) throw();

	void unhandled() throw();
	static sigc::signal<void, MimeDatabaseUpdateError> signal_unhandled()
		throw();

}; /* class MimeDatabaseUpdateError */

/******************************************************************************/
class MimeTypeExistsError : public std::exception {};
/******************************************************************************/

/******************************************************************************/
class MimePackage : public NodeMapExtender {
/******************************************************************************/

public:
	
	explicit MimePackage(const std::string& path, Location type)
		throw(MimeDatabaseLoadError);
	
	virtual ~MimePackage();
	
	bool has_contents() const throw();
	
	void reload() throw(MimeDatabaseLoadError);
	sigc::signal<void> signal_reloaded() throw();
	
	virtual void extend_node_map(MimeNodeMap& node_map) throw();

protected:

	static xmlpp::Document* create_empty_document() throw(xmlpp::exception);

	std::string m_path;
	Location m_type;
	xmlpp::Document* m_document;

	sigc::signal<void> s_reloaded;
	
}; /* class MimePackage */

/******************************************************************************/
class WritableMimePackage : public MimePackage {
/******************************************************************************/
	
public:
	
	WritableMimePackage(const std::string& path, const MimeDirectory& dir,
		Location type) throw(MimeDatabaseLoadError);
	
	void add_type(const MimeType& type) throw(MimeTypeExistsError,
		MimeDatabaseError);

	void replace_type(const MimeType& type, ustring old_name = ustring())
		throw(MimeDatabaseError);

	void remove_type(const ustring& type) throw(MimeDatabaseError);
	
	void clear_types() throw(MimeDatabaseError);
	
	void import_types(const ustring& from) throw(xmlpp::exception,
		MimeDatabaseError);

	void export_types(const ustring& to) const throw(xmlpp::exception);
	
	void update_mime_database() throw(MimeDatabaseUpdateError);

	bool has_default_icon(const MimeType& type) const throw();

	std::string get_default_icon(const MimeType& type) const
		throw(std::invalid_argument);

	void set_default_icon(const MimeType& type, const std::string& path)
		throw(std::invalid_argument, MimeDatabaseError);

	void unset_default_icon(const MimeType& type) throw(MimeDatabaseError);

	void update_icon_cache() throw(MimeDatabaseUpdateError);
	sigc::signal<void> signal_icons_changed() throw();

private:

	void write() throw(MimeDatabaseError);
	
	bool remove_icons(const ustring& path) throw(MimeDatabaseWriteError);
	
	void on_monitor_event(const Gnome::Vfs::MonitorHandle& handle,
		const ustring& monitor_uri, const ustring& info_uri,
		Gnome::Vfs::MonitorEventType type) throw();
	
	static void on_mime_update_exited(GPid pid, int result) throw();
	static void on_theme_update_exited(GPid pid, int result) throw();

	unsigned int m_ignore_events;
	Gnome::Vfs::MonitorHandle m_monitor, m_theme_monitor;
	std::string m_theme_dir, m_icons_dir, m_mime_dir;
	
	sigc::signal<void> s_icons_changed;

}; /* class WritableMimePackage */

} /* namespace assoGiate */

#endif /* LIBASSOGIATE__MIME_PACKAGE_HH */
