// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/RegComp.h"
#include "Puma/ErrorStream.h"
#include "Puma/StrCol.h"
#include <string.h>

namespace Puma {


// Constructor: Compile the regular expression.
RegComp::RegComp (const char *expr, int cflags) {
  if (expr) {
    _cflags = cflags;
    _expr   = StrCol::dup (expr);
    _cerror = regcomp (&_compiled, expr, cflags);
    _eerror = 0;
  }
}


// Constructor: Compile the regular expression.
RegComp::RegComp (RegComp &rc) {
  _cflags = rc._cflags;
  _expr   = StrCol::dup (rc._expr);
  _cerror = regcomp (&_compiled, _expr, _cflags);
  _eerror = 0;
}


// Compile the regular expression.
bool RegComp::compile (const char *expr, int cflags) {
  // Free the old compilation results.
  if (_cerror == 0) 
    regfree (&_compiled);
    
  _cerror = regcomp (&_compiled, expr, cflags);
  _eerror = 0;
    
  if (_cerror == 0) 
    return true;
  else	            
    return false;
}


// Match against a string.
bool RegComp::match (const char *string, int eflags) {
  if (_cerror != 0) 
    return false;
    
  // Match the regular expression against the string.
  _eerror = regexec (&_compiled, string, 0, NULL, eflags);

  if (_eerror == 0) 
    return true;
  else	      
    return false;
}
	

// To be called on errors.
void RegComp::comperror (ErrorStream &err) const {
  int errcode = (_cerror != 0) ? _cerror : _eerror;
    
  if (errcode == 0)
    return;
  if (errcode == REG_NOMATCH) 
    return;	// Not really an error.
    
  int   errlen  = regerror (errcode, &_compiled, NULL, 0);
  char *errmesg = new char[errlen + 1];
  regerror (errcode, &_compiled, errmesg, errlen);
    
  err << sev_error << "RegComp: " << errmesg << endMessage;
  delete[] errmesg;
}


} // namespace Puma
