#!/usr/bin/perl
# Copyright 2007, 2008, 2009
# Frank Terbeck <ft@bewatermyfriend.org>, All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
#   1. Redistributions of source code must retain the above
#      copyright notice, this list of conditions and the following
#      disclaimer.
#   2. Redistributions in binary form must reproduce the above
#      copyright notice, this list of conditions and the following
#      disclaimer in the documentation and/or other materials
#      provided with the distribution.
#
#  THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED
#  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
#  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
#  DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS OF THE
#  PROJECT BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
#  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
#  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
#  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
#  OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
#  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
#  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package ARename;
use warnings;
use strict;

# modules

# These are commonly installed along with Perl:
use Readonly;
use Carp;
use English '-no_match_vars';
use Getopt::Long;
use File::Basename;
use File::Copy;
use Cwd;
use Cwd 'abs_path';

# These are external modules. On debian systems, do:
#   % aptitude install libogg-vorbis-header-perl \
#                      libmp3-tag-perl           \
#                      libaudio-flac-header-perl
#
# If you don't know how to get them for your OS, get them
# from CPAN: <http://cpan.org>.
use MP3::Tag;
use Ogg::Vorbis::Header;
use Audio::FLAC::Header;

# variables
my (
    %cmdline_protect, %conf, %defaults, %hooks, %methods, %parsers, %profiles,
    %opts, %sectconf, %sets,
    $__arename_file, $postproc, $sect,
    @cmdline_profiles, @localizables, @settables, @supported_tags,
);
my ( $NAME, $VERSION ) = qw( unset unset );

# a helper for the testsuite
sub data_reset {
    undef %conf;
    undef %defaults;
    undef %hooks;
    undef %methods;
    undef %parsers;
    undef %profiles;
    undef %opts;
    undef %sectconf;
    undef %sets;
    undef @cmdline_profiles;
    undef @localizables;
    undef @settables;
    undef @supported_tags;

    return 1;
}

set_opt('shutup', 0);

# settings that may occur in [sections]
@localizables = qw(
    copymode
    force
    prefix
    sepreplace
    tnpad
    comp_template
    template
);

@supported_tags = qw(
    album        artist
    compilation
    genre
    tracknumber  tracktitle
    year
);

@settables = qw(
    canonicalize checkprofilerc comp_template
    debug
    hookerrfatal
    prefix
    quiet quiet_skip
    sepreplace
    template tnpad
    usehooks uselocalhooks uselocalrc useprofiles
    verbose warningsautodryrun
);

$postproc = \&arename;

# high level code

sub apply_methods {
    my ($exit) = @_;

    my $file = get_file();

    foreach my $method (sort keys %methods) {
        if ($file =~ m/$method/i) {
            run_hook('pre_method', \$method);
            $methods{$method}->($file);
            run_hook('post_method', \$method);
            if ($exit) {
                exit 0;
            } else {
                return 1;
            }
        }
    }
    return 0;
}
sub __arename_file_eq {
    my ($newname, $oldname) = @_;

    if (file_eq($newname, $oldname)) {
        if (get_opt("quiet")) {
            if (!get_opt("quiet_skip")) {
                print "Skipping: '$oldname'\n";
            }
        } else {
            oprint("'$oldname'\n      would stay the way it is, skipping.\n");
        }
        return 1;
    }

    return 0;
}
sub arename {
    my ($datref, $ext) = @_;
    my ($t, $newname);

    my $file = get_file();

    run_hook('pre_apply_defaults', $datref, \$ext);

    apply_defaults($datref);
    arename_verbosity($datref);

    run_hook('pre_template', $datref, \$ext);

    $t = choose_template($datref);
    $newname = expand_template($t, $datref);
    return if not defined $newname;
    $newname = get_opt("prefix") . "/$newname.$ext";

    run_hook('post_template', $datref, \$ext, \$newname);

    return if (__arename_file_eq($newname, $file));

    if (-e $newname && !get_opt("force")) {
        oprint("'$newname' exists." . (get_opt("quiet") ? q{ } : "\n      ")
            . "use '-f' to force overwriting.\n");
        return 0;
    }

    ensure_dir(dirname($newname));

    run_hook('post_ensure_dir', $datref, \$ext, \$newname);

    if (get_opt("quiet")) {
        print "'$newname'\n" if (!get_opt('shutup'));
    } else {
        my ($mode);
        if (get_opt("copymode")) {
            $mode = 'cp';
        } else {
            $mode = 'mv';
        }
        oprint("$mode '$file' \\\n         '$newname'\n");
    }

    if (!get_opt("dryrun")) {
        if (!get_opt("copymode")) {
            xrename($file, $newname);
        } else {
            xcopy($file, $newname);
        }
    }

    run_hook('post_rename', $datref, \$ext, \$newname);

    return 1;
}

sub apply_defaults {
    my ($datref) = @_;
    my ($value);

    foreach my $key (get_default_keys()) {
        if (!defined $datref->{$key}) {
            run_hook('apply_defaults', $datref, \$key);

            $value = get_defaults($key);
            oprint_verbose(qq{Setting ($key) to "$value".\n});
            $datref->{$key} = $value;
        }
    }

    return 1;
}
sub tag_supported {
    my ($tag) = @_;

    foreach my $sub (@supported_tags) {
        if ($tag eq $sub) {
            return 1;
        }
    }

    return 0;
}

sub arename_verbosity {
    my ($datref) = @_;

    return 0 if (!get_opt('verbose'));

    oprint("Artist     : " . getdat($datref, "artist")      . "\n");
    oprint("Compilation: " . getdat($datref, "compilation") . "\n");
    oprint("Album      : " . getdat($datref, "album")       . "\n");
    oprint("Tracktitle : " . getdat($datref, "tracktitle")  . "\n");
    oprint("Tracknumber: " . getdat($datref, "tracknumber") . "\n");
    oprint("Genre      : " . getdat($datref, "genre")       . "\n");
    oprint("Year       : " . getdat($datref, "year")        . "\n");

    return 1;
}
sub getdat {
    my ($datref, $tag) = @_;

    return defined $datref->{$tag} ? q{"} . $datref->{$tag} . q{"} : "(undefined)";
}

sub process_file {
    my ($file) = @_;

    set_file($file);

    run_hook('next_file_early');

    if (!get_opt("quiet")) {
        print "Processing: $file\n";
    }
    if (-l $file) {
        owarn("Refusing to handle symbolic links ($file).\n");
        return 0;
    }
    if (! -r $file) {
        owarn(qq{Can't read "$file": $!\n});
        return 0;
    }

    if (get_opt('canonicalize')) {
        my $f = abs_path($file);
        run_hook('canonicalize', \$f);
        set_file($f);
    }

    run_hook('next_file_late');

    if (!apply_methods(0)) {
        run_hook('filetype_unknown');
        process_warn();
    } else {
        run_hook('file_done');
    }

    return 1;
}

sub get_profile_list {
    my @list = ();
    my $wd = getcwd();
    my %seen = ();

    # make sure $wd ends in *one* slash
    $wd =~ s/\/+$//;
    $wd .= q{/};

    foreach my $profile (sort keys %profiles) {
        odebug(qq{get_profile_list(): Checking "$profile" patterns...\n});

        foreach my $pat (@{ $profiles{$profile} }) {
            odebug("get_profile_list(): ($wd) =~ ($pat)...\n");

            if ($wd =~ m/^$pat/) {
                odebug("get_profile_list(): MATCHED.\n");
                push @list, $profile;
                last;
            }

        }
    }

    @list = sort grep { ! $seen{ $_ }++ } (@list, @cmdline_profiles);
    odebug("get_profile_list(): (" . join(q{,}, @list) . ")\n");
    return @list;
}

sub set_default_options {
    if (defined $ENV{'ARENAME_LOAD_QUIET'}
        && $ENV{'ARENAME_LOAD_QUIET'} eq '1') {
        set_opt("load_quiet",       1);
    } else {
        set_opt("load_quiet",       0);
    }
    set_opt("canonicalize",         0);
    set_opt("copymode",             0);
    set_opt("checkprofilerc",       1);
    set_opt("debug",                0);
    set_opt("dryrun",               0);
    set_opt("force",                0);
    set_opt("hookerrfatal",         1);
    set_opt("oprefix",              q{  -!- });
    set_opt("prefix" ,              q{.});
    set_opt("quiet",                0);
    set_opt("quiet_skip",           0);
    set_opt("readstdin",            0);
    set_opt("sepreplace",           q{_});
    set_opt("tnpad",                2);
    set_opt("usehooks",             1);
    set_opt("uselocalhooks",        0);
    set_opt("uselocalrc",           0);
    set_opt("useprofiles",          1);
    set_opt("verbose",              0);
    set_opt("warningsautodryrun",   1);
    set_opt("comp_template",        q{va/&album/&tracknumber - &artist - &tracktitle});
    set_opt("template",             q{&artist[1]/&artist/&album/&tracknumber - &tracktitle});

    return 1;
}
sub set_default_methods {

    %methods = (
        '\.flac$' => \&ARename::process_flac,
        '\.mp3$'  => \&ARename::process_mp3,
        '\.ogg$'  => \&ARename::process_ogg,
    );

    return 1;
}
sub set_nameversion {
    my ($n, $v) = @_;

    $NAME    = $n;
    $VERSION = $v;
    return 1;
}
sub set_postproc {
    my ($p) = @_;

    $postproc = $p;
    return 1;
}

sub usage {
    print " Usage:\n  $NAME [OPTION(s)] FILE(s)...\n\n";
    print "    --compare-versions           Compare versions of script and module.\n";
    print "    --copy, -c                   Copy files, rather than renaming.\n";
    print "    --debug, -D                  Enable debugging output.\n";
    print "    --disable-hooks, -H          Disable *all* hooks.\n";
    print "    --disable-profiles, -N       Deactivate all profiles.\n";
    print "    --dryrun, -d                 Go into dryrun mode.\n";
    print "    --enable-hooks               Enable hooks, if the configuration disabled\n";
    print "                                 them.\n";
    print "    --force, -f                  Overwrite files if needed.\n";
    print "    --help, -h                   Display this help text.\n";
    print "    --list-cfg, -L               List current configuration.\n";
    print "    --list-profiles, -S          Show a list of defined profiles\n";
    print "    --quiet, -q                  Enable quiet output.\n";
    print "    --read-local, -l             Read local rc, if it exists.\n";
    print "    --stdin, -s                  Read file names from stdin.\n";
    print "    --uber-quiet, -Q             Don't display skips in quiet mode.\n";
    print "    --version, -V                Display version infomation.\n";
    print "    --verbose, -v                Enable verbose output.\n";
    print "\n";
    print "    --rc <file>                  Read file instead of ~/.arenamerc.\n";
    print "    --post-rc <file>             Read file after ~/.arenamerc.\n";
    print "\n";
    print "    --prefix, -p <prefix>        Define a prefix for destination files.\n";
    print "    --profile, -P <prof(s),...>  Comma seperated list of profiles to activate\n";
    print "                                 forcibly.\n";
    print "\n";
    print "    --compilation-template -T <template>\n";
    print "                                 Define a compilation template.\n";
    print "    --template, -t <template>    Define a generic template.\n";
    print "\n";
    print "    --userset, -u <var=value...> Set a user variable.\n";
    print "\n";

    return 1;
}

# template handling

sub choose_template {
    my ($datref) = @_;

    if (defined $datref->{compilation}
        && $datref->{compilation} ne $datref->{artist}) {

        return get_opt("comp_template");
    } else {
        return get_opt("template");
    }
}
sub __template_create_token {
    my ($datref, $tag, $len) = @_;
    my ($token, $val, $pad);

    if ($len > 0) {
        $token = substr $datref->{$tag}, 0, $len;
    } else {
        if ($tag eq 'tracknumber') {

            if ($datref->{$tag} =~ m/^([^\/]*)\/.*$/) {
                $val = $1;
            } else {
                $val = $datref->{$tag};
            }

            $pad = get_opt('tnpad');
            $token = sprintf q{%0} . ($pad ne q{0} ? "$pad" : q{} ) . q{d}, $val;
        } else {
            $token = $datref->{$tag};
        }
    }

    return $token;
}
sub __template_token_sepreplace {
    my ($tokref) = @_;
    my ($sr);

    if (${ $tokref } =~ m{/}) {
        $sr = get_opt("sepreplace");
        oprint_verbose("Found directory seperator in token.\n");
        oprint_verbose(qq{Replacing with "$sr".\n});
        ${ $tokref } =~ s{/}{$sr}g;
    }

    return 1;
}
sub expand_template {
    my ($template, $datref) = @_;

    run_hook('pre_expand_template', \$template, $datref);

    foreach my $tag (@supported_tags) {
        my ($len, $token);

        while ($template =~ m/&$tag(\[(\d+)\]|)/) {
            $len = 0;
            if (defined $2) { $len = $2; }

            if (!defined $datref->{$tag} || $datref->{$tag} eq q{}) {
                my $file = get_file();
                owarn("-!- $file\n");
                owarn(" -> $tag not defined, but required by template. Giving up.\n");
                return;
            }

            run_hook('expand_template_next_tag', \$template, \$tag, \$len, $datref);

            $token = __template_create_token($datref, $tag, $len);
            __template_token_sepreplace(\$token);

            run_hook('expand_template_postprocess_tag', \$template, \$token, \$tag, \$len, $datref);

            $template =~ s/&$tag(\[(\d+)\]|)/$token/;
        }
    }

    run_hook('post_expand_template', \$template, $datref);

    return $template;
}

# processing audio files

sub handle_vorbistag {
    my ($datref, $tag, $value) = @_;
    my ($realtag);

    run_hook('pre_handle_vorbistag', \$tag, \$value, $datref);

    if (!(
            $tag =~ m/^ALBUM$/i         ||
            $tag =~ m/^ARTIST$/i        ||
            $tag =~ m/^TITLE$/i         ||
            $tag =~ m/^TRACKNUMBER$/i   ||
            $tag =~ m/^DATE$/i          ||
            $tag =~ m/^GENRE$/i         ||
            $tag =~ m/^ALBUMARTIST$/i
        )) { return 0; }

    if ($tag =~ m/^ALBUM$/i) {
        $realtag = 'album';
    } elsif ($tag =~ m/^ARTIST$/i) {
        $realtag = 'artist';
    } elsif ($tag =~ m/^TITLE$/i) {
        $realtag = 'tracktitle';
    } elsif ($tag =~ m/^TRACKNUMBER$/i) {
        $realtag = 'tracknumber';
    } elsif ($tag =~ m/^DATE$/i) {
        $realtag = 'year';
    } elsif ($tag =~ m/^GENRE$/i) {
        $realtag = 'genre';
    } elsif ($tag =~ m/^ALBUMARTIST$/i) {
        $realtag = 'compilation';
    } else {
        croak("This should not happen. Report this BUG. ($tag, $value)");
    }

    if (!defined $datref->{$realtag}) {
        $datref->{$realtag} = $value;
    }

    run_hook('post_handle_vorbistag', \$tag, \$value, \$realtag, $datref);

    return 1;
}
sub process_flac {
    my ($flac, %data, $tags);

    my $file = get_file();

    run_hook('pre_process_flac');

    $flac = Audio::FLAC::Header->new($file);

    if (!defined $flac) {
        oprint(qq{Failed to open "$file".\n});
        oprint("Reason: $!\n");
        return 0;
    }

    $tags = $flac->tags();

    foreach my $tag (keys %{ $tags }) {
        handle_vorbistag(\%data, $tag, $tags->{$tag});
    }

    $postproc->(\%data, 'flac');

    run_hook('post_process_flac');

    return 1;
}
sub process_mp3 {
    my ($mp3, %data, $info);

    my $file = get_file();

    run_hook('pre_process_mp3');

    $mp3 = MP3::Tag->new($file);

    if (!defined $mp3) {
        oprint(qq{Failed to open "$file".\n});
        oprint("Reason: $!\n");
        return 0;
    }

    $mp3->get_tags;

    run_hook('pre_handle_mp3tag', $mp3, \%data);
    if (exists $mp3->{ID3v2}) {
        ($data{artist},      $info) = $mp3->{ID3v2}->get_frame("TPE1");
        ($data{compilation}, $info) = $mp3->{ID3v2}->get_frame("TPE2");
        ($data{album},       $info) = $mp3->{ID3v2}->get_frame("TALB");
        ($data{tracktitle},  $info) = $mp3->{ID3v2}->get_frame("TIT2");
        ($data{tracknumber}, $info) = $mp3->{ID3v2}->get_frame("TRCK");
        ($data{genre},       $info) = $mp3->{ID3v2}->get_frame("TCON");
        ($data{year},        $info) = $mp3->{ID3v2}->get_frame("TYER");
    } elsif (exists $mp3->{ID3v1}) {
        if ($NAME eq 'arename') {
            oprint("Only found ID3v1 tag.\n");
        }
        $data{artist}      = $mp3->{ID3v1}->artist;
        $data{album}       = $mp3->{ID3v1}->album;
        $data{tracktitle}  = $mp3->{ID3v1}->title;
        $data{tracknumber} = $mp3->{ID3v1}->track;
        $data{genre}       = $mp3->{ID3v1}->genre;
        $data{year}        = $mp3->{ID3v1}->year;
    }
    run_hook('post_handle_mp3tag', $mp3, \%data);

    $mp3->close();

    $postproc->(\%data, 'mp3');

    run_hook('post_process_mp3');

    return 1;
}
sub process_ogg {
    my ($ogg, %data, @tags);

    my $file = get_file();

    run_hook('pre_process_ogg');

    $ogg = Ogg::Vorbis::Header->load($file);

    if (!defined $ogg) {
        oprint(qq{Failed to open "$file".\n});
        oprint("Reason: $!\n");
        return 0;
    }

    @tags = $ogg->comment_tags;

    foreach my $tag (@tags) {
        next if ($tag eq q{});
        handle_vorbistag(\%data, $tag, join(q{ }, $ogg->comment($tag)));
    }

    $postproc->(\%data, 'ogg');

    run_hook('post_process_ogg');

    return 1;
}
sub process_warn {
    my $file = get_file();

    owarn(qq{No method for handling "$file".\n});
    return 1;
}

# output subroutines

sub odebug {
    my ($string) = @_;

    # we cannot use the normal API here, because we're using odebug()
    # in the API's subroutines. That would recurse endlessly.
    return 0 if (!$conf{'debug'} || $conf{'shutup'});
    print "DEBUG: $string";

    return 1;
}
sub oprint {
    my ($string) = @_;

    return 0 if (get_opt('shutup'));
    print get_opt("oprefix") . $string;

    return 1;
}
sub oprint_verbose {
    my ($string) = @_;

    return 0 if (!get_opt('verbose'));
    oprint($string);

    return 1;
}
sub owarn {
    my ($string) = @_;

    return 0 if (get_opt('shutup'));
    warn(get_opt("oprefix") . $string);

    return 1;
}
sub owarn_verbose {
    my ($string) = @_;

    return 0 if (!get_opt('verbose'));
    owarn($string);

    return 1;
}

# config file processing

Readonly::Scalar my $FIND_RC => 0;              # find .arenamerc
Readonly::Scalar my $FIND_HOOKS => 1;           # find .arename.hooks
Readonly::Scalar my $FIND_PROFRC => 2;          # find .arename.profilename
Readonly::Scalar my $FIND_PROFHOOKS => 3;       # find .arename.profilename.hooks
Readonly::Scalar my $FIND_NONAME => q{};        # file name to find not further specified

sub __home_find_file {
    # TODO: Someone should clean this up. Seriously.
    my ($home, $name, $dotname) = @_;
    my ($etcdir, $dotdir, $fn, $xdg);

    $etcdir = "$home/etc/arename";
    $dotdir = "$home/.arename";

    # Support for $XDG_CONFIG_HOME.
    if (defined $ENV{'XDG_CONFIG_HOME'}) {
        $xdg= $ENV{'XDG_CONFIG_HOME'} . '/arename';
        if (-d $xdg) {
            odebug(qq{__home_find_file() using "$xdg"\n});
            $fn = "$xdg/$name";

            if (-e $fn) {
                return $fn;
            } else {
                goto error;
            }
        }
    }
    $xdg = "$home/.config/arename";

    # If $XDG_CONFIG_HOME didn't stick, use the old ways:
    # if ~/etc/arename/ exists, we read *everything* from there.
    # if not, but ~/.arename/, we read everything from there.
    # if both are not there, we're looking for stuff like ~/.arenamerc.
    if (-d $etcdir) {
        odebug(qq{__home_find_file() using "$etcdir"\n});
        $fn = "$etcdir/$name";

        if (-e $fn) {
            return $fn;
        }
    } elsif (-d $xdg) {
        odebug(qq{__home_find_file() using "$xdg"\n});
        $fn = "$xdg/$name";

        if (-e $fn) {
            return $fn;
        }
    } elsif (-d $dotdir) {
        odebug(qq{__home_find_file() using "$dotdir"\n});
        $fn = "$dotdir/$name";

        if (-e $fn) {
            return $fn;
        }
    } else {
        odebug(qq{__home_find_file() using "$home"\n});
        $fn = "$home/$dotname";

        if (-e $fn) {
            return $fn;
        }
    }

error:
    return q{};
}
sub home_find_file {
    my ($code, $spec) = @_;
    my ($name, $home);

    $home = $ENV{'HOME'};
    $home =~ s/\/+$//;

    if ($code == $FIND_RC) {
        $name = __home_find_file($home, 'rc', '.arenamerc');
    } elsif ($code == $FIND_HOOKS) {
        $name = __home_find_file($home, 'hooks', '.arename.hooks');
    } elsif ($code == $FIND_PROFRC) {
        $name = __home_find_file(
            $home,
            sprintf('profile.%s', $spec),
            sprintf('.arename.%s', $spec)
        );
    } elsif ($code == $FIND_PROFHOOKS) {
        $name = __home_find_file(
            $home,
            sprintf('profile.%s.hooks', $spec),
            sprintf('.arename.%s.hooks', $spec)
        );
    } else {
        croak("home_find_file(): unknown code ($code). Please report!\n");
    }

    odebug(qq{home_find_file($code, "$spec") returning ($name)\n});
    return $name;
}

sub __is_comment_or_blank {
    my ($string) = @_;

    if ($string =~ m/^\s*#/ || $string =~ m/^\s*$/) {
        return 1;
    }

    return 0;
}
sub __remove_leading_backslash {
    my ($strref) = @_;

    if (defined ${ $strref }) {
        ${ $strref } =~ s/^\\//;
    }
    return 1;
}
sub __remove_leading_whitespace {
    my ($strref) = @_;

    ${ $strref } =~ s/^\s*//;
    return 1;
}
sub __rcload_stats {
    my ($desc, $count, $warnings) = @_;

    if (!get_opt('load_quiet')) {
        oprint("Read $desc.\n");
        oprint("$count valid items.\n");
    }
    if ($warnings > 0) {
        oprint("$warnings warnings.\n");
        if (get_opt('warningsautodryrun') && !get_opt('dryrun')) {
            owarn("Encountered warnings in $desc; enabling 'dryrun'.\n");
            owarn("  (See 'warningsautodryrun' option.)\n");
            set_opt('dryrun', 1);
        }
    }

    return 1;
}
sub __rcload {
    my ($file, $desc) = @_;
    my ($fh, $retval);
    my ($count, $warnings, $lnum) = (0, 0, 0);

    if (!open($fh, q{<}, $file)) {      ## no critic
        ## use critic
        carp("Failed to read $desc ($file).\n");
        owarn("Reason: $!\n");
        return 1;
    }

    print qq{Reading "$file"...\n} if (!get_opt('shutup')
                                    && !get_opt('load_quiet'));

    while (my $line = <$fh>) {
        chomp $line;
        $lnum++;

        if (__is_comment_or_blank($line)) {
            next;
        }

        __remove_leading_whitespace(\$line);

        my ($key,$val) = split /\s+/, $line, 2;

        __remove_leading_backslash(\$val);

        $retval = parse($file, $lnum, $count, $key, $val);
        if ($retval < 0) {
            owarn("$file,$lnum: invalid line '$line'.\n");
            return -1;
        } elsif ($retval > 1) {
            owarn("unknown-line:$file,$lnum: $line\n");
            $warnings++;
        } elsif ($retval > 0) {
            owarn("warning:$file,$lnum: $line\n");
            $warnings++;
        } else {
            $count++;
        }
    }
    close $fh or owarn(qq{Failed to close "$file": $!\n});

    sect_reset();
    __rcload_stats($desc, $count, $warnings);

    return 0;
}
sub rcload {
    my ($rc, $desc) = @_;
    my ($retval);

    $retval = __rcload($rc, $desc);

    if ($retval < 0) {
        croak(qq{Error(s) in "$rc". Aborting.\n});
    } elsif ($retval > 0) {
        owarn("Error opening configuration file.\n");
    }

    return 1;
}
sub read_rcs {
    my ($rc);

    if (cmdopts("rc")) {
        $rc = cmdoptstr("rc");
    } else {
        $rc = home_find_file($FIND_RC, $FIND_NONAME);
    }

    if ($rc eq q{}) {
        owarn("main configuration file not found. Using defaults.\n");
    } else {
        rcload($rc, "main configuration");
    }

    if (cmdopts("post-rc")) {
        $rc = cmdoptstr("post-rc");
        rcload($rc, "additional configuration");
    }

    if (get_opt('useprofiles')) {
        foreach my $profile (get_profile_list()) {
            $rc = home_find_file($FIND_PROFRC, $profile);
            if ($rc ne q{}) {
                rcload($rc, qq{configuration for profile "$profile"});
            } else {
                owarn(qq{Profile "$profile" active, but no configuration found.\n});
            }
        }
    }

    if (get_opt('uselocalrc') && -r "./.arename.local") {
        $rc = "./.arename.local";
        rcload($rc, "local configuration");
    }

    return 1;
}

%parsers = (
    '^\s*\[.*\]\s*$'        => \&parse_new_section,
    '^canonicalize$'        => \&parse_bool,
    '^copymode$'            => \&parse_bool,
    '^checkprofilerc$'      => \&parse_bool,
    '^comp_template$'       => \&parse_string,
    '^default_.*$'          => \&parse_defaultvalues,
    '^debug$'               => \&parse_bool,
    '^hookerrfatal$'        => \&parse_bool,
    '^prefix$'              => \&parse_string,
    '^profile$'             => \&parse_profile,
    '^quiet$'               => \&parse_bool,
    '^quiet_skip$'          => \&parse_bool,
    '^sepreplace$'          => \&parse_string,
    '^set$'                 => \&parse_set,
    '^template$'            => \&parse_string,
    '^tnpad$'               => \&parse_integer,
    '^usehooks$'            => \&parse_bool,
    '^uselocalhooks$'       => \&parse_bool,
    '^uselocalrc$'          => \&parse_bool,
    '^useprofiles$'         => \&parse_bool,
    '^verbose$'             => \&parse_bool,
    '^warningsautodryrun$'  => \&parse_bool,
);

sub parse {
    my ($file, $lnum, $count, $key, $val) = @_;

    foreach my $pattern (sort keys %parsers) {
        if ($key =~ m/$pattern/) {

            return $parsers{$pattern}->(
                        $file, $lnum, $count,
                        $key, (defined $val ? $val : q{})
                   );
        }
    }

    # return 2, to tell __rcload() that we had no parser, which matched
    # this line, unknown lines should be warnings, not fatal error.
    return 2;
}

# parser sub functions
#   return values:
#       all okay: =0
#       warning : >0
#       fatal   : <0
sub parse_bool {
    my ($file, $lnum, $count, $key, $val) = @_;

    if (!defined $val || $val eq q{}
       || $val =~ m/^true$/i || $val eq q{1}) {

        $val = 1;
    } elsif ($val =~ m/^false$/i || $val eq q{0}) {
        $val = 0;
    } else {
        owarn("$file,$lnum: unknown boolean value for '$key': '$val'\n");
        return -1
    }

    oprint_verbose(qq{boolean option "$key" = '} . ($val ? 'true' : 'false' ) . qq{'\n});

    set_opt($key, $val);

    return 0;
}
sub parse_defaultvalues {
    my ($file, $lnum, $count, $key, $val) = @_;

    $key =~ s/^default_//;

    if (!tag_supported($key)) {
        owarn("$file,$lnum: Default for unsupported tag found: '$key'\n");
        return -1;
    }

    oprint_verbose(qq{default for "$key" = '$val'\n});

    set_defaults($key, $val);

    return 0;
}
sub parse_integer {
    my ($file, $lnum, $count, $key, $val) = @_;

    if ($val ne q{} && $val !~ m/^\d+$/) {
        owarn("$file,$lnum: Broken integer value for '$key': '$val'\n");
        return -1;
    }

    $val = 0 if ($val eq q{});

    oprint_verbose(qq{integer option "$key" = $val\n});

    set_opt($key, $val);

    return 0;
}
sub parse_profile {
    my ($file, $lnum, $count, $key, $val) = @_;
    my ($name, $pat, $home);

    ($name, $pat) = $val =~ m/([^\s]+)\s+(.*)/;
    if (!defined $name || !defined $pat || $name eq q{} || $pat eq q{}) {
        owarn(qq{Could not parse profile value "$val"\n});
        return 1;
    }

    if ($name =~ m/[^a-zA-Z0-9_-]/) {
        owarn("Disallowed charaters in profile name ($name)!\n");
        return 1;
    }

    if (get_opt('checkprofilerc') && home_find_file($FIND_PROFRC, $name) eq q{}) {
        owarn(qq{Could not find config file for profile "$name".\n});
        return 1;
    }

    $home = $ENV{'HOME'};
    $home =~ s/\/+$//;
    $home .= q{/};

    $pat =~ s/^\\//;        # throw away a leading backslash
    $pat =~ s/^~\//$home/;  # ~/ -> $HOME

    oprint_verbose(qq{Adding pattern "$pat" to profile "$name".\n});
    push @{ $profiles{$name} }, $pat;

    return 0;
}
sub parse_string {
    my ($file, $lnum, $count, $key, $val) = @_;

    oprint_verbose(qq{string option "$key" = '$val'\n});

    set_opt($key, $val);

    return 0;
}
sub parse_new_section {
    my ($file, $lnum, $count, $key, $val) = @_;

    my ($s) = $key =~ m/^\s*\[(.*)\]\s*$/;

    if (!defined $s) {
        owarn("Broken section start: ($key)\n");
        return -1;
    }

    $s =~ s/^~\//$ENV{HOME}\//;
    sect_set($s);

    oprint_verbose(qq{Switching section: "$s"\n});

    return 0;
}
sub parse_set {
    my ($file, $lnum, $count, $key, $val) = @_;

    my ($name, $value) = $val =~ m/\s*(\w+)\s*=\s*\\?(.*)/;
    if (!defined $name || !defined $value) {
        owarn("Broken user setting: ($val)\n");
        return 1;
    }

    oprint_verbose(qq{user setting "$name" = '$value'\n});

    return user_set($name, $value);
}

# handling commandline options
Readonly::Scalar my $CMDLINE_PROTECT => 1;

sub checkstropts {
    my (@o) = @_;

    foreach my $opt (@o) {
        if (exists $opts{$opt} && !defined $opts{$opt}) {
            owarn(" -$opt *requires* a string argument!\n");
        }
    }

    return 1;
}
sub cmdopts {
    my (@args) = @_;

    foreach my $opt (@args) {
        return 0 if (!defined $opts{$opt});
    }

    odebug("@args options given!\n");
    return 1;
}
sub cmdopts_or {
    my (@args) = @_;

    foreach my $opt (@args) {
        return 1 if (defined $opts{$opt});
    }

    return 0;
}
sub cmdoptstr {
    my ($opt) = @_;

    return $opts{$opt};
}

sub __handle_options {
    my ($opt, $key, $val) = @_;

    if (!defined $val) {
        odebug("$opt: \"$key\"\n");
    } else {
        odebug("$opt: \"$key\" ($val)\n");
    }

    if ($opt eq 'userset') {
        $sets{$key} = $val;
    } else {
        $opts{$opt} = $key;
    }

    return 0;
}

sub read_cmdline_options {
    my ($tmp, $rc);

    if ($#main::ARGV == -1) {
        $opts{'help'} = 1;
    } else {
        Getopt::Long::Configure('require_order', 'no_gnu_compat',
            'auto_abbrev', 'no_ignore_case', 'bundling');

        $rc = GetOptions(
            "compare-versions" => \&__handle_options,
            "copy|c" => \&__handle_options,
            "debug|D" => \&__handle_options,
            "disable-hooks|H" => \&__handle_options,
            "disable-profiles|N" => \&__handle_options,
            "dryrun|d" => \&__handle_options,
            "enable-hooks" => \&__handle_options,
            "force|f" => \&__handle_options,
            "help|h" => \&__handle_options,
            "list-cfg|L" => \&__handle_options,
            "list-profiles|S" => \&__handle_options,
            "quiet|q" => \&__handle_options,
            "read-local|l" => \&__handle_options,
            "stdin|s" => \&__handle_options,
            "uber-quiet|Q" => \&__handle_options,
            "verbose|v" => \&__handle_options,
            "version|V" => \&__handle_options,
            "rc=s" => \&__handle_options,
            "post-rc=s" => \&__handle_options,
            "prefix|p=s" => \&__handle_options,
            "profile|P=s" => \&__handle_options,
            "compilation-template|T=s" => \&__handle_options,
            "template|t=s" => \&__handle_options,
            "userset|u=s%" => \&__handle_options,
        );
        if (!$rc) {
            croak("    Try $NAME -h\n");
        }
    }

    # turn on debugging early
    __set_opt("debug", 1, $CMDLINE_PROTECT) if (cmdopts("debug"));

    set_opt('shutup', 1, $CMDLINE_PROTECT)
        if (cmdopts("list-cfg") || cmdopts("list-profiles"));

    if (cmdopts("help")) {
        usage();
        exit 0;
    }

    if (cmdopts_or("quiet", "uber-quiet") && cmdopts("verbose")) {
        croak("Verbose *and* quiet? Please decide!\n");
    }

    if (cmdopts("version")) {
        print "$NAME $VERSION\n";
        exit 0;
    }

    if (cmdopts("compare-versions")) {
        print "scriptver : $VERSION\n";
        show_version();
        exit 0;
    }

    $tmp = cmdoptstr("profile");
    if (defined $tmp && $tmp ne q{}) {
        @cmdline_profiles = split /,/, $tmp;
    }
    __set_opt("useprofiles", 0, $CMDLINE_PROTECT)
        if (cmdopts("disable-profiles"));
    __set_opt("readstdin", 1, $CMDLINE_PROTECT)
        if (cmdopts("stdin"));
    __set_opt("uselocalrc", 1, $CMDLINE_PROTECT)
        if (cmdopts("read-local"));

    if ($#main::ARGV < 0 && !cmdopts("list-cfg")
        && !cmdopts("list-profiles") && !get_opt('readstdin')) {
        croak("No input files given; try $NAME -h.\n");
    }

    return 1;
}

sub read_cmdline_options_late {
    __set_opt("verbose", 1, $CMDLINE_PROTECT)
        if (cmdopts("verbose"));
    __set_opt("quiet", 1, $CMDLINE_PROTECT)
        if (cmdopts_or("quiet", "uber-quiet"));
    __set_opt("quiet_skip", 1, $CMDLINE_PROTECT)
        if (cmdopts("uber-quiet"));
    __set_opt("force", 1, $CMDLINE_PROTECT)
        if (cmdopts("force"));
    __set_opt("dryrun", 1, $CMDLINE_PROTECT)
        if (cmdopts("dryrun"));
    __set_opt("copymode", 1, $CMDLINE_PROTECT)
        if (cmdopts("copy"));
    __set_opt("template", cmdoptstr("template"), $CMDLINE_PROTECT)
        if (cmdopts("template"));
    __set_opt("comp_template", cmdoptstr("compilation-template"),
              $CMDLINE_PROTECT)
        if (cmdopts("compilation-template"));
    __set_opt("prefix", cmdoptstr("prefix"), $CMDLINE_PROTECT)
        if (cmdopts("prefix"));
    disable_hooks() if (cmdopts("disable-hooks"));
    enable_hooks() if (cmdopts("enable-hooks"));
    if (cmdopts("list-profiles")) {
        dump_profiles();
        exit 0;
    }
    if (cmdopts("list-cfg")) {
        dump_config();
        exit 0;
    }

    return 1;
}

# file system related code

sub ensure_dir {
    # think: mkdir -p /foo/bar/baz
    my ($wantdir) = @_;
    my (@parts, $sofar);

    if (-d $wantdir) {
        return 1;
    }

    if ($wantdir =~ q{^/}) {
        $sofar = q{/};
    } else {
        $sofar = q{};
    }

    @parts = split /\//, $wantdir;
    foreach my $part (@parts) {
        if ($part eq q{}) {
            next;
        }
        $sofar = (
                  $sofar eq q{}
                    ? $part
                    : (
                        $sofar eq q{/}
                          ? q{/} . $part
                          : $sofar . q{/} . $part
                      )
                 );

        if (!-d $sofar) {
            if ((get_opt("dryrun") || get_opt("verbose")) && !get_opt("quiet")) {
                oprint(qq{mkdir "$sofar"\n});
            }
            if (!get_opt("dryrun")) {
                mkdir $sofar or croak("Could not mkdir($sofar).\n" .
                                      "Reason: $!\n");
            }
        }
    }

    return 1;
}
sub file_eq {
    my ($f0, $f1) = @_;
    my (@stat0, @stat1);

    if (!-e $f0 || !-e $f1) {
        # one of the two doesn't even exist, can't be the same then.
        return 0;
    }

    @stat0 = stat $f0 or croak("Could not stat($f0): $!\n");
    @stat1 = stat $f1 or croak("Could not stat($f1): $!\n");

    if ($stat0[0] == $stat1[0] && $stat0[1] == $stat1[1]) {
        # device and inode are the same. same file.
        return 1;
    }

    return 0;
}
sub xrename {
    # a rename() replacement, that implements renames across
    # filesystems via File::copy() + unlink().
    # This assumes, that source and destination directory are
    # there, because it stat()s them, to check if it can use
    # rename().
    my ($src, $dest) = @_;
    my (@stat0, @stat1, $d0, $d1, $cause);

    $d0 = dirname($src);
    $d1 = dirname($dest);
    @stat0 = stat $d0 or croak("Could not stat($d0): $!\n");
    @stat1 = stat $d1 or croak("Could not stat($d1): $!\n");

    if ($stat0[0] == $stat1[0]) {
        $cause = 'rename';
        rename $src, $dest or goto err;
    } else {
        $cause = 'copy';
        copy($src, $dest) or goto err;
        $cause = 'unlink';
        unlink $src or goto dir;
    }

    return 0;

err:
    croak("Could not rename($src, $dest);\n" .
          "Reason: $cause(): $!\n");
}
sub xcopy {
    my ($src, $dest) = @_;

    return copy($src, $dest) or croak("Could not copy($src, $dest): $!\n");
}

# {get,set}_opt() API

sub get_opt {
    my ($opt) = @_;
    my ($section) = (undef);

    odebug("GET OPTION ($opt)\n");
    if (defined $cmdline_protect{$opt} && $cmdline_protect{$opt} eq 'yes') {
        return $conf{$opt};
    }

    if (is_locopt($opt)) {
        $section = section_matches(get_file());
    }

    if (defined $section && $sectconf{$section}{$opt}) {
        odebug("returning $conf{$opt} (section: $section)\n");
        return $sectconf{$section}{$opt};
    } else {
        if (defined $conf{$opt}) {
            odebug("returning $conf{$opt}\n");
        }
        return $conf{$opt};
    }
}
sub set_opt {
    my ($opt, $val) = @_;

    if (
        ($opt eq 'verbose' && $val == 1 && get_opt('quiet')) ||
        ($opt eq 'quiet' && $val == 1 && get_opt('verbose'))
       ) {

        return 0 if ($opt eq 'quiet' && cmdopts("verbose"));
        return 0 if ($opt eq 'verbose' &&
            (cmdopts("quiet") || cmdopts("uber-quiet")));

        croak("verbose and quiet set at the same time. Check your config.\n");
    }

    return 0 if (!__set_opt($opt, $val));

    if ($opt eq 'quiet_skip' && $val == 1 && !get_opt('quiet')) {
        __set_opt('quiet', 1);
    }

    return 1;
}
sub __set_opt {
    my ($opt, $val, $protect) = @_;

    my $s = sect_get();

    if (defined $s && !is_locopt($opt)) {
        owarn(qq{"$opt" is *not* a localizable setting (will not set to $val).\n});
        return 0;
    }

    if ((!defined $protect || $protect == 0)
        && defined $cmdline_protect{$opt} && $cmdline_protect{$opt} eq 'yes') {

        owarn(qq{"$opt" was set on the command line. Not overwriting.\n});
        return 0;
    }

    if (!defined $s) {
        odebug("set_opt() ($opt) = ($val)\n");
        $conf{$opt} = $val;
    } else {
        odebug("set_opt() ($opt) = ($val) [$s]\n");
        $sectconf{$s}{$opt} = $val;
    }
    if (defined $protect && $protect > 0) {
        odebug("Protecting $opt from being overidden later.\n");
        $cmdline_protect{$opt} = 'yes';
    }

    return 1;
}

# accessing the currently processed audio file's name

sub get_file {
    return $__arename_file;
}
sub set_file {
    my ($fname) = @_;

    $__arename_file = $fname;
    return 1;
}

# default_* code

sub get_defaults {
    my ($key) = @_;

    return $defaults{$key};
}
sub get_default_keys {

    return sort keys %defaults;
}
sub set_defaults {
    my ($key, $val) = @_;

    $defaults{$key} = $val;
    return 1;
}

# user-defined-variable API

sub user_get {
    my ($opt) = @_;

    return $sets{$opt}
}
sub user_set {
    my ($opt, $val) = @_;

    if (defined sect_get()) {
        owarn("User-variables in sections are not valid.\n");
        return 1;
    }

    $sets{$opt} = $val;
    return 0;
}

# section handling code

sub is_locopt {
    my ($opt) = @_;

    foreach my $lo (@localizables) {
        if ($lo eq $opt) {
            return 1;
        }
    }

    return 0;
}
sub section_matches {
    my ($filename) = @_;

    if (!defined $filename) {
        return;
    }

    # The block in the sort call makes sure we always get the longest
    # section names first; that way /foo/bar/ supersedes /foo/.
    foreach my $section (sort { length $b <=> length $a } keys %sectconf) {
        my $substring = substr $filename, 0, length $section;
        odebug("<$section> ($filename) eq [generated from $filename] ($substring)\n");

        if ($substring eq $section) {
            odebug("$section MATCHED! returning it.\n");
            return $section;
        }
    }

    return;
}
sub sect_get {
    return $sect;
}
sub sect_set {
    my ($section) = @_;

    $sect = $section;
    return 1;
}
sub sect_reset {
    $sect = undef;
    return 1;
}

# hooks code

sub enable_hooks {
    __set_opt("usehooks", 1);
    __set_opt("uselocalhooks", 1);

    return 1;
}
sub disable_hooks {
    __set_opt("usehooks", 0);
    __set_opt("uselocalhooks", 0);

    return 1;
}
sub __read_hook_file {
    my ($file) = @_;
    my ($rc);

    if ($file eq q{}) {
        return 1;
    }

    if (! -e $file) {
        owarn_verbose("Hook file not found ($file).\n");
        return 1;
    }

    $rc = do $file;

    if (!defined $rc && $EVAL_ERROR) {
        owarn("Could not parse hooks file ($file):\n   - $@\n");
        if (get_opt("hookerrfatal")) {
            exit 1;
        } else {
            return 0;
        }
    } elsif (!defined $rc) {
        owarn("Could not read hooks file ($file):\n   - $!\n");
        if (get_opt("hookerrfatal")) {
            exit 1;
        } else {
            return 0;
        }
    } elsif ($rc != 1 && !get_opt('load_quiet')) {
        owarn(qq{Reading hooks file "$file" did not return 1.\n});
        owarn("  While this is not a fatal problem, it is good practice, to let\n");
        owarn("  perl script files return 1. Just put a '1;' into the last line\n");
        owarn("  of this file to get rid of this warning. Subroutines like\n");
        owarn("  register_hook() and remove_hook() return 1 by default, so sticking\n");
        owarn("  them into the last line of the file will do the trick, too.\n");
    }

    oprint("Hook file read ($file).\n") if (!get_opt('load_quiet'));
    return 1;
}
sub read_hook_files {
    if (get_opt("usehooks")) {
        __read_hook_file( home_find_file($FIND_HOOKS, $FIND_NONAME) );
    }

    if (get_opt('useprofiles')) {
        foreach my $profile (get_profile_list()) {
            __read_hook_file( home_find_file($FIND_PROFHOOKS, $profile) );
        }
    }

    if (get_opt("uselocalhooks")) {
        __read_hook_file("./.arename.hooks.local");
    }

    return 1;
}
sub register_hook {
    my ($namespace, $funref) = @_;

    if (!defined &{ $funref }) {
        owarn("Trying to register undefined subroutine in namespace: $namespace; Ignoring.\n");
        return 1;
    }
    push @{ $hooks{$namespace} }, $funref;

    return 1;
}
sub remove_hook {
    my ($namespace, $funref) = @_;

    for my $i (0 .. scalar @{ $hooks{$namespace} } - 1) {
        if ($funref == $hooks{$namespace}[$i]) {
            # found; remove and rerun ourself to be sure
            # the coderef is not registered more than once
            # in this namespace.
            splice @{ $hooks{$namespace} }, $i, 1;
            remove_hook($namespace, $funref);
            return 1;
        }
    }
    return 1;
}
sub run_hook {
    my ($ns, @args) = @_;

    my $namespace = $ns;
    shift;

    if (!defined $hooks{$namespace} || scalar @{ $hooks{$namespace} } == 0) {
        return 0;
    }

    foreach my $funref (@{ $hooks{$namespace} }) {
        $funref->($namespace, @args);
    }

    return 1
}
sub startup_hook {
    return run_hook(
                'startup',
                \$NAME, \$VERSION,
                \%conf, \%methods,
                \@supported_tags, \@main::ARGV
           );
}

# configuration dumping code

sub dump_string {
    my ($s) = @_;

    if ($s =~ m/^\s/) {
        return "\\$s";
    } else {
        return "$s";
    }
}
sub __dump_config {
    my (
        $formatfull, $formatnoval,
        $hashref,
        $keylistref, $excludelistref
    ) = @_;

    KEYLOOP: foreach my $key (sort @{ $keylistref }) {
        foreach my $exclude (sort @{ $excludelistref }) {
            if ($key eq $exclude) {
                next KEYLOOP;
            }
        }

        if (!exists ${ $hashref }{$key } && !defined ${ $hashref }{$key}) {
            next KEYLOOP;
        }

        my $val = dump_string(${ $hashref }{$key});

        if ($val eq q{}) {
            printf "$formatnoval", $key;
        } else {
            printf "$formatfull", $key, $val;
        }
    }

    return 1;
}
sub __dump_config_print {
    my ($hashref, $string) = @_;

    if (%{ $hashref }) {
        print "$string";
    }

    return 1;
}
sub dump_config {
    my ($vffull, $vfnoval) = ( "%-18s   %s\n", "%s\n");
    my ($tffull, $tfnoval) = ( "%-13s   %s\n", "%s\n");

    __dump_config($vffull, $vfnoval, \%conf, \@settables, [ "comp_template", "template" ]);
    print "\n";
    __dump_config($tffull, $tfnoval, \%conf, [ "comp_template", "template" ], [ ]);

    __dump_config_print(\%defaults, "\n# default values for missing tags\n\n");
    __dump_config("default_%s %s\n", "default_%s\n", \%defaults, [ keys %defaults ], [ ]);

    __dump_config_print(\%sets, "\n# user defined settings (see manual for details)\n\n");
    __dump_config("set %s = %s\n", "set %s =\n", \%sets, [ keys %sets ], [ ]);

    __dump_config_print(\%profiles, "\n# profile definitions (commented out on purpose)\n\n");
    foreach my $key (sort keys %profiles) {
        foreach my $pat (@{ $profiles{$key} }) {
            print "#profile $key $pat\n";
        }
    }

    __dump_config_print(\%sectconf, "\n# section definition(s)\n");
    foreach my $sect (sort keys %sectconf) {
        print "\n[$sect]\n";

        __dump_config($vffull, $vfnoval, \%{ $sectconf{$sect} },
            [ keys %{ $sectconf{$sect} } ], [ "comp_template", "template" ]);
        __dump_config($tffull, $tfnoval, \%{ $sectconf{$sect} },
            [ "comp_template", "template" ], [ ]);
    }

    exit 0;
}
sub dump_profiles {
    foreach my $profile (sort keys %profiles) {
        print "$profile\n";
    }

    return 1;
}
sub show_version {
    ## no critic
    print 'ARename.pm: 3.1' . "\n";
    ## use critic
    return 1;
};
1;
