#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Copyright 2021, Nils Hilbricht, Germany ( https://www.hilbricht.net )

This file is part of the Laborejo Software Suite ( https://www.laborejo.org ).

This application is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import logging; logger = logging.getLogger(__name__); logger.info("import")

#Standard Library
import datetime

#Third Party
from PyQt5 import QtCore, QtGui, QtWidgets

#Engine
from engine.config import METADATA  #includes METADATA only. No other environmental setup is executed.
import engine.api as api



class SessionButton(QtWidgets.QPushButton):

    def __init__(self, sessionDict):
        self.sessionDict = sessionDict
        super().__init__(sessionDict["nsmSessionName"])
        self.clicked.connect(self.openSession)
        #self.setFlat(True)

        font = self.font()
        font.setPixelSize(font.pixelSize() * 1.2 )
        self.setFont(font)

        #width = self.fontMetrics().boundingRect(sessionDict["nsmSessionName"]).width()+10
        #width = self.fontMetrics().boundingRect(longestSessionName).width()+10
        #width = parent.geometry().width()
        #self.setFixedSize(width, 40)
        self.setFixedHeight(40)


    def openSession(self):
        name = self.sessionDict["nsmSessionName"]
        api.sessionOpen(name)

class QuickSessionController(object):
    """Controls the widget, but does not subclass"""

    def __init__(self, mainWindow):
        self.mainWindow = mainWindow
        self.layout = mainWindow.ui.quickSessionChooser.layout()
        #self.layout.setAlignment(QtCore.Qt.AlignHCenter)
        newSessionButton = mainWindow.ui.quickNewSession

        font = newSessionButton.font()
        font.setPixelSize(font.pixelSize() * 1.4)
        newSessionButton.setFont(font)

        newSessionButton.setFixedHeight(40)
        newSessionButton.setFocus(True) #Enter on program start creates a new session.
        newSessionButton.clicked.connect(self._newTimestampSession)
        api.callbacks.sessionsChanged.append(self._reactCallback_sessionsChanged)

        #self.layout.geometry().width() #very small
        #self.mainWindow.ui.quickSessionChooser.geometry().width() #too small
        #self.mainWindow.ui.scrollArea.geometry().width()
        #mainWindow.geometry().width()


        logger.info("Quick Session Chooser ready")

    def _clear(self):
        """Clear everything but the spacer item"""
        for child in self.mainWindow.ui.quickSessionChooser.children():
            if type(child) is SessionButton:
                self.layout.removeWidget(child)
                child.setParent(None)
                del child



    def _reactCallback_sessionsChanged(self, sessionDicts:list):
        """Main callback for new, added, removed, moved sessions etc."""
        logger.info("Rebuilding session buttons")
        self._clear() #except the space

        spacer = self.layout.takeAt(0)

        #longestSessionName = ""
        #for sessionDict in sessionDicts:
        #    if len(sessionDict["nsmSessionName"]) > len(longestSessionName):
        #        longestSessionName = sessionDict["nsmSessionName"]

        for sessionDict in sorted(sessionDicts, key=lambda d: d["nsmSessionName"]):
            self.layout.addWidget(SessionButton(sessionDict))

        #Finally add vertical spacer
        #spacerItem = QtWidgets.QSpacerItem(1, 1, QtWidgets.QSizePolicy.Minimum, QtWidgets.QSizePolicy.Expanding)  #int w, int h, QSizePolicy::Policy hPolicy = QSizePolicy::Minimum, QSizePolicy::Policy vPolicy = QSizePolicy::Minimum
        self.layout.addItem(spacer)

    def _newTimestampSession(self):
        nsmExecutables = api.getNsmExecutables() #type set, cached, very fast.
        con = METADATA["preferredClients"]["data"]
        data = METADATA["preferredClients"]["connections"]
        startclients = []
        if con in nsmExecutables:
            startclients.append(con)
        if data in nsmExecutables:
            startclients.append(data)

        #now = datetime.datetime.now().replace(second=0, microsecond=0).isoformat()[:-3]
        now = datetime.datetime.now().replace(microsecond=0).isoformat()
        name = now
        api.sessionNew(name, startclients)
