/*
 * This file is part of ACGVision, SaaS system-monitoring software
 * Copyright (C) 2009 ACGCenter
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.acgvision.agent.action;

import com.acgvision.agent.run.Main;
import com.acgvision.agent.run.Task;
import com.acgvision.core.ws.AutomaticAction;


import com.acgvision.core.ws.Average;
import com.acgvision.core.ws.Command;
import com.acgvision.core.ws.Control;
import com.acgvision.core.ws.Cpu;
import com.acgvision.core.ws.FileSystem;
import com.acgvision.core.ws.Job;
import com.acgvision.core.ws.LimitValue;
import com.acgvision.core.ws.LogFile;
import com.acgvision.core.ws.Memory;
import com.acgvision.core.ws.Monitor;
import com.acgvision.core.ws.Script;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.HierarchicalConfiguration.Node;
import org.apache.commons.configuration.HierarchicalINIConfiguration;
import org.apache.log4j.Logger;

/**
 *
 * @author Rémi Debay <remi.debay@acgcenter.com>
 */
public class Check {

    private static final Logger logger = Logger.getLogger(Check.class);
    private final static int NULLINPUT = -1;
    private final static String COMMANDSECTION = "Commands";


        private static String removeInvalidXMLCharacters(String s) {
        StringBuilder out = new StringBuilder();                // Used to hold the output.
    	int codePoint;                                          // Used to reference the current character.
    	//String ss = "\ud801\udc00";                           // This is actualy one unicode character, represented by two code units!!!.

		int i=0;
    	while(i<s.length()) {
    	//	System.out.println("i=" + i);
    		codePoint = s.codePointAt(i);
                // This is the unicode code of the character.
                if ((codePoint == 0x9) ||          				    // Consider testing larger ranges first to improve speed.
                                (codePoint == 0xA) ||
                                (codePoint == 0xD) ||
                                ((codePoint >= 0x20) && (codePoint <= 0xD7FF)) ||
                                ((codePoint >= 0xE000) && (codePoint <= 0xFFFD)) ||
                                ((codePoint >= 0x10000) && (codePoint <= 0x10FFFF))) {                    
                        out.append(Character.toChars(codePoint));
                }else{
                        if(codePoint==0x1B){
                            logger.debug("0x1b correctly handled");
                        }
                }
                i+= Character.charCount(codePoint);                 // Increment with the number of code units(java chars) needed to represent a Unicode char.
    	}
    	return out.toString();
    }

    /**
     * Executes a command
     * @param command Command to execute
     * @return the value
     * @throws java.io.IOException
     * @throws java.lang.InterruptedException
     */
    private static ExecResult run(Command command) throws IOException, InterruptedException {
        if (command == null) {
            return new ExecResult(NULLINPUT, "");
        }
        if (command.getCommand() == null || command.getCommand().isEmpty()) {
            return new ExecResult(NULLINPUT, "");
        }
        if (command.getDirectory() == null || command.getDirectory().isEmpty()) {
            return new ExecResult(NULLINPUT, "");
        }
        Process proc = null;

        try {
            if (logger.isInfoEnabled()) {
                logger.info("Excution de la commande :" + command.getCommand());
            }

            if (command.getDirectory() != null) {
                File file = new File(command.getDirectory());


                proc = Runtime.getRuntime().exec(command.getCommand(), null, file);
            } else {
                proc = Runtime.getRuntime().exec(command.getCommand(), null);
            }

            proc.waitFor();
//            if (logger.isDebugEnabled()) {
//                logger.debug(proc.getOutputStream());
//            }
            //lecture de la sortie par un buffer
            String output = "";
            try {

                BufferedReader reader = new BufferedReader(new InputStreamReader(proc.getInputStream()));
                String line = null;
                try {
                    while ((line = reader.readLine()) != null) {
                        // Traitement du flux de sortie de l'application si besoin est
                        output += new String(line.getBytes(),"UTF-8") + "\n";
                    }
                } finally {
                    reader.close();
                }
            } catch (IOException ioe) {
                ioe.printStackTrace();
            }

            //ExecResult result = new ExecResult(proc.exitValue(),proc.getOutputStream().toString());
            //output =output.getBytes(),"UTF-8");
            ExecResult result = new ExecResult(proc.exitValue(),removeInvalidXMLCharacters(output));

            if (logger.isInfoEnabled()) {
                logger.info("Resultat de la commande :" + result.getCode());
                logger.info("Sortie de la commande :" + result.getText());
            }
            proc.destroy();

            return result;
        } catch (IOException ex) {
            logger.error("Erreur lors de l'éxécution de la commande :" + command.getCommand());
            logger.error("Dans le répertoire :" + command.getDirectory());
            logger.error(proc.getOutputStream());
            logger.error(proc.getErrorStream());
            throw ex;
        }
    }

    /**
     * Execute the given command and wait until it ends
     * @param command to execute
     * @return exitValue
     * @throws java.io.IOException if an error occurs while running the command.
     * @throws java.lang.InterruptedException
     */
    private static ExecResult run(String command) throws IOException, InterruptedException {
        Process proc = null;
        try {
            if (logger.isInfoEnabled()) {
                logger.info("Excution de la commande :" + command);
            }
            proc = Runtime.getRuntime().exec(command);
            proc.waitFor();

            String output = "";
            try {

                BufferedReader reader = new BufferedReader(new InputStreamReader(proc.getInputStream()));
                String line = null;
                try {
                    while ((line = reader.readLine()) != null) {
                        // Traitement du flux de sortie de l'application si besoin est
                        output += line + "\n";
                    }
                } finally {
                    reader.close();
                }
            } catch (IOException ioe) {
                ioe.printStackTrace();
            }
            //ExecResult result = new ExecResult(proc.exitValue(),proc.getOutputStream().toString());
            ExecResult result = new ExecResult(proc.exitValue(),removeInvalidXMLCharacters(output));
            if (logger.isInfoEnabled()) {
                logger.info("Resultat de la commande :" + result.getCode());
                logger.info("Sortie de la commande :" + result.getText());
            }
            proc.destroy();

            return result;
        } catch (IOException ex) {
            logger.error("Erreur lors de l'éxécution de la commande :" + command);
            logger.error(ex);
            logger.error(proc.getOutputStream());
            logger.error(proc.getErrorStream());
            throw ex;
        } catch (InterruptedException ex) {
            logger.error("Erreur lors de l'éxécution de la commande (Interrompue) :" + command);
            logger.error(proc.getOutputStream());
            logger.error(proc.getErrorStream());
            logger.error(ex);
            throw ex;
        }
    }

    /**
     * Gets cpu current value
     * @param cpu
     * @param configFile
     * @return exit value of the command
     */
    public static ExecResult get(final Cpu cpu, final HierarchicalINIConfiguration configFile) {
        if (cpu == null) {
            return new ExecResult(NULLINPUT, "");
        }
        if (logger.isDebugEnabled()) {
            logger.debug("measure cpu");
        }
        ExecResult value = null;
        try {
            value = run(configFile.getSection(COMMANDSECTION).getString("cpu", "scripts/Linux/acgMeasureCpu.ksh"));
        } catch (IOException ex) {
            logger.error(ex);
        } catch (InterruptedException ex) {
            logger.error(ex);
        }
        return value;
    }

    /**
     * Gets memory curreent value
     * @param mem
     * @param configFile
     * @return exit value of the command
     */
    public static ExecResult get(final Memory mem, final HierarchicalINIConfiguration configFile) {
        if (mem == null) {
            return new ExecResult(NULLINPUT, "");
        }

        if (logger.isDebugEnabled()) {
            logger.debug("measure Memory");
        }
        ExecResult value = null;
        try {
            value = run(configFile.getSection(COMMANDSECTION).getString("memory", "scripts/Linux/acgMeasureMemory.ksh"));

        } catch (IOException ex) {
            logger.error(ex);
        } catch (InterruptedException ex) {
            logger.error(ex);
        }
        return value;
    }

    /**
     * Gets the File System current value
     * @param fsys
     * @param configFile
     * @return exit value of the command
     */
    public static ExecResult get(final FileSystem fsys, final HierarchicalINIConfiguration configFile) {
        if (fsys == null) {
            return new ExecResult(NULLINPUT, "");
        }
        if (fsys.getFileSystem() == null) {
            return new ExecResult(NULLINPUT, "");
        }
        if (fsys.getFileSystem().isEmpty()) {
            return new ExecResult(NULLINPUT, "");
        }
        if (logger.isDebugEnabled()) {
            logger.debug("measure fileSystem");
        }
        if (fsys.getFileSystem() == null) {
            return new ExecResult(NULLINPUT, "");
        }

        ExecResult value = null;
        try {

            value = run(configFile.getSection(COMMANDSECTION).getString("filesystem", "scripts/Linux/acgMeasureFileSystem.ksh") + " " + fsys.getFileSystem());

        } catch (IOException ex) {
            logger.error(ex);
        } catch (InterruptedException ex) {
            logger.error(ex);
        }
        return value;
    }

    /**
     * Gets the script
     * @param sc
     * @return exit value of the command
     */
    public static ExecResult get(final Script sc) {
        if (sc == null) {
            return new ExecResult(NULLINPUT, "");
        }
        if (sc.getCommand() == null) {
            return new ExecResult(NULLINPUT, "");
        }
        if (logger.isDebugEnabled()) {
            logger.debug("measure Script");
        }
        ExecResult value = null;
        try {

            value = run(sc.getCommand());

        } catch (IOException ex) {
            logger.error(ex);
        } catch (InterruptedException ex) {
            logger.error(ex);
        }
        return value;
    }

   /**
    * Vérifie la prsénce d'une erreur sur une moyennede mesure
    * @param v moyenne
    * @param m Monitor concerné
    * @param a 
    */
    public static void check(final int v, final Monitor m, final Average a) {
        if (Main.getConfigFile() != null) {
            Averager averager = new Averager(Main.getConfigFile());
            if (averager.check(a, m)) {
                if (!getState(m, a)) {//si l'état est à false on lance une erreur
                    logger.info("An event occured for monitor:" + m.getName());
                    if(com.acgvision.agent.run.Main.sender.sendEvent(a, m)){
                        setState(m, a, true);
                        executeAutomaticAction(m,a);
                    }else{
                        //une erreur est survenue pendant l'envoi :
                        logger.warn("Une erreur est survenue pendant l'envoi, les éventuelles actions automatiques ne sont pas éxécutées");
                    }
                } else {
                    if (logger.isDebugEnabled()) {
                        logger.debug(m + " still has an error." + a);
                    }
                }
            } else {
                if (logger.isDebugEnabled()) {
                    logger.debug("Monitors average is OK");
                }
                if (getState(m, a)) {//Le job est en erreur
                    logger.info(m + "Closing event");
                    com.acgvision.agent.run.Main.sender.closeEvent(a, m);
                    setState(m, a, false);//on passe le job à faux
                }
            }
        }
    }

    /**
     * Checks if v is valid according to the limit and throw a new event if not
     * @param v the value to check
     * @param m
     * @param a
     */
    public static void check(ExecResult v, final Monitor m, final LimitValue a) {
        //tests if it is higher than a top limit or lower than a down limit
        if (logger.isDebugEnabled()) {
            logger.debug("Monitor :" + m.getName() + "\n" +
                    "LimitValue :" + a.getValue() + "\n" +
                    "Value :" + v.getCode() + "\n" +
                    "High :" + a.isHigh() + "\n" +
                    "Equal :" + a.isEqual() + "\n" +
                    "Down :" + a.isUnder());
        }
        if (a.isHigh() && (v.getCode() > a.getValue()) ||
                a.isUnder() && (v.getCode() < a.getValue()) ||
                a.isEqual() && (v.getCode() == a.getValue())) {


            if (!getState(m, a)) {
                logger.warn("An event occured for monitor:" + m.getName());
                boolean result = false;
                if (v.getText() != null) {
                   result= com.acgvision.agent.run.Main.sender.sendEvent(a, m, v.getText());
                } else {
                    result=com.acgvision.agent.run.Main.sender.sendEvent(a, m);
                }
                if(result){
                    setState(m, a, true);
                    executeAutomaticAction(m,a);
                }else{
                    logger.info("An error occured while openning the event. Keeping event non-openned.");
                }
            } else {
                if (logger.isDebugEnabled()) {
                    logger.debug("The monitor still has an error.");
                }
            }
        } else {
            if (logger.isDebugEnabled()) {
                logger.debug("Monitor limit is ok.");
            }
            if (getState(m, a)) {//Le job est en erreur
                if (logger.isInfoEnabled()) {
                    logger.info("Closing event");
                }
                if(com.acgvision.agent.run.Main.sender.closeEvent(a, m))              setState(m, a, false);//on passe le job à faux
                else if(logger.isInfoEnabled())logger.info("An error occured while closing event. Keeping event non-closed.");
            }
        }
    }

    /**
     * Checks if v is valid according to the list and throw a new event if not
     * @param v the value to check
     * @param m
     */
    public static void check(ExecResult v, Monitor m) {
        if (m != null && v != null) {
            if (logger.isDebugEnabled()) {
                logger.debug("check");
            }
            if (m.getAverages() != null && !m.getAverages().isEmpty()) {
                //on enregistre la valeur pour préparer les moyennes
                Averager averager = new Averager(Main.getConfigFile());
                averager.save(m, v.getCode());

                Iterator<Average> ita = m.getAverages().iterator();
                while (ita.hasNext()) {
                    Average avg = ita.next();
                    if (logger.isInfoEnabled()) {
                        logger.info(m.getName() + " - Checking an average. " + avg.getName());
                    }
                    check(v.getCode(), m, avg);
                }
            }

            Iterator<LimitValue> itlv = m.getLimitValues().iterator();
            while (itlv.hasNext()) {
                LimitValue lm = itlv.next();
                if (logger.isInfoEnabled()) {
                    logger.info(m.getName() + " - Checking a limit. " + lm.getName());
                }
                check(v, m, lm);
            }
        }
    }

    public static void check(Job job) {
        throw new UnsupportedOperationException("Trying to check a job");
    }

    public static void check(LogFile logfile) {
        throw new UnsupportedOperationException("Trying to check a job");
    }

    /**
     * Sets a control value reminder
     * @param m
     * @param c Control to write
     * @param v value to write
     */
    public static void setState(Monitor m, Control c, boolean v) {
        logger.debug("saveState");
        if (c != null && m != null) {
            if (c.getId() != null && m.getId() != null) {
                HierarchicalINIConfiguration configFile = Main.getConfigFile();
                String section;
                section = String.valueOf(m.getId());
                String key;
                key = String.valueOf(c.getId());

                if (Main.getConfigFile() != null) {
                    try {
                        if (configFile.getSection(section).isEmpty()) {//la section n'existe pas
                            //la section n'existe pas.
                            logger.debug("First Value to be saved");
                            List<Node> nodes = new ArrayList<Node>();//
                            nodes.add(new Node(section));
                            configFile.addNodes(section, nodes);
                            configFile.getSection(section).addProperty(key, v);
                            logger.debug("Enregistrement du fichier de config.");
                        } else {//la section n'existe pas, on la sauve
                            if (configFile.getSection(section).containsKey(key)) {
                                //la clé existe on la modifie
                                logger.debug("Known key.");
                                configFile.getSection(section).setProperty(key, v);
                            } else {
                                //la clé n'existe pas, on l'ajoute
                                logger.debug("Unknown key.");
                                configFile.getSection(section).addProperty(key, v);
                            }
                        }
                        configFile.save();
                    } catch (ConfigurationException ex) {
                        logger.error("an error occured while saving job's state onto config file");
                        logger.error(ex);
                    }
                }
            }
        }
    }

    /**
     * Returns the state of the current job with the control
     * @param m Monitor to measure
     * @param c Control to check
     * @return true if the control is on error, false if not or unknown
     */
    public static boolean getState(Monitor m, Control c) {
        if (logger.isDebugEnabled()) {
            logger.debug("getState");
        }
        if (c == null || m == null) {
            return false;
        }
        if (c.getId() == null || m.getId() == null) {
            return false;
        }
        if (Main.getConfigFile() == null) {
            logger.error("Config file is null!!!!");
            return false;
        }

        //on défini les variables
        HierarchicalINIConfiguration configFile = Main.getConfigFile();
        String section = m.getId().toString();
        String key = c.getId().toString();
        //si la section n'existe pas on erreurise
        if (configFile.getSection(section) == null) {
            if (logger.isDebugEnabled()) {
                logger.debug("Section does not exists.");
            }
            return false;
        }
        if (!configFile.getSection(section).containsKey(key)) {
            if (logger.isDebugEnabled()) {
                logger.debug("Key does not exists.");
            }
            return false;
        }
        //la clé existe, on li la valeur et on la retourne
        if (logger.isDebugEnabled()) {
            logger.debug("State is " + configFile.getSection(section).getBoolean(key));
        }
        return configFile.getSection(section).getBoolean(key);
    }

    /**
     * Execute une action automatique si le contrôle le nécessite
     * @param monitor
     * @param control
     */
    private static void executeAutomaticAction(Monitor monitor, Control control) {
        if (control.getAutomaticAction() != null) {
            logger.debug("Le contrôle a une action automatique prévue.");
            //Vérification des données de l'action automatique.
            AutomaticAction act = control.getAutomaticAction();
            if (act.getNumber() > 0 && act.getCommand() != null) {
                try {
                    //Execution de la commande
                    for (int i = 0; i < act.getNumber(); i++) {
                        ExecResult result = run(act.getCommand());
                        if (logger.isInfoEnabled()) {
                            logger.info("L'éxécution automatique a retoruné : " + result.getCode());
                            logger.info(result.getText());
                        }
                        //Envoi des informations au core
                        if(!com.acgvision.agent.run.Main.sender.sendEvent(control, monitor, act, result.getText())){
                            //une erreur est survenue pendant l'envoi :
                            logger.warn("Une erreur est survenue pendant l'envoi, les éventuelles actions automatiques ne sont pas éxécutées");
                        }
                    }

                } catch (IOException ex) {
                    logger.error("An error Occured while executing the automatic Action", ex);
                } catch (InterruptedException ex) {
                    logger.error("An error Occured while executing the automatic Action", ex);
                }
            } else {
                if (act.getNumber() < 0) {
                    logger.error("Laction automatique a un nombre négatif");
                }
                if (act.getCommand() == null) {
                    logger.error("Laction automatique a une commande nulle");
                }
            }
        } else {
            logger.debug("Le contrôle n'a pas d'action automatique");
        }
    }
}
