#include <hash.h>
#include <jvmti.h>

extern jvmtiEnv* jvmti;

typedef struct hashnode hashnode;
struct hashnode {
    void* key;
    void* el;
    hashnode* next;
};

struct hashtab {
    size_t           size;      /** size of table */
    size_t           cardinal;  /** number of elements in hashtable */
    jmphash_f        hashfun;   /** Hash function used. */
    jmphash_cmp_f    cmp;       /** Comparison function used. */
    hashnode       **vec;       /** The table */
};

#define HASH(tab, el) \
  ((tab)->hashfun (el, (tab)->size))

/** Good sizes of the hash table.
 *  primes, doubling in size so if we need to restructure 
 *  we know that (aprox) half of it is used...
 */
static unsigned long   primes[] =
{ 11UL, 23UL, 47UL, 97UL, 197UL, 307UL,
  617UL, 1237UL, 2477UL, 4957UL, 9923UL,
  19853UL, 39709UL, 79423UL, 158849UL,
  317701UL, 635413UL, 1270849UL,
  2541701UL, 5083423UL };

static size_t make_prime (size_t num) {
    int i;
    for (i = 0; i < sizeof (primes) / sizeof (int) && primes[i] < num; ++i)
        ;
    if (i < sizeof (primes) / sizeof (int))
        return primes[i];
    else
        return num;
}

hashtab* jmphash_new (size_t size, jmphash_f hashfun, jmphash_cmp_f cmp) {
    size_t i;
    hashtab* htab;
    size = make_prime (size);
    (*jvmti)->Allocate (jvmti, sizeof(*htab), 
			(unsigned char**)&htab);
    (*jvmti)->Allocate (jvmti, size * sizeof(hashnode*), 
			(unsigned char**)&htab->vec);
    for (i = 0; i < size; i++)
	htab->vec[i] = NULL;
    htab->size = size;
    htab->cardinal = 0;
    htab->hashfun = hashfun;
    htab->cmp = cmp;
    return htab;
}

void jmphash_free (hashtab* htab) {
    size_t i;
    hashnode* cur;
    hashnode* next;

    for (i = 0; i < htab->size; ++i) {
        cur = htab->vec[i];
        while (cur != NULL) {
            next = cur->next;
	    (*jvmti)->Deallocate(jvmti, (unsigned char*)cur);
            cur = next;
        }
    }    
    (*jvmti)->Deallocate(jvmti, (unsigned char*)htab->vec);    
    (*jvmti)->Deallocate(jvmti, (unsigned char*)htab);    
}

static void restructure (hashtab *htab) {
    hashtab *ntab = NULL;
    hashnode *cur;
    hashnode **tmpvec;
    size_t i;
    int factor;
    size_t tmpsize;
  
    for (factor = 8; ntab == NULL && factor > 1; factor /= 2)
	ntab = jmphash_new (factor * htab->size, htab->hashfun, htab->cmp);

    if (ntab == NULL)
	return;

    for (i = 0; i < htab->size; ++i) {
	cur = htab->vec[i];

	while (cur != NULL) {
	    jmphash_insert (ntab, cur->key, cur->el);
	    cur = cur->next;
        }
    }

    tmpvec = htab->vec;
    tmpsize = htab->size;
    htab->vec = ntab->vec;
    htab->size = ntab->size;
    ntab->vec = tmpvec;
    ntab->size = tmpsize;
    jmphash_free (ntab);
}


void jmphash_insert (hashtab* htab, void* key, void* el) {
    size_t i;
    hashnode* hnode;

    if (htab->cardinal >= htab->size * 10)
        restructure (htab);
    
    (*jvmti)->Allocate (jvmti, sizeof(*hnode), (unsigned char**)&hnode);
    hnode->key = key;
    hnode->el = el;
    i = HASH (htab, key);
    hnode->next = htab->vec[i];
    htab->vec[i] = hnode;
    ++htab->cardinal;
}

void* jmphash_search (hashtab* htab, void* key) {
    hashnode* hnode;
    hnode = htab->vec[HASH (htab, key)];
    while ((hnode != NULL) && (htab->cmp (hnode->key, key)))
        hnode = hnode->next;
    if (hnode == NULL)
	return NULL;
    return hnode->el;
}

size_t jmphash_size (hashtab* htab) {
    return htab->cardinal;    
}

void jmphash_for_each (hashtab* htab, jmphash_iter_f proc, void* data) {
    size_t i;
    hashnode* hnode;

    for (i = 0; i < htab->size; ++i) {
        for (hnode = htab->vec[i]; hnode != NULL; hnode = hnode->next)
            proc (hnode->key, hnode->el, data);
    }
}
