/*
 * Copyright © 2015 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Andreas Pokorny <andreas.pokorny@canonical.com>
 */

#include "unity_input_service.h"
#include "input_configuration.h"
#include "dbus_message_handle.h"
#include "dbus_connection_thread.h"
#include "scoped_dbus_error.h"

#include "unity_input_service_introspection.h" // autogenerated

namespace
{

char const* const dbus_input_interface = "com.canonical.Unity.Input";
char const* const dbus_input_path = "/com/canonical/Unity/Input";
char const* const dbus_input_service_name = "com.canonical.Unity.Input";

}

usc::UnityInputService::UnityInputService(
    std::shared_ptr<usc::DBusConnectionThread> const& dbus,
    std::shared_ptr<usc::InputConfiguration> const& input_config)
    : dbus{dbus},
      input_config{input_config}
{
    auto const& connection = dbus->connection();
    connection.request_name(dbus_input_service_name);
    connection.add_filter(handle_dbus_message_thunk, this);
}

::DBusHandlerResult usc::UnityInputService::handle_dbus_message_thunk(
    ::DBusConnection* connection, DBusMessage* message, void* user_data)
{
    auto const dbus_input_service = static_cast<usc::UnityInputService*>(user_data);
    return dbus_input_service->handle_dbus_message(connection, message, user_data);
}

void usc::UnityInputService::handle_message(DBusMessage* message, void (usc::InputConfiguration::* method)(bool))
{
    ScopedDBusError args_error;
    dbus_bool_t flag{true};
    dbus_message_get_args(
        message, &args_error, DBUS_TYPE_BOOLEAN, &flag, DBUS_TYPE_INVALID);

    if (!args_error)
    {
        (input_config.get()->*method)(flag);

        DBusMessageHandle reply{dbus_message_new_method_return(message)};
        dbus_connection_send(dbus->connection(), reply, nullptr);
    }
}

void usc::UnityInputService::handle_message(DBusMessage* message, void (usc::InputConfiguration::* method)(double))
{
    ScopedDBusError args_error;
    double value{true};
    dbus_message_get_args(
        message, &args_error, DBUS_TYPE_DOUBLE, &value, DBUS_TYPE_INVALID);

    if (!args_error)
    {
        (input_config.get()->*method)(value);

        DBusMessageHandle reply{dbus_message_new_method_return(message)};
        dbus_connection_send(dbus->connection(), reply, nullptr);
    }
}

void usc::UnityInputService::handle_message(DBusMessage* message, void (usc::InputConfiguration::* method)(int32_t))
{
    ScopedDBusError args_error;
    int32_t value{true};
    dbus_message_get_args(
        message, &args_error, DBUS_TYPE_INT32, &value, DBUS_TYPE_INVALID);

    if (!args_error)
    {
        (input_config.get()->*method)(value);

        DBusMessageHandle reply{dbus_message_new_method_return(message)};
        dbus_connection_send(dbus->connection(), reply, nullptr);
    }
}

DBusHandlerResult usc::UnityInputService::handle_dbus_message(
    ::DBusConnection* connection, DBusMessage* message, void* user_data)
{
    ScopedDBusError args_error;

    if (dbus_message_is_method_call(message, "org.freedesktop.DBus.Introspectable", "Introspect"))
    {
        DBusMessageHandle reply{
            dbus_message_new_method_return(message),
            DBUS_TYPE_STRING, &unity_input_service_introspection,
            DBUS_TYPE_INVALID};

        dbus_connection_send(connection, reply, nullptr);
    }
    else if (dbus_message_is_method_call(message, dbus_input_interface, "setMousePrimaryButton"))
        handle_message(message, &InputConfiguration::set_mouse_primary_button);
    else if (dbus_message_is_method_call(message, dbus_input_interface, "setMouseCursorSpeed"))
        handle_message(message, &InputConfiguration::set_mouse_cursor_speed);
    else if (dbus_message_is_method_call(message, dbus_input_interface, "setMouseScrollSpeed"))
        handle_message(message, &InputConfiguration::set_mouse_scroll_speed);
    else if (dbus_message_is_method_call(message, dbus_input_interface, "setTouchpadPrimaryButton"))
        handle_message(message, &InputConfiguration::set_touchpad_primary_button);
    else if (dbus_message_is_method_call(message, dbus_input_interface, "setTouchpadCursorSpeed"))
        handle_message(message, &InputConfiguration::set_touchpad_cursor_speed);
    else if (dbus_message_is_method_call(message, dbus_input_interface, "setTouchpadScrollSpeed"))
        handle_message(message, &InputConfiguration::set_touchpad_scroll_speed);
    else if (dbus_message_is_method_call(message, dbus_input_interface, "setTouchpadDisableWhileTyping"))
        handle_message(message, &InputConfiguration::set_disable_touchpad_while_typing);
    else if (dbus_message_is_method_call(message, dbus_input_interface, "setTouchpadTapToClick"))
        handle_message(message, &InputConfiguration::set_tap_to_click);
    else if (dbus_message_is_method_call(message, dbus_input_interface, "setTouchpadTwoFingerScroll"))
        handle_message(message, &InputConfiguration::set_two_finger_scroll);
    else if (dbus_message_is_method_call(message, dbus_input_interface, "setTouchpadDisableWithMouse"))
        handle_message(message, &InputConfiguration::set_disable_touchpad_with_mouse);
    else if (dbus_message_get_type(message) == DBUS_MESSAGE_TYPE_METHOD_CALL)
    {
         DBusMessageHandle reply{
             dbus_message_new_error(message, DBUS_ERROR_FAILED, "Not supported")};

        dbus_connection_send(connection, reply, nullptr);
    }

    if (args_error)
    {
         DBusMessageHandle reply{
             dbus_message_new_error(message, DBUS_ERROR_FAILED, "Invalid arguments")};

        dbus_connection_send(connection, reply, nullptr);
    }

    return DBUS_HANDLER_RESULT_HANDLED;
}

