#include "TintWindow.hpp"

#include "TintAbout.hpp"
#include "TintRegistration.hpp"
#include "../reg/Components.hpp"
#include "../FileImageResource.hpp"
#include "../../../nuvola/nuvola.hpp"

#include "wx/config.h"
#include "wx/button.h"
#include "wx/sizer.h"
#include "wx/textdlg.h"
#include "wx/filedlg.h"
#include "wx/msgdlg.h"
#include "wx/dcmemory.h"

using namespace indii::tint;
using namespace indii::tint::gui;
using namespace indii::tint::reg;

TintWindow::TintWindow(wxWindow* parent, ImageResource* res,
    ClusterModel* model, const Mode mode, TintWindowObserver* obs) :
    wxFrame(parent, wxID_ANY, _("tintii"), wxDefaultPosition, wxDefaultSize,
    wxCAPTION|wxMINIMIZE_BOX|wxMAXIMIZE_BOX|wxRESIZE_BORDER), res(res),
    model(model), obs(obs), own(false), mode(mode), isSaved(false),
    havePath(false) {
  model->setForDialog();
  watch(model);
  init(mode);
}

TintWindow::TintWindow(wxWindow* parent, const wxString& path,
    const Mode mode, TintWindowObserver* obs) : wxFrame(parent, wxID_ANY, path,
    wxDefaultPosition, wxDefaultSize), obs(obs) {
  res = new FileImageResource(path);
  model = new ClusterModel(res);

  model->prepare();
  model->cluster();
  model->lock();

  own = true;
  this->mode = mode;
  isSaved = false;
  havePath = false;
  init(mode);

  model->unlock();
  watch(model);
}

TintWindow::~TintWindow() {
  if (own) {
    delete model;
    delete res;
  }
}

bool TintWindow::IsSaved() {
  return isSaved;
}

int TintWindow::GetReturnCode() {
  return returnCode;
}

void TintWindow::SetReturnCode(const int returnCode) {
  this->returnCode = returnCode;
}

void TintWindow::OnSystemClose(wxCloseEvent& evt) {
  bool vetoed = false;
  if (mode == STANDALONE) {
    /* check for save */
    if (!isSaved && evt.CanVeto()) {
      int ret = wxMessageBox(_("Save changes to image before closing?"),
          _("Save changes?"), wxYES_NO|wxCANCEL|wxICON_QUESTION, this);
      if (ret == wxYES) {
        wxCommandEvent cmdEvt;
        OnButtonSave(cmdEvt);
        if (!isSaved) {
          evt.Veto();
          vetoed = true;
        }
      } else if (ret == wxCANCEL) {
        evt.Veto();
        vetoed = true;
      }
    }
    SetReturnCode(isSaved ? wxID_OK : wxID_CANCEL);
  }

  if (!vetoed) {
    if (obs != NULL) {
      obs->notifyClose(this);
    }
    Destroy();
  }
}

void TintWindow::OnAllowNotebookDnD(wxAuiNotebookEvent& evt) {
  evt.Allow();
}

void TintWindow::OnButtonSave(wxCommandEvent& evt) {
  if (!havePath) {
    OnButtonSaveAs(evt);
  } else {
    save();
    if (!isSaved) {
      havePath = false;
      wxMessageBox(_("Error. File not saved."), _("Error"), wxOK|wxICON_HAND,
          this);
    }
  }
}

void TintWindow::OnButtonSaveAs(wxCommandEvent& evt) {
  wxString path = wxFileSelector(wxFileSelectorPromptStr, wxEmptyString,
      _("tintiied.jpg"), _("jpg"), _("Image Files ") +
      wxImage::GetImageExtWildcard() + _("|All files (*)|*"),
      wxFD_SAVE|wxFD_OVERWRITE_PROMPT, this);
  if (!path.empty()) {
    havePath = true;
    this->path = path;
    OnButtonSave(evt);
  }
}

void TintWindow::OnButtonClose(wxCommandEvent& evt) {
  Close();
}

void TintWindow::OnButtonOk(wxCommandEvent& evt) {
  SetReturnCode(wxID_OK);
  Close();
}

void TintWindow::OnButtonCancel(wxCommandEvent& evt) {
  SetReturnCode(wxID_CANCEL);
  Close();
}

void TintWindow::OnButtonAbout(wxCommandEvent& evt) {
  showAboutDialog();
}

void TintWindow::OnButtonDefault(wxCommandEvent& evt) {
  model->setDefaults();
}

void TintWindow::OnButZoomOut(wxCommandEvent& evt) {
  previewPane->getPreviewImage()->zoomOut();
}

void TintWindow::OnButZoomIn(wxCommandEvent& evt) {
  previewPane->getPreviewImage()->zoomIn();
}

void TintWindow::OnButZoomNormal(wxCommandEvent& evt) {
  previewPane->getPreviewImage()->zoomNormal();
}

void TintWindow::OnButZoomFit(wxCommandEvent& evt) {
  previewPane->getPreviewImage()->zoomFit();
}

void TintWindow::notifyClusterChange(const unsigned int i) {
  isSaved = false;
}

void TintWindow::notifyNumClustersChange() {
  isSaved = false;
}

void TintWindow::notifyNumRepetitionsChange() {
  isSaved = false;
}

void TintWindow::notifySaturationThresholdChange() {
  isSaved = false;
}

void TintWindow::notifyMaxPixelsChange() {
  isSaved = false;
}

void TintWindow::notifySaturationDecayChange() {
  isSaved = false;
}

void TintWindow::notifyCentroidDecayChange() {
  isSaved = false;
}

void TintWindow::notifySaturationSoftnessChange() {
  isSaved = false;
}

void TintWindow::notifyCentroidSoftnessChange() {
  isSaved = false;
}

void TintWindow::notifyGreyscaleChange() {
  isSaved = false;
}

void TintWindow::init(const Mode mode) {
  /* extra space for floating pane decorations */
  static const int PANE_EXTRA_WIDTH = 0;
  static const int PANE_EXTRA_HEIGHT = 0;
   
  /* AUI manager */
  manager = new wxAuiManager(this);
  
  int width, height;
  wxAuiPaneInfo paneInfo;
  paneInfo.DefaultPane();
  paneInfo.CloseButton(false);
  
  /* preview pane */
  previewPane = new PreviewPane(this, res, model);
  
  width = previewPane->GetSize().GetWidth() + PANE_EXTRA_WIDTH;
  height = previewPane->GetSize().GetHeight() + PANE_EXTRA_HEIGHT;
  
  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.CaptionVisible(false);
  paneInfo.Centre();
  paneInfo.PaneBorder(false);
  manager->AddPane(previewPane, paneInfo);

  paneInfo.CaptionVisible(true);
  paneInfo.PaneBorder(true);
  
  /* thumbnail pane */
  thumbPane = new ThumbPane(this, res, model);
  width = thumbPane->GetSize().GetWidth() + PANE_EXTRA_WIDTH;
  height = thumbPane->GetSize().GetHeight() + PANE_EXTRA_HEIGHT;

  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.Caption(_("Thumbs"));
  paneInfo.Left();
  manager->AddPane(thumbPane, paneInfo);

  /* hsl pane */
  /*hslPane = new HSLPane(this, res, model);
  width = hslPane->GetSize().GetWidth() + PANE_EXTRA_WIDTH;
  height = hslPane->GetSize().GetHeight() + PANE_EXTRA_HEIGHT;

  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.Caption(_("HSL"));
  paneInfo.Left();
  manager->AddPane(hslPane, paneInfo);*/

  /* post-processing pane */
  postPane = new PostProcessPane(this, model);
  width = postPane->GetSize().GetWidth() + PANE_EXTRA_WIDTH;
  height = postPane->GetSize().GetHeight() + PANE_EXTRA_HEIGHT;

  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.Caption(_("Post-processing"));
  paneInfo.Right();
  manager->AddPane(postPane, paneInfo);

  /* pre-processing pane */
  prePane = new PreProcessPane(this, model);
  width = prePane->GetSize().GetWidth() + PANE_EXTRA_WIDTH;
  height = prePane->GetSize().GetHeight() + PANE_EXTRA_HEIGHT;

  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.Caption(_("Colour detection"));
  paneInfo.Right();
  manager->AddPane(prePane, paneInfo);

  /* channel mixer pane */
  mixerPane = new ChannelMixerPane(this, model);
  width = mixerPane->GetSize().GetWidth() + PANE_EXTRA_WIDTH;
  height = mixerPane->GetSize().GetHeight() + PANE_EXTRA_HEIGHT;

  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));
  paneInfo.Caption(_("Channel mixer"));
  paneInfo.Right();
  manager->AddPane(mixerPane, paneInfo);

  /* menu bar for Mac */
  #ifdef __WXMAC__
  if (mode != EMBED) {
    wxMenuBar* mbMenu = new wxMenuBar();

    wxMenu* mnFile = new wxMenu();
    mnFile->Append(wxID_OPEN);
    mnFile->AppendSeparator();
    mnFile->Append(wxID_SAVE);
    mnFile->Append(wxID_SAVEAS);
    mnFile->Append(wxID_EXIT); // moved to program menu
    mnFile->Append(wxID_ABOUT); // moved to program menu
    
    wxMenu* mnView = new wxMenu();
    mnView->Append(ID_BUT_SPLASH, _("Splash"));
    mnView->AppendSeparator();
    mnView->Append(ID_BUT_ZOOM_IN, _("Zoom &In\tCtrl++"));
    mnView->Append(ID_BUT_ZOOM_OUT, _("Zoom &Out\tCtrl+-"));
    mnView->Append(ID_BUT_ZOOM_NORMAL, _("&Normal"));
    mnView->Append(ID_BUT_ZOOM_FIT, _("&Fit"));

    mbMenu->Append(mnFile, _("File"));
    mbMenu->Append(mnView, _("View"));
    
    this->SetMenuBar(mbMenu);
  }
  #endif
  
  /* toolbar & dialog buttons */
  static unsigned int PADDING = 4;
  wxPanel* buttonPanel = new wxPanel(this);
  wxBoxSizer* buttonSizer = new wxBoxSizer(wxHORIZONTAL);
  
  wxToolBar* tbTop = this->CreateToolBar(wxNO_BORDER|wxTB_HORIZONTAL|wxTB_TEXT);
  tbTop->SetToolBitmapSize(wxSize(32,32));
  if (mode == EMBED) {
    tbTop->AddTool(ID_BUT_ZOOM_IN, _("Zoom In"), *_img_nuvola_viewmagplus);
    tbTop->AddTool(ID_BUT_ZOOM_OUT, _("Zoom Out"), *_img_nuvola_viewmagminus);
    tbTop->AddTool(ID_BUT_ZOOM_NORMAL, _("Normal"), *_img_nuvola_viewmag1);
    tbTop->AddTool(ID_BUT_ZOOM_FIT, _("Fit"), *_img_nuvola_viewmagfit);

    wxButton* butDefault = new wxButton(buttonPanel, ID_BUT_DEFAULT,
        _("Defaults"));
    wxButton* butAbout = new wxButton(buttonPanel, wxID_ABOUT);
    wxButton* butOk = new wxButton(buttonPanel, wxID_OK);
    wxButton* butCancel = new wxButton(buttonPanel, wxID_CANCEL);
    buttonSizer->Add(butDefault, 0, wxALIGN_RIGHT|wxALL, PADDING);
    buttonSizer->AddStretchSpacer(1);
    buttonSizer->Add(butAbout, 0, wxALIGN_RIGHT|wxALL, PADDING);
    buttonSizer->Add(butOk, 0, wxALIGN_RIGHT|wxALL, PADDING);
    buttonSizer->Add(butCancel, 0, wxALIGN_RIGHT|wxALL, PADDING);
  } else {
    tbTop->AddTool(wxID_SAVE, _("Save"), *_img_nuvola_filesave);
    tbTop->AddTool(wxID_SAVEAS, _("Save As"), *_img_nuvola_filesaveas);
    tbTop->AddSeparator();
    tbTop->AddTool(ID_BUT_ZOOM_IN, _("Zoom In"), *_img_nuvola_viewmagplus);
    tbTop->AddTool(ID_BUT_ZOOM_OUT, _("Zoom Out"), *_img_nuvola_viewmagminus);
    tbTop->AddTool(ID_BUT_ZOOM_NORMAL, _("Normal"), *_img_nuvola_viewmag1);
    tbTop->AddTool(ID_BUT_ZOOM_FIT, _("Fit"), *_img_nuvola_viewmagfit);
    if (!Components::allActivated()) {
      tbTop->AddSeparator();
      tbTop->AddTool(ID_BUT_REGISTER, _("Register"), *_img_nuvola_password);
    }
    
    wxButton* butDefault = new wxButton(buttonPanel, ID_BUT_DEFAULT,
        _("Restore Defaults"));
    #ifndef __WXMAC__
    wxButton* butClose = new wxButton(buttonPanel, wxID_CLOSE);
    #endif
    buttonSizer->Add(butDefault, 0, wxALIGN_RIGHT|wxALL, PADDING);
    buttonSizer->AddStretchSpacer(1);
    #ifndef __WXMAC__
    buttonSizer->Add(butClose, 0, wxALIGN_RIGHT|wxALL, PADDING);
    #endif
  }
  tbTop->Realize();

  buttonPanel->SetSizerAndFit(buttonSizer);
  paneInfo.PaneBorder(false);
  paneInfo.ToolbarPane();
  paneInfo.Floatable(false);
  paneInfo.Fixed();
  paneInfo.Gripper(false);
  paneInfo.Bottom();

  width = buttonPanel->GetSize().GetWidth();
  height = buttonPanel->GetSize().GetHeight();
  paneInfo.MinSize(wxSize(width, height));
  paneInfo.BestSize(wxSize(width, height));

  manager->AddPane(buttonPanel, paneInfo);
  
  SetSize(800,620);
  manager->Update();
  previewPane->getPreviewImage()->zoomFit();

  SetReturnCode(wxID_CANCEL);
}

void TintWindow::save() {
  /* pre-condition */
  assert (havePath);
  
  unsigned width = res->getWidth();
  unsigned height = res->getHeight();
  channel a(height, width);

  wxImage fg(*res->get());
  wxImage bg(fg.ConvertToGreyscale());
  model->calcAlpha(width, height, a);
  applyAlpha(fg, a);

  wxBitmap bmp(res->getWidth(), res->getHeight());
  wxMemoryDC dc(bmp);
  dc.DrawBitmap(bg, 0, 0);
  dc.DrawBitmap(fg, 0, 0);

  wxImage img(bmp.ConvertToImage());
  
  isSaved = img.SaveFile(path);
}

BEGIN_EVENT_TABLE(TintWindow, wxFrame)
//EVT_AUINOTEBOOK_ALLOW_DND(wxID_ANY, TintWindow::OnAllowNotebookDnD)
EVT_CLOSE(TintWindow::OnSystemClose)
EVT_TOOL(wxID_SAVE, TintWindow::OnButtonSave)
EVT_TOOL(wxID_SAVEAS, TintWindow::OnButtonSaveAs)
EVT_TOOL(ID_BUT_ZOOM_OUT, TintWindow::OnButZoomOut)
EVT_TOOL(ID_BUT_ZOOM_IN, TintWindow::OnButZoomIn)
EVT_TOOL(ID_BUT_ZOOM_NORMAL, TintWindow::OnButZoomNormal)
EVT_TOOL(ID_BUT_ZOOM_FIT, TintWindow::OnButZoomFit)
EVT_BUTTON(wxID_CLOSE, TintWindow::OnButtonClose)
EVT_BUTTON(wxID_OK, TintWindow::OnButtonOk)
EVT_BUTTON(wxID_CANCEL, TintWindow::OnButtonCancel)
EVT_BUTTON(wxID_ABOUT, TintWindow::OnButtonAbout)
EVT_BUTTON(ID_BUT_DEFAULT, TintWindow::OnButtonDefault)
END_EVENT_TABLE()
