/* Copyright (C) 2007-2010 Open Information Security Foundation
 *
 * You can copy, redistribute or modify this Program under the terms of
 * the GNU General Public License version 2 as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * version 2 along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

/**
 * \file
 *
 * \author Anoop Saldanha <poonaatsoc@gmail.com>
 */

#include "suricata-common.h"
#include "suricata.h"

#include "decode.h"

#include "detect.h"
#include "detect-engine.h"
#include "detect-parse.h"
#include "detect-content.h"
#include "detect-pcre.h"
#include "detect-isdataat.h"
#include "detect-bytetest.h"
#include "detect-bytejump.h"
#include "detect-byte-extract.h"

#include "util-spm.h"
#include "util-spm-bm.h"

#include "stream-tcp-reassemble.h"
#include "stream-tcp.h"

#include "app-layer.h"
#include "app-layer-dcerpc.h"
#include "decode-tcp.h"
#include "flow-util.h"
#include "util-debug.h"

#include "util-unittest.h"
#include "util-unittest-helper.h"

/**
 * \brief Run the dce stub match functions for the dce stub based keywords.
 *
 *        The following keywords are inspected:
 *        - content
 *        - isdaatat
 *        - pcre
 *        - bytejump
 *        - bytetest
 *
 *        All keywords are evaluated against the dce stub data.
 *
 *        For accounting the last match in relative matching,
 *        det_ctx->payload_offset var is used.
 *
 * \param de_ctx        Detection engine context.
 * \param det_ctx       Detection engine thread context.
 * \param s             Signature to inspect.
 * \param sm            SigMatch to inspect.
 * \param f             Flow
 * \param payload       Pointer to the dce stub to inspect.
 * \param payload_len   Length of the payload
 *
 * \retval 0 No match.
 * \retval 1 Match.
 */
static int DoInspectDcePayload(DetectEngineCtx *de_ctx,
                               DetectEngineThreadCtx *det_ctx, Signature *s,
                               SigMatch *sm, Flow *f, uint8_t *stub,
                               uint32_t stub_len, DCERPCState *dcerpc_state)
{
    SCEnter();

    det_ctx->inspection_recursion_counter++;

    if (det_ctx->inspection_recursion_counter == de_ctx->inspection_recursion_limit) {
        det_ctx->discontinue_matching = 1;
        SCReturnInt(0);
    }

    if (sm == NULL || stub_len == 0) {
        SCReturnInt(0);
    }

    switch(sm->type) {
        case DETECT_CONTENT:
        {
            DetectContentData *cd = NULL;
            cd = (DetectContentData *)sm->ctx;
            SCLogDebug("inspecting content %"PRIu32" stub_len %"PRIu32,
                       cd->id, stub_len);

            /* rule parsers should take care of this */
#ifdef DEBUG
            BUG_ON(cd->depth != 0 && cd->depth <= cd->offset);
#endif

            /* search for our pattern, checking the matches recursively.
             * if we match we look for the next SigMatch as well */
            uint8_t *found = NULL;
            uint32_t offset = 0;
            uint32_t depth = stub_len;
            uint32_t prev_offset = 0; /**< used in recursive searching */
            uint32_t prev_payload_offset = det_ctx->payload_offset;

            do {
                if (cd->flags & DETECT_CONTENT_DISTANCE ||
                    cd->flags & DETECT_CONTENT_WITHIN) {
                    SCLogDebug("prev_payload_offset %"PRIu32,
                               prev_payload_offset);

                    offset = prev_payload_offset;
                    depth = stub_len;

                    int distance = cd->distance;
                    if (cd->flags & DETECT_CONTENT_DISTANCE) {
                        if (cd->flags & DETECT_CONTENT_DISTANCE_BE) {
                            distance = det_ctx->bj_values[cd->distance];
                        }
                        if (distance < 0 && (uint32_t)(abs(distance)) > offset)
                            offset = 0;
                        else
                            offset += distance;

                        SCLogDebug("cd->distance %"PRIi32", offset %"PRIu32", depth %"PRIu32,
                                   cd->distance, offset, depth);
                    }

                    if (cd->flags & DETECT_CONTENT_WITHIN) {
                        if (cd->flags & DETECT_CONTENT_WITHIN_BE) {
                            if ((int32_t)depth > (int32_t)(prev_payload_offset + det_ctx->bj_values[cd->within] + distance)) {
                                depth = prev_payload_offset + det_ctx->bj_values[cd->within] + distance;
                            }
                        } else {
                            if ((int32_t)depth > (int32_t)(prev_payload_offset + cd->within + distance)) {
                                depth = prev_payload_offset + cd->within + distance;
                            }

                            SCLogDebug("cd->within %"PRIi32", prev_payload_offset "
                                       "%"PRIu32", depth %"PRIu32, cd->within,
                                       prev_payload_offset, depth);
                        }
                    }

                    if (cd->flags & DETECT_CONTENT_DEPTH_BE) {
                        if ((det_ctx->bj_values[cd->depth] + prev_payload_offset) < depth) {
                            depth = prev_payload_offset + det_ctx->bj_values[cd->depth];
                         }
                    } else {
                        if (cd->depth != 0) {
                            if ((cd->depth + prev_payload_offset) < depth) {
                                depth = prev_payload_offset + cd->depth;
                            }

                            SCLogDebug("cd->depth %"PRIu32", depth %"PRIu32,
                                       cd->depth, depth);
                        }
                    }

                    if (cd->flags & DETECT_CONTENT_OFFSET_BE) {
                        if (det_ctx->bj_values[cd->offset] > offset)
                            offset = det_ctx->bj_values[cd->offset];
                    } else {
                        if (cd->offset > offset) {
                            offset = cd->offset;
                            SCLogDebug("setting offset %"PRIu32, offset);
                        }
                    }

                /* implied no relative matches */
                } else {
                    /* set depth */
                    if (cd->flags & DETECT_CONTENT_DEPTH_BE) {
                        depth = det_ctx->bj_values[cd->depth];
                    } else {
                        if (cd->depth != 0) {
                            depth = cd->depth;
                        }
                    }

                    /* set offset */
                    if (cd->flags & DETECT_CONTENT_OFFSET_BE)
                        offset = det_ctx->bj_values[cd->offset];
                    else
                        offset = cd->offset;
                    prev_payload_offset = 0;
                }

                /* update offset with prev_offset if we're searching for
                 * matches after the first occurence. */
                SCLogDebug("offset %"PRIu32", prev_offset %"PRIu32, offset,
                           prev_offset);
                if (prev_offset != 0)
                    offset = prev_offset;

                SCLogDebug("offset %"PRIu32", depth %"PRIu32, offset, depth);

                if (depth > stub_len)
                    depth = stub_len;

                /* if offset is bigger than depth we can never match on a
                 * pattern.  We can however, "match" on a negated pattern. */
                /* \todo why isn't it >= ?.  Same question applies for
                 * detect-engine-dcepayload.c and detect-engine-uri.c */
                if (offset > depth || depth == 0) {
                    if (cd->flags & DETECT_CONTENT_NEGATED) {
                        goto match;
                    } else {
                        SCReturnInt(0);
                    }
                }

                uint8_t *sstub = stub + offset;
                uint32_t sstub_len = depth - offset;
                uint32_t match_offset = 0;
                SCLogDebug("sstub_len %"PRIu32, sstub_len);
#ifdef DEBUG
                BUG_ON(sstub_len > stub_len);
#endif

                /* do the actual search */
                if (cd->flags & DETECT_CONTENT_NOCASE) {
                    found = BoyerMooreNocase(cd->content, cd->content_len, sstub,
                                             sstub_len, cd->bm_ctx->bmGs,
                                             cd->bm_ctx->bmBc);
                } else {
                    found = BoyerMoore(cd->content, cd->content_len, sstub,
                                       sstub_len, cd->bm_ctx->bmGs,
                                       cd->bm_ctx->bmBc);
                }

                /* next we evaluate the result in combination with the
                 * negation flag. */
                SCLogDebug("found %p cd negated %s", found,
                           cd->flags & DETECT_CONTENT_NEGATED ? "true" : "false");

                if (found == NULL && !(cd->flags & DETECT_CONTENT_NEGATED)) {
                    SCReturnInt(0);
                } else if (found == NULL && cd->flags & DETECT_CONTENT_NEGATED) {
                    goto match;
                } else if (found != NULL && cd->flags & DETECT_CONTENT_NEGATED) {
                    SCLogDebug("content %"PRIu32" matched at offset %"PRIu32", but "
                               "negated so no match", cd->id, match_offset);
                    det_ctx->discontinue_matching = 1;
                    SCReturnInt(0);
                } else {
                    match_offset = (uint32_t)((found - stub) + cd->content_len);
                    SCLogDebug("content %"PRIu32" matched at offset %"PRIu32"",
                               cd->id, match_offset);
                    det_ctx->payload_offset = match_offset;

                    if (!(cd->flags & DETECT_CONTENT_RELATIVE_NEXT)) {
                        SCLogDebug("no relative match coming up, so this is a match");
                        goto match;
                    }

                    /* bail out if we have no next match. Technically this is an
                     * error, as the current cd has the DETECT_CONTENT_RELATIVE_NEXT
                     * flag set. */
                    if (sm->next == NULL) {
                        SCReturnInt(0);
                    }

                    SCLogDebug("content %"PRIu32, cd->id);

                    /* see if the next payload keywords match. If not, we will
                     * search for another occurence of this content and see
                     * if the others match then until we run out of matches */
                    int r = DoInspectDcePayload(de_ctx, det_ctx, s, sm->next,
                                                f, stub, stub_len, dcerpc_state);
                    if (r == 1) {
                        SCReturnInt(1);
                    }

                    if (det_ctx->discontinue_matching)
                        SCReturnInt(0);

                    /* set the previous match offset to the start of this match + 1 */
                    prev_offset = (match_offset - (cd->content_len - 1));
                    SCLogDebug("trying to see if there is another match after "
                               "prev_offset %"PRIu32, prev_offset);
                }

            } while(1);
        }

        case DETECT_ISDATAAT:
        {
            SCLogDebug("inspecting isdataat");

            DetectIsdataatData *id = (DetectIsdataatData *)sm->ctx;
            if (id->flags & ISDATAAT_RELATIVE) {
                if (det_ctx->payload_offset + id->dataat > stub_len) {
                    SCLogDebug("det_ctx->payload_offset + id->dataat "
                               "%"PRIu32" > %"PRIu32,
                               det_ctx->payload_offset + id->dataat, stub_len);
                    SCReturnInt(0);
                } else {
                    SCLogDebug("relative isdataat match");
                    goto match;
                }
            } else {
                if (id->dataat < stub_len) {
                    SCLogDebug("absolute isdataat match");
                    goto match;
                } else {
                    SCLogDebug("absolute isdataat mismatch, id->isdataat %"PRIu32", "
                               "stub_len %"PRIu32"", id->dataat, stub_len);
                    SCReturnInt(0);
                }
            }
        }

        case DETECT_PCRE:
        {
            SCLogDebug("inspecting pcre");
            DetectPcreData *pe = (DetectPcreData *)sm->ctx;
            uint32_t prev_payload_offset = det_ctx->payload_offset;
            uint32_t prev_offset = 0;
            int r = 0;

            det_ctx->pcre_match_start_offset = 0;
            do {
                r = DetectPcrePayloadMatch(det_ctx, s, sm, NULL, f,
                                           stub, stub_len);
                if (r == 0) {
                    det_ctx->discontinue_matching = 1;
                    SCReturnInt(0);
                }

                if (!(pe->flags & DETECT_PCRE_RELATIVE_NEXT)) {
                    SCLogDebug("no relative match coming up, so this is a match");
                    goto match;
                }

                /* save it, in case we need to do a pcre match once again */
                prev_offset = det_ctx->pcre_match_start_offset;

                /* see if the next payload keywords match. If not, we will
                 * search for another occurence of this pcre and see
                 * if the others match, until we run out of matches */
                r = DoInspectDcePayload(de_ctx, det_ctx, s, sm->next, f, stub,
                                        stub_len, dcerpc_state);
                if (r == 1) {
                    SCReturnInt(1);
                }

                if (det_ctx->discontinue_matching)
                    SCReturnInt(0);

                det_ctx->payload_offset = prev_payload_offset;
                det_ctx->pcre_match_start_offset = prev_offset;
            } while (1);
        }
        case DETECT_BYTETEST:
        {
            DetectBytetestData *data = (DetectBytetestData *)sm->ctx;
            uint8_t flags = data->flags;
            int32_t offset = data->offset;;
            uint64_t value = data->value;
            if (flags & DETECT_BYTETEST_OFFSET_BE) {
                offset = det_ctx->bj_values[offset];
            }
            if (flags & DETECT_BYTETEST_VALUE_BE) {
                value = det_ctx->bj_values[value];
            }

            /* if we have dce enabled we will have to use the endianness
             * specified by the dce header */
            if (flags & DETECT_BYTETEST_DCE) {
                /* enable the endianness flag temporarily.  once we are done
                 * processing we reset the flags to the original value*/
                flags |= ((dcerpc_state->dcerpc.dcerpchdr.packed_drep[0] & 0x10) ?
                          DETECT_BYTETEST_LITTLE: 0);
            }

            if (DetectBytetestDoMatch(det_ctx, s, sm, stub, stub_len, flags, offset, value) != 1) {
                SCReturnInt(0);
            }

            goto match;
        }

        case DETECT_BYTEJUMP:
        {
            DetectBytejumpData *data = (DetectBytejumpData *)sm->ctx;
            uint8_t flags = data->flags;
            int32_t offset = data->offset;;
            if (flags & DETECT_BYTEJUMP_OFFSET_BE) {
                offset = det_ctx->bj_values[offset];
            }

            /* if we have dce enabled we will have to use the endianness
             * specified by the dce header */
            if (flags & DETECT_BYTEJUMP_DCE) {
                /* enable the endianness flag temporarily.  once we are done
                 * processing we reset the flags to the original value*/
                flags |= ((dcerpc_state->dcerpc.dcerpchdr.packed_drep[0] & 0x10) ?
                          DETECT_BYTEJUMP_LITTLE : 0);
            }

            if (DetectBytejumpDoMatch(det_ctx, s, sm, stub, stub_len, flags, offset) != 1) {
                SCReturnInt(0);
            }

            goto match;
        }

        case DETECT_BYTE_EXTRACT:
        {
            DetectByteExtractData *bed = (DetectByteExtractData *)sm->ctx;
            uint8_t endian = bed->endian;

            /* if we have dce enabled we will have to use the endianness
             * specified by the dce header */
            if (bed->flags & DETECT_BYTE_EXTRACT_FLAG_ENDIAN &&
                endian == DETECT_BYTE_EXTRACT_ENDIAN_DCE) {

                /* enable the endianness flag temporarily.  once we are done
                 * processing we reset the flags to the original value*/
                endian |= ((dcerpc_state->dcerpc.dcerpchdr.packed_drep[0] == 0x10) ?
                           DETECT_BYTE_EXTRACT_ENDIAN_LITTLE : DETECT_BYTE_EXTRACT_ENDIAN_BIG);
            }
            if (DetectByteExtractDoMatch(det_ctx, sm, s, stub, stub_len,
                                         &det_ctx->bj_values[bed->local_id], endian) != 1) {
                SCReturnInt(0);
            }

            goto match;
        }

        /* we should never get here, but bail out just in case */
        default:
        {
            SCLogDebug("sm->type %u", sm->type);
#ifdef DEBUG
            BUG_ON(1);
#endif
        }
    }

    SCReturnInt(0);

match:
    /* this sigmatch matched, inspect the next one. If it was the last,
     * the payload portion of the signature matched. */
    if (sm->next != NULL) {
        int r = DoInspectDcePayload(de_ctx, det_ctx, s, sm->next, f, stub,
                                    stub_len, dcerpc_state);
        SCReturnInt(r);
    } else {
        SCReturnInt(1);
    }
}

/**
 * \brief Do the content inspection & validation for a signature against dce stub.
 *
 * \param de_ctx  Detection engine context.
 * \param det_ctx Detection engine thread context.
 * \param s       Signature to inspect.
 * \param sm      SigMatch to inspect.
 * \param f       Flow.
 * \param flags   App layer flags.
 * \param state   App layer state.
 *
 *  \retval 0 No match.
 *  \retval 1 Match.
 */
int DetectEngineInspectDcePayload(DetectEngineCtx *de_ctx,
                                  DetectEngineThreadCtx *det_ctx, Signature *s,
                                  Flow *f, uint8_t flags, void *alstate)
{
    SCEnter();
    DCERPCState *dcerpc_state = (DCERPCState *)alstate;
    uint8_t *dce_stub_data = NULL;
    uint16_t dce_stub_data_len;
    int r = 0;

    if (s->sm_lists[DETECT_SM_LIST_DMATCH] == NULL || dcerpc_state == NULL) {
        SCReturnInt(0);
    }

    if (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
        dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh != 0) {
        /* the request stub and stub_len */
        dce_stub_data = dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer;
        dce_stub_data_len = dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len;

        det_ctx->payload_offset = 0;
        det_ctx->discontinue_matching = 0;
        det_ctx->inspection_recursion_counter = 0;

        r = DoInspectDcePayload(de_ctx, det_ctx, s, s->sm_lists[DETECT_SM_LIST_DMATCH], f,
                                dce_stub_data, dce_stub_data_len, dcerpc_state);
        if (r == 1) {
            SCReturnInt(1);
        }
    }

    if (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer != NULL &&
        dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) {
        /* the response stub and stub_len */
        dce_stub_data = dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer;
        dce_stub_data_len = dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer_len;

        det_ctx->payload_offset = 0;
        det_ctx->discontinue_matching = 0;
        det_ctx->inspection_recursion_counter = 0;

        r = DoInspectDcePayload(de_ctx, det_ctx, s, s->sm_lists[DETECT_SM_LIST_DMATCH], f,
                                dce_stub_data, dce_stub_data_len, dcerpc_state);
        if (r == 1) {
            SCReturnInt(1);
        }
    }

    SCReturnInt(0);
}

/**************************************Unittests*******************************/

#ifdef UNITTESTS

/**
 * \test Test the working of detection engien with respect to dce keywords.
 */
int DcePayloadTest01(void)
{
    int result = 0;
    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x6a, 0x28, 0x19, 0x39, 0x0c, 0xb1, 0xd0, 0x11,
        0x9b, 0xa8, 0x00, 0xc0, 0x4f, 0xd9, 0x2e, 0xf5,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_len = sizeof(bind);

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x48, 0x1a, 0x00, 0x00,
        0x0c, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x6c, 0x73, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x82, 0x7c, 0x83, 0x46, 0x0f, 0xf7,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xcf, 0x80, 0x98, 0x6d, 0xfe, 0xb0, 0x90, 0xd1,
        0xcf, 0x86, 0x0f, 0x52, 0x2c, 0x23, 0x66, 0x28,
        0x27, 0x30, 0x48, 0x55, 0x42, 0x6a, 0x48, 0x4b,
        0x68, 0x22, 0x2e, 0x23, 0x64, 0x33, 0x2c, 0x2d,
        0x5c, 0x51, 0x48, 0x55, 0x24, 0x67, 0x6c, 0x4c,
        0x45, 0x71, 0x35, 0x72, 0x5a, 0x48, 0x5e, 0x35,
        0x61, 0x78, 0x35, 0x42, 0x2c, 0x7a, 0x75, 0x61,
        0x5b, 0x4e, 0x76, 0x30, 0x26, 0x2f, 0x2a, 0x34,
        0x48, 0x29, 0x25, 0x6e, 0x5c, 0x3a, 0x6c, 0x3e,
        0x79, 0x4e, 0x2a, 0x21, 0x6f, 0x6f, 0x34, 0x46,
        0x43, 0x26, 0x5b, 0x35, 0x78, 0x27, 0x69, 0x23,
        0x72, 0x21, 0x69, 0x56, 0x6a, 0x7d, 0x4b, 0x5e,
        0x65, 0x37, 0x60, 0x44, 0x7c, 0x5d, 0x5b, 0x72,
        0x7d, 0x73, 0x7b, 0x47, 0x57, 0x21, 0x41, 0x38,
        0x76, 0x38, 0x76, 0x5c, 0x58, 0x32, 0x4a, 0x37,
        0x2f, 0x40, 0x4b, 0x4c, 0x3d, 0x41, 0x33, 0x56,
        0x73, 0x38, 0x61, 0x71, 0x24, 0x49, 0x4c, 0x4a,
        0x44, 0x2e, 0x3a, 0x3f, 0x74, 0x54, 0x4c, 0x65,
        0x54, 0x2d, 0x3b, 0x28, 0x41, 0x45, 0x49, 0x2c,
        0x6e, 0x48, 0x44, 0x43, 0x37, 0x3d, 0x7b, 0x6d,
        0x2b, 0x4b, 0x32, 0x5a, 0x31, 0x61, 0x6e, 0x2b,
        0x27, 0x50, 0x6b, 0x66, 0x76, 0x4e, 0x55, 0x35,
        0x2b, 0x72, 0x2d, 0x5e, 0x42, 0x3e, 0x5a, 0x5d,
        0x36, 0x45, 0x32, 0x3a, 0x58, 0x78, 0x78, 0x3e,
        0x60, 0x6c, 0x5d, 0x63, 0x41, 0x7c, 0x52, 0x21,
        0x75, 0x6a, 0x5a, 0x70, 0x55, 0x45, 0x76, 0x58,
        0x33, 0x40, 0x38, 0x39, 0x21, 0x37, 0x7d, 0x77,
        0x21, 0x70, 0x2b, 0x72, 0x29, 0x6a, 0x31, 0x5f,
        0x38, 0x4a, 0x66, 0x65, 0x62, 0x2c, 0x39, 0x52,
        0x5f, 0x2a, 0x2b, 0x63, 0x4f, 0x76, 0x43, 0x25,
        0x6a, 0x50, 0x37, 0x52, 0x5e, 0x23, 0x3c, 0x42,
        0x28, 0x75, 0x75, 0x42, 0x25, 0x23, 0x28, 0x56,
        0x6c, 0x46, 0x5c, 0x5e, 0x6b, 0x7d, 0x48, 0x24,
        0x77, 0x6c, 0x70, 0x62, 0x2e, 0x28, 0x7d, 0x6b,
        0x69, 0x4a, 0x75, 0x3d, 0x5d, 0x56, 0x21, 0x49,
        0x56, 0x47, 0x64, 0x2b, 0x4c, 0x52, 0x43, 0x60,
        0x77, 0x49, 0x46, 0x46, 0x33, 0x2c, 0x4b, 0x4b,
        0x3d, 0x63, 0x5d, 0x33, 0x78, 0x76, 0x51, 0x56,
        0x77, 0x3c, 0x72, 0x74, 0x52, 0x27, 0x40, 0x6c,
        0x42, 0x79, 0x49, 0x24, 0x62, 0x5e, 0x26, 0x31,
        0x5c, 0x22, 0x2b, 0x4c, 0x64, 0x49, 0x52, 0x45,
        0x47, 0x49, 0x3a, 0x2a, 0x51, 0x71, 0x22, 0x22,
        0x70, 0x24, 0x34, 0x67, 0x4b, 0x6d, 0x58, 0x29,
        0x63, 0x26, 0x7b, 0x6f, 0x38, 0x78, 0x25, 0x62,
        0x4d, 0x3a, 0x7d, 0x40, 0x23, 0x57, 0x67, 0x33,
        0x38, 0x31, 0x4e, 0x54, 0x3c, 0x4b, 0x48, 0x69,
        0x3c, 0x39, 0x31, 0x2b, 0x26, 0x70, 0x44, 0x66,
        0x4a, 0x37, 0x2b, 0x75, 0x36, 0x45, 0x59, 0x34,
        0x3e, 0x3e, 0x29, 0x70, 0x71, 0x5a, 0x55, 0x49,
        0x3e, 0x4b, 0x68, 0x4e, 0x75, 0x70, 0x3c, 0x5c,
        0x50, 0x58, 0x28, 0x75, 0x3c, 0x2a, 0x41, 0x70,
        0x2f, 0x2b, 0x37, 0x26, 0x75, 0x71, 0x55, 0x22,
        0x3a, 0x44, 0x30, 0x48, 0x5d, 0x2f, 0x6c, 0x44,
        0x28, 0x4b, 0x34, 0x45, 0x21, 0x60, 0x44, 0x36,
        0x7b, 0x32, 0x39, 0x5f, 0x6d, 0x3f, 0x68, 0x73,
        0x25, 0x45, 0x56, 0x7c, 0x78, 0x7a, 0x49, 0x6a,
        0x46, 0x3d, 0x2d, 0x33, 0x6c, 0x6f, 0x23, 0x77,
        0x38, 0x33, 0x36, 0x74, 0x7b, 0x57, 0x4b, 0x6d,
        0x27, 0x75, 0x24, 0x6e, 0x43, 0x61, 0x4d, 0x44,
        0x6d, 0x27, 0x48, 0x58, 0x5e, 0x7b, 0x26, 0x6a,
        0x50, 0x7c, 0x51, 0x23, 0x3c, 0x4f, 0x37, 0x4c,
        0x47, 0x3e, 0x45, 0x56, 0x22, 0x33, 0x7c, 0x66,
        0x35, 0x54, 0x7a, 0x6e, 0x5a, 0x24, 0x70, 0x62,
        0x29, 0x3f, 0x69, 0x79, 0x24, 0x43, 0x41, 0x24,
        0x65, 0x25, 0x62, 0x4f, 0x73, 0x3e, 0x2b, 0x36,
        0x46, 0x69, 0x27, 0x55, 0x2a, 0x6e, 0x24, 0x6c,
        0x7d, 0x64, 0x7c, 0x61, 0x26, 0x67, 0x2a, 0x53,
        0x73, 0x60, 0x28, 0x2d, 0x6b, 0x44, 0x54, 0x61,
        0x34, 0x53, 0x22, 0x59, 0x6d, 0x73, 0x56, 0x55,
        0x25, 0x2c, 0x38, 0x4a, 0x3b, 0x4e, 0x78, 0x46,
        0x54, 0x6e, 0x6d, 0x4f, 0x47, 0x4f, 0x4f, 0x5a,
        0x67, 0x77, 0x39, 0x66, 0x28, 0x29, 0x4e, 0x43,
        0x55, 0x6e, 0x60, 0x59, 0x28, 0x3b, 0x65, 0x62,
        0x61, 0x5a, 0x29, 0x6e, 0x79, 0x60, 0x41, 0x53,
        0x2f, 0x5d, 0x44, 0x36, 0x7b, 0x3e, 0x7c, 0x2b,
        0x77, 0x36, 0x70, 0x3f, 0x40, 0x55, 0x48, 0x67,
        0x4b, 0x4d, 0x5d, 0x51, 0x79, 0x76, 0x48, 0x4a,
        0x2d, 0x21, 0x60, 0x40, 0x46, 0x55, 0x7a, 0x60,
        0x22, 0x25, 0x3f, 0x4b, 0x54, 0x6a, 0x6a, 0x3c,
        0x77, 0x22, 0x5b, 0x43, 0x67, 0x58, 0x71, 0x22,
        0x79, 0x4b, 0x32, 0x61, 0x44, 0x4d, 0x6f, 0x42,
        0x33, 0x2d, 0x53, 0x35, 0x3d, 0x6f, 0x57, 0x48,
        0x33, 0x3b, 0x5a, 0x53, 0x3f, 0x4e, 0x3f, 0x6b,
        0x4c, 0x27, 0x26, 0x3b, 0x73, 0x49, 0x22, 0x55,
        0x79, 0x2f, 0x47, 0x2f, 0x55, 0x5a, 0x7a, 0x71,
        0x6c, 0x31, 0x43, 0x40, 0x56, 0x7b, 0x21, 0x7a,
        0x6d, 0x4c, 0x43, 0x5e, 0x38, 0x47, 0x29, 0x38,
        0x62, 0x49, 0x45, 0x78, 0x70, 0x2b, 0x2e, 0x65,
        0x47, 0x71, 0x58, 0x79, 0x39, 0x67, 0x7d, 0x6d,
        0x6a, 0x67, 0x4a, 0x71, 0x27, 0x35, 0x2a, 0x4c,
        0x3e, 0x58, 0x55, 0x30, 0x4d, 0x75, 0x77, 0x48,
        0x5f, 0x4b, 0x59, 0x34, 0x65, 0x68, 0x57, 0x59,
        0x63, 0x23, 0x47, 0x38, 0x47, 0x5e, 0x56, 0x28,
        0x79, 0x58, 0x3e, 0x39, 0x66, 0x77, 0x67, 0x33,
        0x29, 0x61, 0x24, 0x7d, 0x37, 0x44, 0x37, 0x67,
        0x3a, 0x58, 0x76, 0x21, 0x51, 0x59, 0x61, 0x73,
        0x66, 0x75, 0x71, 0x53, 0x4d, 0x24, 0x2d, 0x4b,
        0x29, 0x30, 0x32, 0x26, 0x59, 0x64, 0x27, 0x55,
        0x2c, 0x5a, 0x4c, 0x3c, 0x6c, 0x53, 0x56, 0x4b,
        0x3e, 0x55, 0x2e, 0x44, 0x38, 0x6b, 0x47, 0x76,
        0x2d, 0x2c, 0x3f, 0x4d, 0x22, 0x7b, 0x6d, 0x61,
        0x34, 0x6b, 0x50, 0x73, 0x28, 0x6d, 0x41, 0x71,
        0x21, 0x76, 0x52, 0x2a, 0x6d, 0x53, 0x2a, 0x74,
        0x28, 0x27, 0x62, 0x2a, 0x66, 0x25, 0x6e, 0x5e,
        0x37, 0x4f, 0x27, 0x72, 0x28, 0x47, 0x63, 0x6e,
        0x5a, 0x6a, 0x41, 0x35, 0x3a, 0x42, 0x3f, 0x27,
        0x75, 0x3e, 0x26, 0x3e, 0x6b, 0x55, 0x59, 0x60,
        0x24, 0x70, 0x49, 0x3c, 0x4e, 0x2c, 0x39, 0x7a,
        0x36, 0x6c, 0x27, 0x3e, 0x6a, 0x4a, 0x59, 0x5a,
        0x3e, 0x21, 0x73, 0x4e, 0x59, 0x6e, 0x3d, 0x32,
        0x27, 0x45, 0x49, 0x58, 0x7d, 0x37, 0x39, 0x77,
        0x28, 0x51, 0x79, 0x54, 0x2b, 0x78, 0x46, 0x5a,
        0x21, 0x75, 0x33, 0x21, 0x63, 0x5a, 0x7b, 0x3e,
        0x33, 0x4f, 0x67, 0x75, 0x3a, 0x50, 0x48, 0x60,
        0x26, 0x64, 0x76, 0x5c, 0x42, 0x5c, 0x72, 0x38,
        0x6c, 0x52, 0x21, 0x2b, 0x25, 0x6b, 0x7c, 0x6b,
        0x2d, 0x5e, 0x63, 0x2a, 0x4c, 0x26, 0x5b, 0x4c,
        0x58, 0x52, 0x51, 0x55, 0x31, 0x79, 0x6c, 0x53,
        0x62, 0x3a, 0x36, 0x46, 0x7a, 0x29, 0x27, 0x78,
        0x1a, 0xbf, 0x49, 0x74, 0x68, 0x24, 0x51, 0x44,
        0x5b, 0x3e, 0x34, 0x44, 0x29, 0x5e, 0x4f, 0x2a,
        0xe9, 0x3f, 0xf8, 0xff, 0xff, 0x52, 0x7d, 0x47,
        0x67, 0x40, 0x27, 0x5e, 0x47, 0x46, 0x6d, 0x72,
        0x5d, 0x49, 0x26, 0x45, 0x33, 0x6b, 0x4d, 0x4a,
        0x6f, 0x62, 0x60, 0x45, 0x62, 0x27, 0x27, 0x7d,
        0x6a, 0x41, 0x2c, 0x6c, 0x5b, 0x2a, 0x2b, 0x36,
        0x29, 0x58, 0x7a, 0x4c, 0x6e, 0x2d, 0x74, 0x5c,
        0x38, 0x22, 0x5f, 0x49, 0x63, 0x43, 0x5b, 0x67
    };
    uint32_t request2_len = sizeof(request2);

    uint8_t request3[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x26, 0x65, 0x3c, 0x6e, 0x6d, 0x64, 0x24, 0x39,
        0x56, 0x43, 0x3e, 0x61, 0x5c, 0x54, 0x42, 0x23,
        0x75, 0x6b, 0x71, 0x27, 0x66, 0x2e, 0x6e, 0x3d,
        0x58, 0x23, 0x54, 0x77, 0x3b, 0x52, 0x6b, 0x50,
        0x3b, 0x74, 0x2c, 0x54, 0x25, 0x5c, 0x51, 0x7c,
        0x29, 0x7c, 0x5f, 0x4a, 0x35, 0x5c, 0x3d, 0x3f,
        0x33, 0x55, 0x3b, 0x5a, 0x57, 0x31, 0x59, 0x4f,
        0x6d, 0x6d, 0x7b, 0x3e, 0x38, 0x4d, 0x68, 0x75,
        0x64, 0x21, 0x50, 0x63, 0x47, 0x42, 0x56, 0x39,
        0x6c, 0x6f, 0x61, 0x53, 0x32, 0x56, 0x43, 0x52,
        0x43, 0x67, 0x26, 0x45, 0x28, 0x6b, 0x77, 0x28,
        0x7c, 0x64, 0x61, 0x24, 0x38, 0x6b, 0x59, 0x2a,
        0x4f, 0x6e, 0x5b, 0x57, 0x24, 0x54, 0x33, 0x37,
        0x47, 0x58, 0x4b, 0x58, 0x3d, 0x21, 0x38, 0x7c,
        0x2c, 0x24, 0x5f, 0x67, 0x3a, 0x41, 0x3e, 0x2a,
        0x72, 0x66, 0x2d, 0x6b, 0x66, 0x7b, 0x2b, 0x75,
        0x78, 0x2f, 0x4d, 0x4c, 0x51, 0x70, 0x5d, 0x55,
        0x54, 0x3c, 0x63, 0x46, 0x6b, 0x64, 0x4d, 0x25,
        0x45, 0x21, 0x34, 0x65, 0x48, 0x32, 0x58, 0x4c,
        0x70, 0x4c, 0x4c, 0x75, 0x5c, 0x77, 0x68, 0x78,
        0x34, 0x5c, 0x2d, 0x39, 0x58, 0x3b, 0x40, 0x71,
        0x77, 0x47, 0x32, 0x2e, 0x3c, 0x61, 0x6f, 0x6d,
        0x5f, 0x43, 0x74, 0x36, 0x4f, 0x21, 0x44, 0x66,
        0x36, 0x62, 0x30, 0x29, 0x5a, 0x34, 0x66, 0x4e,
        0x51, 0x23, 0x4e, 0x38, 0x51, 0x78, 0x74, 0x58,
        0x2e, 0x6d, 0x51, 0x49, 0x55, 0x73, 0x2a, 0x71,
        0x3c, 0x74, 0x38, 0x6f, 0x5d, 0x4b, 0x74, 0x68,
        0x65, 0x4a, 0x58, 0x41, 0x55, 0x29, 0x42, 0x69,
        0x55, 0x3b, 0x2b, 0x47, 0x64, 0x3b, 0x77, 0x72,
        0x74, 0x38, 0x53, 0x5c, 0x69, 0x49, 0x49, 0x5b,
        0x31, 0x41, 0x6a, 0x4e, 0x2c, 0x6a, 0x63, 0x3f,
        0x58, 0x4e, 0x25, 0x3e, 0x57, 0x41, 0x61, 0x26,
        0x5e, 0x24, 0x69, 0x7a, 0x38, 0x60, 0x73, 0x70,
        0x7d, 0x63, 0x34, 0x78, 0x4d, 0x50, 0x35, 0x69,
        0x49, 0x22, 0x45, 0x44, 0x3f, 0x6e, 0x75, 0x64,
        0x57, 0x3a, 0x61, 0x60, 0x34, 0x21, 0x61, 0x21,
        0x2a, 0x78, 0x7b, 0x52, 0x43, 0x50, 0x5b, 0x76,
        0x5f, 0x4b, 0x6a, 0x5d, 0x23, 0x5b, 0x57, 0x40,
        0x53, 0x51, 0x33, 0x21, 0x35, 0x7d, 0x31, 0x46,
        0x65, 0x52, 0x28, 0x25, 0x30, 0x5a, 0x37, 0x7c,
        0x2c, 0x3d, 0x2a, 0x48, 0x24, 0x5a, 0x2f, 0x47,
        0x64, 0x73, 0x64, 0x3d, 0x7a, 0x5b, 0x34, 0x5e,
        0x42, 0x22, 0x32, 0x47, 0x6e, 0x58, 0x3b, 0x3e,
        0x25, 0x2f, 0x58, 0x78, 0x42, 0x66, 0x71, 0x56,
        0x2a, 0x66, 0x66, 0x5b, 0x55, 0x35, 0x7a, 0x41,
        0x7c, 0x7c, 0x6a, 0x2d, 0x59, 0x25, 0x22, 0x34,
        0x5a, 0x61, 0x37, 0x48, 0x39, 0x31, 0x4a, 0x55,
        0x6a, 0x68, 0x40, 0x2f, 0x45, 0x69, 0x46, 0x25,
        0x51, 0x7d, 0x4f, 0x71, 0x21, 0x33, 0x55, 0x50,
        0x56, 0x5f, 0x75, 0x27, 0x64, 0x36, 0x7a, 0x39,
        0x40, 0x6a, 0x77, 0x38, 0x5d, 0x39, 0x30, 0x5e,
        0x74, 0x54, 0x24, 0x3f, 0x3d, 0x79, 0x3b, 0x27,
        0x7d, 0x68, 0x7d, 0x40, 0x71, 0x7a, 0x65, 0x54,
        0x50, 0x66, 0x33, 0x3c, 0x42, 0x69, 0x6e, 0x3c,
        0x63, 0x63, 0x69, 0x7a, 0x5e, 0x7b, 0x76, 0x26,
        0x71, 0x6f, 0x4a, 0x6d, 0x70, 0x73, 0x66, 0x3b,
        0x26, 0x70, 0x43, 0x5b, 0x52, 0x4c, 0x6d, 0x51,
        0x2a, 0x66, 0x6c, 0x3e, 0x68, 0x6a, 0x31, 0x41,
        0x79, 0x72, 0x37, 0x47, 0x7d, 0x2b, 0x3c, 0x40,
        0x6b, 0x75, 0x56, 0x70, 0x7b, 0x2d, 0x5f, 0x33,
        0x30, 0x30, 0x21, 0x35, 0x7a, 0x7a, 0x67, 0x48,
        0x5e, 0x3b, 0x73, 0x50, 0x54, 0x47, 0x23, 0x2b,
        0x4c, 0x4e, 0x2f, 0x24, 0x44, 0x34, 0x23, 0x5d,
        0x76, 0x51, 0x5a, 0x73, 0x72, 0x3e, 0x47, 0x77,
        0x40, 0x28, 0x65, 0x2e, 0x2a, 0x75, 0x3c, 0x2a,
        0x27, 0x4a, 0x3f, 0x3c, 0x66, 0x2d, 0x21, 0x79,
        0x2d, 0x2b, 0x78, 0x7c, 0x5a, 0x73, 0x46, 0x6b,
        0x39, 0x65, 0x5e, 0x3d, 0x38, 0x40, 0x32, 0x3e,
        0x21, 0x62, 0x34, 0x41, 0x58, 0x53, 0x67, 0x34,
        0x58, 0x56, 0x61, 0x5b, 0x3e, 0x4e, 0x2c, 0x5b,
        0x73, 0x35, 0x34, 0x35, 0x21, 0x3a, 0x61, 0x5f,
        0x6e, 0x45, 0x78, 0x44, 0x28, 0x23, 0x48, 0x65,
        0x53, 0x47, 0x6e, 0x2c, 0x38, 0x5e, 0x2c, 0x57,
        0x58, 0x30, 0x7a, 0x3b, 0x4b, 0x4a, 0x74, 0x7d,
        0x3e, 0x4d, 0x30, 0x24, 0x76, 0x66, 0x6d, 0x2e,
        0x74, 0x75, 0x28, 0x48, 0x5c, 0x23, 0x6c, 0x46,
        0x27, 0x46, 0x6e, 0x34, 0x63, 0x21, 0x58, 0x54,
        0x50, 0x2f, 0x40, 0x47, 0x40, 0x32, 0x36, 0x48,
        0x5f, 0x7d, 0x4a, 0x41, 0x6e, 0x60, 0x2c, 0x4a,
        0x6a, 0x67, 0x6c, 0x41, 0x27, 0x23, 0x30, 0x48,
        0x6a, 0x49, 0x73, 0x26, 0x77, 0x75, 0x4d, 0x65,
        0x5b, 0x34, 0x79, 0x67, 0x61, 0x5b, 0x5c, 0x2b,
        0x71, 0x3f, 0x62, 0x51, 0x3a, 0x53, 0x42, 0x26,
        0x6f, 0x36, 0x57, 0x3f, 0x2b, 0x34, 0x24, 0x30,
        0x60, 0x55, 0x70, 0x65, 0x70, 0x57, 0x5d, 0x68,
        0x36, 0x52, 0x5d, 0x3f, 0x6a, 0x3a, 0x33, 0x31,
        0x6c, 0x4e, 0x57, 0x79, 0x49, 0x79, 0x69, 0x71,
        0x6f, 0x70, 0x6a, 0x76, 0x4b, 0x2f, 0x33, 0x51,
        0x68, 0x30, 0x2e, 0x77, 0x78, 0x55, 0x2f, 0x53,
        0x52, 0x5e, 0x57, 0x60, 0x3b, 0x6f, 0x69, 0x61,
        0x6c, 0x60, 0x5a, 0x34, 0x5a, 0x35, 0x4b, 0x28,
        0x54, 0x32, 0x6a, 0x35, 0x36, 0x6d, 0x68, 0x47,
        0x5c, 0x74, 0x2e, 0x5f, 0x6c, 0x6d, 0x55, 0x42,
        0x77, 0x64, 0x7d, 0x53, 0x4d, 0x39, 0x2c, 0x41,
        0x42, 0x23, 0x3a, 0x73, 0x40, 0x60, 0x5d, 0x38,
        0x6d, 0x36, 0x56, 0x57, 0x2a, 0x28, 0x3d, 0x3b,
        0x5c, 0x75, 0x35, 0x2d, 0x69, 0x2d, 0x44, 0x51,
        0x27, 0x63, 0x66, 0x33, 0x46, 0x42, 0x2e, 0x36,
        0x6b, 0x7b, 0x2c, 0x23, 0x3b, 0x5a, 0x50, 0x2a,
        0x65, 0x28, 0x3b, 0x3c, 0x51, 0x3f, 0x4d, 0x63,
        0x38, 0x25, 0x74, 0x2e, 0x51, 0x22, 0x31, 0x74,
        0x35, 0x33, 0x23, 0x2d, 0x3f, 0x77, 0x26, 0x2c,
        0x55, 0x6d, 0x27, 0x39, 0x79, 0x76, 0x63, 0x4b,
        0x43, 0x4a, 0x3a, 0x6b, 0x59, 0x55, 0x65, 0x26,
        0x2f, 0x3f, 0x56, 0x67, 0x5a, 0x77, 0x71, 0x22,
        0x51, 0x2b, 0x6d, 0x4c, 0x2c, 0x57, 0x66, 0x76,
        0x37, 0x70, 0x5f, 0x52, 0x29, 0x44, 0x52, 0x22,
        0x57, 0x37, 0x27, 0x79, 0x29, 0x5c, 0x57, 0x3b,
        0x54, 0x3c, 0x3f, 0x53, 0x35, 0x27, 0x5e, 0x7c,
        0x49, 0x77, 0x57, 0x5a, 0x22, 0x76, 0x7c, 0x5b,
        0x2f, 0x53, 0x5e, 0x55, 0x6d, 0x64, 0x67, 0x34,
        0x41, 0x23, 0x76, 0x67, 0x23, 0x78, 0x6a, 0x63,
        0x27, 0x68, 0x43, 0x7d, 0x58, 0x49, 0x2d, 0x79,
        0x2e, 0x75, 0x60, 0x6b, 0x34, 0x48, 0x6f, 0x4a,
        0x6c, 0x48, 0x40, 0x68, 0x5f, 0x35, 0x25, 0x6c,
        0x38, 0x5c, 0x30, 0x32, 0x4c, 0x36, 0x31, 0x29,
        0x74, 0x4a, 0x55, 0x56, 0x6d, 0x4e, 0x23, 0x54,
        0x2e, 0x69, 0x78, 0x61, 0x76, 0x66, 0x22, 0x44,
        0x73, 0x25, 0x44, 0x29, 0x2a, 0x28, 0x3b, 0x67,
        0x48, 0x58, 0x37, 0x4a, 0x76, 0x76, 0x51, 0x4a,
        0x61, 0x70, 0x51, 0x74, 0x40, 0x23, 0x29, 0x63,
        0x69, 0x4a, 0x29, 0x23, 0x34, 0x6a, 0x3b, 0x25,
        0x28, 0x54, 0x45, 0x33, 0x28, 0x44, 0x30, 0x61,
        0x5b, 0x60, 0x51, 0x3f, 0x68, 0x50, 0x70, 0x3d,
        0x58, 0x2e, 0x6e, 0x59, 0x5a, 0x62, 0x66, 0x4d,
        0x7a, 0x2e, 0x37, 0x37, 0x3d, 0x7b, 0x74, 0x79,
        0x48, 0x45, 0x77, 0x56, 0x33, 0x76, 0x71, 0x60,
        0x74, 0x3f, 0x61, 0x22, 0x52, 0x51, 0x71, 0x69
    };
    uint32_t request3_len = sizeof(request3);

    uint8_t request4[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x75, 0x3e, 0x76, 0x3e, 0x66, 0x6b, 0x6b, 0x3e,
        0x6d, 0x59, 0x38, 0x2b, 0x63, 0x4d, 0x2c, 0x73,
        0x54, 0x57, 0x34, 0x25, 0x5b, 0x42, 0x7d, 0x5d,
        0x37, 0x34, 0x2c, 0x79, 0x24, 0x4b, 0x74, 0x73,
        0x25, 0x36, 0x73, 0x3a, 0x2c, 0x55, 0x69, 0x3c,
        0x58, 0x67, 0x33, 0x53, 0x67, 0x5c, 0x61, 0x7b,
        0x44, 0x2e, 0x42, 0x2d, 0x6b, 0x50, 0x55, 0x24,
        0x70, 0x58, 0x60, 0x38, 0x42, 0x45, 0x70, 0x6d,
        0x2f, 0x27, 0x27, 0x2c, 0x21, 0x6d, 0x57, 0x6e,
        0x43, 0x3c, 0x5b, 0x27, 0x7a, 0x34, 0x49, 0x5a,
        0x69, 0x30, 0x3f, 0x6f, 0x77, 0x70, 0x39, 0x2d,
        0x51, 0x74, 0x4b, 0x25, 0x70, 0x51, 0x64, 0x4d,
        0x75, 0x52, 0x5e, 0x3e, 0x37, 0x30, 0x5d, 0x3b,
        0x2c, 0x72, 0x25, 0x6c, 0x6f, 0x79, 0x69, 0x3c,
        0x5b, 0x73, 0x3d, 0x41, 0x28, 0x28, 0x64, 0x60,
        0x4b, 0x7a, 0x2c, 0x4a, 0x6b, 0x3d, 0x2e, 0x6c,
        0x7a, 0x54, 0x70, 0x61, 0x6f, 0x4b, 0x40, 0x28,
        0x59, 0x31, 0x25, 0x21, 0x57, 0x79, 0x4b, 0x31,
        0x6f, 0x4e, 0x71, 0x2b, 0x3c, 0x24, 0x30, 0x28,
        0x3c, 0x61, 0x28, 0x4b, 0x35, 0x61, 0x4d, 0x55,
        0x5e, 0x66, 0x34, 0x5f, 0x61, 0x70, 0x7b, 0x67,
        0x51, 0x55, 0x68, 0x78, 0x26, 0x3a, 0x27, 0x4e,
        0x71, 0x79, 0x4f, 0x67, 0x2c, 0x5a, 0x79, 0x75,
        0x59, 0x3a, 0x33, 0x4a, 0x36, 0x71, 0x72, 0x6d,
        0x49, 0x3e, 0x53, 0x59, 0x2b, 0x2b, 0x27, 0x4e,
        0x50, 0x5d, 0x21, 0x55, 0x64, 0x4b, 0x72, 0x73,
        0x25, 0x55, 0x26, 0x4f, 0x3a, 0x21, 0x54, 0x29,
        0x4f, 0x64, 0x51, 0x59, 0x60, 0x7b, 0x7c, 0x6f,
        0x3e, 0x65, 0x74, 0x6a, 0x5b, 0x52, 0x2c, 0x56,
        0x4e, 0x45, 0x53, 0x4b, 0x7c, 0x38, 0x49, 0x4b,
        0x4e, 0x4f, 0x4a, 0x47, 0x5e, 0x7c, 0x46, 0x3b,
        0x67, 0x2e, 0x43, 0x79, 0x35, 0x55, 0x59, 0x6d,
        0x38, 0x70, 0x2f, 0x59, 0x4f, 0x27, 0x63, 0x40,
        0x66, 0x2d, 0x39, 0x4f, 0x3d, 0x2e, 0x4c, 0x67,
        0x71, 0x7d, 0x34, 0x22, 0x52, 0x4e, 0x36, 0x7b,
        0x2c, 0x39, 0x4d, 0x42, 0x60, 0x75, 0x74, 0x72,
        0x4f, 0x72, 0x68, 0x3a, 0x51, 0x31, 0x2d, 0x21,
        0x4a, 0x35, 0x47, 0x6d, 0x69, 0x3c, 0x50, 0x4c,
        0x59, 0x66, 0x4c, 0x71, 0x24, 0x3a, 0x36, 0x67,
        0x24, 0x5a, 0x59, 0x28, 0x7c, 0x21, 0x5e, 0x77,
        0x68, 0x5e, 0x7b, 0x6e, 0x56, 0x62, 0x36, 0x29,
        0x6f, 0x4f, 0x5d, 0x57, 0x56, 0x2b, 0x75, 0x2a,
        0x2c, 0x69, 0x63, 0x51, 0x74, 0x6e, 0x5e, 0x46,
        0x50, 0x28, 0x2c, 0x3b, 0x32, 0x53, 0x28, 0x78,
        0x59, 0x72, 0x39, 0x5e, 0x44, 0x5c, 0x77, 0x60,
        0x72, 0x44, 0x3b, 0x75, 0x68, 0x39, 0x55, 0x3e,
        0x44, 0x50, 0x76, 0x3c, 0x48, 0x46, 0x43, 0x22,
        0x56, 0x27, 0x21, 0x31, 0x33, 0x4a, 0x5a, 0x74,
        0x41, 0x58, 0x3f, 0x39, 0x29, 0x71, 0x73, 0x30,
        0x57, 0x70, 0x33, 0x62, 0x7b, 0x4a, 0x75, 0x3e,
        0x4d, 0x4c, 0x4e, 0x55, 0x63, 0x38, 0x66, 0x7d,
        0x68, 0x7d, 0x6f, 0x23, 0x55, 0x50, 0x3d, 0x34,
        0x46, 0x5e, 0x2f, 0x55, 0x27, 0x62, 0x68, 0x7c,
        0x6c, 0x21, 0x2b, 0x63, 0x4b, 0x47, 0x6b, 0x6a,
        0x5b, 0x7b, 0x5c, 0x71, 0x37, 0x7c, 0x52, 0x2b,
        0x2f, 0x4a, 0x47, 0x70, 0x78, 0x50, 0x2f, 0x75,
        0x28, 0x4c, 0x60, 0x4c, 0x4c, 0x54, 0x6b, 0x68,
        0x63, 0x4f, 0x47, 0x39, 0x2a, 0x70, 0x51, 0x7d,
        0x28, 0x59, 0x52, 0x46, 0x4b, 0x38, 0x27, 0x49,
        0x50, 0x5d, 0x25, 0x22, 0x5f, 0x48, 0x2c, 0x2f,
        0x67, 0x59, 0x5d, 0x7d, 0x21, 0x3d, 0x72, 0x4f,
        0x5c, 0x5b, 0x41, 0x47, 0x5f, 0x56, 0x69, 0x42,
        0x55, 0x60, 0x68, 0x4b, 0x77, 0x44, 0x4c, 0x3b,
        0x7d, 0x5a, 0x58, 0x43, 0x7a, 0x33, 0x22, 0x58,
        0x58, 0x6f, 0x74, 0x53, 0x57, 0x6d, 0x6e, 0x29,
        0x6b, 0x33, 0x71, 0x68, 0x29, 0x48, 0x67, 0x35,
        0x52, 0x41, 0x6b, 0x36, 0x4f, 0x46, 0x31, 0x24,
        0x73, 0x56, 0x40, 0x48, 0x37, 0x51, 0x24, 0x2a,
        0x59, 0x21, 0x74, 0x76, 0x25, 0x2e, 0x4a, 0x74,
        0x32, 0x29, 0x5f, 0x57, 0x7c, 0x58, 0x30, 0x2c,
        0x7b, 0x70, 0x5b, 0x51, 0x73, 0x27, 0x4a, 0x28,
        0x77, 0x2a, 0x43, 0x28, 0x2e, 0x32, 0x3d, 0x38,
        0x36, 0x2e, 0x6b, 0x40, 0x6c, 0x76, 0x54, 0x66,
        0x4a, 0x5c, 0x25, 0x62, 0x2e, 0x61, 0x48, 0x30,
        0x28, 0x41, 0x40, 0x69, 0x3c, 0x39, 0x36, 0x4b,
        0x64, 0x50, 0x76, 0x3d, 0x52, 0x50, 0x77, 0x33,
        0x3b, 0x65, 0x59, 0x31, 0x5c, 0x48, 0x6a, 0x74,
        0x78, 0x5b, 0x74, 0x60, 0x47, 0x27, 0x60, 0x22,
        0x4a, 0x72, 0x25, 0x34, 0x5d, 0x3a, 0x21, 0x66,
        0x61, 0x7b, 0x34, 0x41, 0x3b, 0x3a, 0x27, 0x44,
        0x48, 0x7c, 0x7a, 0x74, 0x3a, 0x68, 0x59, 0x48,
        0x61, 0x32, 0x49, 0x61, 0x40, 0x22, 0x33, 0x75,
        0x29, 0x76, 0x5b, 0x24, 0x5b, 0x5c, 0x76, 0x5c,
        0x28, 0x75, 0x36, 0x26, 0x2c, 0x65, 0x5e, 0x51,
        0x7b, 0x3a, 0x7d, 0x4f, 0x35, 0x73, 0x6b, 0x5b,
        0x5c, 0x37, 0x35, 0x6b, 0x41, 0x35, 0x40, 0x3a,
        0x22, 0x28, 0x6c, 0x71, 0x46, 0x68, 0x7b, 0x66,
        0x56, 0x24, 0x7c, 0x54, 0x28, 0x30, 0x22, 0x4e,
        0x3c, 0x65, 0x69, 0x36, 0x44, 0x53, 0x3d, 0x6c,
        0x5f, 0x73, 0x6c, 0x6f, 0x5e, 0x27, 0x23, 0x4e,
        0x60, 0x45, 0x2f, 0x3d, 0x37, 0x28, 0x51, 0x29,
        0x77, 0x6a, 0x6b, 0x2a, 0x2a, 0x51, 0x26, 0x4c,
        0x4e, 0x71, 0x77, 0x73, 0x71, 0x2d, 0x5a, 0x2c,
        0x23, 0x3d, 0x5f, 0x62, 0x63, 0x2e, 0x72, 0x2a,
        0x75, 0x66, 0x43, 0x56, 0x5f, 0x21, 0x64, 0x66,
        0x35, 0x3b, 0x7a, 0x45, 0x3f, 0x4f, 0x57, 0x22,
        0x5a, 0x45, 0x65, 0x37, 0x58, 0x5b, 0x43, 0x66,
        0x4f, 0x5d, 0x6e, 0x41, 0x41, 0x62, 0x5e, 0x39,
        0x65, 0x6f, 0x43, 0x4b, 0x5e, 0x51, 0x42, 0x3f,
        0x2d, 0x68, 0x4b, 0x6e, 0x46, 0x6f, 0x21, 0x44,
        0x3c, 0x22, 0x46, 0x31, 0x31, 0x2e, 0x56, 0x2e,
        0x77, 0x48, 0x68, 0x23, 0x4a, 0x36, 0x52, 0x5d,
        0x61, 0x47, 0x71, 0x2e, 0x3a, 0x4a, 0x5b, 0x56,
        0x6b, 0x52, 0x2a, 0x4c, 0x4f, 0x24, 0x34, 0x60,
        0x70, 0x58, 0x7a, 0x76, 0x4b, 0x68, 0x24, 0x5f,
        0x51, 0x6d, 0x75, 0x45, 0x48, 0x21, 0x53, 0x4d,
        0x27, 0x75, 0x5f, 0x50, 0x3e, 0x40, 0x3f, 0x5e,
        0x64, 0x41, 0x5f, 0x68, 0x48, 0x30, 0x71, 0x4b,
        0x66, 0x2c, 0x2f, 0x76, 0x4b, 0x23, 0x46, 0x34,
        0x50, 0x58, 0x52, 0x69, 0x2b, 0x6e, 0x7a, 0x33,
        0x53, 0x43, 0x43, 0x35, 0x54, 0x30, 0x73, 0x63,
        0x3b, 0x43, 0x52, 0x29, 0x45, 0x37, 0x71, 0x79,
        0x5a, 0x26, 0x24, 0x72, 0x73, 0x4e, 0x44, 0x38,
        0x5b, 0x71, 0x36, 0x3a, 0x4f, 0x5b, 0x71, 0x28,
        0x71, 0x79, 0x72, 0x40, 0x6e, 0x51, 0x72, 0x29,
        0x3d, 0x4f, 0x33, 0x22, 0x73, 0x5a, 0x30, 0x71,
        0x58, 0x54, 0x59, 0x48, 0x29, 0x2b, 0x5c, 0x73,
        0x6f, 0x4e, 0x60, 0x2a, 0x72, 0x39, 0x50, 0x59,
        0x6f, 0x48, 0x3e, 0x62, 0x6c, 0x62, 0x49, 0x6c,
        0x2c, 0x3f, 0x43, 0x3f, 0x32, 0x7c, 0x6f, 0x6c,
        0x39, 0x26, 0x26, 0x7b, 0x5d, 0x65, 0x6f, 0x41,
        0x7c, 0x42, 0x2b, 0x65, 0x6f, 0x3e, 0x7b, 0x69,
        0x46, 0x4d, 0x68, 0x68, 0x5a, 0x33, 0x25, 0x5d,
        0x6f, 0x48, 0x7c, 0x77, 0x7d, 0x3f, 0x4e, 0x30,
        0x69, 0x65, 0x28, 0x2e, 0x34, 0x34, 0x41, 0x43,
        0x5e, 0x30, 0x23, 0x3b, 0x60, 0x79, 0x5b, 0x26,
        0x7c, 0x77, 0x3e, 0x43, 0x24, 0x31, 0x3a, 0x56,
        0x24, 0x3c, 0x60, 0x3f, 0x60, 0x55, 0x6a, 0x68
    };
    uint32_t request4_len = sizeof(request4);

    uint8_t request5[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x69, 0x3e, 0x72, 0x44, 0x31, 0x6b, 0x28, 0x2f,
        0x79, 0x37, 0x58, 0x5d, 0x5f, 0x68, 0x71, 0x47,
        0x7a, 0x68, 0x7c, 0x6c, 0x65, 0x3c, 0x74, 0x67,
        0x59, 0x5c, 0x3d, 0x28, 0x65, 0x28, 0x58, 0x74,
        0x44, 0x62, 0x2e, 0x36, 0x54, 0x2f, 0x24, 0x34,
        0x4b, 0x6d, 0x3a, 0x7b, 0x60, 0x71, 0x5a, 0x77,
        0x4a, 0x27, 0x25, 0x70, 0x75, 0x56, 0x78, 0x73,
        0x2e, 0x38, 0x6c, 0x70, 0x66, 0x7b, 0x7b, 0x2d,
        0x78, 0x27, 0x65, 0x63, 0x58, 0x4f, 0x7d, 0x5c,
        0x31, 0x3e, 0x36, 0x6e, 0x65, 0x61, 0x2e, 0x4e,
        0x26, 0x68, 0x2b, 0x33, 0x7d, 0x54, 0x2c, 0x28,
        0x47, 0x3a, 0x31, 0x47, 0x56, 0x32, 0x74, 0x51,
        0x79, 0x65, 0x42, 0x45, 0x60, 0x55, 0x6f, 0x48,
        0x61, 0x23, 0x72, 0x62, 0x74, 0x3a, 0x5a, 0x26,
        0x2d, 0x41, 0x58, 0x62, 0x75, 0x4b, 0x37, 0x2e,
        0x3f, 0x2a, 0x6e, 0x2e, 0x2c, 0x43, 0x6f, 0x53,
        0x5f, 0x48, 0x7a, 0x53, 0x7b, 0x54, 0x28, 0x30,
        0x2b, 0x7a, 0x34, 0x33, 0x28, 0x2b, 0x23, 0x23,
        0x72, 0x38, 0x25, 0x30, 0x35, 0x66, 0x76, 0x46,
        0x2a, 0x57, 0x7a, 0x60, 0x38, 0x5a, 0x26, 0x4f,
        0x78, 0x43, 0x2c, 0x7d, 0x3d, 0x76, 0x7d, 0x66,
        0x48, 0x7d, 0x3e, 0x59, 0x31, 0x58, 0x6b, 0x30,
        0x76, 0x45, 0x6e, 0x70, 0x72, 0x5f, 0x3c, 0x70,
        0x6d, 0x77, 0x42, 0x75, 0x42, 0x73, 0x68, 0x5e,
        0x5f, 0x72, 0x2b, 0x2a, 0x70, 0x38, 0x7a, 0x4c,
        0x58, 0x2e, 0x5e, 0x2d, 0x2d, 0x78, 0x67, 0x5a,
        0x77, 0x34, 0x5a, 0x50, 0x76, 0x2d, 0x2b, 0x77,
        0x37, 0x6e, 0x38, 0x2d, 0x7b, 0x44, 0x78, 0x67,
        0x52, 0x57, 0x79, 0x43, 0x7d, 0x6d, 0x4d, 0x32,
        0x23, 0x37, 0x51, 0x4b, 0x41, 0x60, 0x6e, 0x53,
        0x4e, 0x78, 0x37, 0x37, 0x60, 0x56, 0x64, 0x52,
        0x25, 0x46, 0x53, 0x5f, 0x2b, 0x56, 0x2b, 0x3b,
        0x40, 0x37, 0x33, 0x37, 0x23, 0x43, 0x36, 0x6b,
        0x6b, 0x5d, 0x35, 0x28, 0x7d, 0x6a, 0x2c, 0x68,
        0x28, 0x4b, 0x4a, 0x6c, 0x27, 0x35, 0x51, 0x66,
        0x30, 0x39, 0x28, 0x4d, 0x61, 0x2f, 0x64, 0x36,
        0x59, 0x39, 0x68, 0x4b, 0x24, 0x51, 0x7b, 0x6e,
        0x38, 0x49, 0x55, 0x72, 0x5f, 0x33, 0x5c, 0x26,
        0x45, 0x2f, 0x71, 0x66, 0x33, 0x3d, 0x36, 0x68,
        0x65, 0x48, 0x42, 0x40, 0x58, 0x61, 0x4f, 0x50,
        0x70, 0x5e, 0x3c, 0x5d, 0x56, 0x43, 0x4c, 0x41,
        0x45, 0x54, 0x76, 0x4b, 0x21, 0x25, 0x45, 0x4c,
        0x5e, 0x58, 0x23, 0x7d, 0x34, 0x61, 0x5c, 0x53,
        0x2a, 0x47, 0x37, 0x22, 0x6d, 0x31, 0x42, 0x6e,
        0x22, 0x72, 0x62, 0x55, 0x59, 0x66, 0x28, 0x73,
        0x55, 0x50, 0x5c, 0x6f, 0x52, 0x40, 0x3e, 0x3b,
        0x44, 0x2a, 0x51, 0x3d, 0x4d, 0x47, 0x3a, 0x57,
        0x3e, 0x29, 0x29, 0x7d, 0x40, 0x36, 0x41, 0x3f,
        0x58, 0x77, 0x3b, 0x41, 0x2d, 0x64, 0x5a, 0x72,
        0x7c, 0x7d, 0x30, 0x68, 0x54, 0x34, 0x40, 0x21,
        0x7d, 0x2b, 0x2d, 0x2b, 0x6d, 0x5f, 0x49, 0x57,
        0x68, 0x65, 0x79, 0x2c, 0x21, 0x41, 0x31, 0x55,
        0x27, 0x4d, 0x78, 0x55, 0x2f, 0x61, 0x62, 0x78,
        0x58, 0x25, 0x3a, 0x4b, 0x3e, 0x67, 0x44, 0x7c,
        0x7d, 0x52, 0x3d, 0x3e, 0x3b, 0x62, 0x2d, 0x28,
        0x48, 0x70, 0x2c, 0x79, 0x31, 0x5a, 0x5e, 0x3f,
        0x6a, 0x30, 0x78, 0x41, 0x44, 0x60, 0x4e, 0x63,
        0x63, 0x2e, 0x31, 0x79, 0x2b, 0x47, 0x57, 0x26,
        0x22, 0x6a, 0x46, 0x43, 0x70, 0x30, 0x51, 0x7d,
        0x21, 0x3c, 0x68, 0x74, 0x40, 0x5a, 0x6e, 0x71,
        0x3f, 0x76, 0x73, 0x2e, 0x29, 0x3f, 0x6a, 0x55,
        0x21, 0x72, 0x65, 0x75, 0x5e, 0x6b, 0x39, 0x6e,
        0x3e, 0x76, 0x42, 0x41, 0x65, 0x3f, 0x2b, 0x37,
        0x70, 0x7a, 0x7a, 0x29, 0x50, 0x66, 0x21, 0x67,
        0x3f, 0x54, 0x32, 0x5f, 0x73, 0x27, 0x59, 0x6f,
        0x39, 0x4b, 0x4e, 0x23, 0x54, 0x3b, 0x39, 0x21,
        0x38, 0x41, 0x33, 0x44, 0x57, 0x6b, 0x51, 0x30,
        0x6a, 0x76, 0x62, 0x2c, 0x5c, 0x5e, 0x49, 0x3e,
        0x59, 0x38, 0x5e, 0x4a, 0x59, 0x77, 0x34, 0x25,
        0x4f, 0x76, 0x6a, 0x68, 0x6f, 0x73, 0x7c, 0x3d,
        0x2d, 0x64, 0x6c, 0x7a, 0x3d, 0x2c, 0x26, 0x28,
        0x58, 0x2b, 0x4b, 0x45, 0x68, 0x38, 0x74, 0x63,
        0x7b, 0x4a, 0x63, 0x52, 0x26, 0x54, 0x3c, 0x46,
        0x77, 0x2d, 0x6b, 0x78, 0x63, 0x7b, 0x6a, 0x50,
        0x26, 0x42, 0x62, 0x63, 0x65, 0x6b, 0x63, 0x54,
        0x4d, 0x47, 0x59, 0x48, 0x2e, 0x60, 0x7c, 0x4d,
        0x33, 0x4d, 0x61, 0x72, 0x76, 0x72, 0x21, 0x4d,
        0x2b, 0x43, 0x58, 0x47, 0x4a, 0x36, 0x2d, 0x7b,
        0x32, 0x72, 0x21, 0x78, 0x22, 0x38, 0x2c, 0x7a,
        0x34, 0x44, 0x45, 0x66, 0x31, 0x7b, 0x37, 0x68,
        0x62, 0x65, 0x62, 0x6d, 0x4e, 0x7c, 0x75, 0x38,
        0x2a, 0x73, 0x27, 0x64, 0x33, 0x4f, 0x21, 0x41,
        0x7c, 0x41, 0x3f, 0x60, 0x68, 0x34, 0x72, 0x5b,
        0x38, 0x33, 0x6f, 0x65, 0x3e, 0x5a, 0x7d, 0x25,
        0x49, 0x50, 0x60, 0x36, 0x59, 0x5e, 0x6b, 0x25,
        0x66, 0x7a, 0x7d, 0x71, 0x40, 0x6c, 0x2c, 0x6e,
        0x6a, 0x5a, 0x24, 0x5a, 0x76, 0x21, 0x67, 0x39,
        0x4b, 0x4a, 0x31, 0x24, 0x66, 0x66, 0x2e, 0x58,
        0x43, 0x46, 0x75, 0x6c, 0x47, 0x28, 0x4f, 0x21,
        0x75, 0x77, 0x6f, 0x71, 0x48, 0x3f, 0x4d, 0x4c,
        0x51, 0x37, 0x3b, 0x41, 0x4d, 0x41, 0x48, 0x28,
        0x71, 0x24, 0x2f, 0x7a, 0x22, 0x49, 0x4a, 0x39,
        0x44, 0x43, 0x68, 0x21, 0x3a, 0x34, 0x4e, 0x52,
        0x7a, 0x60, 0x71, 0x61, 0x6d, 0x51, 0x58, 0x2a,
        0x59, 0x4c, 0x4a, 0x59, 0x6b, 0x77, 0x78, 0x2e,
        0x27, 0x78, 0x76, 0x48, 0x4f, 0x46, 0x79, 0x2c,
        0x54, 0x42, 0x7b, 0x2c, 0x52, 0x41, 0x54, 0x2b,
        0x2c, 0x33, 0x6b, 0x70, 0x77, 0x2e, 0x2e, 0x41,
        0x25, 0x7a, 0x48, 0x6e, 0x71, 0x55, 0x6a, 0x43,
        0x5a, 0x2c, 0x6c, 0x76, 0x6d, 0x71, 0x72, 0x4d,
        0x76, 0x5b, 0x7b, 0x22, 0x4b, 0x45, 0x31, 0x30,
        0x26, 0x53, 0x75, 0x3f, 0x26, 0x59, 0x36, 0x2f,
        0x68, 0x4f, 0x34, 0x5e, 0x2b, 0x30, 0x63, 0x68,
        0x7b, 0x32, 0x5e, 0x77, 0x7d, 0x7b, 0x53, 0x5f,
        0x63, 0x53, 0x77, 0x7a, 0x7d, 0x35, 0x28, 0x3e,
        0x41, 0x6f, 0x5b, 0x31, 0x78, 0x7b, 0x2b, 0x51,
        0x23, 0x43, 0x46, 0x6a, 0x32, 0x32, 0x25, 0x45,
        0x57, 0x43, 0x22, 0x50, 0x60, 0x32, 0x70, 0x2e,
        0x79, 0x2e, 0x6b, 0x33, 0x67, 0x6c, 0x43, 0x5b,
        0x3b, 0x68, 0x53, 0x53, 0x6a, 0x48, 0x59, 0x5f,
        0x30, 0x72, 0x7d, 0x6b, 0x37, 0x24, 0x75, 0x52,
        0x50, 0x2b, 0x75, 0x35, 0x24, 0x3b, 0x6e, 0x53,
        0x56, 0x34, 0x23, 0x54, 0x65, 0x4f, 0x78, 0x3e,
        0x46, 0x7d, 0x25, 0x3f, 0x2f, 0x49, 0x6b, 0x49,
        0x47, 0x45, 0x24, 0x38, 0x3b, 0x68, 0x6c, 0x4f,
        0x29, 0x21, 0x50, 0x32, 0x67, 0x47, 0x5a, 0x72,
        0x76, 0x21, 0x39, 0x67, 0x3c, 0x72, 0x47, 0x43,
        0x4a, 0x2e, 0x31, 0x32, 0x34, 0x3c, 0x53, 0x2d,
        0x22, 0x5b, 0x5b, 0x6a, 0x77, 0x75, 0x31, 0x68,
        0x30, 0x45, 0x43, 0x5f, 0x60, 0x5d, 0x56, 0x67,
        0x66, 0x55, 0x6a, 0x72, 0x77, 0x7b, 0x44, 0x61,
        0x22, 0x64, 0x36, 0x39, 0x6e, 0x44, 0x37, 0x54,
        0x45, 0x46, 0x6f, 0x58, 0x35, 0x51, 0x3c, 0x62,
        0x49, 0x3a, 0x50, 0x58, 0x56, 0x5d, 0x77, 0x6f,
        0x56, 0x64, 0x7b, 0x49, 0x39, 0x21, 0x31, 0x2d,
        0x5f, 0x56, 0x56, 0x33, 0x31, 0x69, 0x4a, 0x52,
        0x62, 0x5b, 0x6e, 0x65, 0x7c, 0x3d, 0x31, 0x55,
        0x3d, 0x75, 0x25, 0x61, 0x50, 0x71, 0x45, 0x29
    };
    uint32_t request5_len = sizeof(request5);

    uint8_t request6[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x5b, 0x56, 0x3d, 0x5a, 0x6b, 0x43, 0x73, 0x26,
        0x65, 0x3b, 0x38, 0x79, 0x26, 0x5e, 0x60, 0x59,
        0x40, 0x71, 0x7c, 0x72, 0x28, 0x29, 0x69, 0x32,
        0x72, 0x5a, 0x6c, 0x55, 0x43, 0x65, 0x3f, 0x4a,
        0x21, 0x66, 0x59, 0x30, 0x76, 0x39, 0x21, 0x69,
        0x4b, 0x25, 0x5d, 0x6e, 0x5f, 0x24, 0x2b, 0x38,
        0x70, 0x78, 0x35, 0x7d, 0x39, 0x36, 0x31, 0x72,
        0x44, 0x49, 0x45, 0x3d, 0x25, 0x50, 0x24, 0x3b,
        0x52, 0x27, 0x66, 0x46, 0x5d, 0x4f, 0x34, 0x50,
        0x26, 0x5a, 0x25, 0x3e, 0x3f, 0x34, 0x4b, 0x35,
        0x77, 0x3a, 0x3f, 0x3e, 0x23, 0x4e, 0x30, 0x23,
        0x70, 0x72, 0x33, 0x34, 0x60, 0x2a, 0x4a, 0x32,
        0x6e, 0x29, 0x54, 0x73, 0x5f, 0x26, 0x71, 0x3a,
        0x78, 0x5d, 0x3f, 0x31, 0x48, 0x59, 0x61, 0x44,
        0x5c, 0x38, 0x4f, 0x41, 0x73, 0x67, 0x62, 0x73,
        0x33, 0x52, 0x77, 0x73, 0x57, 0x49, 0x7a, 0x59,
        0x26, 0x21, 0x34, 0x38, 0x2b, 0x5f, 0x5f, 0x37,
        0x74, 0x28, 0x46, 0x3d, 0x43, 0x42, 0x26, 0x66,
        0x63, 0x37, 0x6d, 0x2a, 0x65, 0x3f, 0x71, 0x2d,
        0x4c, 0x72, 0x29, 0x4b, 0x3a, 0x77, 0x64, 0x6a,
        0x6b, 0x42, 0x70, 0x5c, 0x51, 0x38, 0x71, 0x25,
        0x4c, 0x7c, 0x6f, 0x74, 0x71, 0x39, 0x71, 0x25,
        0x3f, 0x62, 0x23, 0x45, 0x5f, 0x77, 0x59, 0x56,
        0x56, 0x67, 0x78, 0x3a, 0x2e, 0x4e, 0x27, 0x59,
        0x65, 0x2f, 0x64, 0x3c, 0x62, 0x40, 0x69, 0x52,
        0x36, 0x49, 0x3e, 0x3b, 0x2c, 0x47, 0x4f, 0x3e,
        0x61, 0x78, 0x2d, 0x45, 0x71, 0x3f, 0x7b, 0x55,
        0x34, 0x36, 0x47, 0x5e, 0x36, 0x51, 0x3d, 0x5a,
        0x4b, 0x75, 0x44, 0x72, 0x61, 0x44, 0x71, 0x4e,
        0x42, 0x6a, 0x2c, 0x34, 0x40, 0x3b, 0x40, 0x31,
        0x31, 0x75, 0x4b, 0x32, 0x71, 0x69, 0x3a, 0x5d,
        0x31, 0x25, 0x53, 0x2a, 0x61, 0x54, 0x68, 0x2a,
        0x76, 0x71, 0x57, 0x67, 0x56, 0x23, 0x7d, 0x70,
        0x7d, 0x28, 0x57, 0x5f, 0x2f, 0x4c, 0x71, 0x2e,
        0x40, 0x63, 0x49, 0x5b, 0x7c, 0x7b, 0x56, 0x76,
        0x77, 0x46, 0x69, 0x56, 0x3d, 0x75, 0x31, 0x3b,
        0x35, 0x40, 0x37, 0x2c, 0x51, 0x37, 0x49, 0x6a,
        0x79, 0x68, 0x53, 0x31, 0x4c, 0x6f, 0x57, 0x4c,
        0x48, 0x31, 0x6a, 0x30, 0x2b, 0x69, 0x30, 0x56,
        0x58, 0x4b, 0x76, 0x3b, 0x60, 0x6d, 0x35, 0x4d,
        0x74, 0x2f, 0x74, 0x2c, 0x54, 0x4f, 0x6e, 0x3f,
        0x38, 0x56, 0x5c, 0x67, 0x2b, 0x4a, 0x35, 0x30,
        0x67, 0x7d, 0x58, 0x24, 0x59, 0x54, 0x48, 0x2e,
        0x28, 0x7d, 0x6e, 0x51, 0x55, 0x68, 0x56, 0x54,
        0x59, 0x31, 0x4a, 0x65, 0x5a, 0x5e, 0x27, 0x76,
        0x76, 0x65, 0x6d, 0x2f, 0x75, 0x63, 0x67, 0x52,
        0x5e, 0x29, 0x58, 0x3d, 0x5c, 0x3f, 0x54, 0x7c,
        0x67, 0x21, 0x6e, 0x75, 0x67, 0x35, 0x77, 0x31,
        0x3d, 0x26, 0x3f, 0x60, 0x45, 0x2d, 0x2b, 0x45,
        0x5d, 0x3f, 0x55, 0x73, 0x59, 0x4c, 0x5e, 0x6c,
        0x30, 0x4a, 0x4e, 0x47, 0x55, 0x42, 0x6a, 0x4b,
        0x32, 0x3c, 0x75, 0x6e, 0x36, 0x51, 0x5f, 0x4c,
        0x68, 0x72, 0x72, 0x27, 0x3b, 0x51, 0x59, 0x7b,
        0x68, 0x7b, 0x3b, 0x54, 0x35, 0x37, 0x7c, 0x44,
        0x43, 0x36, 0x4c, 0x4f, 0x67, 0x62, 0x4e, 0x39,
        0x4b, 0x7a, 0x49, 0x36, 0x68, 0x38, 0x4c, 0x4a,
        0x64, 0x33, 0x35, 0x2f, 0x3e, 0x5c, 0x58, 0x61,
        0x23, 0x5b, 0x50, 0x6e, 0x34, 0x44, 0x60, 0x28,
        0x54, 0x41, 0x5c, 0x31, 0x53, 0x2d, 0x58, 0x58,
        0x54, 0x28, 0x77, 0x51, 0x6f, 0x64, 0x4c, 0x68,
        0x34, 0x79, 0x45, 0x66, 0x2c, 0x26, 0x77, 0x64,
        0x5f, 0x6c, 0x3b, 0x71, 0x28, 0x4d, 0x68, 0x2a,
        0x6b, 0x37, 0x6a, 0x34, 0x51, 0x27, 0x2a, 0x46,
        0x3a, 0x2e, 0x35, 0x21, 0x21, 0x79, 0x51, 0x44,
        0x58, 0x5d, 0x6f, 0x65, 0x6b, 0x76, 0x68, 0x3a,
        0x43, 0x70, 0x36, 0x41, 0x6b, 0x56, 0x64, 0x75,
        0x5b, 0x37, 0x24, 0x56, 0x7c, 0x6e, 0x6c, 0x41,
        0x3a, 0x60, 0x56, 0x38, 0x55, 0x63, 0x77, 0x4d,
        0x6e, 0x50, 0x3c, 0x3d, 0x7a, 0x44, 0x71, 0x42,
        0x4b, 0x55, 0x75, 0x72, 0x61, 0x60, 0x65, 0x6f,
        0x7a, 0x26, 0x64, 0x46, 0x67, 0x74, 0x29, 0x2a,
        0x5b, 0x62, 0x41, 0x28, 0x62, 0x30, 0x34, 0x33,
        0x40, 0x79, 0x7a, 0x38, 0x56, 0x38, 0x73, 0x22,
        0x7a, 0x7d, 0x73, 0x2a, 0x2a, 0x28, 0x2b, 0x63,
        0x27, 0x6f, 0x3d, 0x3e, 0x2c, 0x56, 0x23, 0x32,
        0x4b, 0x3b, 0x58, 0x4d, 0x72, 0x4c, 0x49, 0x6f,
        0x30, 0x76, 0x23, 0x21, 0x21, 0x3c, 0x49, 0x56,
        0x7a, 0x56, 0x79, 0x2f, 0x50, 0x7a, 0x5b, 0x21,
        0x21, 0x4a, 0x48, 0x61, 0x33, 0x52, 0x49, 0x2e,
        0x30, 0x7d, 0x2c, 0x2d, 0x67, 0x23, 0x55, 0x62,
        0x66, 0x52, 0x5a, 0x61, 0x75, 0x63, 0x3c, 0x39,
        0x69, 0x41, 0x31, 0x6b, 0x4e, 0x6f, 0x25, 0x34,
        0x74, 0x30, 0x21, 0x3a, 0x40, 0x72, 0x44, 0x40,
        0x60, 0x4c, 0x53, 0x74, 0x42, 0x64, 0x44, 0x49,
        0x76, 0x67, 0x21, 0x79, 0x36, 0x3c, 0x37, 0x70,
        0x4f, 0x58, 0x29, 0x71, 0x2a, 0x3a, 0x4d, 0x5d,
        0x67, 0x68, 0x52, 0x63, 0x23, 0x24, 0x4b, 0x21,
        0x3f, 0x68, 0x69, 0x6c, 0x66, 0x66, 0x42, 0x28,
        0x59, 0x35, 0x34, 0x6f, 0x2d, 0x6a, 0x25, 0x66,
        0x34, 0x54, 0x5d, 0x50, 0x26, 0x41, 0x22, 0x4f,
        0x34, 0x79, 0x3c, 0x50, 0x68, 0x2d, 0x5f, 0x7b,
        0x63, 0x7d, 0x58, 0x2e, 0x73, 0x46, 0x2f, 0x54,
        0x61, 0x27, 0x74, 0x45, 0x23, 0x72, 0x31, 0x7d,
        0x63, 0x4b, 0x43, 0x5e, 0x44, 0x54, 0x2c, 0x38,
        0x58, 0x24, 0x75, 0x6c, 0x50, 0x3c, 0x23, 0x5f,
        0x35, 0x57, 0x4f, 0x7b, 0x2f, 0x57, 0x29, 0x73,
        0x58, 0x2a, 0x66, 0x3e, 0x49, 0x42, 0x5a, 0x6b,
        0x75, 0x6a, 0x38, 0x3f, 0x73, 0x44, 0x42, 0x46,
        0x2d, 0x39, 0x66, 0x5b, 0x28, 0x3e, 0x63, 0x62,
        0x53, 0x75, 0x65, 0x64, 0x79, 0x32, 0x35, 0x71,
        0x22, 0x6a, 0x7b, 0x41, 0x2b, 0x26, 0x43, 0x79,
        0x58, 0x6f, 0x71, 0x25, 0x24, 0x34, 0x72, 0x5b,
        0x4a, 0x2c, 0x5c, 0x77, 0x23, 0x42, 0x27, 0x6a,
        0x67, 0x51, 0x5f, 0x3c, 0x75, 0x2c, 0x3f, 0x43,
        0x45, 0x5b, 0x48, 0x65, 0x6f, 0x6c, 0x27, 0x65,
        0x21, 0x3e, 0x33, 0x37, 0x5f, 0x2b, 0x2e, 0x24,
        0x22, 0x47, 0x4e, 0x33, 0x5b, 0x7b, 0x21, 0x3c,
        0x53, 0x69, 0x2e, 0x31, 0x3d, 0x48, 0x57, 0x3a,
        0x56, 0x48, 0x6b, 0x47, 0x5d, 0x33, 0x41, 0x6c,
        0x66, 0x4c, 0x61, 0x67, 0x32, 0x69, 0x53, 0x2c,
        0x2f, 0x3e, 0x36, 0x68, 0x37, 0x28, 0x40, 0x21,
        0x76, 0x27, 0x44, 0x26, 0x24, 0x6a, 0x30, 0x75,
        0x2a, 0x73, 0x48, 0x36, 0x52, 0x4a, 0x3b, 0x51,
        0x4e, 0x2f, 0x23, 0x36, 0x4b, 0x49, 0x33, 0x5a,
        0x70, 0x2c, 0x54, 0x5b, 0x67, 0x48, 0x53, 0x5d,
        0x21, 0x3e, 0x6b, 0x52, 0x6a, 0x3c, 0x48, 0x29,
        0x68, 0x27, 0x32, 0x75, 0x61, 0x7c, 0x51, 0x2e,
        0x7b, 0x49, 0x2f, 0x5b, 0x3d, 0x74, 0x5a, 0x28,
        0x26, 0x29, 0x2c, 0x30, 0x54, 0x74, 0x45, 0x55,
        0x4a, 0x3d, 0x39, 0x35, 0x66, 0x56, 0x28, 0x6d,
        0x6e, 0x38, 0x7b, 0x2b, 0x40, 0x31, 0x56, 0x61,
        0x74, 0x2b, 0x79, 0x5f, 0x63, 0x51, 0x53, 0x52,
        0x7d, 0x73, 0x4e, 0x2e, 0x45, 0x3b, 0x22, 0x28,
        0x6c, 0x2b, 0x47, 0x21, 0x50, 0x2a, 0x7c, 0x45,
        0x48, 0x57, 0x3e, 0x2f, 0x6d, 0x66, 0x6c, 0x51,
        0x23, 0x6c, 0x37, 0x4d, 0x4b, 0x4b, 0x66, 0x55,
        0x69, 0x2e, 0x4a, 0x69, 0x71, 0x7c, 0x71, 0x30,
        0x5c, 0x43, 0x46, 0x63, 0x5a, 0x23, 0x75, 0x40
    };
    uint32_t request6_len = sizeof(request6);

    uint8_t request7[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x5d, 0x32, 0x55, 0x71, 0x51, 0x45, 0x4e, 0x54,
        0x34, 0x21, 0x46, 0x77, 0x5e, 0x5b, 0x75, 0x62,
        0x2b, 0x5c, 0x34, 0x26, 0x72, 0x2b, 0x2c, 0x64,
        0x4b, 0x65, 0x56, 0x72, 0x31, 0x7d, 0x6a, 0x5f,
        0x70, 0x26, 0x32, 0x29, 0x7d, 0x21, 0x5b, 0x3e,
        0x5e, 0x53, 0x3d, 0x48, 0x5e, 0x2a, 0x4c, 0x37,
        0x3d, 0x59, 0x79, 0x21, 0x4f, 0x56, 0x79, 0x2a,
        0x4e, 0x28, 0x61, 0x7d, 0x2c, 0x58, 0x2f, 0x78,
        0x5c, 0x3f, 0x5c, 0x42, 0x6d, 0x2f, 0x71, 0x54,
        0x25, 0x31, 0x73, 0x38, 0x6c, 0x31, 0x5a, 0x2e,
        0x42, 0x5b, 0x2d, 0x41, 0x24, 0x4c, 0x37, 0x40,
        0x39, 0x7d, 0x2a, 0x67, 0x60, 0x6a, 0x7a, 0x62,
        0x24, 0x4e, 0x3f, 0x2e, 0x69, 0x35, 0x28, 0x65,
        0x77, 0x53, 0x23, 0x44, 0x59, 0x71, 0x31, 0x5c,
        0x40, 0x5d, 0x3a, 0x27, 0x46, 0x55, 0x30, 0x56,
        0x21, 0x74, 0x3e, 0x73, 0x41, 0x22, 0x52, 0x68,
        0x40, 0x6c, 0x37, 0x3e, 0x62, 0x5a, 0x2e, 0x21,
        0x23, 0x33, 0x27, 0x73, 0x68, 0x26, 0x60, 0x67,
        0x70, 0x58, 0x50, 0x42, 0x58, 0x27, 0x3a, 0x35,
        0x6f, 0x51, 0x62, 0x78, 0x25, 0x2c, 0x7b, 0x66,
        0x34, 0x6a, 0x5a, 0x39, 0x60, 0x70, 0x41, 0x2d,
        0x65, 0x26, 0x5a, 0x67, 0x58, 0x2d, 0x3e, 0x56,
        0x6d, 0x30, 0x4b, 0x4d, 0x5d, 0x45, 0x41, 0x3d,
        0x6e, 0x27, 0x4e, 0x5a, 0x7d, 0x2e, 0x62, 0x4d,
        0x42, 0x70, 0x31, 0x24, 0x73, 0x5c, 0x78, 0x77,
        0x50, 0x73, 0x27, 0x48, 0x3d, 0x35, 0x2c, 0x4b,
        0x40, 0x2d, 0x25, 0x77, 0x5d, 0x3d, 0x6b, 0x50,
        0x6f, 0x57, 0x73, 0x2f, 0x4f, 0x6e, 0x4c, 0x6e,
        0x56, 0x7b, 0x55, 0x3c, 0x6d, 0x60, 0x47, 0x53,
        0x56, 0x39, 0x3b, 0x51, 0x61, 0x71, 0x75, 0x73,
        0x6b, 0x70, 0x58, 0x5f, 0x2c, 0x27, 0x74, 0x49,
        0x2c, 0x2b, 0x53, 0x2d, 0x5b, 0x79, 0x43, 0x34,
        0x39, 0x5a, 0x38, 0x3e, 0x2d, 0x66, 0x70, 0x3d,
        0x49, 0x51, 0x29, 0x4d, 0x5d, 0x4c, 0x57, 0x4a,
        0x2f, 0x41, 0x69, 0x56, 0x57, 0x77, 0x49, 0x58,
        0x75, 0x28, 0x29, 0x4a, 0x6d, 0x54, 0x4f, 0x4f,
        0x3f, 0x58, 0x5f, 0x58, 0x6f, 0x39, 0x22, 0x4d,
        0x5d, 0x31, 0x75, 0x43, 0x2f, 0x7d, 0x31, 0x3d,
        0x4c, 0x4d, 0x76, 0x74, 0x4d, 0x57, 0x3b, 0x56,
        0x57, 0x48, 0x2b, 0x5d, 0x32, 0x67, 0x51, 0x6e,
        0x60, 0x39, 0x6f, 0x64, 0x38, 0x37, 0x52, 0x4b,
        0x52, 0x42, 0x32, 0x4f, 0x24, 0x53, 0x31, 0x6e,
        0x4a, 0x68, 0x2f, 0x28, 0x2e, 0x27, 0x49, 0x75,
        0x77, 0x75, 0x26, 0x47, 0x7c, 0x5d, 0x72, 0x5a,
        0x77, 0x50, 0x2e, 0x6c, 0x27, 0x68, 0x6b, 0x7b,
        0x27, 0x63, 0x21, 0x3d, 0x30, 0x2d, 0x5c, 0x67,
        0x4d, 0x41, 0x79, 0x47, 0x42, 0x50, 0x6d, 0x32,
        0x74, 0x39, 0x62, 0x4d, 0x5f, 0x65, 0x78, 0x4f,
        0x67, 0x3a, 0x60, 0x26, 0x45, 0x61, 0x7c, 0x61,
        0x63, 0x40, 0x46, 0x79, 0x52, 0x47, 0x57, 0x49,
        0x53, 0x4c, 0x48, 0x36, 0x67, 0x47, 0x5c, 0x71,
        0x50, 0x4d, 0x4f, 0x58, 0x26, 0x40, 0x6d, 0x54,
        0x55, 0x67, 0x66, 0x23, 0x70, 0x23, 0x68, 0x70,
        0x4d, 0x2c, 0x7a, 0x3d, 0x60, 0x51, 0x35, 0x64,
        0x56, 0x2f, 0x26, 0x6d, 0x72, 0x6a, 0x59, 0x34,
        0x3a, 0x73, 0x4b, 0x27, 0x33, 0x61, 0x26, 0x45,
        0x61, 0x28, 0x74, 0x22, 0x54, 0x50, 0x2e, 0x39,
        0x6a, 0x2c, 0x27, 0x59, 0x26, 0x73, 0x44, 0x71,
        0x67, 0x4c, 0x37, 0x74, 0x2c, 0x63, 0x52, 0x2a,
        0x60, 0x4f, 0x7b, 0x32, 0x39, 0x21, 0x79, 0x54,
        0x79, 0x6d, 0x28, 0x27, 0x3a, 0x6a, 0x7d, 0x40,
        0x6a, 0x4f, 0x4b, 0x46, 0x61, 0x36, 0x6a, 0x22,
        0x3f, 0x77, 0x2d, 0x6a, 0x3b, 0x73, 0x71, 0x72,
        0x3c, 0x21, 0x2e, 0x3f, 0x33, 0x25, 0x76, 0x64,
        0x64, 0x70, 0x43, 0x32, 0x44, 0x73, 0x61, 0x51,
        0x3c, 0x3b, 0x45, 0x3a, 0x68, 0x46, 0x5b, 0x6e,
        0x36, 0x47, 0x4d, 0x38, 0x26, 0x4f, 0x5c, 0x7d,
        0x73, 0x29, 0x24, 0x78, 0x44, 0x75, 0x40, 0x42,
        0x41, 0x2a, 0x73, 0x2b, 0x24, 0x38, 0x51, 0x67,
        0x36, 0x67, 0x2f, 0x70, 0x58, 0x54, 0x6e, 0x5d,
        0x3b, 0x41, 0x59, 0x76, 0x7d, 0x2d, 0x40, 0x70,
        0x29, 0x4a, 0x4a, 0x31, 0x79, 0x2c, 0x4e, 0x22,
        0x31, 0x59, 0x31, 0x3c, 0x2f, 0x21, 0x29, 0x3f,
        0x65, 0x6c, 0x38, 0x55, 0x4f, 0x27, 0x66, 0x66,
        0x34, 0x45, 0x49, 0x41, 0x56, 0x24, 0x2e, 0x40,
        0x36, 0x23, 0x5a, 0x46, 0x40, 0x23, 0x7b, 0x2d,
        0x69, 0x54, 0x6c, 0x51, 0x58, 0x73, 0x56, 0x60,
        0x5f, 0x60, 0x63, 0x5f, 0x77, 0x6a, 0x4c, 0x2c,
        0x35, 0x39, 0x60, 0x73, 0x63, 0x3e, 0x2d, 0x55,
        0x5a, 0x26, 0x4b, 0x43, 0x3b, 0x56, 0x33, 0x58,
        0x74, 0x51, 0x4f, 0x5c, 0x2a, 0x44, 0x78, 0x66,
        0x78, 0x71, 0x40, 0x29, 0x5e, 0x26, 0x57, 0x51,
        0x49, 0x30, 0x29, 0x73, 0x38, 0x56, 0x6c, 0x41,
        0x78, 0x3d, 0x61, 0x3d, 0x2c, 0x33, 0x46, 0x57,
        0x54, 0x63, 0x3e, 0x79, 0x55, 0x4a, 0x7d, 0x2e,
        0x2a, 0x3c, 0x77, 0x47, 0x35, 0x29, 0x5a, 0x6d,
        0x69, 0x48, 0x6b, 0x73, 0x7d, 0x4f, 0x5f, 0x6f,
        0x3a, 0x7a, 0x4e, 0x54, 0x59, 0x38, 0x62, 0x44,
        0x72, 0x51, 0x57, 0x6a, 0x74, 0x54, 0x4f, 0x77,
        0x6b, 0x66, 0x4a, 0x6b, 0x39, 0x29, 0x69, 0x60,
        0x71, 0x52, 0x6a, 0x32, 0x66, 0x6c, 0x25, 0x76,
        0x27, 0x7a, 0x2c, 0x38, 0x72, 0x4e, 0x5f, 0x40,
        0x26, 0x74, 0x6a, 0x5e, 0x42, 0x38, 0x78, 0x34,
        0x4f, 0x4f, 0x35, 0x27, 0x39, 0x62, 0x52, 0x61,
        0x37, 0x54, 0x47, 0x38, 0x70, 0x31, 0x7a, 0x66,
        0x69, 0x72, 0x24, 0x52, 0x2a, 0x2a, 0x78, 0x72,
        0x2b, 0x2e, 0x2a, 0x57, 0x4a, 0x21, 0x52, 0x3c,
        0x2a, 0x2f, 0x24, 0x58, 0x34, 0x3c, 0x42, 0x5c,
        0x5b, 0x78, 0x27, 0x55, 0x63, 0x58, 0x3e, 0x26,
        0x50, 0x2c, 0x72, 0x60, 0x36, 0x6c, 0x46, 0x58,
        0x63, 0x59, 0x23, 0x2a, 0x2d, 0x63, 0x6a, 0x68,
        0x69, 0x74, 0x3f, 0x49, 0x4f, 0x48, 0x4a, 0x3b,
        0x59, 0x56, 0x77, 0x43, 0x6d, 0x57, 0x28, 0x5f,
        0x39, 0x73, 0x28, 0x74, 0x3c, 0x4f, 0x43, 0x48,
        0x6a, 0x57, 0x5d, 0x41, 0x73, 0x3f, 0x41, 0x7c,
        0x65, 0x5e, 0x2d, 0x38, 0x72, 0x3a, 0x53, 0x3e,
        0x33, 0x47, 0x69, 0x6a, 0x6e, 0x78, 0x67, 0x5d,
        0x35, 0x3b, 0x3f, 0x23, 0x7c, 0x71, 0x3d, 0x7c,
        0x3a, 0x3c, 0x75, 0x6e, 0x00, 0x00, 0x00, 0x00,
        0x50, 0x6a, 0x40, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x50, 0x6a, 0x40, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x50, 0x6a, 0x40, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x50, 0x6a, 0x40, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x50, 0x80, 0x23, 0x00, 0xdf, 0xaf, 0xff, 0x33,
        0x9b, 0x78, 0x70, 0x43, 0xc5, 0x0a, 0x4d, 0x98,
        0x96, 0x02, 0x64, 0x92, 0xc1, 0xee, 0x70, 0x32
    };
    uint32_t request7_len = sizeof(request7);

    uint8_t request8[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x65, 0xc1, 0xef, 0x7b, 0xd6, 0xaa, 0xd6, 0x09,
        0x21, 0xf6, 0xe7, 0xd1, 0x4c, 0xdf, 0x6a, 0x2d,
        0x0a, 0xfb, 0x43, 0xea, 0xda, 0x07, 0x24, 0x84,
        0x88, 0x52, 0x9e, 0xa8, 0xa1, 0x7f, 0x4b, 0x60,
        0xec, 0x94, 0x57, 0x33, 0x06, 0x93, 0x92, 0x25,
        0xd6, 0xac, 0xdc, 0x89, 0x68, 0x5e, 0xbb, 0x32,
        0x2b, 0x17, 0x68, 0xf2, 0x06, 0xb7, 0x86, 0xac,
        0x81, 0xfe, 0x52, 0x27, 0xf5, 0x80, 0x11, 0x0d,
        0x4e, 0x2e, 0x1b, 0xa3, 0x44, 0x8a, 0x58, 0xed,
        0xf3, 0x9c, 0xe9, 0x31, 0x01, 0x72, 0xa6, 0xab,
        0xfa, 0xa8, 0x05, 0x00, 0x37, 0x60, 0x6b, 0x81,
        0xef, 0xf4, 0x96, 0x9a, 0xf7, 0x67, 0x95, 0x27,
        0x7a, 0x25, 0xef, 0x6f, 0x0e, 0xff, 0x2d, 0x15,
        0x7f, 0x23, 0x1c, 0xa7, 0x56, 0x94, 0x4a, 0x18,
        0x98, 0xc6, 0xd8, 0xd2, 0x29, 0x5b, 0x57, 0xb8,
        0x5d, 0x3a, 0x93, 0x58, 0x45, 0x77, 0x36, 0xe3,
        0xd1, 0x36, 0x87, 0xff, 0xe3, 0x94, 0x0f, 0x00,
        0xe6, 0x7c, 0x1a, 0x92, 0xc1, 0x5f, 0x40, 0xc3,
        0xa3, 0x25, 0xce, 0xd4, 0xaf, 0x39, 0xeb, 0x17,
        0xcf, 0x22, 0x43, 0xd9, 0x0c, 0xce, 0x37, 0x86,
        0x46, 0x54, 0xd6, 0xce, 0x00, 0x30, 0x36, 0xae,
        0xf9, 0xb5, 0x2b, 0x11, 0xa0, 0xfe, 0xa3, 0x4b,
        0x2e, 0x05, 0xbe, 0x54, 0xa9, 0xd8, 0xa5, 0x76,
        0x83, 0x5b, 0x63, 0x01, 0x1c, 0xd4, 0x56, 0x72,
        0xcd, 0xdc, 0x4a, 0x1d, 0x77, 0xda, 0x8a, 0x9e,
        0xba, 0xcb, 0x6c, 0xe8, 0x19, 0x5d, 0x68, 0xef,
        0x8e, 0xbc, 0x6a, 0x05, 0x53, 0x0b, 0xc7, 0xc5,
        0x96, 0x84, 0x04, 0xd9, 0xda, 0x4c, 0x42, 0x31,
        0xd9, 0xbd, 0x99, 0x06, 0xf7, 0xa3, 0x0a, 0x19,
        0x49, 0x07, 0x77, 0xf0, 0xdb, 0x7c, 0x43, 0xfa,
        0xb2, 0xad, 0xb0, 0xfa, 0x87, 0x52, 0xba, 0xc9,
        0x94, 0x61, 0xdc, 0xcf, 0x16, 0xac, 0x0f, 0x4a,
        0xa3, 0x6b, 0x5b, 0x6e, 0x27, 0x86, 0x1f, 0xfe,
        0x4d, 0x28, 0x3a, 0xa5, 0x10, 0x54, 0x6d, 0xed,
        0x53, 0xf9, 0x73, 0xc6, 0x6e, 0xa8, 0xc0, 0x97,
        0xcf, 0x56, 0x3b, 0x61, 0xdf, 0xab, 0x83, 0x18,
        0xe8, 0x09, 0xee, 0x6a, 0xb7, 0xf5, 0xc9, 0x62,
        0x55, 0x2d, 0xc7, 0x0c, 0x0d, 0xa0, 0x22, 0xd8,
        0xd4, 0xd6, 0xb2, 0x12, 0x21, 0xd7, 0x73, 0x3e,
        0x41, 0xb0, 0x5c, 0xd4, 0xcf, 0x98, 0xf3, 0x70,
        0xe6, 0x08, 0xe6, 0x2a, 0x4f, 0x24, 0x85, 0xe8,
        0x74, 0xa8, 0x41, 0x5f, 0x0e, 0xfd, 0xf1, 0xf3,
        0xbe, 0x9b, 0x14, 0xfd, 0xc0, 0x73, 0x11, 0xff,
        0xa5, 0x5b, 0x06, 0x34, 0xc3, 0x6c, 0x28, 0x42,
        0x07, 0xfe, 0x8a, 0xa5, 0xbe, 0x72, 0x7a, 0xf7,
        0xfa, 0x25, 0xec, 0x35, 0x5e, 0x98, 0x71, 0x50,
        0x60, 0x35, 0x76, 0x53, 0x40, 0x1a, 0x34, 0xa5,
        0x99, 0x09, 0xa2, 0xc6, 0xca, 0xa5, 0xce, 0x08,
        0x50, 0x45, 0xab, 0x8d, 0xfb, 0xe3, 0xb8, 0xe4,
        0x8a, 0x61, 0x48, 0x14, 0x6e, 0xf7, 0x58, 0x71,
        0xe5, 0x2e, 0xbc, 0x12, 0xd1, 0x25, 0xe9, 0x65,
        0x7a, 0xa1, 0x27, 0xbe, 0x3b, 0x8b, 0xe8, 0xe7,
        0xbc, 0xe1, 0x05, 0xe7, 0x92, 0xeb, 0xb9, 0xdf,
        0x5d, 0x53, 0x74, 0xc0, 0x63, 0x97, 0x80, 0xb8,
        0x3c, 0xae, 0xf3, 0xf2, 0x09, 0x12, 0x81, 0x6c,
        0x69, 0x10, 0x6f, 0xf6, 0xbe, 0x03, 0x7b, 0x88,
        0xcf, 0x26, 0x6b, 0x51, 0x06, 0x23, 0x68, 0x03,
        0xa1, 0xb7, 0xd3, 0x0c, 0xca, 0xbf, 0x29, 0x01,
        0xa9, 0x61, 0x34, 0x75, 0x98, 0x1e, 0x05, 0x59,
        0xb3, 0x46, 0x44, 0xff, 0x2b, 0x98, 0x04, 0x88,
        0x89, 0xfd, 0x7f, 0xd5, 0x19, 0x8a, 0xa6, 0xf3,
        0xd9, 0x44, 0xd5, 0xf9, 0x3a, 0x3c, 0xec, 0xd9,
        0x9b, 0x8c, 0x93, 0x93, 0x2b, 0x44, 0x86, 0x8b,
        0x80, 0x83, 0x23, 0x00, 0xdf, 0xaf, 0xff, 0x33,
        0x9b, 0x78, 0x70, 0x43, 0xf1, 0x55, 0x87, 0xb1,
        0xa1, 0xb3, 0x8e, 0x79, 0x02, 0x70, 0x82, 0x6c,
        0x0b, 0xc1, 0xef, 0x96, 0xf1, 0xef, 0xdd, 0xa2,
        0x69, 0x86, 0xc7, 0x85, 0x09, 0x7e, 0xf0, 0x2f,
        0x8e, 0xa0, 0x5f, 0xea, 0x39, 0x2e, 0x24, 0xf0,
        0x82, 0x30, 0x26, 0xa8, 0xa1, 0x4f, 0xc6, 0x5c,
        0xec, 0x94, 0x87, 0x52, 0x9b, 0x93, 0x92, 0xf3,
        0xa3, 0x1b, 0xc7, 0x8f, 0x9e, 0xb3, 0xbb, 0x32,
        0x2b, 0x17, 0x54, 0xf2, 0x06, 0x0c, 0x86, 0x92,
        0x0f, 0xb8, 0xe0, 0x27, 0x50, 0xaa, 0xeb, 0xf5,
        0x4e, 0x2b, 0x1b, 0xb2, 0x44, 0xe6, 0x58, 0x02,
        0xd7, 0x65, 0xdc, 0x31, 0x01, 0xec, 0xa6, 0xab,
        0xfa, 0xa8, 0x05, 0x00, 0x37, 0x60, 0x4f, 0xa1,
        0x3c, 0x4f, 0x7a, 0x9a, 0x10, 0x67, 0x95, 0xc2,
        0x5b, 0x25, 0xef, 0x76, 0x0e, 0xff, 0x2d, 0x15,
        0x7f, 0x23, 0x1c, 0x77, 0x56, 0x94, 0x4a, 0x18,
        0x98, 0xc6, 0xd8, 0xd2, 0x29, 0x44, 0x57, 0xb8,
        0x40, 0x3a, 0x93, 0x58, 0x45, 0x77, 0x36, 0x36,
        0x07, 0x35, 0x2a, 0xff, 0x00, 0x94, 0x5c, 0x80,
        0xe6, 0x7c, 0x1a, 0x92, 0xc1, 0x5f, 0x40, 0xc3,
        0xbc, 0xf8, 0xce, 0x05, 0x77, 0x39, 0x40, 0x17,
        0xcf, 0x63, 0x43, 0x77, 0x27, 0xce, 0x37, 0x86,
        0x46, 0x54, 0xd6, 0xce, 0x00, 0x30, 0x36, 0xae,
        0x9f, 0x24, 0x2b, 0x5a, 0xa0, 0xfe, 0xa3, 0x4b,
        0x2e, 0x7e, 0xf7, 0x54, 0xa9, 0xd8, 0xa5, 0x76,
        0x83, 0x7b, 0x63, 0x01, 0x1c, 0xd4, 0x56, 0x17,
        0x02, 0xdc, 0x4a, 0x89, 0x77, 0xda, 0x8f, 0x9e,
        0xba, 0xcb, 0x37, 0xe8, 0x19, 0x5d, 0x68, 0x38,
        0x8e, 0xbc, 0x6a, 0x05, 0x53, 0x0b, 0xc7, 0xc5,
        0x96, 0x84, 0x5a, 0xd9, 0x6d, 0x4c, 0x42, 0x31,
        0xd9, 0xf2, 0x99, 0x06, 0xf7, 0x0c, 0x99, 0xbe,
        0x49, 0x07, 0x77, 0xf0, 0x8b, 0x7c, 0x43, 0xfa,
        0xb2, 0xad, 0xb0, 0xfa, 0x87, 0x52, 0xba, 0xc9,
        0x94, 0x61, 0xdc, 0xcf, 0x16, 0xac, 0x0f, 0x4a,
        0xa3, 0x6b, 0x5b, 0x6e, 0x27, 0x86, 0x1f, 0xfe,
        0x4d, 0x28, 0x3a, 0xa5, 0x10, 0x98, 0x6d, 0xed,
        0x53, 0xf9, 0x73, 0xc6, 0xa5, 0xa8, 0xf7, 0x66,
        0xcf, 0x56, 0x3b, 0x61, 0xdf, 0xab, 0x83, 0x18,
        0xe8, 0x09, 0xee, 0x6a, 0xb7, 0xf5, 0xc9, 0x62,
        0x55, 0x2d, 0xc7, 0x0c, 0x0d, 0xa0, 0x22, 0xd8,
        0xd4, 0xd6, 0xb2, 0x12, 0x21, 0xd7, 0x73, 0x3e,
        0x41, 0xb0, 0x5c, 0xd4, 0xcf, 0x98, 0xf3, 0x70,
        0xe6, 0x08, 0xe6, 0x2a, 0x4f, 0x92, 0x85, 0xe8,
        0x74, 0xa8, 0x41, 0x5f, 0x0e, 0xfd, 0xf1, 0xf3,
        0xbe, 0x9b, 0x14, 0xfd, 0xc0, 0x73, 0x11, 0xff,
        0xa5, 0x5b, 0x06, 0x34, 0xc3, 0x5d, 0x28, 0x42,
        0x34, 0xfe, 0x8a, 0xa5, 0xbe, 0x72, 0x7a, 0xf7,
        0xfa, 0x25, 0x2b, 0x35, 0x5e, 0x98, 0x71, 0x50,
        0x2c, 0x35, 0x76, 0x53, 0x4e, 0x1a, 0x34, 0xa5,
        0x99, 0x09, 0xa2, 0xc6, 0xca, 0xa5, 0xce, 0x08,
        0x50, 0x45, 0xab, 0x8d, 0xfb, 0xe3, 0xb8, 0xe4,
        0x8a, 0x61, 0x48, 0x14, 0x6e, 0xf7, 0x58, 0x71,
        0xe5, 0x2e, 0xbc, 0x12, 0xd1, 0x25, 0xe9, 0x65,
        0x7a, 0xa1, 0x27, 0xbe, 0x3b, 0x8b, 0xe8, 0xe7,
        0xbc, 0x77, 0x05, 0xe7, 0x92, 0xeb, 0xb9, 0xdf,
        0x5d, 0x53, 0x74, 0xc0, 0x63, 0x97, 0x80, 0xb8,
        0x3c, 0xae, 0xf3, 0xf2, 0x09, 0x12, 0x81, 0x6c,
        0x69, 0x10, 0x6f, 0xf6, 0xbe, 0x03, 0x7b, 0x88,
        0xcf, 0x26, 0x6b, 0x51, 0x06, 0x23, 0x68, 0x03,
        0xa1, 0xb7, 0xd3, 0x0c, 0xca, 0xbf, 0x29, 0x01,
        0xa9, 0x61, 0x34, 0x75, 0x98, 0x1e, 0x6f, 0x59,
        0xb3, 0x46, 0x44, 0xff, 0x2b, 0x98, 0x04, 0x88,
        0x89, 0xfd, 0x1c, 0xd5, 0x19, 0x8a, 0xa6, 0xf3,
        0xd9, 0x44, 0xd5, 0xf9, 0x79, 0x26, 0x46, 0xf7
    };
    uint32_t request8_len = sizeof(request8);

    uint8_t request9[] = {
        0x05, 0x00, 0x00, 0x02, 0x10, 0x00, 0x00, 0x00,
        0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xbf, 0xa1, 0x12, 0x73, 0x23, 0x44, 0x86, 0x8b,
        0x50, 0x6a, 0x40, 0x00
    };
    uint32_t request9_len = sizeof(request9);

    TcpSession ssn;
    Packet *p[11];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|26 d0 cf 80|\"; distance:0; sid:1;)";
    char *sig2 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|43 5b 67 26 65|\"; distance:0; sid:2;)";
    char *sig3 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|71 69 75 3e|\"; distance:0; sid:3;)";
    char *sig4 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|6a 68 69 3e 72|\"; distance:0; sid:4;)";
    char *sig5 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|61 50 71 45 29 5b 56 3d 5a|\"; distance:0; sid:5;)";
    char *sig6 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|23 75 40 5d 32 55|\"; distance:0; sid:6;)";
    char *sig7 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|ee 70 32 65 c1|\"; distance:0; sid:7;)";
    char *sig8 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|79 26 46 f7 bf a1|\"; distance:0; sid:8;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 11; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }
    p[1]->flowflags |= FLOW_PKT_TOCLIENT;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig2);
    s = s->next;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig3);
    s = s->next;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig4);
    s = s->next;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig5);
    s = s->next;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig6);
    s = s->next;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig7);
    s = s->next;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig8);
    s = s->next;
    if (s == NULL)
        goto end;


    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, bind, bind_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if ((PacketAlertCheck(p[0], 1))) {
        printf("sid 1 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 2))) {
        printf("sid 2 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 3))) {
        printf("sid 3 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 4))) {
        printf("sid 4 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 5))) {
        printf("sid 5 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 6))) {
        printf("sid 6 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 7))) {
        printf("sid 7 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 8))) {
        printf("sid 8 matched but shouldn't have for packet 0: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, bind_ack, bind_ack_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[1]);
    if ((PacketAlertCheck(p[1], 1))) {
        printf("sid 1 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 2))) {
        printf("sid 2 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 3))) {
        printf("sid 3 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 4))) {
        printf("sid 4 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 5))) {
        printf("sid 5 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 6))) {
        printf("sid 6 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 7))) {
        printf("sid 7 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 8))) {
        printf("sid 8 matched but shouldn't have for packet 1: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[2]);
    if ((PacketAlertCheck(p[2], 1))) {
        printf("sid 1 matched but shouldn't have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 2))) {
        printf("sid 2 matched but shouldn't have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 3))) {
        printf("sid 3 matched but shouldn't have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 4))) {
        printf("sid 4 matched but shouldn't have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 5))) {
        printf("sid 5 matched but shouldn't have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 6))) {
        printf("sid 6 matched but shouldn't have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 7))) {
        printf("sid 7 matched but shouldn't have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 8))) {
        printf("sid 8 matched but shouldn't have for packet 2: ");
        goto end;
    }

    SCLogDebug("sending request 2");
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[3]);
    if (!(PacketAlertCheck(p[3], 1))) {
        printf("sid 1 didn't match but should have for packet 3: ");
        goto end;
    }
    if ((PacketAlertCheck(p[3], 2))) {
        printf("sid 2 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if ((PacketAlertCheck(p[3], 3))) {
        printf("sid 3 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if ((PacketAlertCheck(p[3], 4))) {
        printf("sid 4 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if ((PacketAlertCheck(p[3], 5))) {
        printf("sid 5 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if ((PacketAlertCheck(p[3], 6))) {
        printf("sid 6 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if ((PacketAlertCheck(p[3], 7))) {
        printf("sid 7 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if ((PacketAlertCheck(p[3], 8))) {
        printf("sid 8 matched but shouldn't have for packet 3: ");
        goto end;
    }

    SCLogDebug("sending request 3");
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request3, request3_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SCLogDebug("inspecting packet 4");
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[4]);
    if ((PacketAlertCheck(p[4], 1))) {
        printf("sid 1 matched but shouldn't have for packet 4: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[4], 2))) {
        printf("sid 2 didn't match but should have for packet 4: ");
        goto end;
    }
    if ((PacketAlertCheck(p[4], 3))) {
        printf("sid 3 matched but shouldn't have for packet 4: ");
        goto end;
    }
    if ((PacketAlertCheck(p[4], 4))) {
        printf("sid 4 matched but shouldn't have for packet 4: ");
        goto end;
    }
    if ((PacketAlertCheck(p[4], 5))) {
        printf("sid 5 matched but shouldn't have for packet 4: ");
        goto end;
    }
    if ((PacketAlertCheck(p[4], 6))) {
        printf("sid 6 matched but shouldn't have for packet 4: ");
        goto end;
    }
    if ((PacketAlertCheck(p[4], 7))) {
        printf("sid 7 matched but shouldn't have for packet 4: ");
        goto end;
    }
    if ((PacketAlertCheck(p[4], 8))) {
        printf("sid 8 matched but shouldn't have for packet 4: ");
        goto end;
    }

    SCLogDebug("sending request 4");
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request4, request4_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[5]);
    if ((PacketAlertCheck(p[5], 1))) {
        printf("sid 1 matched but shouldn't have for packet 5: ");
        goto end;
    }
    if ((PacketAlertCheck(p[5], 2))) {
        printf("sid 2 matched but shouldn't have for packet 5: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[5], 3))) {
        printf("sid 3 didn't match but should have packet 5: ");
        goto end;
    }
    if ((PacketAlertCheck(p[5], 4))) {
        printf("sid 4 matched but shouldn't have for packet 5: ");
        goto end;
    }
    if ((PacketAlertCheck(p[5], 5))) {
        printf("sid 5 matched but shouldn't have for packet 5: ");
        goto end;
    }
    if ((PacketAlertCheck(p[5], 6))) {
        printf("sid 6 matched but shouldn't have for packet 5: ");
        goto end;
    }
    if ((PacketAlertCheck(p[5], 7))) {
        printf("sid 7 matched but shouldn't have for packet 5: ");
        goto end;
    }
    if ((PacketAlertCheck(p[5], 8))) {
        printf("sid 8 matched but shouldn't have for packet 5: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request5, request5_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[6]);
    if ((PacketAlertCheck(p[6], 1))) {
        printf("sid 1 matched but shouldn't have for packet 6: ");
        goto end;
    }
    if ((PacketAlertCheck(p[6], 2))) {
        printf("sid 2 matched but shouldn't have for packet 6: ");
        goto end;
    }
    if ((PacketAlertCheck(p[6], 3))) {
        printf("sid 3 matched but shouldn't have for packet 6: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[6], 4))) {
        printf("sid 4 didn't match but should have packet 6: ");
        goto end;
    }
    if ((PacketAlertCheck(p[6], 5))) {
        printf("sid 5 matched but shouldn't have for packet 6: ");
        goto end;
    }
    if ((PacketAlertCheck(p[6], 6))) {
        printf("sid 6 matched but shouldn't have for packet 6: ");
        goto end;
    }
    if ((PacketAlertCheck(p[6], 7))) {
        printf("sid 7 matched but shouldn't have for packet 6: ");
        goto end;
    }
    if ((PacketAlertCheck(p[6], 8))) {
        printf("sid 8 matched but shouldn't have for packet 6: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request6, request6_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[7]);
    if ((PacketAlertCheck(p[7], 1))) {
        printf("sid 1 matched but shouldn't have for packet 7: ");
        goto end;
    }
    if ((PacketAlertCheck(p[7], 2))) {
        printf("sid 2 matched but shouldn't have for packet 7: ");
        goto end;
    }
    if ((PacketAlertCheck(p[7], 3))) {
        printf("sid 3 matched but shouldn't have for packet 7: ");
        goto end;
    }
    if ((PacketAlertCheck(p[7], 4))) {
        printf("sid 4 matched but shouldn't have for packet 7: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[7], 5))) {
        printf("sid 5 didn't match but should have paket 7: ");
        goto end;
    }
    if ((PacketAlertCheck(p[7], 6))) {
        printf("sid 6 matched but shouldn't have for packet 7: ");
        goto end;
    }
    if ((PacketAlertCheck(p[7], 7))) {
        printf("sid 7 matched but shouldn't have for packet 7: ");
        goto end;
    }
    if ((PacketAlertCheck(p[7], 8))) {
        printf("sid 8 matched but shouldn't have for packet 7: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request7, request7_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[8]);
    if ((PacketAlertCheck(p[8], 1))) {
        printf("sid 1 matched but shouldn't have for packet 8: ");
        goto end;
    }
    if ((PacketAlertCheck(p[8], 2))) {
        printf("sid 2 matched but shouldn't have for packet 8: ");
        goto end;
    }
    if ((PacketAlertCheck(p[8], 3))) {
        printf("sid 3 matched but shouldn't have for packet 8: ");
        goto end;
    }
    if ((PacketAlertCheck(p[8], 4))) {
        printf("sid 4 matched but shouldn't have for packet 8: ");
        goto end;
    }
    if ((PacketAlertCheck(p[8], 5))) {
        printf("sid 5 matched but shouldn't have for packet 8: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[8], 6))) {
        printf("sid 6 didn't match but should have paket 8: ");
        goto end;
    }
    if ((PacketAlertCheck(p[8], 7))) {
        printf("sid 7 matched but shouldn't have for packet 8: ");
        goto end;
    }
    if ((PacketAlertCheck(p[8], 8))) {
        printf("sid 8 matched but shouldn't have for packet 8: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request8, request8_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[9]);
    if ((PacketAlertCheck(p[9], 1))) {
        printf("sid 1 matched but shouldn't have for packet 9: ");
        goto end;
    }
    if ((PacketAlertCheck(p[9], 2))) {
        printf("sid 2 matched but shouldn't have for packet 9: ");
        goto end;
    }
    if ((PacketAlertCheck(p[9], 3))) {
        printf("sid 3 matched but shouldn't have for packet 9: ");
        goto end;
    }
    if ((PacketAlertCheck(p[9], 4))) {
        printf("sid 4 matched but shouldn't have for packet 9: ");
        goto end;
    }
    if ((PacketAlertCheck(p[9], 5))) {
        printf("sid 5 matched but shouldn't have for packet 9: ");
        goto end;
    }
    if ((PacketAlertCheck(p[9], 6))) {
        printf("sid 6 matched but shouldn't have for packet 9: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[9], 7))) {
        printf("sid 7 didn't match but should have for packet 9: ");
        goto end;
    }
    if ((PacketAlertCheck(p[9], 8))) {
        printf("sid 8 matched but shouldn't have for packet 9: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request9, request9_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[10]);
    if ((PacketAlertCheck(p[10], 1))) {
        printf("sid 1 matched but shouldn't have for packet 10: ");
        goto end;
    }
    if ((PacketAlertCheck(p[10], 2))) {
        printf("sid 2 matched but shouldn't have for packet 10: ");
        goto end;
    }
    if ((PacketAlertCheck(p[10], 3))) {
        printf("sid 3 matched but shouldn't have for packet 10: ");
        goto end;
    }
    if ((PacketAlertCheck(p[10], 4))) {
        printf("sid 4 matched but shouldn't have for packet 10: ");
        goto end;
    }
    if ((PacketAlertCheck(p[10], 5))) {
        printf("sid 5 matched but shouldn't have for packet 10: ");
        goto end;
    }
    if ((PacketAlertCheck(p[10], 6))) {
        printf("sid 6 matched but shouldn't have for packet 10: ");
        goto end;
    }
    if ((PacketAlertCheck(p[10], 7))) {
        printf("sid 7 matched but shouldn't have for packet 10: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[10], 8))) {
        printf("sid 8 didn't match but should have for paket 10: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 11);
    return result;
}

/**
 * \test Test the working of detection engien with respect to dce keywords.
 */
int DcePayloadTest02(void)
{
    int result = 0;
    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x6a, 0x28, 0x19, 0x39, 0x0c, 0xb1, 0xd0, 0x11,
        0x9b, 0xa8, 0x00, 0xc0, 0x4f, 0xd9, 0x2e, 0xf5,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_len = sizeof(bind);

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x48, 0x1a, 0x00, 0x00,
        0x0c, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x6c, 0x73, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x82, 0x7c, 0x83, 0x46, 0x0f, 0xf7,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xcf, 0x80, 0x98, 0x6d, 0xfe, 0xb0, 0x90, 0xd1,
        0xcf, 0x86, 0x0f, 0x52, 0x2c, 0x23, 0x66, 0x28,
        0x27, 0x30, 0x48, 0x55, 0x42, 0x6a, 0x48, 0x4b,
        0x68, 0x22, 0x2e, 0x23, 0x64, 0x33, 0x2c, 0x2d,
        0x5c, 0x51, 0x48, 0x55, 0x24, 0x67, 0x6c, 0x4c,
        0x45, 0x71, 0x35, 0x72, 0x5a, 0x48, 0x5e, 0x35,
        0x61, 0x78, 0x35, 0x42, 0x2c, 0x7a, 0x75, 0x61,
        0x5b, 0x4e, 0x76, 0x30, 0x26, 0x2f, 0x2a, 0x34,
        0x48, 0x29, 0x25, 0x6e, 0x5c, 0x3a, 0x6c, 0x3e,
        0x79, 0x4e, 0x2a, 0x21, 0x6f, 0x6f, 0x34, 0x46,
        0x43, 0x26, 0x5b, 0x35, 0x78, 0x27, 0x69, 0x23,
        0x72, 0x21, 0x69, 0x56, 0x6a, 0x7d, 0x4b, 0x5e,
        0x65, 0x37, 0x60, 0x44, 0x7c, 0x5d, 0x5b, 0x72,
        0x7d, 0x73, 0x7b, 0x47, 0x57, 0x21, 0x41, 0x38,
        0x76, 0x38, 0x76, 0x5c, 0x58, 0x32, 0x4a, 0x37,
        0x2f, 0x40, 0x4b, 0x4c, 0x3d, 0x41, 0x33, 0x56,
        0x73, 0x38, 0x61, 0x71, 0x24, 0x49, 0x4c, 0x4a,
        0x44, 0x2e, 0x3a, 0x3f, 0x74, 0x54, 0x4c, 0x65,
        0x54, 0x2d, 0x3b, 0x28, 0x41, 0x45, 0x49, 0x2c,
        0x6e, 0x48, 0x44, 0x43, 0x37, 0x3d, 0x7b, 0x6d,
        0x2b, 0x4b, 0x32, 0x5a, 0x31, 0x61, 0x6e, 0x2b,
        0x27, 0x50, 0x6b, 0x66, 0x76, 0x4e, 0x55, 0x35,
        0x2b, 0x72, 0x2d, 0x5e, 0x42, 0x3e, 0x5a, 0x5d,
        0x36, 0x45, 0x32, 0x3a, 0x58, 0x78, 0x78, 0x3e,
        0x60, 0x6c, 0x5d, 0x63, 0x41, 0x7c, 0x52, 0x21,
        0x75, 0x6a, 0x5a, 0x70, 0x55, 0x45, 0x76, 0x58,
        0x33, 0x40, 0x38, 0x39, 0x21, 0x37, 0x7d, 0x77,
        0x21, 0x70, 0x2b, 0x72, 0x29, 0x6a, 0x31, 0x5f,
        0x38, 0x4a, 0x66, 0x65, 0x62, 0x2c, 0x39, 0x52,
        0x5f, 0x2a, 0x2b, 0x63, 0x4f, 0x76, 0x43, 0x25,
        0x6a, 0x50, 0x37, 0x52, 0x5e, 0x23, 0x3c, 0x42,
        0x28, 0x75, 0x75, 0x42, 0x25, 0x23, 0x28, 0x56,
        0x6c, 0x46, 0x5c, 0x5e, 0x6b, 0x7d, 0x48, 0x24,
        0x77, 0x6c, 0x70, 0x62, 0x2e, 0x28, 0x7d, 0x6b,
        0x69, 0x4a, 0x75, 0x3d, 0x5d, 0x56, 0x21, 0x49,
        0x56, 0x47, 0x64, 0x2b, 0x4c, 0x52, 0x43, 0x60,
        0x77, 0x49, 0x46, 0x46, 0x33, 0x2c, 0x4b, 0x4b,
        0x3d, 0x63, 0x5d, 0x33, 0x78, 0x76, 0x51, 0x56,
        0x77, 0x3c, 0x72, 0x74, 0x52, 0x27, 0x40, 0x6c,
        0x42, 0x79, 0x49, 0x24, 0x62, 0x5e, 0x26, 0x31,
        0x5c, 0x22, 0x2b, 0x4c, 0x64, 0x49, 0x52, 0x45,
        0x47, 0x49, 0x3a, 0x2a, 0x51, 0x71, 0x22, 0x22,
        0x70, 0x24, 0x34, 0x67, 0x4b, 0x6d, 0x58, 0x29,
        0x63, 0x26, 0x7b, 0x6f, 0x38, 0x78, 0x25, 0x62,
        0x4d, 0x3a, 0x7d, 0x40, 0x23, 0x57, 0x67, 0x33,
        0x38, 0x31, 0x4e, 0x54, 0x3c, 0x4b, 0x48, 0x69,
        0x3c, 0x39, 0x31, 0x2b, 0x26, 0x70, 0x44, 0x66,
        0x4a, 0x37, 0x2b, 0x75, 0x36, 0x45, 0x59, 0x34,
        0x3e, 0x3e, 0x29, 0x70, 0x71, 0x5a, 0x55, 0x49,
        0x3e, 0x4b, 0x68, 0x4e, 0x75, 0x70, 0x3c, 0x5c,
        0x50, 0x58, 0x28, 0x75, 0x3c, 0x2a, 0x41, 0x70,
        0x2f, 0x2b, 0x37, 0x26, 0x75, 0x71, 0x55, 0x22,
        0x3a, 0x44, 0x30, 0x48, 0x5d, 0x2f, 0x6c, 0x44,
        0x28, 0x4b, 0x34, 0x45, 0x21, 0x60, 0x44, 0x36,
        0x7b, 0x32, 0x39, 0x5f, 0x6d, 0x3f, 0x68, 0x73,
        0x25, 0x45, 0x56, 0x7c, 0x78, 0x7a, 0x49, 0x6a,
        0x46, 0x3d, 0x2d, 0x33, 0x6c, 0x6f, 0x23, 0x77,
        0x38, 0x33, 0x36, 0x74, 0x7b, 0x57, 0x4b, 0x6d,
        0x27, 0x75, 0x24, 0x6e, 0x43, 0x61, 0x4d, 0x44,
        0x6d, 0x27, 0x48, 0x58, 0x5e, 0x7b, 0x26, 0x6a,
        0x50, 0x7c, 0x51, 0x23, 0x3c, 0x4f, 0x37, 0x4c,
        0x47, 0x3e, 0x45, 0x56, 0x22, 0x33, 0x7c, 0x66,
        0x35, 0x54, 0x7a, 0x6e, 0x5a, 0x24, 0x70, 0x62,
        0x29, 0x3f, 0x69, 0x79, 0x24, 0x43, 0x41, 0x24,
        0x65, 0x25, 0x62, 0x4f, 0x73, 0x3e, 0x2b, 0x36,
        0x46, 0x69, 0x27, 0x55, 0x2a, 0x6e, 0x24, 0x6c,
        0x7d, 0x64, 0x7c, 0x61, 0x26, 0x67, 0x2a, 0x53,
        0x73, 0x60, 0x28, 0x2d, 0x6b, 0x44, 0x54, 0x61,
        0x34, 0x53, 0x22, 0x59, 0x6d, 0x73, 0x56, 0x55,
        0x25, 0x2c, 0x38, 0x4a, 0x3b, 0x4e, 0x78, 0x46,
        0x54, 0x6e, 0x6d, 0x4f, 0x47, 0x4f, 0x4f, 0x5a,
        0x67, 0x77, 0x39, 0x66, 0x28, 0x29, 0x4e, 0x43,
        0x55, 0x6e, 0x60, 0x59, 0x28, 0x3b, 0x65, 0x62,
        0x61, 0x5a, 0x29, 0x6e, 0x79, 0x60, 0x41, 0x53,
        0x2f, 0x5d, 0x44, 0x36, 0x7b, 0x3e, 0x7c, 0x2b,
        0x77, 0x36, 0x70, 0x3f, 0x40, 0x55, 0x48, 0x67,
        0x4b, 0x4d, 0x5d, 0x51, 0x79, 0x76, 0x48, 0x4a,
        0x2d, 0x21, 0x60, 0x40, 0x46, 0x55, 0x7a, 0x60,
        0x22, 0x25, 0x3f, 0x4b, 0x54, 0x6a, 0x6a, 0x3c,
        0x77, 0x22, 0x5b, 0x43, 0x67, 0x58, 0x71, 0x22,
        0x79, 0x4b, 0x32, 0x61, 0x44, 0x4d, 0x6f, 0x42,
        0x33, 0x2d, 0x53, 0x35, 0x3d, 0x6f, 0x57, 0x48,
        0x33, 0x3b, 0x5a, 0x53, 0x3f, 0x4e, 0x3f, 0x6b,
        0x4c, 0x27, 0x26, 0x3b, 0x73, 0x49, 0x22, 0x55,
        0x79, 0x2f, 0x47, 0x2f, 0x55, 0x5a, 0x7a, 0x71,
        0x6c, 0x31, 0x43, 0x40, 0x56, 0x7b, 0x21, 0x7a,
        0x6d, 0x4c, 0x43, 0x5e, 0x38, 0x47, 0x29, 0x38,
        0x62, 0x49, 0x45, 0x78, 0x70, 0x2b, 0x2e, 0x65,
        0x47, 0x71, 0x58, 0x79, 0x39, 0x67, 0x7d, 0x6d,
        0x6a, 0x67, 0x4a, 0x71, 0x27, 0x35, 0x2a, 0x4c,
        0x3e, 0x58, 0x55, 0x30, 0x4d, 0x75, 0x77, 0x48,
        0x5f, 0x4b, 0x59, 0x34, 0x65, 0x68, 0x57, 0x59,
        0x63, 0x23, 0x47, 0x38, 0x47, 0x5e, 0x56, 0x28,
        0x79, 0x58, 0x3e, 0x39, 0x66, 0x77, 0x67, 0x33,
        0x29, 0x61, 0x24, 0x7d, 0x37, 0x44, 0x37, 0x67,
        0x3a, 0x58, 0x76, 0x21, 0x51, 0x59, 0x61, 0x73,
        0x66, 0x75, 0x71, 0x53, 0x4d, 0x24, 0x2d, 0x4b,
        0x29, 0x30, 0x32, 0x26, 0x59, 0x64, 0x27, 0x55,
        0x2c, 0x5a, 0x4c, 0x3c, 0x6c, 0x53, 0x56, 0x4b,
        0x3e, 0x55, 0x2e, 0x44, 0x38, 0x6b, 0x47, 0x76,
        0x2d, 0x2c, 0x3f, 0x4d, 0x22, 0x7b, 0x6d, 0x61,
        0x34, 0x6b, 0x50, 0x73, 0x28, 0x6d, 0x41, 0x71,
        0x21, 0x76, 0x52, 0x2a, 0x6d, 0x53, 0x2a, 0x74,
        0x28, 0x27, 0x62, 0x2a, 0x66, 0x25, 0x6e, 0x5e,
        0x37, 0x4f, 0x27, 0x72, 0x28, 0x47, 0x63, 0x6e,
        0x5a, 0x6a, 0x41, 0x35, 0x3a, 0x42, 0x3f, 0x27,
        0x75, 0x3e, 0x26, 0x3e, 0x6b, 0x55, 0x59, 0x60,
        0x24, 0x70, 0x49, 0x3c, 0x4e, 0x2c, 0x39, 0x7a,
        0x36, 0x6c, 0x27, 0x3e, 0x6a, 0x4a, 0x59, 0x5a,
        0x3e, 0x21, 0x73, 0x4e, 0x59, 0x6e, 0x3d, 0x32,
        0x27, 0x45, 0x49, 0x58, 0x7d, 0x37, 0x39, 0x77,
        0x28, 0x51, 0x79, 0x54, 0x2b, 0x78, 0x46, 0x5a,
        0x21, 0x75, 0x33, 0x21, 0x63, 0x5a, 0x7b, 0x3e,
        0x33, 0x4f, 0x67, 0x75, 0x3a, 0x50, 0x48, 0x60,
        0x26, 0x64, 0x76, 0x5c, 0x42, 0x5c, 0x72, 0x38,
        0x6c, 0x52, 0x21, 0x2b, 0x25, 0x6b, 0x7c, 0x6b,
        0x2d, 0x5e, 0x63, 0x2a, 0x4c, 0x26, 0x5b, 0x4c,
        0x58, 0x52, 0x51, 0x55, 0x31, 0x79, 0x6c, 0x53,
        0x62, 0x3a, 0x36, 0x46, 0x7a, 0x29, 0x27, 0x78,
        0x1a, 0xbf, 0x49, 0x74, 0x68, 0x24, 0x51, 0x44,
        0x5b, 0x3e, 0x34, 0x44, 0x29, 0x5e, 0x4f, 0x2a,
        0xe9, 0x3f, 0xf8, 0xff, 0xff, 0x52, 0x7d, 0x47,
        0x67, 0x40, 0x27, 0x5e, 0x47, 0x46, 0x6d, 0x72,
        0x5d, 0x49, 0x26, 0x45, 0x33, 0x6b, 0x4d, 0x4a,
        0x6f, 0x62, 0x60, 0x45, 0x62, 0x27, 0x27, 0x7d,
        0x6a, 0x41, 0x2c, 0x6c, 0x5b, 0x2a, 0x2b, 0x36,
        0x29, 0x58, 0x7a, 0x4c, 0x6e, 0x2d, 0x74, 0x5c,
        0x38, 0x22, 0x5f, 0x49, 0x63, 0x43, 0x5b, 0x67
    };
    uint32_t request2_len = sizeof(request2);


    TcpSession ssn;
    Packet *p[4];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_iface:3919286a-b10c-11d0-9ba8-00c04fd92ef5; "
        "sid:1;)";
    char *sig2 = "alert tcp any any -> any any (dce_stub_data; "
        "dce_stub_data; content:\"|2d 5e 63 2a 4c|\"; distance:0; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 4; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }
    p[1]->flowflags |= FLOW_PKT_TOCLIENT;
    p[1]->flowflags &=~ FLOW_PKT_TOSERVER;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig2);
    s = s->next;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, bind, bind_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if (PacketAlertCheck(p[0], 1)) {
        printf("sid 1 didn't match but should have for packet 0: ");
        goto end;
    }
    if (PacketAlertCheck(p[0], 2)) {
        printf("sid 2 matched but shouldn't have for packet 0: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, bind_ack, bind_ack_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[1]);
    if ((PacketAlertCheck(p[1], 1))) {
        printf("sid 1 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 2))) {
        printf("sid 2 matched but shouldn't have for packet 1: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[2]);
    if (!(PacketAlertCheck(p[2], 1))) {
        printf("sid 1 didn't match but should have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 2))) {
        printf("sid 2 matched but shouldn't have for packet 2: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[3]);
    if ((PacketAlertCheck(p[3], 1))) {
        printf("sid 1 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[3], 2))) {
        printf("sid 2 didn't match but should have for packet 3: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 4);
    return result;
}

/**
 * \test Test the working of detection engien with respect to dce keywords.
 */
int DcePayloadTest03(void)
{
    int result = 0;
    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x6a, 0x28, 0x19, 0x39, 0x0c, 0xb1, 0xd0, 0x11,
        0x9b, 0xa8, 0x00, 0xc0, 0x4f, 0xd9, 0x2e, 0xf5,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_len = sizeof(bind);

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x48, 0x1a, 0x00, 0x00,
        0x0c, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x6c, 0x73, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x82, 0x7c, 0x83, 0x46, 0x0f, 0xf7,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xcf, 0x80, 0x98, 0x6d, 0xfe, 0xb0, 0x90, 0xd1,
        0xcf, 0x86, 0x0f, 0x52, 0x2c, 0x23, 0x66, 0x28,
        0x27, 0x30, 0x48, 0x55, 0x42, 0x6a, 0x48, 0x4b,
        0x68, 0x22, 0x2e, 0x23, 0x64, 0x33, 0x2c, 0x2d,
        0x5c, 0x51, 0x48, 0x55, 0x24, 0x67, 0x6c, 0x4c,
        0x45, 0x71, 0x35, 0x72, 0x5a, 0x48, 0x5e, 0x35,
        0x61, 0x78, 0x35, 0x42, 0x2c, 0x7a, 0x75, 0x61,
        0x5b, 0x4e, 0x76, 0x30, 0x26, 0x2f, 0x2a, 0x34,
        0x48, 0x29, 0x25, 0x6e, 0x5c, 0x3a, 0x6c, 0x3e,
        0x79, 0x4e, 0x2a, 0x21, 0x6f, 0x6f, 0x34, 0x46,
        0x43, 0x26, 0x5b, 0x35, 0x78, 0x27, 0x69, 0x23,
        0x72, 0x21, 0x69, 0x56, 0x6a, 0x7d, 0x4b, 0x5e,
        0x65, 0x37, 0x60, 0x44, 0x7c, 0x5d, 0x5b, 0x72,
        0x7d, 0x73, 0x7b, 0x47, 0x57, 0x21, 0x41, 0x38,
        0x76, 0x38, 0x76, 0x5c, 0x58, 0x32, 0x4a, 0x37,
        0x2f, 0x40, 0x4b, 0x4c, 0x3d, 0x41, 0x33, 0x56,
        0x73, 0x38, 0x61, 0x71, 0x24, 0x49, 0x4c, 0x4a,
        0x44, 0x2e, 0x3a, 0x3f, 0x74, 0x54, 0x4c, 0x65,
        0x54, 0x2d, 0x3b, 0x28, 0x41, 0x45, 0x49, 0x2c,
        0x6e, 0x48, 0x44, 0x43, 0x37, 0x3d, 0x7b, 0x6d,
        0x2b, 0x4b, 0x32, 0x5a, 0x31, 0x61, 0x6e, 0x2b,
        0x27, 0x50, 0x6b, 0x66, 0x76, 0x4e, 0x55, 0x35,
        0x2b, 0x72, 0x2d, 0x5e, 0x42, 0x3e, 0x5a, 0x5d,
        0x36, 0x45, 0x32, 0x3a, 0x58, 0x78, 0x78, 0x3e,
        0x60, 0x6c, 0x5d, 0x63, 0x41, 0x7c, 0x52, 0x21,
        0x75, 0x6a, 0x5a, 0x70, 0x55, 0x45, 0x76, 0x58,
        0x33, 0x40, 0x38, 0x39, 0x21, 0x37, 0x7d, 0x77,
        0x21, 0x70, 0x2b, 0x72, 0x29, 0x6a, 0x31, 0x5f,
        0x38, 0x4a, 0x66, 0x65, 0x62, 0x2c, 0x39, 0x52,
        0x5f, 0x2a, 0x2b, 0x63, 0x4f, 0x76, 0x43, 0x25,
        0x6a, 0x50, 0x37, 0x52, 0x5e, 0x23, 0x3c, 0x42,
        0x28, 0x75, 0x75, 0x42, 0x25, 0x23, 0x28, 0x56,
        0x6c, 0x46, 0x5c, 0x5e, 0x6b, 0x7d, 0x48, 0x24,
        0x77, 0x6c, 0x70, 0x62, 0x2e, 0x28, 0x7d, 0x6b,
        0x69, 0x4a, 0x75, 0x3d, 0x5d, 0x56, 0x21, 0x49,
        0x56, 0x47, 0x64, 0x2b, 0x4c, 0x52, 0x43, 0x60,
        0x77, 0x49, 0x46, 0x46, 0x33, 0x2c, 0x4b, 0x4b,
        0x3d, 0x63, 0x5d, 0x33, 0x78, 0x76, 0x51, 0x56,
        0x77, 0x3c, 0x72, 0x74, 0x52, 0x27, 0x40, 0x6c,
        0x42, 0x79, 0x49, 0x24, 0x62, 0x5e, 0x26, 0x31,
        0x5c, 0x22, 0x2b, 0x4c, 0x64, 0x49, 0x52, 0x45,
        0x47, 0x49, 0x3a, 0x2a, 0x51, 0x71, 0x22, 0x22,
        0x70, 0x24, 0x34, 0x67, 0x4b, 0x6d, 0x58, 0x29,
        0x63, 0x26, 0x7b, 0x6f, 0x38, 0x78, 0x25, 0x62,
        0x4d, 0x3a, 0x7d, 0x40, 0x23, 0x57, 0x67, 0x33,
        0x38, 0x31, 0x4e, 0x54, 0x3c, 0x4b, 0x48, 0x69,
        0x3c, 0x39, 0x31, 0x2b, 0x26, 0x70, 0x44, 0x66,
        0x4a, 0x37, 0x2b, 0x75, 0x36, 0x45, 0x59, 0x34,
        0x3e, 0x3e, 0x29, 0x70, 0x71, 0x5a, 0x55, 0x49,
        0x3e, 0x4b, 0x68, 0x4e, 0x75, 0x70, 0x3c, 0x5c,
        0x50, 0x58, 0x28, 0x75, 0x3c, 0x2a, 0x41, 0x70,
        0x2f, 0x2b, 0x37, 0x26, 0x75, 0x71, 0x55, 0x22,
        0x3a, 0x44, 0x30, 0x48, 0x5d, 0x2f, 0x6c, 0x44,
        0x28, 0x4b, 0x34, 0x45, 0x21, 0x60, 0x44, 0x36,
        0x7b, 0x32, 0x39, 0x5f, 0x6d, 0x3f, 0x68, 0x73,
        0x25, 0x45, 0x56, 0x7c, 0x78, 0x7a, 0x49, 0x6a,
        0x46, 0x3d, 0x2d, 0x33, 0x6c, 0x6f, 0x23, 0x77,
        0x38, 0x33, 0x36, 0x74, 0x7b, 0x57, 0x4b, 0x6d,
        0x27, 0x75, 0x24, 0x6e, 0x43, 0x61, 0x4d, 0x44,
        0x6d, 0x27, 0x48, 0x58, 0x5e, 0x7b, 0x26, 0x6a,
        0x50, 0x7c, 0x51, 0x23, 0x3c, 0x4f, 0x37, 0x4c,
        0x47, 0x3e, 0x45, 0x56, 0x22, 0x33, 0x7c, 0x66,
        0x35, 0x54, 0x7a, 0x6e, 0x5a, 0x24, 0x70, 0x62,
        0x29, 0x3f, 0x69, 0x79, 0x24, 0x43, 0x41, 0x24,
        0x65, 0x25, 0x62, 0x4f, 0x73, 0x3e, 0x2b, 0x36,
        0x46, 0x69, 0x27, 0x55, 0x2a, 0x6e, 0x24, 0x6c,
        0x7d, 0x64, 0x7c, 0x61, 0x26, 0x67, 0x2a, 0x53,
        0x73, 0x60, 0x28, 0x2d, 0x6b, 0x44, 0x54, 0x61,
        0x34, 0x53, 0x22, 0x59, 0x6d, 0x73, 0x56, 0x55,
        0x25, 0x2c, 0x38, 0x4a, 0x3b, 0x4e, 0x78, 0x46,
        0x54, 0x6e, 0x6d, 0x4f, 0x47, 0x4f, 0x4f, 0x5a,
        0x67, 0x77, 0x39, 0x66, 0x28, 0x29, 0x4e, 0x43,
        0x55, 0x6e, 0x60, 0x59, 0x28, 0x3b, 0x65, 0x62,
        0x61, 0x5a, 0x29, 0x6e, 0x79, 0x60, 0x41, 0x53,
        0x2f, 0x5d, 0x44, 0x36, 0x7b, 0x3e, 0x7c, 0x2b,
        0x77, 0x36, 0x70, 0x3f, 0x40, 0x55, 0x48, 0x67,
        0x4b, 0x4d, 0x5d, 0x51, 0x79, 0x76, 0x48, 0x4a,
        0x2d, 0x21, 0x60, 0x40, 0x46, 0x55, 0x7a, 0x60,
        0x22, 0x25, 0x3f, 0x4b, 0x54, 0x6a, 0x6a, 0x3c,
        0x77, 0x22, 0x5b, 0x43, 0x67, 0x58, 0x71, 0x22,
        0x79, 0x4b, 0x32, 0x61, 0x44, 0x4d, 0x6f, 0x42,
        0x33, 0x2d, 0x53, 0x35, 0x3d, 0x6f, 0x57, 0x48,
        0x33, 0x3b, 0x5a, 0x53, 0x3f, 0x4e, 0x3f, 0x6b,
        0x4c, 0x27, 0x26, 0x3b, 0x73, 0x49, 0x22, 0x55,
        0x79, 0x2f, 0x47, 0x2f, 0x55, 0x5a, 0x7a, 0x71,
        0x6c, 0x31, 0x43, 0x40, 0x56, 0x7b, 0x21, 0x7a,
        0x6d, 0x4c, 0x43, 0x5e, 0x38, 0x47, 0x29, 0x38,
        0x62, 0x49, 0x45, 0x78, 0x70, 0x2b, 0x2e, 0x65,
        0x47, 0x71, 0x58, 0x79, 0x39, 0x67, 0x7d, 0x6d,
        0x6a, 0x67, 0x4a, 0x71, 0x27, 0x35, 0x2a, 0x4c,
        0x3e, 0x58, 0x55, 0x30, 0x4d, 0x75, 0x77, 0x48,
        0x5f, 0x4b, 0x59, 0x34, 0x65, 0x68, 0x57, 0x59,
        0x63, 0x23, 0x47, 0x38, 0x47, 0x5e, 0x56, 0x28,
        0x79, 0x58, 0x3e, 0x39, 0x66, 0x77, 0x67, 0x33,
        0x29, 0x61, 0x24, 0x7d, 0x37, 0x44, 0x37, 0x67,
        0x3a, 0x58, 0x76, 0x21, 0x51, 0x59, 0x61, 0x73,
        0x66, 0x75, 0x71, 0x53, 0x4d, 0x24, 0x2d, 0x4b,
        0x29, 0x30, 0x32, 0x26, 0x59, 0x64, 0x27, 0x55,
        0x2c, 0x5a, 0x4c, 0x3c, 0x6c, 0x53, 0x56, 0x4b,
        0x3e, 0x55, 0x2e, 0x44, 0x38, 0x6b, 0x47, 0x76,
        0x2d, 0x2c, 0x3f, 0x4d, 0x22, 0x7b, 0x6d, 0x61,
        0x34, 0x6b, 0x50, 0x73, 0x28, 0x6d, 0x41, 0x71,
        0x21, 0x76, 0x52, 0x2a, 0x6d, 0x53, 0x2a, 0x74,
        0x28, 0x27, 0x62, 0x2a, 0x66, 0x25, 0x6e, 0x5e,
        0x37, 0x4f, 0x27, 0x72, 0x28, 0x47, 0x63, 0x6e,
        0x5a, 0x6a, 0x41, 0x35, 0x3a, 0x42, 0x3f, 0x27,
        0x75, 0x3e, 0x26, 0x3e, 0x6b, 0x55, 0x59, 0x60,
        0x24, 0x70, 0x49, 0x3c, 0x4e, 0x2c, 0x39, 0x7a,
        0x36, 0x6c, 0x27, 0x3e, 0x6a, 0x4a, 0x59, 0x5a,
        0x3e, 0x21, 0x73, 0x4e, 0x59, 0x6e, 0x3d, 0x32,
        0x27, 0x45, 0x49, 0x58, 0x7d, 0x37, 0x39, 0x77,
        0x28, 0x51, 0x79, 0x54, 0x2b, 0x78, 0x46, 0x5a,
        0x21, 0x75, 0x33, 0x21, 0x63, 0x5a, 0x7b, 0x3e,
        0x33, 0x4f, 0x67, 0x75, 0x3a, 0x50, 0x48, 0x60,
        0x26, 0x64, 0x76, 0x5c, 0x42, 0x5c, 0x72, 0x38,
        0x6c, 0x52, 0x21, 0x2b, 0x25, 0x6b, 0x7c, 0x6b,
        0x2d, 0x5e, 0x63, 0x2a, 0x4c, 0x26, 0x5b, 0x4c,
        0x58, 0x52, 0x51, 0x55, 0x31, 0x79, 0x6c, 0x53,
        0x62, 0x3a, 0x36, 0x46, 0x7a, 0x29, 0x27, 0x78,
        0x1a, 0xbf, 0x49, 0x74, 0x68, 0x24, 0x51, 0x44,
        0x5b, 0x3e, 0x34, 0x44, 0x29, 0x5e, 0x4f, 0x2a,
        0xe9, 0x3f, 0xf8, 0xff, 0xff, 0x52, 0x7d, 0x47,
        0x67, 0x40, 0x27, 0x5e, 0x47, 0x46, 0x6d, 0x72,
        0x5d, 0x49, 0x26, 0x45, 0x33, 0x6b, 0x4d, 0x4a,
        0x6f, 0x62, 0x60, 0x45, 0x62, 0x27, 0x27, 0x7d,
        0x6a, 0x41, 0x2c, 0x6c, 0x5b, 0x2a, 0x2b, 0x36,
        0x29, 0x58, 0x7a, 0x4c, 0x6e, 0x2d, 0x74, 0x5c,
        0x38, 0x22, 0x5f, 0x49, 0x63, 0x43, 0x5b, 0x67
    };
    uint32_t request2_len = sizeof(request2);


    TcpSession ssn;
    Packet *p[4];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_iface:3919286a-b10c-11d0-9ba8-00c04fd92ef4; "
        "dce_stub_data; sid:1;)";
    char *sig2 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|2d 5e 63 2a 4c|\"; distance:0; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 4; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }
    p[1]->flowflags |= FLOW_PKT_TOCLIENT;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig2);
    s = s->next;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, bind, bind_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if ((PacketAlertCheck(p[0], 1))) {
        printf("sid 1 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 2))) {
        printf("sid 2 matched but shouldn't have for packet 0: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, bind_ack, bind_ack_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[1]);
    if ((PacketAlertCheck(p[1], 1))) {
        printf("sid 1 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 2))) {
        printf("sid 2 matched but shouldn't have for packet 1: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[2]);
    if ((PacketAlertCheck(p[2], 1))) {
        printf("sid 1 matched but shouldn't have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 2))) {
        printf("sid 2 matched but shouldn't have for packet 2: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[3]);
    if ((PacketAlertCheck(p[3], 1))) {
        printf("sid 1 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[3], 2))) {
        printf("sid 2 didn't match but should have for packet 3: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 4);
    return result;
}

/**
 * \test Test the working of detection engien with respect to dce keywords.
 */
int DcePayloadTest04(void)
{
    int result = 0;
    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x6a, 0x28, 0x19, 0x39, 0x0c, 0xb1, 0xd0, 0x11,
        0x9b, 0xa8, 0x00, 0xc0, 0x4f, 0xd9, 0x2e, 0xf5,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_len = sizeof(bind);

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x48, 0x1a, 0x00, 0x00,
        0x0c, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x6c, 0x73, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x82, 0x7c, 0x83, 0x46, 0x0f, 0xf7,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xcf, 0x80, 0x98, 0x6d, 0xfe, 0xb0, 0x90, 0xd1,
        0xcf, 0x86, 0x0f, 0x52, 0x2c, 0x23, 0x66, 0x28,
        0x27, 0x30, 0x48, 0x55, 0x42, 0x6a, 0x48, 0x4b,
        0x68, 0x22, 0x2e, 0x23, 0x64, 0x33, 0x2c, 0x2d,
        0x5c, 0x51, 0x48, 0x55, 0x24, 0x67, 0x6c, 0x4c,
        0x45, 0x71, 0x35, 0x72, 0x5a, 0x48, 0x5e, 0x35,
        0x61, 0x78, 0x35, 0x42, 0x2c, 0x7a, 0x75, 0x61,
        0x5b, 0x4e, 0x76, 0x30, 0x26, 0x2f, 0x2a, 0x34,
        0x48, 0x29, 0x25, 0x6e, 0x5c, 0x3a, 0x6c, 0x3e,
        0x79, 0x4e, 0x2a, 0x21, 0x6f, 0x6f, 0x34, 0x46,
        0x43, 0x26, 0x5b, 0x35, 0x78, 0x27, 0x69, 0x23,
        0x72, 0x21, 0x69, 0x56, 0x6a, 0x7d, 0x4b, 0x5e,
        0x65, 0x37, 0x60, 0x44, 0x7c, 0x5d, 0x5b, 0x72,
        0x7d, 0x73, 0x7b, 0x47, 0x57, 0x21, 0x41, 0x38,
        0x76, 0x38, 0x76, 0x5c, 0x58, 0x32, 0x4a, 0x37,
        0x2f, 0x40, 0x4b, 0x4c, 0x3d, 0x41, 0x33, 0x56,
        0x73, 0x38, 0x61, 0x71, 0x24, 0x49, 0x4c, 0x4a,
        0x44, 0x2e, 0x3a, 0x3f, 0x74, 0x54, 0x4c, 0x65,
        0x54, 0x2d, 0x3b, 0x28, 0x41, 0x45, 0x49, 0x2c,
        0x6e, 0x48, 0x44, 0x43, 0x37, 0x3d, 0x7b, 0x6d,
        0x2b, 0x4b, 0x32, 0x5a, 0x31, 0x61, 0x6e, 0x2b,
        0x27, 0x50, 0x6b, 0x66, 0x76, 0x4e, 0x55, 0x35,
        0x2b, 0x72, 0x2d, 0x5e, 0x42, 0x3e, 0x5a, 0x5d,
        0x36, 0x45, 0x32, 0x3a, 0x58, 0x78, 0x78, 0x3e,
        0x60, 0x6c, 0x5d, 0x63, 0x41, 0x7c, 0x52, 0x21,
        0x75, 0x6a, 0x5a, 0x70, 0x55, 0x45, 0x76, 0x58,
        0x33, 0x40, 0x38, 0x39, 0x21, 0x37, 0x7d, 0x77,
        0x21, 0x70, 0x2b, 0x72, 0x29, 0x6a, 0x31, 0x5f,
        0x38, 0x4a, 0x66, 0x65, 0x62, 0x2c, 0x39, 0x52,
        0x5f, 0x2a, 0x2b, 0x63, 0x4f, 0x76, 0x43, 0x25,
        0x6a, 0x50, 0x37, 0x52, 0x5e, 0x23, 0x3c, 0x42,
        0x28, 0x75, 0x75, 0x42, 0x25, 0x23, 0x28, 0x56,
        0x6c, 0x46, 0x5c, 0x5e, 0x6b, 0x7d, 0x48, 0x24,
        0x77, 0x6c, 0x70, 0x62, 0x2e, 0x28, 0x7d, 0x6b,
        0x69, 0x4a, 0x75, 0x3d, 0x5d, 0x56, 0x21, 0x49,
        0x56, 0x47, 0x64, 0x2b, 0x4c, 0x52, 0x43, 0x60,
        0x77, 0x49, 0x46, 0x46, 0x33, 0x2c, 0x4b, 0x4b,
        0x3d, 0x63, 0x5d, 0x33, 0x78, 0x76, 0x51, 0x56,
        0x77, 0x3c, 0x72, 0x74, 0x52, 0x27, 0x40, 0x6c,
        0x42, 0x79, 0x49, 0x24, 0x62, 0x5e, 0x26, 0x31,
        0x5c, 0x22, 0x2b, 0x4c, 0x64, 0x49, 0x52, 0x45,
        0x47, 0x49, 0x3a, 0x2a, 0x51, 0x71, 0x22, 0x22,
        0x70, 0x24, 0x34, 0x67, 0x4b, 0x6d, 0x58, 0x29,
        0x63, 0x26, 0x7b, 0x6f, 0x38, 0x78, 0x25, 0x62,
        0x4d, 0x3a, 0x7d, 0x40, 0x23, 0x57, 0x67, 0x33,
        0x38, 0x31, 0x4e, 0x54, 0x3c, 0x4b, 0x48, 0x69,
        0x3c, 0x39, 0x31, 0x2b, 0x26, 0x70, 0x44, 0x66,
        0x4a, 0x37, 0x2b, 0x75, 0x36, 0x45, 0x59, 0x34,
        0x3e, 0x3e, 0x29, 0x70, 0x71, 0x5a, 0x55, 0x49,
        0x3e, 0x4b, 0x68, 0x4e, 0x75, 0x70, 0x3c, 0x5c,
        0x50, 0x58, 0x28, 0x75, 0x3c, 0x2a, 0x41, 0x70,
        0x2f, 0x2b, 0x37, 0x26, 0x75, 0x71, 0x55, 0x22,
        0x3a, 0x44, 0x30, 0x48, 0x5d, 0x2f, 0x6c, 0x44,
        0x28, 0x4b, 0x34, 0x45, 0x21, 0x60, 0x44, 0x36,
        0x7b, 0x32, 0x39, 0x5f, 0x6d, 0x3f, 0x68, 0x73,
        0x25, 0x45, 0x56, 0x7c, 0x78, 0x7a, 0x49, 0x6a,
        0x46, 0x3d, 0x2d, 0x33, 0x6c, 0x6f, 0x23, 0x77,
        0x38, 0x33, 0x36, 0x74, 0x7b, 0x57, 0x4b, 0x6d,
        0x27, 0x75, 0x24, 0x6e, 0x43, 0x61, 0x4d, 0x44,
        0x6d, 0x27, 0x48, 0x58, 0x5e, 0x7b, 0x26, 0x6a,
        0x50, 0x7c, 0x51, 0x23, 0x3c, 0x4f, 0x37, 0x4c,
        0x47, 0x3e, 0x45, 0x56, 0x22, 0x33, 0x7c, 0x66,
        0x35, 0x54, 0x7a, 0x6e, 0x5a, 0x24, 0x70, 0x62,
        0x29, 0x3f, 0x69, 0x79, 0x24, 0x43, 0x41, 0x24,
        0x65, 0x25, 0x62, 0x4f, 0x73, 0x3e, 0x2b, 0x36,
        0x46, 0x69, 0x27, 0x55, 0x2a, 0x6e, 0x24, 0x6c,
        0x7d, 0x64, 0x7c, 0x61, 0x26, 0x67, 0x2a, 0x53,
        0x73, 0x60, 0x28, 0x2d, 0x6b, 0x44, 0x54, 0x61,
        0x34, 0x53, 0x22, 0x59, 0x6d, 0x73, 0x56, 0x55,
        0x25, 0x2c, 0x38, 0x4a, 0x3b, 0x4e, 0x78, 0x46,
        0x54, 0x6e, 0x6d, 0x4f, 0x47, 0x4f, 0x4f, 0x5a,
        0x67, 0x77, 0x39, 0x66, 0x28, 0x29, 0x4e, 0x43,
        0x55, 0x6e, 0x60, 0x59, 0x28, 0x3b, 0x65, 0x62,
        0x61, 0x5a, 0x29, 0x6e, 0x79, 0x60, 0x41, 0x53,
        0x2f, 0x5d, 0x44, 0x36, 0x7b, 0x3e, 0x7c, 0x2b,
        0x77, 0x36, 0x70, 0x3f, 0x40, 0x55, 0x48, 0x67,
        0x4b, 0x4d, 0x5d, 0x51, 0x79, 0x76, 0x48, 0x4a,
        0x2d, 0x21, 0x60, 0x40, 0x46, 0x55, 0x7a, 0x60,
        0x22, 0x25, 0x3f, 0x4b, 0x54, 0x6a, 0x6a, 0x3c,
        0x77, 0x22, 0x5b, 0x43, 0x67, 0x58, 0x71, 0x22,
        0x79, 0x4b, 0x32, 0x61, 0x44, 0x4d, 0x6f, 0x42,
        0x33, 0x2d, 0x53, 0x35, 0x3d, 0x6f, 0x57, 0x48,
        0x33, 0x3b, 0x5a, 0x53, 0x3f, 0x4e, 0x3f, 0x6b,
        0x4c, 0x27, 0x26, 0x3b, 0x73, 0x49, 0x22, 0x55,
        0x79, 0x2f, 0x47, 0x2f, 0x55, 0x5a, 0x7a, 0x71,
        0x6c, 0x31, 0x43, 0x40, 0x56, 0x7b, 0x21, 0x7a,
        0x6d, 0x4c, 0x43, 0x5e, 0x38, 0x47, 0x29, 0x38,
        0x62, 0x49, 0x45, 0x78, 0x70, 0x2b, 0x2e, 0x65,
        0x47, 0x71, 0x58, 0x79, 0x39, 0x67, 0x7d, 0x6d,
        0x6a, 0x67, 0x4a, 0x71, 0x27, 0x35, 0x2a, 0x4c,
        0x3e, 0x58, 0x55, 0x30, 0x4d, 0x75, 0x77, 0x48,
        0x5f, 0x4b, 0x59, 0x34, 0x65, 0x68, 0x57, 0x59,
        0x63, 0x23, 0x47, 0x38, 0x47, 0x5e, 0x56, 0x28,
        0x79, 0x58, 0x3e, 0x39, 0x66, 0x77, 0x67, 0x33,
        0x29, 0x61, 0x24, 0x7d, 0x37, 0x44, 0x37, 0x67,
        0x3a, 0x58, 0x76, 0x21, 0x51, 0x59, 0x61, 0x73,
        0x66, 0x75, 0x71, 0x53, 0x4d, 0x24, 0x2d, 0x4b,
        0x29, 0x30, 0x32, 0x26, 0x59, 0x64, 0x27, 0x55,
        0x2c, 0x5a, 0x4c, 0x3c, 0x6c, 0x53, 0x56, 0x4b,
        0x3e, 0x55, 0x2e, 0x44, 0x38, 0x6b, 0x47, 0x76,
        0x2d, 0x2c, 0x3f, 0x4d, 0x22, 0x7b, 0x6d, 0x61,
        0x34, 0x6b, 0x50, 0x73, 0x28, 0x6d, 0x41, 0x71,
        0x21, 0x76, 0x52, 0x2a, 0x6d, 0x53, 0x2a, 0x74,
        0x28, 0x27, 0x62, 0x2a, 0x66, 0x25, 0x6e, 0x5e,
        0x37, 0x4f, 0x27, 0x72, 0x28, 0x47, 0x63, 0x6e,
        0x5a, 0x6a, 0x41, 0x35, 0x3a, 0x42, 0x3f, 0x27,
        0x75, 0x3e, 0x26, 0x3e, 0x6b, 0x55, 0x59, 0x60,
        0x24, 0x70, 0x49, 0x3c, 0x4e, 0x2c, 0x39, 0x7a,
        0x36, 0x6c, 0x27, 0x3e, 0x6a, 0x4a, 0x59, 0x5a,
        0x3e, 0x21, 0x73, 0x4e, 0x59, 0x6e, 0x3d, 0x32,
        0x27, 0x45, 0x49, 0x58, 0x7d, 0x37, 0x39, 0x77,
        0x28, 0x51, 0x79, 0x54, 0x2b, 0x78, 0x46, 0x5a,
        0x21, 0x75, 0x33, 0x21, 0x63, 0x5a, 0x7b, 0x3e,
        0x33, 0x4f, 0x67, 0x75, 0x3a, 0x50, 0x48, 0x60,
        0x26, 0x64, 0x76, 0x5c, 0x42, 0x5c, 0x72, 0x38,
        0x6c, 0x52, 0x21, 0x2b, 0x25, 0x6b, 0x7c, 0x6b,
        0x2d, 0x5e, 0x63, 0x2a, 0x4c, 0x26, 0x5b, 0x4c,
        0x58, 0x52, 0x51, 0x55, 0x31, 0x79, 0x6c, 0x53,
        0x62, 0x3a, 0x36, 0x46, 0x7a, 0x29, 0x27, 0x78,
        0x1a, 0xbf, 0x49, 0x74, 0x68, 0x24, 0x51, 0x44,
        0x5b, 0x3e, 0x34, 0x44, 0x29, 0x5e, 0x4f, 0x2a,
        0xe9, 0x3f, 0xf8, 0xff, 0xff, 0x52, 0x7d, 0x47,
        0x67, 0x40, 0x27, 0x5e, 0x47, 0x46, 0x6d, 0x72,
        0x5d, 0x49, 0x26, 0x45, 0x33, 0x6b, 0x4d, 0x4a,
        0x6f, 0x62, 0x60, 0x45, 0x62, 0x27, 0x27, 0x7d,
        0x6a, 0x41, 0x2c, 0x6c, 0x5b, 0x2a, 0x2b, 0x36,
        0x29, 0x58, 0x7a, 0x4c, 0x6e, 0x2d, 0x74, 0x5c,
        0x38, 0x22, 0x5f, 0x49, 0x63, 0x43, 0x5b, 0x67
    };
    uint32_t request2_len = sizeof(request2);


    TcpSession ssn;
    Packet *p[4];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_iface:3919286a-b10c-11d0-9ba8-00c04fd92ef5; "
        "dce_stub_data; content:\"|91 27 27 40|\"; distance:0; sid:1;)";
    char *sig2 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|2d 5e 63 2a 4c|\"; distance:0; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 4; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }
    p[1]->flowflags |= FLOW_PKT_TOCLIENT;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig2);
    s = s->next;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, bind, bind_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if ((PacketAlertCheck(p[0], 1))) {
        printf("sid 1 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 2))) {
        printf("sid 2 matched but shouldn't have for packet 0: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, bind_ack, bind_ack_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[1]);
    if ((PacketAlertCheck(p[1], 1))) {
        printf("sid 1 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 2))) {
        printf("sid 2 matched but shouldn't have for packet 1: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[2]);
    if (!(PacketAlertCheck(p[2], 1))) {
        printf("sid 1 didn't match but should have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 2))) {
        printf("sid 2 matched but shouldn't have for packet 2: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[3]);
    if ((PacketAlertCheck(p[3], 1))) {
        printf("sid 1 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[3], 2))) {
        printf("sid 2 didn't match but should have for packet 3: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 4);
    return result;
}

/**
 * \test Test the working of detection engien with respect to dce keywords.
 */
int DcePayloadTest05(void)
{
    int result = 0;
    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x6a, 0x28, 0x19, 0x39, 0x0c, 0xb1, 0xd0, 0x11,
        0x9b, 0xa8, 0x00, 0xc0, 0x4f, 0xd9, 0x2e, 0xf5,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_len = sizeof(bind);

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x48, 0x1a, 0x00, 0x00,
        0x0c, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x6c, 0x73, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x82, 0x7c, 0x83, 0x46, 0x0f, 0xf7,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xcf, 0x80, 0x98, 0x6d, 0xfe, 0xb0, 0x90, 0xd1,
        0xcf, 0x86, 0x0f, 0x52, 0x2c, 0x23, 0x66, 0x28,
        0x27, 0x30, 0x48, 0x55, 0x42, 0x6a, 0x48, 0x4b,
        0x68, 0x22, 0x2e, 0x23, 0x64, 0x33, 0x2c, 0x2d,
        0x5c, 0x51, 0x48, 0x55, 0x24, 0x67, 0x6c, 0x4c,
        0x45, 0x71, 0x35, 0x72, 0x5a, 0x48, 0x5e, 0x35,
        0x61, 0x78, 0x35, 0x42, 0x2c, 0x7a, 0x75, 0x61,
        0x5b, 0x4e, 0x76, 0x30, 0x26, 0x2f, 0x2a, 0x34,
        0x48, 0x29, 0x25, 0x6e, 0x5c, 0x3a, 0x6c, 0x3e,
        0x79, 0x4e, 0x2a, 0x21, 0x6f, 0x6f, 0x34, 0x46,
        0x43, 0x26, 0x5b, 0x35, 0x78, 0x27, 0x69, 0x23,
        0x72, 0x21, 0x69, 0x56, 0x6a, 0x7d, 0x4b, 0x5e,
        0x65, 0x37, 0x60, 0x44, 0x7c, 0x5d, 0x5b, 0x72,
        0x7d, 0x73, 0x7b, 0x47, 0x57, 0x21, 0x41, 0x38,
        0x76, 0x38, 0x76, 0x5c, 0x58, 0x32, 0x4a, 0x37,
        0x2f, 0x40, 0x4b, 0x4c, 0x3d, 0x41, 0x33, 0x56,
        0x73, 0x38, 0x61, 0x71, 0x24, 0x49, 0x4c, 0x4a,
        0x44, 0x2e, 0x3a, 0x3f, 0x74, 0x54, 0x4c, 0x65,
        0x54, 0x2d, 0x3b, 0x28, 0x41, 0x45, 0x49, 0x2c,
        0x6e, 0x48, 0x44, 0x43, 0x37, 0x3d, 0x7b, 0x6d,
        0x2b, 0x4b, 0x32, 0x5a, 0x31, 0x61, 0x6e, 0x2b,
        0x27, 0x50, 0x6b, 0x66, 0x76, 0x4e, 0x55, 0x35,
        0x2b, 0x72, 0x2d, 0x5e, 0x42, 0x3e, 0x5a, 0x5d,
        0x36, 0x45, 0x32, 0x3a, 0x58, 0x78, 0x78, 0x3e,
        0x60, 0x6c, 0x5d, 0x63, 0x41, 0x7c, 0x52, 0x21,
        0x75, 0x6a, 0x5a, 0x70, 0x55, 0x45, 0x76, 0x58,
        0x33, 0x40, 0x38, 0x39, 0x21, 0x37, 0x7d, 0x77,
        0x21, 0x70, 0x2b, 0x72, 0x29, 0x6a, 0x31, 0x5f,
        0x38, 0x4a, 0x66, 0x65, 0x62, 0x2c, 0x39, 0x52,
        0x5f, 0x2a, 0x2b, 0x63, 0x4f, 0x76, 0x43, 0x25,
        0x6a, 0x50, 0x37, 0x52, 0x5e, 0x23, 0x3c, 0x42,
        0x28, 0x75, 0x75, 0x42, 0x25, 0x23, 0x28, 0x56,
        0x6c, 0x46, 0x5c, 0x5e, 0x6b, 0x7d, 0x48, 0x24,
        0x77, 0x6c, 0x70, 0x62, 0x2e, 0x28, 0x7d, 0x6b,
        0x69, 0x4a, 0x75, 0x3d, 0x5d, 0x56, 0x21, 0x49,
        0x56, 0x47, 0x64, 0x2b, 0x4c, 0x52, 0x43, 0x60,
        0x77, 0x49, 0x46, 0x46, 0x33, 0x2c, 0x4b, 0x4b,
        0x3d, 0x63, 0x5d, 0x33, 0x78, 0x76, 0x51, 0x56,
        0x77, 0x3c, 0x72, 0x74, 0x52, 0x27, 0x40, 0x6c,
        0x42, 0x79, 0x49, 0x24, 0x62, 0x5e, 0x26, 0x31,
        0x5c, 0x22, 0x2b, 0x4c, 0x64, 0x49, 0x52, 0x45,
        0x47, 0x49, 0x3a, 0x2a, 0x51, 0x71, 0x22, 0x22,
        0x70, 0x24, 0x34, 0x67, 0x4b, 0x6d, 0x58, 0x29,
        0x63, 0x26, 0x7b, 0x6f, 0x38, 0x78, 0x25, 0x62,
        0x4d, 0x3a, 0x7d, 0x40, 0x23, 0x57, 0x67, 0x33,
        0x38, 0x31, 0x4e, 0x54, 0x3c, 0x4b, 0x48, 0x69,
        0x3c, 0x39, 0x31, 0x2b, 0x26, 0x70, 0x44, 0x66,
        0x4a, 0x37, 0x2b, 0x75, 0x36, 0x45, 0x59, 0x34,
        0x3e, 0x3e, 0x29, 0x70, 0x71, 0x5a, 0x55, 0x49,
        0x3e, 0x4b, 0x68, 0x4e, 0x75, 0x70, 0x3c, 0x5c,
        0x50, 0x58, 0x28, 0x75, 0x3c, 0x2a, 0x41, 0x70,
        0x2f, 0x2b, 0x37, 0x26, 0x75, 0x71, 0x55, 0x22,
        0x3a, 0x44, 0x30, 0x48, 0x5d, 0x2f, 0x6c, 0x44,
        0x28, 0x4b, 0x34, 0x45, 0x21, 0x60, 0x44, 0x36,
        0x7b, 0x32, 0x39, 0x5f, 0x6d, 0x3f, 0x68, 0x73,
        0x25, 0x45, 0x56, 0x7c, 0x78, 0x7a, 0x49, 0x6a,
        0x46, 0x3d, 0x2d, 0x33, 0x6c, 0x6f, 0x23, 0x77,
        0x38, 0x33, 0x36, 0x74, 0x7b, 0x57, 0x4b, 0x6d,
        0x27, 0x75, 0x24, 0x6e, 0x43, 0x61, 0x4d, 0x44,
        0x6d, 0x27, 0x48, 0x58, 0x5e, 0x7b, 0x26, 0x6a,
        0x50, 0x7c, 0x51, 0x23, 0x3c, 0x4f, 0x37, 0x4c,
        0x47, 0x3e, 0x45, 0x56, 0x22, 0x33, 0x7c, 0x66,
        0x35, 0x54, 0x7a, 0x6e, 0x5a, 0x24, 0x70, 0x62,
        0x29, 0x3f, 0x69, 0x79, 0x24, 0x43, 0x41, 0x24,
        0x65, 0x25, 0x62, 0x4f, 0x73, 0x3e, 0x2b, 0x36,
        0x46, 0x69, 0x27, 0x55, 0x2a, 0x6e, 0x24, 0x6c,
        0x7d, 0x64, 0x7c, 0x61, 0x26, 0x67, 0x2a, 0x53,
        0x73, 0x60, 0x28, 0x2d, 0x6b, 0x44, 0x54, 0x61,
        0x34, 0x53, 0x22, 0x59, 0x6d, 0x73, 0x56, 0x55,
        0x25, 0x2c, 0x38, 0x4a, 0x3b, 0x4e, 0x78, 0x46,
        0x54, 0x6e, 0x6d, 0x4f, 0x47, 0x4f, 0x4f, 0x5a,
        0x67, 0x77, 0x39, 0x66, 0x28, 0x29, 0x4e, 0x43,
        0x55, 0x6e, 0x60, 0x59, 0x28, 0x3b, 0x65, 0x62,
        0x61, 0x5a, 0x29, 0x6e, 0x79, 0x60, 0x41, 0x53,
        0x2f, 0x5d, 0x44, 0x36, 0x7b, 0x3e, 0x7c, 0x2b,
        0x77, 0x36, 0x70, 0x3f, 0x40, 0x55, 0x48, 0x67,
        0x4b, 0x4d, 0x5d, 0x51, 0x79, 0x76, 0x48, 0x4a,
        0x2d, 0x21, 0x60, 0x40, 0x46, 0x55, 0x7a, 0x60,
        0x22, 0x25, 0x3f, 0x4b, 0x54, 0x6a, 0x6a, 0x3c,
        0x77, 0x22, 0x5b, 0x43, 0x67, 0x58, 0x71, 0x22,
        0x79, 0x4b, 0x32, 0x61, 0x44, 0x4d, 0x6f, 0x42,
        0x33, 0x2d, 0x53, 0x35, 0x3d, 0x6f, 0x57, 0x48,
        0x33, 0x3b, 0x5a, 0x53, 0x3f, 0x4e, 0x3f, 0x6b,
        0x4c, 0x27, 0x26, 0x3b, 0x73, 0x49, 0x22, 0x55,
        0x79, 0x2f, 0x47, 0x2f, 0x55, 0x5a, 0x7a, 0x71,
        0x6c, 0x31, 0x43, 0x40, 0x56, 0x7b, 0x21, 0x7a,
        0x6d, 0x4c, 0x43, 0x5e, 0x38, 0x47, 0x29, 0x38,
        0x62, 0x49, 0x45, 0x78, 0x70, 0x2b, 0x2e, 0x65,
        0x47, 0x71, 0x58, 0x79, 0x39, 0x67, 0x7d, 0x6d,
        0x6a, 0x67, 0x4a, 0x71, 0x27, 0x35, 0x2a, 0x4c,
        0x3e, 0x58, 0x55, 0x30, 0x4d, 0x75, 0x77, 0x48,
        0x5f, 0x4b, 0x59, 0x34, 0x65, 0x68, 0x57, 0x59,
        0x63, 0x23, 0x47, 0x38, 0x47, 0x5e, 0x56, 0x28,
        0x79, 0x58, 0x3e, 0x39, 0x66, 0x77, 0x67, 0x33,
        0x29, 0x61, 0x24, 0x7d, 0x37, 0x44, 0x37, 0x67,
        0x3a, 0x58, 0x76, 0x21, 0x51, 0x59, 0x61, 0x73,
        0x66, 0x75, 0x71, 0x53, 0x4d, 0x24, 0x2d, 0x4b,
        0x29, 0x30, 0x32, 0x26, 0x59, 0x64, 0x27, 0x55,
        0x2c, 0x5a, 0x4c, 0x3c, 0x6c, 0x53, 0x56, 0x4b,
        0x3e, 0x55, 0x2e, 0x44, 0x38, 0x6b, 0x47, 0x76,
        0x2d, 0x2c, 0x3f, 0x4d, 0x22, 0x7b, 0x6d, 0x61,
        0x34, 0x6b, 0x50, 0x73, 0x28, 0x6d, 0x41, 0x71,
        0x21, 0x76, 0x52, 0x2a, 0x6d, 0x53, 0x2a, 0x74,
        0x28, 0x27, 0x62, 0x2a, 0x66, 0x25, 0x6e, 0x5e,
        0x37, 0x4f, 0x27, 0x72, 0x28, 0x47, 0x63, 0x6e,
        0x5a, 0x6a, 0x41, 0x35, 0x3a, 0x42, 0x3f, 0x27,
        0x75, 0x3e, 0x26, 0x3e, 0x6b, 0x55, 0x59, 0x60,
        0x24, 0x70, 0x49, 0x3c, 0x4e, 0x2c, 0x39, 0x7a,
        0x36, 0x6c, 0x27, 0x3e, 0x6a, 0x4a, 0x59, 0x5a,
        0x3e, 0x21, 0x73, 0x4e, 0x59, 0x6e, 0x3d, 0x32,
        0x27, 0x45, 0x49, 0x58, 0x7d, 0x37, 0x39, 0x77,
        0x28, 0x51, 0x79, 0x54, 0x2b, 0x78, 0x46, 0x5a,
        0x21, 0x75, 0x33, 0x21, 0x63, 0x5a, 0x7b, 0x3e,
        0x33, 0x4f, 0x67, 0x75, 0x3a, 0x50, 0x48, 0x60,
        0x26, 0x64, 0x76, 0x5c, 0x42, 0x5c, 0x72, 0x38,
        0x6c, 0x52, 0x21, 0x2b, 0x25, 0x6b, 0x7c, 0x6b,
        0x2d, 0x5e, 0x63, 0x2a, 0x4c, 0x26, 0x5b, 0x4c,
        0x58, 0x52, 0x51, 0x55, 0x31, 0x79, 0x6c, 0x53,
        0x62, 0x3a, 0x36, 0x46, 0x7a, 0x29, 0x27, 0x78,
        0x1a, 0xbf, 0x49, 0x74, 0x68, 0x24, 0x51, 0x44,
        0x5b, 0x3e, 0x34, 0x44, 0x29, 0x5e, 0x4f, 0x2a,
        0xe9, 0x3f, 0xf8, 0xff, 0xff, 0x52, 0x7d, 0x47,
        0x67, 0x40, 0x27, 0x5e, 0x47, 0x46, 0x6d, 0x72,
        0x5d, 0x49, 0x26, 0x45, 0x33, 0x6b, 0x4d, 0x4a,
        0x6f, 0x62, 0x60, 0x45, 0x62, 0x27, 0x27, 0x7d,
        0x6a, 0x41, 0x2c, 0x6c, 0x5b, 0x2a, 0x2b, 0x36,
        0x29, 0x58, 0x7a, 0x4c, 0x6e, 0x2d, 0x74, 0x5c,
        0x38, 0x22, 0x5f, 0x49, 0x63, 0x43, 0x5b, 0x67
    };
    uint32_t request2_len = sizeof(request2);

    TcpSession ssn;
    Packet *p[4];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_iface:3919286a-b10c-11d0-9ba8-00c04fd92ef4; "
        "dce_stub_data; content:\"|91 27 27 40|\"; distance:0; sid:1;)";
    char *sig2 = "alert tcp any any -> any any (dce_stub_data; "
        "dce_stub_data; content:\"|2d 5e 63 2a 4c|\"; distance:0; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 4; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }
    p[1]->flowflags |= FLOW_PKT_TOCLIENT;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig2);
    s = s->next;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, bind, bind_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if ((PacketAlertCheck(p[0], 1))) {
        printf("sid 1 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 2))) {
        printf("sid 2 matched but shouldn't have for packet 0: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, bind_ack, bind_ack_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[1]);
    if ((PacketAlertCheck(p[1], 1))) {
        printf("sid 1 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 2))) {
        printf("sid 2 matched but shouldn't have for packet 1: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[2]);
    if ((PacketAlertCheck(p[2], 1))) {
        printf("sid 1 matched but shouldn't have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 2))) {
        printf("sid 2 matched but shouldn't have for packet 2: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[3]);
    if ((PacketAlertCheck(p[3], 1))) {
        printf("sid 2 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[3], 2))) {
        printf("sid 2 didn't match but should have for packet 3: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 4);
    return result;
}

/**
 * \test Test the working of detection engien with respect to dce keywords.
 */
int DcePayloadTest06(void)
{
    int result = 0;
    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x6a, 0x28, 0x19, 0x39, 0x0c, 0xb1, 0xd0, 0x11,
        0x9b, 0xa8, 0x00, 0xc0, 0x4f, 0xd9, 0x2e, 0xf5,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_len = sizeof(bind);

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x48, 0x1a, 0x00, 0x00,
        0x0c, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x6c, 0x73, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x82, 0x7c, 0x83, 0x46, 0x0f, 0xf7,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xcf, 0x80, 0x98, 0x6d, 0xfe, 0xb0, 0x90, 0xd1,
        0xcf, 0x86, 0x0f, 0x52, 0x2c, 0x23, 0x66, 0x28,
        0x27, 0x30, 0x48, 0x55, 0x42, 0x6a, 0x48, 0x4b,
        0x68, 0x22, 0x2e, 0x23, 0x64, 0x33, 0x2c, 0x2d,
        0x5c, 0x51, 0x48, 0x55, 0x24, 0x67, 0x6c, 0x4c,
        0x45, 0x71, 0x35, 0x72, 0x5a, 0x48, 0x5e, 0x35,
        0x61, 0x78, 0x35, 0x42, 0x2c, 0x7a, 0x75, 0x61,
        0x5b, 0x4e, 0x76, 0x30, 0x26, 0x2f, 0x2a, 0x34,
        0x48, 0x29, 0x25, 0x6e, 0x5c, 0x3a, 0x6c, 0x3e,
        0x79, 0x4e, 0x2a, 0x21, 0x6f, 0x6f, 0x34, 0x46,
        0x43, 0x26, 0x5b, 0x35, 0x78, 0x27, 0x69, 0x23,
        0x72, 0x21, 0x69, 0x56, 0x6a, 0x7d, 0x4b, 0x5e,
        0x65, 0x37, 0x60, 0x44, 0x7c, 0x5d, 0x5b, 0x72,
        0x7d, 0x73, 0x7b, 0x47, 0x57, 0x21, 0x41, 0x38,
        0x76, 0x38, 0x76, 0x5c, 0x58, 0x32, 0x4a, 0x37,
        0x2f, 0x40, 0x4b, 0x4c, 0x3d, 0x41, 0x33, 0x56,
        0x73, 0x38, 0x61, 0x71, 0x24, 0x49, 0x4c, 0x4a,
        0x44, 0x2e, 0x3a, 0x3f, 0x74, 0x54, 0x4c, 0x65,
        0x54, 0x2d, 0x3b, 0x28, 0x41, 0x45, 0x49, 0x2c,
        0x6e, 0x48, 0x44, 0x43, 0x37, 0x3d, 0x7b, 0x6d,
        0x2b, 0x4b, 0x32, 0x5a, 0x31, 0x61, 0x6e, 0x2b,
        0x27, 0x50, 0x6b, 0x66, 0x76, 0x4e, 0x55, 0x35,
        0x2b, 0x72, 0x2d, 0x5e, 0x42, 0x3e, 0x5a, 0x5d,
        0x36, 0x45, 0x32, 0x3a, 0x58, 0x78, 0x78, 0x3e,
        0x60, 0x6c, 0x5d, 0x63, 0x41, 0x7c, 0x52, 0x21,
        0x75, 0x6a, 0x5a, 0x70, 0x55, 0x45, 0x76, 0x58,
        0x33, 0x40, 0x38, 0x39, 0x21, 0x37, 0x7d, 0x77,
        0x21, 0x70, 0x2b, 0x72, 0x29, 0x6a, 0x31, 0x5f,
        0x38, 0x4a, 0x66, 0x65, 0x62, 0x2c, 0x39, 0x52,
        0x5f, 0x2a, 0x2b, 0x63, 0x4f, 0x76, 0x43, 0x25,
        0x6a, 0x50, 0x37, 0x52, 0x5e, 0x23, 0x3c, 0x42,
        0x28, 0x75, 0x75, 0x42, 0x25, 0x23, 0x28, 0x56,
        0x6c, 0x46, 0x5c, 0x5e, 0x6b, 0x7d, 0x48, 0x24,
        0x77, 0x6c, 0x70, 0x62, 0x2e, 0x28, 0x7d, 0x6b,
        0x69, 0x4a, 0x75, 0x3d, 0x5d, 0x56, 0x21, 0x49,
        0x56, 0x47, 0x64, 0x2b, 0x4c, 0x52, 0x43, 0x60,
        0x77, 0x49, 0x46, 0x46, 0x33, 0x2c, 0x4b, 0x4b,
        0x3d, 0x63, 0x5d, 0x33, 0x78, 0x76, 0x51, 0x56,
        0x77, 0x3c, 0x72, 0x74, 0x52, 0x27, 0x40, 0x6c,
        0x42, 0x79, 0x49, 0x24, 0x62, 0x5e, 0x26, 0x31,
        0x5c, 0x22, 0x2b, 0x4c, 0x64, 0x49, 0x52, 0x45,
        0x47, 0x49, 0x3a, 0x2a, 0x51, 0x71, 0x22, 0x22,
        0x70, 0x24, 0x34, 0x67, 0x4b, 0x6d, 0x58, 0x29,
        0x63, 0x26, 0x7b, 0x6f, 0x38, 0x78, 0x25, 0x62,
        0x4d, 0x3a, 0x7d, 0x40, 0x23, 0x57, 0x67, 0x33,
        0x38, 0x31, 0x4e, 0x54, 0x3c, 0x4b, 0x48, 0x69,
        0x3c, 0x39, 0x31, 0x2b, 0x26, 0x70, 0x44, 0x66,
        0x4a, 0x37, 0x2b, 0x75, 0x36, 0x45, 0x59, 0x34,
        0x3e, 0x3e, 0x29, 0x70, 0x71, 0x5a, 0x55, 0x49,
        0x3e, 0x4b, 0x68, 0x4e, 0x75, 0x70, 0x3c, 0x5c,
        0x50, 0x58, 0x28, 0x75, 0x3c, 0x2a, 0x41, 0x70,
        0x2f, 0x2b, 0x37, 0x26, 0x75, 0x71, 0x55, 0x22,
        0x3a, 0x44, 0x30, 0x48, 0x5d, 0x2f, 0x6c, 0x44,
        0x28, 0x4b, 0x34, 0x45, 0x21, 0x60, 0x44, 0x36,
        0x7b, 0x32, 0x39, 0x5f, 0x6d, 0x3f, 0x68, 0x73,
        0x25, 0x45, 0x56, 0x7c, 0x78, 0x7a, 0x49, 0x6a,
        0x46, 0x3d, 0x2d, 0x33, 0x6c, 0x6f, 0x23, 0x77,
        0x38, 0x33, 0x36, 0x74, 0x7b, 0x57, 0x4b, 0x6d,
        0x27, 0x75, 0x24, 0x6e, 0x43, 0x61, 0x4d, 0x44,
        0x6d, 0x27, 0x48, 0x58, 0x5e, 0x7b, 0x26, 0x6a,
        0x50, 0x7c, 0x51, 0x23, 0x3c, 0x4f, 0x37, 0x4c,
        0x47, 0x3e, 0x45, 0x56, 0x22, 0x33, 0x7c, 0x66,
        0x35, 0x54, 0x7a, 0x6e, 0x5a, 0x24, 0x70, 0x62,
        0x29, 0x3f, 0x69, 0x79, 0x24, 0x43, 0x41, 0x24,
        0x65, 0x25, 0x62, 0x4f, 0x73, 0x3e, 0x2b, 0x36,
        0x46, 0x69, 0x27, 0x55, 0x2a, 0x6e, 0x24, 0x6c,
        0x7d, 0x64, 0x7c, 0x61, 0x26, 0x67, 0x2a, 0x53,
        0x73, 0x60, 0x28, 0x2d, 0x6b, 0x44, 0x54, 0x61,
        0x34, 0x53, 0x22, 0x59, 0x6d, 0x73, 0x56, 0x55,
        0x25, 0x2c, 0x38, 0x4a, 0x3b, 0x4e, 0x78, 0x46,
        0x54, 0x6e, 0x6d, 0x4f, 0x47, 0x4f, 0x4f, 0x5a,
        0x67, 0x77, 0x39, 0x66, 0x28, 0x29, 0x4e, 0x43,
        0x55, 0x6e, 0x60, 0x59, 0x28, 0x3b, 0x65, 0x62,
        0x61, 0x5a, 0x29, 0x6e, 0x79, 0x60, 0x41, 0x53,
        0x2f, 0x5d, 0x44, 0x36, 0x7b, 0x3e, 0x7c, 0x2b,
        0x77, 0x36, 0x70, 0x3f, 0x40, 0x55, 0x48, 0x67,
        0x4b, 0x4d, 0x5d, 0x51, 0x79, 0x76, 0x48, 0x4a,
        0x2d, 0x21, 0x60, 0x40, 0x46, 0x55, 0x7a, 0x60,
        0x22, 0x25, 0x3f, 0x4b, 0x54, 0x6a, 0x6a, 0x3c,
        0x77, 0x22, 0x5b, 0x43, 0x67, 0x58, 0x71, 0x22,
        0x79, 0x4b, 0x32, 0x61, 0x44, 0x4d, 0x6f, 0x42,
        0x33, 0x2d, 0x53, 0x35, 0x3d, 0x6f, 0x57, 0x48,
        0x33, 0x3b, 0x5a, 0x53, 0x3f, 0x4e, 0x3f, 0x6b,
        0x4c, 0x27, 0x26, 0x3b, 0x73, 0x49, 0x22, 0x55,
        0x79, 0x2f, 0x47, 0x2f, 0x55, 0x5a, 0x7a, 0x71,
        0x6c, 0x31, 0x43, 0x40, 0x56, 0x7b, 0x21, 0x7a,
        0x6d, 0x4c, 0x43, 0x5e, 0x38, 0x47, 0x29, 0x38,
        0x62, 0x49, 0x45, 0x78, 0x70, 0x2b, 0x2e, 0x65,
        0x47, 0x71, 0x58, 0x79, 0x39, 0x67, 0x7d, 0x6d,
        0x6a, 0x67, 0x4a, 0x71, 0x27, 0x35, 0x2a, 0x4c,
        0x3e, 0x58, 0x55, 0x30, 0x4d, 0x75, 0x77, 0x48,
        0x5f, 0x4b, 0x59, 0x34, 0x65, 0x68, 0x57, 0x59,
        0x63, 0x23, 0x47, 0x38, 0x47, 0x5e, 0x56, 0x28,
        0x79, 0x58, 0x3e, 0x39, 0x66, 0x77, 0x67, 0x33,
        0x29, 0x61, 0x24, 0x7d, 0x37, 0x44, 0x37, 0x67,
        0x3a, 0x58, 0x76, 0x21, 0x51, 0x59, 0x61, 0x73,
        0x66, 0x75, 0x71, 0x53, 0x4d, 0x24, 0x2d, 0x4b,
        0x29, 0x30, 0x32, 0x26, 0x59, 0x64, 0x27, 0x55,
        0x2c, 0x5a, 0x4c, 0x3c, 0x6c, 0x53, 0x56, 0x4b,
        0x3e, 0x55, 0x2e, 0x44, 0x38, 0x6b, 0x47, 0x76,
        0x2d, 0x2c, 0x3f, 0x4d, 0x22, 0x7b, 0x6d, 0x61,
        0x34, 0x6b, 0x50, 0x73, 0x28, 0x6d, 0x41, 0x71,
        0x21, 0x76, 0x52, 0x2a, 0x6d, 0x53, 0x2a, 0x74,
        0x28, 0x27, 0x62, 0x2a, 0x66, 0x25, 0x6e, 0x5e,
        0x37, 0x4f, 0x27, 0x72, 0x28, 0x47, 0x63, 0x6e,
        0x5a, 0x6a, 0x41, 0x35, 0x3a, 0x42, 0x3f, 0x27,
        0x75, 0x3e, 0x26, 0x3e, 0x6b, 0x55, 0x59, 0x60,
        0x24, 0x70, 0x49, 0x3c, 0x4e, 0x2c, 0x39, 0x7a,
        0x36, 0x6c, 0x27, 0x3e, 0x6a, 0x4a, 0x59, 0x5a,
        0x3e, 0x21, 0x73, 0x4e, 0x59, 0x6e, 0x3d, 0x32,
        0x27, 0x45, 0x49, 0x58, 0x7d, 0x37, 0x39, 0x77,
        0x28, 0x51, 0x79, 0x54, 0x2b, 0x78, 0x46, 0x5a,
        0x21, 0x75, 0x33, 0x21, 0x63, 0x5a, 0x7b, 0x3e,
        0x33, 0x4f, 0x67, 0x75, 0x3a, 0x50, 0x48, 0x60,
        0x26, 0x64, 0x76, 0x5c, 0x42, 0x5c, 0x72, 0x38,
        0x6c, 0x52, 0x21, 0x2b, 0x25, 0x6b, 0x7c, 0x6b,
        0x2d, 0x5e, 0x63, 0x2a, 0x4c, 0x26, 0x5b, 0x4c,
        0x58, 0x52, 0x51, 0x55, 0x31, 0x79, 0x6c, 0x53,
        0x62, 0x3a, 0x36, 0x46, 0x7a, 0x29, 0x27, 0x78,
        0x1a, 0xbf, 0x49, 0x74, 0x68, 0x24, 0x51, 0x44,
        0x5b, 0x3e, 0x34, 0x44, 0x29, 0x5e, 0x4f, 0x2a,
        0xe9, 0x3f, 0xf8, 0xff, 0xff, 0x52, 0x7d, 0x47,
        0x67, 0x40, 0x27, 0x5e, 0x47, 0x46, 0x6d, 0x72,
        0x5d, 0x49, 0x26, 0x45, 0x33, 0x6b, 0x4d, 0x4a,
        0x6f, 0x62, 0x60, 0x45, 0x62, 0x27, 0x27, 0x7d,
        0x6a, 0x41, 0x2c, 0x6c, 0x5b, 0x2a, 0x2b, 0x36,
        0x29, 0x58, 0x7a, 0x4c, 0x6e, 0x2d, 0x74, 0x5c,
        0x38, 0x22, 0x5f, 0x49, 0x63, 0x43, 0x5b, 0x67
    };
    uint32_t request2_len = sizeof(request2);


    TcpSession ssn;
    Packet *p[4];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_iface:3919286a-b10c-11d0-9ba8-00c04fd92ef5; "
        "dce_stub_data; content:\"|91 27 27 30|\"; distance:0; sid:1;)";
    char *sig2 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|2d 5e 63 2a 4c|\"; distance:0; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 4; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }
    p[1]->flowflags |= FLOW_PKT_TOCLIENT;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig2);
    s = s->next;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, bind, bind_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if ((PacketAlertCheck(p[0], 1))) {
        printf("sid 1 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 2))) {
        printf("sid 2 matched but shouldn't have for packet 0: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, bind_ack, bind_ack_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[1]);
    if ((PacketAlertCheck(p[1], 1))) {
        printf("sid 1 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 2))) {
        printf("sid 2 matched but shouldn't have for packet 1: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[2]);
    if ((PacketAlertCheck(p[2], 1))) {
        printf("sid 1 matched but shouldn't have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 2))) {
        printf("sid 2 matched but shouldn't have for packet 2: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[3]);
    if ((PacketAlertCheck(p[3], 1))) {
        printf("sid 1 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if (!(PacketAlertCheck(p[3], 2))) {
        printf("sid 2 didn't match but should have for packet 3: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 4);
    return result;
}

/**
 * \test Test the working of detection engien with respect to dce keywords.
 */
int DcePayloadTest07(void)
{
    int result = 0;
    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x6a, 0x28, 0x19, 0x39, 0x0c, 0xb1, 0xd0, 0x11,
        0x9b, 0xa8, 0x00, 0xc0, 0x4f, 0xd9, 0x2e, 0xf5,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_len = sizeof(bind);

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x48, 0x1a, 0x00, 0x00,
        0x0c, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x6c, 0x73, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x82, 0x7c, 0x83, 0x46, 0x0f, 0xf7,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xcf, 0x80, 0x98, 0x6d, 0xfe, 0xb0, 0x90, 0xd1,
        0xcf, 0x86, 0x0f, 0x52, 0x2c, 0x23, 0x66, 0x28,
        0x27, 0x30, 0x48, 0x55, 0x42, 0x6a, 0x48, 0x4b,
        0x68, 0x22, 0x2e, 0x23, 0x64, 0x33, 0x2c, 0x2d,
        0x5c, 0x51, 0x48, 0x55, 0x24, 0x67, 0x6c, 0x4c,
        0x45, 0x71, 0x35, 0x72, 0x5a, 0x48, 0x5e, 0x35,
        0x61, 0x78, 0x35, 0x42, 0x2c, 0x7a, 0x75, 0x61,
        0x5b, 0x4e, 0x76, 0x30, 0x26, 0x2f, 0x2a, 0x34,
        0x48, 0x29, 0x25, 0x6e, 0x5c, 0x3a, 0x6c, 0x3e,
        0x79, 0x4e, 0x2a, 0x21, 0x6f, 0x6f, 0x34, 0x46,
        0x43, 0x26, 0x5b, 0x35, 0x78, 0x27, 0x69, 0x23,
        0x72, 0x21, 0x69, 0x56, 0x6a, 0x7d, 0x4b, 0x5e,
        0x65, 0x37, 0x60, 0x44, 0x7c, 0x5d, 0x5b, 0x72,
        0x7d, 0x73, 0x7b, 0x47, 0x57, 0x21, 0x41, 0x38,
        0x76, 0x38, 0x76, 0x5c, 0x58, 0x32, 0x4a, 0x37,
        0x2f, 0x40, 0x4b, 0x4c, 0x3d, 0x41, 0x33, 0x56,
        0x73, 0x38, 0x61, 0x71, 0x24, 0x49, 0x4c, 0x4a,
        0x44, 0x2e, 0x3a, 0x3f, 0x74, 0x54, 0x4c, 0x65,
        0x54, 0x2d, 0x3b, 0x28, 0x41, 0x45, 0x49, 0x2c,
        0x6e, 0x48, 0x44, 0x43, 0x37, 0x3d, 0x7b, 0x6d,
        0x2b, 0x4b, 0x32, 0x5a, 0x31, 0x61, 0x6e, 0x2b,
        0x27, 0x50, 0x6b, 0x66, 0x76, 0x4e, 0x55, 0x35,
        0x2b, 0x72, 0x2d, 0x5e, 0x42, 0x3e, 0x5a, 0x5d,
        0x36, 0x45, 0x32, 0x3a, 0x58, 0x78, 0x78, 0x3e,
        0x60, 0x6c, 0x5d, 0x63, 0x41, 0x7c, 0x52, 0x21,
        0x75, 0x6a, 0x5a, 0x70, 0x55, 0x45, 0x76, 0x58,
        0x33, 0x40, 0x38, 0x39, 0x21, 0x37, 0x7d, 0x77,
        0x21, 0x70, 0x2b, 0x72, 0x29, 0x6a, 0x31, 0x5f,
        0x38, 0x4a, 0x66, 0x65, 0x62, 0x2c, 0x39, 0x52,
        0x5f, 0x2a, 0x2b, 0x63, 0x4f, 0x76, 0x43, 0x25,
        0x6a, 0x50, 0x37, 0x52, 0x5e, 0x23, 0x3c, 0x42,
        0x28, 0x75, 0x75, 0x42, 0x25, 0x23, 0x28, 0x56,
        0x6c, 0x46, 0x5c, 0x5e, 0x6b, 0x7d, 0x48, 0x24,
        0x77, 0x6c, 0x70, 0x62, 0x2e, 0x28, 0x7d, 0x6b,
        0x69, 0x4a, 0x75, 0x3d, 0x5d, 0x56, 0x21, 0x49,
        0x56, 0x47, 0x64, 0x2b, 0x4c, 0x52, 0x43, 0x60,
        0x77, 0x49, 0x46, 0x46, 0x33, 0x2c, 0x4b, 0x4b,
        0x3d, 0x63, 0x5d, 0x33, 0x78, 0x76, 0x51, 0x56,
        0x77, 0x3c, 0x72, 0x74, 0x52, 0x27, 0x40, 0x6c,
        0x42, 0x79, 0x49, 0x24, 0x62, 0x5e, 0x26, 0x31,
        0x5c, 0x22, 0x2b, 0x4c, 0x64, 0x49, 0x52, 0x45,
        0x47, 0x49, 0x3a, 0x2a, 0x51, 0x71, 0x22, 0x22,
        0x70, 0x24, 0x34, 0x67, 0x4b, 0x6d, 0x58, 0x29,
        0x63, 0x26, 0x7b, 0x6f, 0x38, 0x78, 0x25, 0x62,
        0x4d, 0x3a, 0x7d, 0x40, 0x23, 0x57, 0x67, 0x33,
        0x38, 0x31, 0x4e, 0x54, 0x3c, 0x4b, 0x48, 0x69,
        0x3c, 0x39, 0x31, 0x2b, 0x26, 0x70, 0x44, 0x66,
        0x4a, 0x37, 0x2b, 0x75, 0x36, 0x45, 0x59, 0x34,
        0x3e, 0x3e, 0x29, 0x70, 0x71, 0x5a, 0x55, 0x49,
        0x3e, 0x4b, 0x68, 0x4e, 0x75, 0x70, 0x3c, 0x5c,
        0x50, 0x58, 0x28, 0x75, 0x3c, 0x2a, 0x41, 0x70,
        0x2f, 0x2b, 0x37, 0x26, 0x75, 0x71, 0x55, 0x22,
        0x3a, 0x44, 0x30, 0x48, 0x5d, 0x2f, 0x6c, 0x44,
        0x28, 0x4b, 0x34, 0x45, 0x21, 0x60, 0x44, 0x36,
        0x7b, 0x32, 0x39, 0x5f, 0x6d, 0x3f, 0x68, 0x73,
        0x25, 0x45, 0x56, 0x7c, 0x78, 0x7a, 0x49, 0x6a,
        0x46, 0x3d, 0x2d, 0x33, 0x6c, 0x6f, 0x23, 0x77,
        0x38, 0x33, 0x36, 0x74, 0x7b, 0x57, 0x4b, 0x6d,
        0x27, 0x75, 0x24, 0x6e, 0x43, 0x61, 0x4d, 0x44,
        0x6d, 0x27, 0x48, 0x58, 0x5e, 0x7b, 0x26, 0x6a,
        0x50, 0x7c, 0x51, 0x23, 0x3c, 0x4f, 0x37, 0x4c,
        0x47, 0x3e, 0x45, 0x56, 0x22, 0x33, 0x7c, 0x66,
        0x35, 0x54, 0x7a, 0x6e, 0x5a, 0x24, 0x70, 0x62,
        0x29, 0x3f, 0x69, 0x79, 0x24, 0x43, 0x41, 0x24,
        0x65, 0x25, 0x62, 0x4f, 0x73, 0x3e, 0x2b, 0x36,
        0x46, 0x69, 0x27, 0x55, 0x2a, 0x6e, 0x24, 0x6c,
        0x7d, 0x64, 0x7c, 0x61, 0x26, 0x67, 0x2a, 0x53,
        0x73, 0x60, 0x28, 0x2d, 0x6b, 0x44, 0x54, 0x61,
        0x34, 0x53, 0x22, 0x59, 0x6d, 0x73, 0x56, 0x55,
        0x25, 0x2c, 0x38, 0x4a, 0x3b, 0x4e, 0x78, 0x46,
        0x54, 0x6e, 0x6d, 0x4f, 0x47, 0x4f, 0x4f, 0x5a,
        0x67, 0x77, 0x39, 0x66, 0x28, 0x29, 0x4e, 0x43,
        0x55, 0x6e, 0x60, 0x59, 0x28, 0x3b, 0x65, 0x62,
        0x61, 0x5a, 0x29, 0x6e, 0x79, 0x60, 0x41, 0x53,
        0x2f, 0x5d, 0x44, 0x36, 0x7b, 0x3e, 0x7c, 0x2b,
        0x77, 0x36, 0x70, 0x3f, 0x40, 0x55, 0x48, 0x67,
        0x4b, 0x4d, 0x5d, 0x51, 0x79, 0x76, 0x48, 0x4a,
        0x2d, 0x21, 0x60, 0x40, 0x46, 0x55, 0x7a, 0x60,
        0x22, 0x25, 0x3f, 0x4b, 0x54, 0x6a, 0x6a, 0x3c,
        0x77, 0x22, 0x5b, 0x43, 0x67, 0x58, 0x71, 0x22,
        0x79, 0x4b, 0x32, 0x61, 0x44, 0x4d, 0x6f, 0x42,
        0x33, 0x2d, 0x53, 0x35, 0x3d, 0x6f, 0x57, 0x48,
        0x33, 0x3b, 0x5a, 0x53, 0x3f, 0x4e, 0x3f, 0x6b,
        0x4c, 0x27, 0x26, 0x3b, 0x73, 0x49, 0x22, 0x55,
        0x79, 0x2f, 0x47, 0x2f, 0x55, 0x5a, 0x7a, 0x71,
        0x6c, 0x31, 0x43, 0x40, 0x56, 0x7b, 0x21, 0x7a,
        0x6d, 0x4c, 0x43, 0x5e, 0x38, 0x47, 0x29, 0x38,
        0x62, 0x49, 0x45, 0x78, 0x70, 0x2b, 0x2e, 0x65,
        0x47, 0x71, 0x58, 0x79, 0x39, 0x67, 0x7d, 0x6d,
        0x6a, 0x67, 0x4a, 0x71, 0x27, 0x35, 0x2a, 0x4c,
        0x3e, 0x58, 0x55, 0x30, 0x4d, 0x75, 0x77, 0x48,
        0x5f, 0x4b, 0x59, 0x34, 0x65, 0x68, 0x57, 0x59,
        0x63, 0x23, 0x47, 0x38, 0x47, 0x5e, 0x56, 0x28,
        0x79, 0x58, 0x3e, 0x39, 0x66, 0x77, 0x67, 0x33,
        0x29, 0x61, 0x24, 0x7d, 0x37, 0x44, 0x37, 0x67,
        0x3a, 0x58, 0x76, 0x21, 0x51, 0x59, 0x61, 0x73,
        0x66, 0x75, 0x71, 0x53, 0x4d, 0x24, 0x2d, 0x4b,
        0x29, 0x30, 0x32, 0x26, 0x59, 0x64, 0x27, 0x55,
        0x2c, 0x5a, 0x4c, 0x3c, 0x6c, 0x53, 0x56, 0x4b,
        0x3e, 0x55, 0x2e, 0x44, 0x38, 0x6b, 0x47, 0x76,
        0x2d, 0x2c, 0x3f, 0x4d, 0x22, 0x7b, 0x6d, 0x61,
        0x34, 0x6b, 0x50, 0x73, 0x28, 0x6d, 0x41, 0x71,
        0x21, 0x76, 0x52, 0x2a, 0x6d, 0x53, 0x2a, 0x74,
        0x28, 0x27, 0x62, 0x2a, 0x66, 0x25, 0x6e, 0x5e,
        0x37, 0x4f, 0x27, 0x72, 0x28, 0x47, 0x63, 0x6e,
        0x5a, 0x6a, 0x41, 0x35, 0x3a, 0x42, 0x3f, 0x27,
        0x75, 0x3e, 0x26, 0x3e, 0x6b, 0x55, 0x59, 0x60,
        0x24, 0x70, 0x49, 0x3c, 0x4e, 0x2c, 0x39, 0x7a,
        0x36, 0x6c, 0x27, 0x3e, 0x6a, 0x4a, 0x59, 0x5a,
        0x3e, 0x21, 0x73, 0x4e, 0x59, 0x6e, 0x3d, 0x32,
        0x27, 0x45, 0x49, 0x58, 0x7d, 0x37, 0x39, 0x77,
        0x28, 0x51, 0x79, 0x54, 0x2b, 0x78, 0x46, 0x5a,
        0x21, 0x75, 0x33, 0x21, 0x63, 0x5a, 0x7b, 0x3e,
        0x33, 0x4f, 0x67, 0x75, 0x3a, 0x50, 0x48, 0x60,
        0x26, 0x64, 0x76, 0x5c, 0x42, 0x5c, 0x72, 0x38,
        0x6c, 0x52, 0x21, 0x2b, 0x25, 0x6b, 0x7c, 0x6b,
        0x2d, 0x5e, 0x63, 0x2a, 0x4c, 0x26, 0x5b, 0x4c,
        0x58, 0x52, 0x51, 0x55, 0x31, 0x79, 0x6c, 0x53,
        0x62, 0x3a, 0x36, 0x46, 0x7a, 0x29, 0x27, 0x78,
        0x1a, 0xbf, 0x49, 0x74, 0x68, 0x24, 0x51, 0x44,
        0x5b, 0x3e, 0x34, 0x44, 0x29, 0x5e, 0x4f, 0x2a,
        0xe9, 0x3f, 0xf8, 0xff, 0xff, 0x52, 0x7d, 0x47,
        0x67, 0x40, 0x27, 0x5e, 0x47, 0x46, 0x6d, 0x72,
        0x5d, 0x49, 0x26, 0x45, 0x33, 0x6b, 0x4d, 0x4a,
        0x6f, 0x62, 0x60, 0x45, 0x62, 0x27, 0x27, 0x7d,
        0x6a, 0x41, 0x2c, 0x6c, 0x5b, 0x2a, 0x2b, 0x36,
        0x29, 0x58, 0x7a, 0x4c, 0x6e, 0x2d, 0x74, 0x5c,
        0x38, 0x22, 0x5f, 0x49, 0x63, 0x43, 0x5b, 0x67
    };
    uint32_t request2_len = sizeof(request2);

    TcpSession ssn;
    Packet *p[4];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_iface:3919286a-b10c-11d0-9ba8-00c04fd92ef5; "
        "dce_stub_data; content:\"|91 27 27 30|\"; distance:0; sid:1;)";
    char *sig2 = "alert tcp any any -> any any (dce_stub_data; "
        "content:\"|2d 5e 63 35 25|\"; distance:0; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 4; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }
    p[1]->flowflags |= FLOW_PKT_TOCLIENT;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    s->next = SigInit(de_ctx, sig2);
    s = s->next;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, bind, bind_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if ((PacketAlertCheck(p[0], 1))) {
        printf("sid 1 matched but shouldn't have for packet 0: ");
        goto end;
    }
    if ((PacketAlertCheck(p[0], 2))) {
        printf("sid 2 matched but shouldn't have for packet 0: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, bind_ack, bind_ack_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[1]);
    if ((PacketAlertCheck(p[1], 1))) {
        printf("sid 1 matched but shouldn't have for packet 1: ");
        goto end;
    }
    if ((PacketAlertCheck(p[1], 2))) {
        printf("sid 2 matched but shouldn't have for packet 1: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[2]);
    if ((PacketAlertCheck(p[2], 1))) {
        printf("sid 1 matched but shouldn't have for packet 2: ");
        goto end;
    }
    if ((PacketAlertCheck(p[2], 2))) {
        printf("sid 2 matched but shouldn't have for packet 2: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[3]);
    if ((PacketAlertCheck(p[3], 1))) {
        printf("sid 1 matched but shouldn't have for packet 3: ");
        goto end;
    }
    if ((PacketAlertCheck(p[3], 2))) {
        printf("sid 2 matched but shouldn't have for packet 3: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 4);
    return result;
}

/**
 * \test Positive test, to test the working of distance and within.
 */
int DcePayloadTest08(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x82, 0x7c, 0x83, 0x46, 0x0f, 0xf7,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p[1];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5d 5b 35|\"; distance:0; content:\"|9e a3|\"; "
        "distance:0; within:2; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 1; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if (!(PacketAlertCheck(p[0], 1))) {
        printf("sid 1 didn't match but should have for packet 0: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 1);
    return result;
}

/**
 * \test Positive test, to test the working of distance and within.
 */
int DcePayloadTest09(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x5d, 0x5b, 0x35, 0x46, 0x9e, 0xa3,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p[1];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5d 5b 35|\"; distance:0; content:\"|9e a3|\"; "
        "distance:0; within:2; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 1; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if (!(PacketAlertCheck(p[0], 1))) {
        printf("sid 1 didn't match but should have for packet 0: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 1);
    return result;
}

/**
 * \test Positive test, to test the working of distance and within.
 */
int DcePayloadTest10(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x5d, 0x5b, 0x35, 0x46, 0x9e, 0xa3,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p[1];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|ad 0d|\"; distance:0; content:\"|ad 0d 00|\"; "
        "distance:-10; within:3; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 1; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if (!(PacketAlertCheck(p[0], 1))) {
        printf("sid 1 didn't match but should have for packet 0: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 1);
    return result;
}

/**
 * \test Postive test to check the working of disance and within across frags.
 */
int DcePayloadTest11(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x82, 0x7c, 0x83, 0x46, 0x0f, 0xf7,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xcf, 0x80, 0x98, 0x6d, 0xfe, 0xb0, 0x90, 0xd1,
        0xcf, 0x86, 0x0f, 0x52, 0x2c, 0x23, 0x66, 0x28,
        0x27, 0x30, 0x48, 0x55, 0x42, 0x6a, 0x48, 0x4b,
        0x68, 0x22, 0x2e, 0x23, 0x64, 0x33, 0x2c, 0x2d,
        0x5c, 0x51, 0x48, 0x55, 0x24, 0x67, 0x6c, 0x4c,
        0x45, 0x71, 0x35, 0x72, 0x5a, 0x48, 0x5e, 0x35,
        0x61, 0x78, 0x35, 0x42, 0x2c, 0x7a, 0x75, 0x61,
        0x5b, 0x4e, 0x76, 0x30, 0x26, 0x2f, 0x2a, 0x34,
        0x48, 0x29, 0x25, 0x6e, 0x5c, 0x3a, 0x6c, 0x3e,
        0x79, 0x4e, 0x2a, 0x21, 0x6f, 0x6f, 0x34, 0x46,
        0x43, 0x26, 0x5b, 0x35, 0x78, 0x27, 0x69, 0x23,
        0x72, 0x21, 0x69, 0x56, 0x6a, 0x7d, 0x4b, 0x5e,
        0x65, 0x37, 0x60, 0x44, 0x7c, 0x5d, 0x5b, 0x72,
        0x7d, 0x73, 0x7b, 0x47, 0x57, 0x21, 0x41, 0x38,
        0x76, 0x38, 0x76, 0x5c, 0x58, 0x32, 0x4a, 0x37,
        0x2f, 0x40, 0x4b, 0x4c, 0x3d, 0x41, 0x33, 0x56,
        0x73, 0x38, 0x61, 0x71, 0x24, 0x49, 0x4c, 0x4a,
        0x44, 0x2e, 0x3a, 0x3f, 0x74, 0x54, 0x4c, 0x65,
        0x54, 0x2d, 0x3b, 0x28, 0x41, 0x45, 0x49, 0x2c,
        0x6e, 0x48, 0x44, 0x43, 0x37, 0x3d, 0x7b, 0x6d,
        0x2b, 0x4b, 0x32, 0x5a, 0x31, 0x61, 0x6e, 0x2b,
        0x27, 0x50, 0x6b, 0x66, 0x76, 0x4e, 0x55, 0x35,
        0x2b, 0x72, 0x2d, 0x5e, 0x42, 0x3e, 0x5a, 0x5d,
        0x36, 0x45, 0x32, 0x3a, 0x58, 0x78, 0x78, 0x3e,
        0x60, 0x6c, 0x5d, 0x63, 0x41, 0x7c, 0x52, 0x21,
        0x75, 0x6a, 0x5a, 0x70, 0x55, 0x45, 0x76, 0x58,
        0x33, 0x40, 0x38, 0x39, 0x21, 0x37, 0x7d, 0x77,
        0x21, 0x70, 0x2b, 0x72, 0x29, 0x6a, 0x31, 0x5f,
        0x38, 0x4a, 0x66, 0x65, 0x62, 0x2c, 0x39, 0x52,
        0x5f, 0x2a, 0x2b, 0x63, 0x4f, 0x76, 0x43, 0x25,
        0x6a, 0x50, 0x37, 0x52, 0x5e, 0x23, 0x3c, 0x42,
        0x28, 0x75, 0x75, 0x42, 0x25, 0x23, 0x28, 0x56,
        0x6c, 0x46, 0x5c, 0x5e, 0x6b, 0x7d, 0x48, 0x24,
        0x77, 0x6c, 0x70, 0x62, 0x2e, 0x28, 0x7d, 0x6b,
        0x69, 0x4a, 0x75, 0x3d, 0x5d, 0x56, 0x21, 0x49,
        0x56, 0x47, 0x64, 0x2b, 0x4c, 0x52, 0x43, 0x60,
        0x77, 0x49, 0x46, 0x46, 0x33, 0x2c, 0x4b, 0x4b,
        0x3d, 0x63, 0x5d, 0x33, 0x78, 0x76, 0x51, 0x56,
        0x77, 0x3c, 0x72, 0x74, 0x52, 0x27, 0x40, 0x6c,
        0x42, 0x79, 0x49, 0x24, 0x62, 0x5e, 0x26, 0x31,
        0x5c, 0x22, 0x2b, 0x4c, 0x64, 0x49, 0x52, 0x45,
        0x47, 0x49, 0x3a, 0x2a, 0x51, 0x71, 0x22, 0x22,
        0x70, 0x24, 0x34, 0x67, 0x4b, 0x6d, 0x58, 0x29,
        0x63, 0x26, 0x7b, 0x6f, 0x38, 0x78, 0x25, 0x62,
        0x4d, 0x3a, 0x7d, 0x40, 0x23, 0x57, 0x67, 0x33,
        0x38, 0x31, 0x4e, 0x54, 0x3c, 0x4b, 0x48, 0x69,
        0x3c, 0x39, 0x31, 0x2b, 0x26, 0x70, 0x44, 0x66,
        0x4a, 0x37, 0x2b, 0x75, 0x36, 0x45, 0x59, 0x34,
        0x3e, 0x3e, 0x29, 0x70, 0x71, 0x5a, 0x55, 0x49,
        0x3e, 0x4b, 0x68, 0x4e, 0x75, 0x70, 0x3c, 0x5c,
        0x50, 0x58, 0x28, 0x75, 0x3c, 0x2a, 0x41, 0x70,
        0x2f, 0x2b, 0x37, 0x26, 0x75, 0x71, 0x55, 0x22,
        0x3a, 0x44, 0x30, 0x48, 0x5d, 0x2f, 0x6c, 0x44,
        0x28, 0x4b, 0x34, 0x45, 0x21, 0x60, 0x44, 0x36,
        0x7b, 0x32, 0x39, 0x5f, 0x6d, 0x3f, 0x68, 0x73,
        0x25, 0x45, 0x56, 0x7c, 0x78, 0x7a, 0x49, 0x6a,
        0x46, 0x3d, 0x2d, 0x33, 0x6c, 0x6f, 0x23, 0x77,
        0x38, 0x33, 0x36, 0x74, 0x7b, 0x57, 0x4b, 0x6d,
        0x27, 0x75, 0x24, 0x6e, 0x43, 0x61, 0x4d, 0x44,
        0x6d, 0x27, 0x48, 0x58, 0x5e, 0x7b, 0x26, 0x6a,
        0x50, 0x7c, 0x51, 0x23, 0x3c, 0x4f, 0x37, 0x4c,
        0x47, 0x3e, 0x45, 0x56, 0x22, 0x33, 0x7c, 0x66,
        0x35, 0x54, 0x7a, 0x6e, 0x5a, 0x24, 0x70, 0x62,
        0x29, 0x3f, 0x69, 0x79, 0x24, 0x43, 0x41, 0x24,
        0x65, 0x25, 0x62, 0x4f, 0x73, 0x3e, 0x2b, 0x36,
        0x46, 0x69, 0x27, 0x55, 0x2a, 0x6e, 0x24, 0x6c,
        0x7d, 0x64, 0x7c, 0x61, 0x26, 0x67, 0x2a, 0x53,
        0x73, 0x60, 0x28, 0x2d, 0x6b, 0x44, 0x54, 0x61,
        0x34, 0x53, 0x22, 0x59, 0x6d, 0x73, 0x56, 0x55,
        0x25, 0x2c, 0x38, 0x4a, 0x3b, 0x4e, 0x78, 0x46,
        0x54, 0x6e, 0x6d, 0x4f, 0x47, 0x4f, 0x4f, 0x5a,
        0x67, 0x77, 0x39, 0x66, 0x28, 0x29, 0x4e, 0x43,
        0x55, 0x6e, 0x60, 0x59, 0x28, 0x3b, 0x65, 0x62,
        0x61, 0x5a, 0x29, 0x6e, 0x79, 0x60, 0x41, 0x53,
        0x2f, 0x5d, 0x44, 0x36, 0x7b, 0x3e, 0x7c, 0x2b,
        0x77, 0x36, 0x70, 0x3f, 0x40, 0x55, 0x48, 0x67,
        0x4b, 0x4d, 0x5d, 0x51, 0x79, 0x76, 0x48, 0x4a,
        0x2d, 0x21, 0x60, 0x40, 0x46, 0x55, 0x7a, 0x60,
        0x22, 0x25, 0x3f, 0x4b, 0x54, 0x6a, 0x6a, 0x3c,
        0x77, 0x22, 0x5b, 0x43, 0x67, 0x58, 0x71, 0x22,
        0x79, 0x4b, 0x32, 0x61, 0x44, 0x4d, 0x6f, 0x42,
        0x33, 0x2d, 0x53, 0x35, 0x3d, 0x6f, 0x57, 0x48,
        0x33, 0x3b, 0x5a, 0x53, 0x3f, 0x4e, 0x3f, 0x6b,
        0x4c, 0x27, 0x26, 0x3b, 0x73, 0x49, 0x22, 0x55,
        0x79, 0x2f, 0x47, 0x2f, 0x55, 0x5a, 0x7a, 0x71,
        0x6c, 0x31, 0x43, 0x40, 0x56, 0x7b, 0x21, 0x7a,
        0x6d, 0x4c, 0x43, 0x5e, 0x38, 0x47, 0x29, 0x38,
        0x62, 0x49, 0x45, 0x78, 0x70, 0x2b, 0x2e, 0x65,
        0x47, 0x71, 0x58, 0x79, 0x39, 0x67, 0x7d, 0x6d,
        0x6a, 0x67, 0x4a, 0x71, 0x27, 0x35, 0x2a, 0x4c,
        0x3e, 0x58, 0x55, 0x30, 0x4d, 0x75, 0x77, 0x48,
        0x5f, 0x4b, 0x59, 0x34, 0x65, 0x68, 0x57, 0x59,
        0x63, 0x23, 0x47, 0x38, 0x47, 0x5e, 0x56, 0x28,
        0x79, 0x58, 0x3e, 0x39, 0x66, 0x77, 0x67, 0x33,
        0x29, 0x61, 0x24, 0x7d, 0x37, 0x44, 0x37, 0x67,
        0x3a, 0x58, 0x76, 0x21, 0x51, 0x59, 0x61, 0x73,
        0x66, 0x75, 0x71, 0x53, 0x4d, 0x24, 0x2d, 0x4b,
        0x29, 0x30, 0x32, 0x26, 0x59, 0x64, 0x27, 0x55,
        0x2c, 0x5a, 0x4c, 0x3c, 0x6c, 0x53, 0x56, 0x4b,
        0x3e, 0x55, 0x2e, 0x44, 0x38, 0x6b, 0x47, 0x76,
        0x2d, 0x2c, 0x3f, 0x4d, 0x22, 0x7b, 0x6d, 0x61,
        0x34, 0x6b, 0x50, 0x73, 0x28, 0x6d, 0x41, 0x71,
        0x21, 0x76, 0x52, 0x2a, 0x6d, 0x53, 0x2a, 0x74,
        0x28, 0x27, 0x62, 0x2a, 0x66, 0x25, 0x6e, 0x5e,
        0x37, 0x4f, 0x27, 0x72, 0x28, 0x47, 0x63, 0x6e,
        0x5a, 0x6a, 0x41, 0x35, 0x3a, 0x42, 0x3f, 0x27,
        0x75, 0x3e, 0x26, 0x3e, 0x6b, 0x55, 0x59, 0x60,
        0x24, 0x70, 0x49, 0x3c, 0x4e, 0x2c, 0x39, 0x7a,
        0x36, 0x6c, 0x27, 0x3e, 0x6a, 0x4a, 0x59, 0x5a,
        0x3e, 0x21, 0x73, 0x4e, 0x59, 0x6e, 0x3d, 0x32,
        0x27, 0x45, 0x49, 0x58, 0x7d, 0x37, 0x39, 0x77,
        0x28, 0x51, 0x79, 0x54, 0x2b, 0x78, 0x46, 0x5a,
        0x21, 0x75, 0x33, 0x21, 0x63, 0x5a, 0x7b, 0x3e,
        0x33, 0x4f, 0x67, 0x75, 0x3a, 0x50, 0x48, 0x60,
        0x26, 0x64, 0x76, 0x5c, 0x42, 0x5c, 0x72, 0x38,
        0x6c, 0x52, 0x21, 0x2b, 0x25, 0x6b, 0x7c, 0x6b,
        0x2d, 0x5e, 0x63, 0x2a, 0x4c, 0x26, 0x5b, 0x4c,
        0x58, 0x52, 0x51, 0x55, 0x31, 0x79, 0x6c, 0x53,
        0x62, 0x3a, 0x36, 0x46, 0x7a, 0x29, 0x27, 0x78,
        0x1a, 0xbf, 0x49, 0x74, 0x68, 0x24, 0x51, 0x44,
        0x5b, 0x3e, 0x34, 0x44, 0x29, 0x5e, 0x4f, 0x2a,
        0xe9, 0x3f, 0xf8, 0xff, 0xff, 0x52, 0x7d, 0x47,
        0x67, 0x40, 0x27, 0x5e, 0x47, 0x46, 0x6d, 0x72,
        0x5d, 0x49, 0x26, 0x45, 0x33, 0x6b, 0x4d, 0x4a,
        0x6f, 0x62, 0x60, 0x45, 0x62, 0x27, 0x27, 0x7d,
        0x6a, 0x41, 0x2c, 0x6c, 0x5b, 0x2a, 0x2b, 0x36,
        0x29, 0x58, 0x7a, 0x4c, 0x6e, 0x2d, 0x74, 0x5c,
        0x38, 0x22, 0x5f, 0x49, 0x63, 0x43, 0x5b, 0x67
    };
    uint32_t request2_len = sizeof(request2);

    TcpSession ssn;
    Packet *p[2];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|af, 26, d0|\"; distance:0; content:\"|80 98 6d|\"; "
        "distance:1; within:3; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 2; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if ((PacketAlertCheck(p[0], 1))) {
        printf("sid 1 matched but shouldn't have for packet 0: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[1]);
    if (!(PacketAlertCheck(p[1], 1))) {
        printf("sid 1 didn't match but should have for pacekt 1: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 2);
    return result;
}

/**
 * \test Negative test the working of contents on stub data with invalid
 *       distance.
 */
int DcePayloadTest12(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x82, 0x7c, 0x83, 0x46, 0x0f, 0xf7,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xcf, 0x80, 0x98, 0x6d, 0xfe, 0xb0, 0x90, 0xd1,
        0xcf, 0x86, 0x0f, 0x52, 0x2c, 0x23, 0x66, 0x28,
        0x27, 0x30, 0x48, 0x55, 0x42, 0x6a, 0x48, 0x4b,
        0x68, 0x22, 0x2e, 0x23, 0x64, 0x33, 0x2c, 0x2d,
        0x5c, 0x51, 0x48, 0x55, 0x24, 0x67, 0x6c, 0x4c,
        0x45, 0x71, 0x35, 0x72, 0x5a, 0x48, 0x5e, 0x35,
        0x61, 0x78, 0x35, 0x42, 0x2c, 0x7a, 0x75, 0x61,
        0x5b, 0x4e, 0x76, 0x30, 0x26, 0x2f, 0x2a, 0x34,
        0x48, 0x29, 0x25, 0x6e, 0x5c, 0x3a, 0x6c, 0x3e,
        0x79, 0x4e, 0x2a, 0x21, 0x6f, 0x6f, 0x34, 0x46,
        0x43, 0x26, 0x5b, 0x35, 0x78, 0x27, 0x69, 0x23,
        0x72, 0x21, 0x69, 0x56, 0x6a, 0x7d, 0x4b, 0x5e,
        0x65, 0x37, 0x60, 0x44, 0x7c, 0x5d, 0x5b, 0x72,
        0x7d, 0x73, 0x7b, 0x47, 0x57, 0x21, 0x41, 0x38,
        0x76, 0x38, 0x76, 0x5c, 0x58, 0x32, 0x4a, 0x37,
        0x2f, 0x40, 0x4b, 0x4c, 0x3d, 0x41, 0x33, 0x56,
        0x73, 0x38, 0x61, 0x71, 0x24, 0x49, 0x4c, 0x4a,
        0x44, 0x2e, 0x3a, 0x3f, 0x74, 0x54, 0x4c, 0x65,
        0x54, 0x2d, 0x3b, 0x28, 0x41, 0x45, 0x49, 0x2c,
        0x6e, 0x48, 0x44, 0x43, 0x37, 0x3d, 0x7b, 0x6d,
        0x2b, 0x4b, 0x32, 0x5a, 0x31, 0x61, 0x6e, 0x2b,
        0x27, 0x50, 0x6b, 0x66, 0x76, 0x4e, 0x55, 0x35,
        0x2b, 0x72, 0x2d, 0x5e, 0x42, 0x3e, 0x5a, 0x5d,
        0x36, 0x45, 0x32, 0x3a, 0x58, 0x78, 0x78, 0x3e,
        0x60, 0x6c, 0x5d, 0x63, 0x41, 0x7c, 0x52, 0x21,
        0x75, 0x6a, 0x5a, 0x70, 0x55, 0x45, 0x76, 0x58,
        0x33, 0x40, 0x38, 0x39, 0x21, 0x37, 0x7d, 0x77,
        0x21, 0x70, 0x2b, 0x72, 0x29, 0x6a, 0x31, 0x5f,
        0x38, 0x4a, 0x66, 0x65, 0x62, 0x2c, 0x39, 0x52,
        0x5f, 0x2a, 0x2b, 0x63, 0x4f, 0x76, 0x43, 0x25,
        0x6a, 0x50, 0x37, 0x52, 0x5e, 0x23, 0x3c, 0x42,
        0x28, 0x75, 0x75, 0x42, 0x25, 0x23, 0x28, 0x56,
        0x6c, 0x46, 0x5c, 0x5e, 0x6b, 0x7d, 0x48, 0x24,
        0x77, 0x6c, 0x70, 0x62, 0x2e, 0x28, 0x7d, 0x6b,
        0x69, 0x4a, 0x75, 0x3d, 0x5d, 0x56, 0x21, 0x49,
        0x56, 0x47, 0x64, 0x2b, 0x4c, 0x52, 0x43, 0x60,
        0x77, 0x49, 0x46, 0x46, 0x33, 0x2c, 0x4b, 0x4b,
        0x3d, 0x63, 0x5d, 0x33, 0x78, 0x76, 0x51, 0x56,
        0x77, 0x3c, 0x72, 0x74, 0x52, 0x27, 0x40, 0x6c,
        0x42, 0x79, 0x49, 0x24, 0x62, 0x5e, 0x26, 0x31,
        0x5c, 0x22, 0x2b, 0x4c, 0x64, 0x49, 0x52, 0x45,
        0x47, 0x49, 0x3a, 0x2a, 0x51, 0x71, 0x22, 0x22,
        0x70, 0x24, 0x34, 0x67, 0x4b, 0x6d, 0x58, 0x29,
        0x63, 0x26, 0x7b, 0x6f, 0x38, 0x78, 0x25, 0x62,
        0x4d, 0x3a, 0x7d, 0x40, 0x23, 0x57, 0x67, 0x33,
        0x38, 0x31, 0x4e, 0x54, 0x3c, 0x4b, 0x48, 0x69,
        0x3c, 0x39, 0x31, 0x2b, 0x26, 0x70, 0x44, 0x66,
        0x4a, 0x37, 0x2b, 0x75, 0x36, 0x45, 0x59, 0x34,
        0x3e, 0x3e, 0x29, 0x70, 0x71, 0x5a, 0x55, 0x49,
        0x3e, 0x4b, 0x68, 0x4e, 0x75, 0x70, 0x3c, 0x5c,
        0x50, 0x58, 0x28, 0x75, 0x3c, 0x2a, 0x41, 0x70,
        0x2f, 0x2b, 0x37, 0x26, 0x75, 0x71, 0x55, 0x22,
        0x3a, 0x44, 0x30, 0x48, 0x5d, 0x2f, 0x6c, 0x44,
        0x28, 0x4b, 0x34, 0x45, 0x21, 0x60, 0x44, 0x36,
        0x7b, 0x32, 0x39, 0x5f, 0x6d, 0x3f, 0x68, 0x73,
        0x25, 0x45, 0x56, 0x7c, 0x78, 0x7a, 0x49, 0x6a,
        0x46, 0x3d, 0x2d, 0x33, 0x6c, 0x6f, 0x23, 0x77,
        0x38, 0x33, 0x36, 0x74, 0x7b, 0x57, 0x4b, 0x6d,
        0x27, 0x75, 0x24, 0x6e, 0x43, 0x61, 0x4d, 0x44,
        0x6d, 0x27, 0x48, 0x58, 0x5e, 0x7b, 0x26, 0x6a,
        0x50, 0x7c, 0x51, 0x23, 0x3c, 0x4f, 0x37, 0x4c,
        0x47, 0x3e, 0x45, 0x56, 0x22, 0x33, 0x7c, 0x66,
        0x35, 0x54, 0x7a, 0x6e, 0x5a, 0x24, 0x70, 0x62,
        0x29, 0x3f, 0x69, 0x79, 0x24, 0x43, 0x41, 0x24,
        0x65, 0x25, 0x62, 0x4f, 0x73, 0x3e, 0x2b, 0x36,
        0x46, 0x69, 0x27, 0x55, 0x2a, 0x6e, 0x24, 0x6c,
        0x7d, 0x64, 0x7c, 0x61, 0x26, 0x67, 0x2a, 0x53,
        0x73, 0x60, 0x28, 0x2d, 0x6b, 0x44, 0x54, 0x61,
        0x34, 0x53, 0x22, 0x59, 0x6d, 0x73, 0x56, 0x55,
        0x25, 0x2c, 0x38, 0x4a, 0x3b, 0x4e, 0x78, 0x46,
        0x54, 0x6e, 0x6d, 0x4f, 0x47, 0x4f, 0x4f, 0x5a,
        0x67, 0x77, 0x39, 0x66, 0x28, 0x29, 0x4e, 0x43,
        0x55, 0x6e, 0x60, 0x59, 0x28, 0x3b, 0x65, 0x62,
        0x61, 0x5a, 0x29, 0x6e, 0x79, 0x60, 0x41, 0x53,
        0x2f, 0x5d, 0x44, 0x36, 0x7b, 0x3e, 0x7c, 0x2b,
        0x77, 0x36, 0x70, 0x3f, 0x40, 0x55, 0x48, 0x67,
        0x4b, 0x4d, 0x5d, 0x51, 0x79, 0x76, 0x48, 0x4a,
        0x2d, 0x21, 0x60, 0x40, 0x46, 0x55, 0x7a, 0x60,
        0x22, 0x25, 0x3f, 0x4b, 0x54, 0x6a, 0x6a, 0x3c,
        0x77, 0x22, 0x5b, 0x43, 0x67, 0x58, 0x71, 0x22,
        0x79, 0x4b, 0x32, 0x61, 0x44, 0x4d, 0x6f, 0x42,
        0x33, 0x2d, 0x53, 0x35, 0x3d, 0x6f, 0x57, 0x48,
        0x33, 0x3b, 0x5a, 0x53, 0x3f, 0x4e, 0x3f, 0x6b,
        0x4c, 0x27, 0x26, 0x3b, 0x73, 0x49, 0x22, 0x55,
        0x79, 0x2f, 0x47, 0x2f, 0x55, 0x5a, 0x7a, 0x71,
        0x6c, 0x31, 0x43, 0x40, 0x56, 0x7b, 0x21, 0x7a,
        0x6d, 0x4c, 0x43, 0x5e, 0x38, 0x47, 0x29, 0x38,
        0x62, 0x49, 0x45, 0x78, 0x70, 0x2b, 0x2e, 0x65,
        0x47, 0x71, 0x58, 0x79, 0x39, 0x67, 0x7d, 0x6d,
        0x6a, 0x67, 0x4a, 0x71, 0x27, 0x35, 0x2a, 0x4c,
        0x3e, 0x58, 0x55, 0x30, 0x4d, 0x75, 0x77, 0x48,
        0x5f, 0x4b, 0x59, 0x34, 0x65, 0x68, 0x57, 0x59,
        0x63, 0x23, 0x47, 0x38, 0x47, 0x5e, 0x56, 0x28,
        0x79, 0x58, 0x3e, 0x39, 0x66, 0x77, 0x67, 0x33,
        0x29, 0x61, 0x24, 0x7d, 0x37, 0x44, 0x37, 0x67,
        0x3a, 0x58, 0x76, 0x21, 0x51, 0x59, 0x61, 0x73,
        0x66, 0x75, 0x71, 0x53, 0x4d, 0x24, 0x2d, 0x4b,
        0x29, 0x30, 0x32, 0x26, 0x59, 0x64, 0x27, 0x55,
        0x2c, 0x5a, 0x4c, 0x3c, 0x6c, 0x53, 0x56, 0x4b,
        0x3e, 0x55, 0x2e, 0x44, 0x38, 0x6b, 0x47, 0x76,
        0x2d, 0x2c, 0x3f, 0x4d, 0x22, 0x7b, 0x6d, 0x61,
        0x34, 0x6b, 0x50, 0x73, 0x28, 0x6d, 0x41, 0x71,
        0x21, 0x76, 0x52, 0x2a, 0x6d, 0x53, 0x2a, 0x74,
        0x28, 0x27, 0x62, 0x2a, 0x66, 0x25, 0x6e, 0x5e,
        0x37, 0x4f, 0x27, 0x72, 0x28, 0x47, 0x63, 0x6e,
        0x5a, 0x6a, 0x41, 0x35, 0x3a, 0x42, 0x3f, 0x27,
        0x75, 0x3e, 0x26, 0x3e, 0x6b, 0x55, 0x59, 0x60,
        0x24, 0x70, 0x49, 0x3c, 0x4e, 0x2c, 0x39, 0x7a,
        0x36, 0x6c, 0x27, 0x3e, 0x6a, 0x4a, 0x59, 0x5a,
        0x3e, 0x21, 0x73, 0x4e, 0x59, 0x6e, 0x3d, 0x32,
        0x27, 0x45, 0x49, 0x58, 0x7d, 0x37, 0x39, 0x77,
        0x28, 0x51, 0x79, 0x54, 0x2b, 0x78, 0x46, 0x5a,
        0x21, 0x75, 0x33, 0x21, 0x63, 0x5a, 0x7b, 0x3e,
        0x33, 0x4f, 0x67, 0x75, 0x3a, 0x50, 0x48, 0x60,
        0x26, 0x64, 0x76, 0x5c, 0x42, 0x5c, 0x72, 0x38,
        0x6c, 0x52, 0x21, 0x2b, 0x25, 0x6b, 0x7c, 0x6b,
        0x2d, 0x5e, 0x63, 0x2a, 0x4c, 0x26, 0x5b, 0x4c,
        0x58, 0x52, 0x51, 0x55, 0x31, 0x79, 0x6c, 0x53,
        0x62, 0x3a, 0x36, 0x46, 0x7a, 0x29, 0x27, 0x78,
        0x1a, 0xbf, 0x49, 0x74, 0x68, 0x24, 0x51, 0x44,
        0x5b, 0x3e, 0x34, 0x44, 0x29, 0x5e, 0x4f, 0x2a,
        0xe9, 0x3f, 0xf8, 0xff, 0xff, 0x52, 0x7d, 0x47,
        0x67, 0x40, 0x27, 0x5e, 0x47, 0x46, 0x6d, 0x72,
        0x5d, 0x49, 0x26, 0x45, 0x33, 0x6b, 0x4d, 0x4a,
        0x6f, 0x62, 0x60, 0x45, 0x62, 0x27, 0x27, 0x7d,
        0x6a, 0x41, 0x2c, 0x6c, 0x5b, 0x2a, 0x2b, 0x36,
        0x29, 0x58, 0x7a, 0x4c, 0x6e, 0x2d, 0x74, 0x5c,
        0x38, 0x22, 0x5f, 0x49, 0x63, 0x43, 0x5b, 0x67
    };
    uint32_t request2_len = sizeof(request2);

    TcpSession ssn;
    Packet *p[2];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|af, 26, d0|\"; distance:0; content:\"|80 98 6d|\"; "
        "distance:2; within:3; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 2; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if ((PacketAlertCheck(p[0], 1))) {
        printf("sid 1 matched but shouldn't have for packet 0: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[1]);
    if ((PacketAlertCheck(p[1], 1))) {
        printf("sid 1 matched but shouldn't have for packet 1: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 2);
    return result;
}


/**
 * \test Test the working of detection engien with respect to dce keywords.
 */
int DcePayloadTest13(void)
{
    int result = 0;
    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x2c, 0xfd, 0xb5, 0x00, 0x40, 0xaa, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x02,
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t response1[] = {
        0x05, 0x00, 0x02, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x30, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xf6, 0x72, 0x28, 0x9c,
        0xf0, 0x57, 0xd8, 0x11, 0xb0, 0x05, 0x00, 0x0c,
        0x29, 0x87, 0xea, 0xe9, 0x00, 0x00, 0x00, 0x00,
    };
    uint32_t response1_len = sizeof(response1);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0xa4, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xf6, 0x72, 0x28, 0x9c,
        0xf0, 0x57, 0xd8, 0x11, 0xb0, 0x05, 0x00, 0x0c,
        0x29, 0x87, 0xea, 0xe9, 0x5c, 0x00, 0x5c, 0x00,
        0xa8, 0xb9, 0x14, 0x00, 0x2e, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00,
        0x53, 0x00, 0x4f, 0x00, 0x46, 0x00, 0x54, 0x00,
        0x57, 0x00, 0x41, 0x00, 0x52, 0x00, 0x45, 0x00,
        0x5c, 0x00, 0x4d, 0x00, 0x69, 0x00, 0x63, 0x00,
        0x72, 0x00, 0x6f, 0x00, 0x73, 0x00, 0x6f, 0x00,
        0x66, 0x00, 0x74, 0x00, 0x5c, 0x00, 0x57, 0x00,
        0x69, 0x00, 0x6e, 0x00, 0x64, 0x00, 0x6f, 0x00,
        0x77, 0x00, 0x73, 0x00, 0x5c, 0x00, 0x43, 0x00,
        0x75, 0x00, 0x72, 0x00, 0x72, 0x00, 0x65, 0x00,
        0x6e, 0x00, 0x74, 0x00, 0x56, 0x00, 0x65, 0x00,
        0x72, 0x00, 0x73, 0x00, 0x69, 0x00, 0x6f, 0x00,
        0x6e, 0x00, 0x5c, 0x00, 0x52, 0x00, 0x75, 0x00,
        0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00,
    };
    uint32_t request2_len = sizeof(request2);

    uint8_t response2[] = {
        0x05, 0x00, 0x02, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x30, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xf7, 0x72, 0x28, 0x9c,
        0xf0, 0x57, 0xd8, 0x11, 0xb0, 0x05, 0x00, 0x0c,
        0x29, 0x87, 0xea, 0xe9, 0x00, 0x00, 0x00, 0x00,
    };
    uint32_t response2_len = sizeof(response2);

    uint8_t request3[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x70, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x58, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xf7, 0x72, 0x28, 0x9c,
        0xf0, 0x57, 0xd8, 0x11, 0xb0, 0x05, 0x00, 0x0c,
        0x29, 0x87, 0xea, 0xe9, 0x0c, 0x00, 0x0c, 0x00,
        0x98, 0xda, 0x14, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x4f, 0x00, 0x73, 0x00, 0x61, 0x00, 0x33, 0x00,
        0x32, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x18, 0x00, 0x00, 0x00, 0x4e, 0x00, 0x54, 0x00,
        0x4f, 0x00, 0x53, 0x00, 0x41, 0x00, 0x33, 0x00,
        0x32, 0x00, 0x2e, 0x00, 0x45, 0x00, 0x58, 0x00,
        0x45, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00,
    };
    uint32_t request3_len = sizeof(request3);

    uint8_t response3[] = {
        0x05, 0x00, 0x02, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x1c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
    };
    uint32_t response3_len = sizeof(response3);

    TcpSession ssn;
    Packet *p[8];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    /* let the 7 and the 8th packet be dummy packets the client sends to the server */
    for (i = 0; i < 8; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }
    p[1]->flowflags |= FLOW_PKT_TOCLIENT;
    p[1]->flowflags &=~ FLOW_PKT_TOSERVER;
    p[3]->flowflags |= FLOW_PKT_TOCLIENT;
    p[3]->flowflags &=~ FLOW_PKT_TOSERVER;
    p[5]->flowflags |= FLOW_PKT_TOCLIENT;
    p[5]->flowflags &=~ FLOW_PKT_TOSERVER;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if (!(PacketAlertCheck(p[0], 1))) {
        printf("sid 1 didn't match but should have for packet 0: ");
        goto end;
    }

    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[6]);
    if ((PacketAlertCheck(p[6], 1))) {
        printf("sid 1 matched but shouldn't have for packet 6: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, response1, response1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[1]);
    if ((PacketAlertCheck(p[1], 1))) {
        printf("sid 1 matched but shouldn't have for packet 1: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    /* we should have a match for the sig once again for the same flow, since
     * the detection engine state for the flow has been reset because of a
     * fresh transaction */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[2]);
    if (!(PacketAlertCheck(p[2], 1))) {
        printf("sid 1 didn't match but should have for packet 2: ");
        goto end;
    }

    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[7]);
    if ((PacketAlertCheck(p[7], 1))) {
        printf("sid 1 matched but shouldn't have for packet 7: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, response2, response2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[3]);
    if ((PacketAlertCheck(p[3], 1))) {
        printf("sid 1 matched but shouldn't have for packet 3: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request3, request3_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    /* we should have a match for the sig once again for the same flow, since
     * the detection engine state for the flow has been reset because of a
     * fresh transaction */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[4]);
    if (!(PacketAlertCheck(p[4], 1))) {
        printf("sid 1 didn't match but should have for packet 4: ");
        goto end;
    }

    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, response3, response3_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[5]);
    if ((PacketAlertCheck(p[5], 1))) {
        printf("sid 1 matched but shouldn't have for packet 5: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 8);
    return result;
}

/**
 * \test Test the working of detection engien with respect to dce keywords.
 */
int DcePayloadTest14(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x76, 0x7e, 0x32, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x5c, 0x00, 0x5c, 0x00, 0x31, 0x00, 0x37, 0x00,
        0x31, 0x00, 0x2e, 0x00, 0x37, 0x00, 0x31, 0x00,
        0x2e, 0x00, 0x38, 0x00, 0x34, 0x00, 0x2e, 0x00,
        0x36, 0x00, 0x37, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x84, 0xf9, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0x14, 0xfa, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x81, 0xbb, 0x7a, 0x36, 0x44, 0x98, 0xf1, 0x35,
        0xad, 0x32, 0x98, 0xf0, 0x38, 0x00, 0x10, 0x03,
        0x02, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_len = sizeof(bind);

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x74, 0x73, 0x00, 0x00,
        0x0d, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x6e, 0x74, 0x73, 0x76, 0x63, 0x73, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x40, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1b, 0x00,
        0x64, 0x7e, 0x32, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x5c, 0x5c, 0x31, 0x37, 0x31, 0x2e, 0x37, 0x31,
        0x2e, 0x38, 0x34, 0x2e, 0x36, 0x37, 0x00, 0x8a,
        0x00, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x0f, 0x00
    };
    uint32_t request2_len = sizeof(request2);

    uint8_t response2[] = {
        0x05, 0x00, 0x02, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x30, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xd2, 0xc4, 0x88, 0x14,
        0xef, 0x31, 0xbb, 0x4d, 0xa8, 0x13, 0xb7, 0x1b,
        0x47, 0x49, 0xb5, 0xd7, 0x00, 0x00, 0x00, 0x00
    };
    uint32_t response2_len = sizeof(response2);

    TcpSession ssn;
    Packet *p[6];
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;
    int i = 0;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    for (i = 0; i < 6; i++) {
        p[i] = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
        p[i]->flow = &f;
        p[i]->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
        p[i]->flowflags |= FLOW_PKT_TOSERVER;
        p[i]->flowflags |= FLOW_PKT_ESTABLISHED;
    }
    p[3]->flowflags |= FLOW_PKT_TOCLIENT;
    p[3]->flowflags &=~ FLOW_PKT_TOSERVER;
    p[5]->flowflags |= FLOW_PKT_TOCLIENT;
    p[5]->flowflags &=~ FLOW_PKT_TOSERVER;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[0]);
    if (!(PacketAlertCheck(p[0], 1))) {
        printf("sid 1 didn't match but should have for packet 0: ");
        goto end;
    }

    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[1]);
    if ((PacketAlertCheck(p[1], 1))) {
        printf("sid 1 matched but shouldn't have for packet 1: ");
        goto end;
    }

    /* bind */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, bind, bind_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[2]);
    if ((PacketAlertCheck(p[2], 1))) {
        printf("sid 1 matched but shouldn't have for packet 2: ");
        goto end;
    }

    /* bind_ack.  A new transaction initiation */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, bind_ack, bind_ack_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[3]);
    if ((PacketAlertCheck(p[3], 1))) {
        printf("sid 1 matched but shouldn't have for packet 3: ");
        goto end;
    }

    /* we should have a match for the sig once again for the same flow, since
     * the detection engine state for the flow has been reset because of a
     * fresh transaction */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request2, request2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[4]);
    if (!(PacketAlertCheck(p[4], 1))) {
        printf("sid 1 didn't match but should have for packet 4: ");
        goto end;
    }

    /* response */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOCLIENT, response2, response2_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p[5]);
    if ((PacketAlertCheck(p[5], 1))) {
        printf("sid 1 matched but shouldn't have for packet 5: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(p, 6);
    return result;
}

/**
 * \test Test the working of byte_test endianness.
 */
int DcePayloadTest15(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x76, 0x7e, 0x32, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x5c, 0x00, 0x5c, 0x00, 0x31, 0x00, 0x37, 0x00,
        0x31, 0x00, 0x2e, 0x00, 0x37, 0x00, 0x31, 0x00,
        0x2e, 0x00, 0x38, 0x00, 0x34, 0x00, 0x2e, 0x00,
        0x36, 0x00, 0x37, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x84, 0xf9, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0x14, 0xfa, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00
    };
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_test:2,=,14080,0,relative,dce; sid:1;)";
    char *sig2 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_test:2,=,46,5,relative,dce; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;
    s->next = SigInit(de_ctx, sig2);
    if (s->next == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if (!(PacketAlertCheck(p, 1))) {
        printf("sid 1 didn't match but should have for packet: ");
        goto end;
    }
    if (!(PacketAlertCheck(p, 2))) {
        printf("sid 2 didn't match but should have for packet: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test the working of byte_test endianness.
 */
int DcePayloadTest16(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x76, 0x7e, 0x32, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x5c, 0x00, 0x5c, 0x00, 0x31, 0x00, 0x37, 0x00,
        0x31, 0x00, 0x2e, 0x00, 0x37, 0x00, 0x31, 0x00,
        0x2e, 0x00, 0x38, 0x00, 0x34, 0x00, 0x2e, 0x00,
        0x36, 0x00, 0x37, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x84, 0xf9, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0x14, 0xfa, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00
    };
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_test:2,=,55,0,relative; sid:1;)";
    char *sig2 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_test:2,=,11776,5,relative; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;
    s->next = SigInit(de_ctx, sig2);
    if (s->next == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if (!(PacketAlertCheck(p, 1))) {
        printf("sid 1 didn't match but should have for packet: ");
        goto end;
    }
    if (!(PacketAlertCheck(p, 2))) {
        printf("sid 2 didn't match but should have for packet: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test the working of byte_test endianness.
 */
int DcePayloadTest17(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x76, 0x7e, 0x32, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x5c, 0x00, 0x5c, 0x00, 0x31, 0x00, 0x37, 0x00,
        0x31, 0x00, 0x2e, 0x00, 0x37, 0x00, 0x31, 0x00,
        0x2e, 0x00, 0x38, 0x00, 0x34, 0x00, 0x2e, 0x00,
        0x36, 0x00, 0x37, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x84, 0xf9, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0x14, 0xfa, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00
    };
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_test:2,=,55,0,relative,big; sid:1;)";
    char *sig2 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_test:2,=,46,5,relative,little; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;
    s->next = SigInit(de_ctx, sig2);
    if (s->next == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if (!(PacketAlertCheck(p, 1))) {
        printf("sid 1 didn't match but should have for packet: ");
        goto end;
    }
    if (!(PacketAlertCheck(p, 2))) {
        printf("sid 2 didn't match but should have for packet: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test the working of byte_jump endianness.
 */
int DcePayloadTest18(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x76, 0x7e, 0x32, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x5c, 0x00, 0x5c, 0x00, 0x31, 0x03, 0x00, 0x03,
        0x00, 0x00, 0x2e, 0x00, 0x37, 0x00, 0x31, 0x00,
        0x2e, 0x00, 0x38, 0x00, 0x34, 0x00, 0x2e, 0x00,
        0x36, 0x00, 0x37, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x84, 0xf9, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0x14, 0xfa, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00
    };
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_jump:2,0,relative,dce; byte_test:2,=,46,0,relative,dce; sid:1;)";
    char *sig2 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_jump:2,2,relative,dce; byte_test:2,=,14080,0,relative; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;
    s->next = SigInit(de_ctx, sig2);
    if (s->next == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if (!(PacketAlertCheck(p, 1))) {
        printf("sid 1 didn't match but should have for packet: ");
        goto end;
    }
    if (!(PacketAlertCheck(p, 2))) {
        printf("sid 2 didn't match but should have for packet: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test the working of byte_jump endianness.
 */
int DcePayloadTest19(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x76, 0x7e, 0x32, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x5c, 0x00, 0x5c, 0x00, 0x31, 0x00, 0x03, 0x00,
        0x03, 0x00, 0x2e, 0x00, 0x37, 0x00, 0x31, 0x00,
        0x2e, 0x00, 0x38, 0x00, 0x34, 0x00, 0x2e, 0x00,
        0x36, 0x00, 0x37, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x84, 0xf9, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0x14, 0xfa, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00
    };
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_jump:2,0,relative; byte_test:2,=,46,0,relative,dce; sid:1;)";
    char *sig2 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_jump:2,2,relative; byte_test:2,=,14080,0,relative; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;
    s->next = SigInit(de_ctx, sig2);
    if (s->next == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if (!(PacketAlertCheck(p, 1))) {
        printf("sid 1 didn't match but should have for packet: ");
        goto end;
    }
    if (!(PacketAlertCheck(p, 2))) {
        printf("sid 2 didn't match but should have for packet: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test the working of byte_jump endianness.
 */
int DcePayloadTest20(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x76, 0x7e, 0x32, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
        0x5c, 0x00, 0x5c, 0x00, 0x31, 0x00, 0x03, 0x03,
        0x00, 0x00, 0x2e, 0x00, 0x37, 0x00, 0x31, 0x00,
        0x2e, 0x00, 0x38, 0x00, 0x34, 0x00, 0x2e, 0x00,
        0x36, 0x00, 0x37, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x84, 0xf9, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
        0x14, 0xfa, 0x7f, 0x01, 0x00, 0x00, 0x00, 0x00
    };
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_jump:2,0,relative,big; byte_test:2,=,46,0,relative,dce; sid:1;)";
    char *sig2 = "alert tcp any any -> any any "
        "(dce_stub_data; content:\"|5c 00 5c 00 31|\"; distance:0; "
        "byte_jump:2,2,little,relative; byte_test:2,=,14080,0,relative; sid:2;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;
    s->next = SigInit(de_ctx, sig2);
    if (s->next == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if (!(PacketAlertCheck(p, 1))) {
        printf("sid 1 didn't match but should have for packet: ");
        goto end;
    }
    if (!(PacketAlertCheck(p, 2))) {
        printf("sid 2 didn't match but should have for packet: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test the working of consecutive relative matches.
 */
int DcePayloadTest21(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x6e, 0x6f, 0x77, 0x20, 0x74, 0x68, 0x69, 0x73, /* "now this" */
        0x20, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20, 0x62, /* " is is b" */
        0x69, 0x67, 0x20, 0x62, 0x69, 0x67, 0x20, 0x73, /* "ig big s" */
        0x74, 0x72, 0x69, 0x6e, 0x67, 0x20, 0x6e, 0x6f, /* "tring no" */
        0x77 };                                         /* "w" */
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(msg:\"testing dce consecutive relative matches\"; dce_stub_data; "
        "content:\"this\"; distance:0; content:\"is\"; within:6; content:\"big\"; within:8; "
        "content:\"string\"; within:8; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if (!(PacketAlertCheck(p, 1))) {
        printf("sid 1 didn't match but should have for packet: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test the working of consecutive relative matches.
 */
int DcePayloadTest22(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x6e, 0x6f, 0x77, 0x20, 0x74, 0x68, 0x69, 0x73,   /* "now this" */
        0x20, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20, 0x69,   /* " is is i" */
        0x73, 0x20, 0x62, 0x69, 0x67, 0x20, 0x62, 0x69,   /* "s big bi" */
        0x67, 0x20, 0x62, 0x69, 0x67, 0x20, 0x73, 0x74,   /* "g big st" */
        0x72, 0x69, 0x6e, 0x67, 0x20, 0x6e, 0x6f, 0x77 }; /* "ring now" */
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(msg:\"testing dce consecutive relative matches\"; dce_stub_data; "
        "content:\"this\"; distance:0; content:\"is\"; within:9; content:\"big\"; within:12; "
        "content:\"string\"; within:8; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if (!(PacketAlertCheck(p, 1))) {
        printf("sid 1 didn't match but should have for packet: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test the working of consecutive relative matches.
 */
int DcePayloadTest23(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x74, 0x68, 0x69, 0x73, 0x20, 0x74, 0x68, 0x69, /* "this thi" */
        0x73, 0x20, 0x6e, 0x6f, 0x77, 0x20, 0x69, 0x73, /* "s now is" */
        0x20, 0x69, 0x73, 0x20, 0x20, 0x20, 0x20, 0x20, /* " is     " */
        0x62, 0x69, 0x67, 0x20, 0x73, 0x74, 0x72, 0x69, /* "big stri" */
        0x6e, 0x67, 0x20, 0x6e, 0x6f, 0x77 };           /* "ng now"   */
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(msg:\"testing dce consecutive relative matches\"; dce_stub_data; "
        "content:\"now\"; distance:0; content:\"this\"; distance:-20; "
        "content:\"is\"; within:12; content:\"big\"; within:8; "
        "content:\"string\"; within:8; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if (!(PacketAlertCheck(p, 1))) {
        printf("sid 1 didn't match but should have for packet: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test the working of consecutive relative matches with offset.
 */
int DcePayloadTest24(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, /* "        " */
        0x20, 0x74, 0x68, 0x75, 0x73, 0x20, 0x74, 0x68, /* " thus th" */
        0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x61, 0x20, /* "us is a " */
        0x62, 0x69, 0x67 };                             /* "big" */
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(msg:\"testing dce consecutive relative matches\"; dce_stub_data; "
        "content:\"thus\"; distance:0; offset:8; content:\"is\"; within:6; "
        "content:\"big\"; within:8; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if (!(PacketAlertCheck(p, 1))) {
        printf("sid 1 didn't match but should have for packet: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest25(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "content:\"one\"; content:\"two\"; "
                                   "content:\"three\"; within:10; "
                                   "content:\"four\"; distance:4; "
                                   "dce_iface:3919286a-b10c-11d0-9ba8-00c04fd92ef5; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] != NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "three", 5) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "four", 4) == 0);
    if (result == 0)
        goto end;

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest26(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_stub_data; "
                                   "content:\"one\"; "
                                   "content:\"two\"; "
                                   "content:\"three\"; within:5; "
                                   "content:\"four\"; distance:10; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] != NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "three", 5) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("four failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "four", 4) == 0);
    if (result == 0)
        goto end;

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest27(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_stub_data; "
                                   "content:\"one\"; distance:10; within:5; "
                                   "content:\"two\"; within:5;"
                                   "content:\"three\"; within:5; "
                                   "content:\"four\"; distance:10; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] != NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "three", 5) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("four failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "four", 4) == 0);
    if (result == 0)
        goto end;

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest28(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_stub_data; "
                                   "content:\"one\"; distance:10; within:5; "
                                   "content:\"two\"; within:5;"
                                   "content:\"three\";"
                                   "content:\"four\";"
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "three", 5) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("four failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "four", 4) == 0);
    if (result == 0)
        goto end;

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest29(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectPcreData *pd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_stub_data; "
                                   "pcre:/boom/; "
                                   "content:\"one\"; distance:10; within:5; "
                                   "content:\"two\"; within:5;"
                                   "content:\"three\";"
                                   "content:\"four\";"
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] != NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_PCRE) {
        result = 0;
        goto end;
    }
    pd = (DetectPcreData *)sm->ctx;
    if (pd->flags & DETECT_PCRE_RAWBYTES ||
        pd->flags & DETECT_PCRE_RELATIVE) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "three", 5) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("four failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "four", 4) == 0);
    if (result == 0)
        goto end;

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest30(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectBytejumpData *bd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_stub_data; "
                                   "byte_jump:2,5; "
                                   "content:\"one\"; distance:10; within:5; "
                                   "content:\"two\"; within:5;"
                                   "content:\"three\";"
                                   "content:\"four\";"
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] != NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_BYTEJUMP) {
        result = 0;
        goto end;
    }
    bd = (DetectBytejumpData *)sm->ctx;
    if (bd->flags & DETECT_BYTEJUMP_BEGIN ||
        bd->flags & DETECT_BYTEJUMP_LITTLE ||
        bd->flags & DETECT_BYTEJUMP_BIG ||
        bd->flags & DETECT_BYTEJUMP_STRING ||
        bd->flags & DETECT_BYTEJUMP_RELATIVE ||
        bd->flags & DETECT_BYTEJUMP_ALIGN ||
        bd->flags & DETECT_BYTEJUMP_DCE ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "three", 5) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("four failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "four", 4) == 0);
    if (result == 0)
        goto end;

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest31(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectBytejumpData *bd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_stub_data; "
                                   "byte_jump:2,5,relative; "
                                   "content:\"one\"; distance:10; within:5; "
                                   "content:\"two\"; within:5;"
                                   "content:\"three\";"
                                   "content:\"four\";"
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_BYTEJUMP) {
        result = 0;
        goto end;
    }
    bd = (DetectBytejumpData *)sm->ctx;
    if (bd->flags & DETECT_BYTEJUMP_BEGIN ||
        bd->flags & DETECT_BYTEJUMP_LITTLE ||
        bd->flags & DETECT_BYTEJUMP_BIG ||
        bd->flags & DETECT_BYTEJUMP_STRING ||
        !(bd->flags & DETECT_BYTEJUMP_RELATIVE) ||
        bd->flags & DETECT_BYTEJUMP_ALIGN ||
        bd->flags & DETECT_BYTEJUMP_DCE ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "three", 5) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("four failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "four", 4) == 0);
    if (result == 0)
        goto end;

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest32(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectBytejumpData *bd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_stub_data; "
                                   "byte_jump:2,5,relative; "
                                   "content:\"one\"; distance:10; within:5; "
                                   "content:\"two\"; within:5;"
                                   "content:\"three\";"
                                   "content:\"four\"; within:4; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_BYTEJUMP) {
        result = 0;
        goto end;
    }
    bd = (DetectBytejumpData *)sm->ctx;
    if (bd->flags & DETECT_BYTEJUMP_BEGIN ||
        bd->flags & DETECT_BYTEJUMP_LITTLE ||
        bd->flags & DETECT_BYTEJUMP_BIG ||
        bd->flags & DETECT_BYTEJUMP_STRING ||
        !(bd->flags & DETECT_BYTEJUMP_RELATIVE) ||
        bd->flags & DETECT_BYTEJUMP_ALIGN ||
        bd->flags & DETECT_BYTEJUMP_DCE ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "three", 5) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("four failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "four", 4) == 0);
    if (result == 0)
        goto end;

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest33(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectPcreData *pd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_stub_data; "
                                   "pcre:/boom/R; "
                                   "content:\"one\"; distance:10; within:5; "
                                   "content:\"two\"; within:5;"
                                   "content:\"three\";"
                                   "content:\"four\"; distance:5;"
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_PCRE) {
        result = 0;
        goto end;
    }
    pd = (DetectPcreData *)sm->ctx;
    if ( pd->flags & DETECT_PCRE_RAWBYTES ||
         !(pd->flags & DETECT_PCRE_RELATIVE)) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "three", 5) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("four failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "four", 4) == 0);
    if (result == 0)
        goto end;

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest34(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectPcreData *pd = NULL;
    DetectBytejumpData *bd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_iface:12345678-1234-1234-1234-123456789012; "
                                   "dce_opnum:10; dce_stub_data; "
                                   "pcre:/boom/R; "
                                   "byte_jump:1,2,relative,align,dce; "
                                   "content:\"one\"; within:4; distance:8; "
                                   "content:\"two\"; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_PCRE) {
        result = 0;
        goto end;
    }
    pd = (DetectPcreData *)sm->ctx;
    if ( pd->flags & DETECT_PCRE_RAWBYTES ||
         !(pd->flags & DETECT_PCRE_RELATIVE)) {
        result = 0;
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_BYTEJUMP) {
        result = 0;
        goto end;
    }
    bd = (DetectBytejumpData *)sm->ctx;
    if (bd->flags & DETECT_BYTEJUMP_BEGIN ||
        bd->flags & DETECT_BYTEJUMP_LITTLE ||
        bd->flags & DETECT_BYTEJUMP_BIG ||
        bd->flags & DETECT_BYTEJUMP_STRING ||
        !(bd->flags & DETECT_BYTEJUMP_RELATIVE) ||
        !(bd->flags & DETECT_BYTEJUMP_ALIGN) ||
        !(bd->flags & DETECT_BYTEJUMP_DCE) ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest35(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectBytetestData *bd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_iface:12345678-1234-1234-1234-123456789012; "
                                   "dce_opnum:10; dce_stub_data; "
                                   "byte_test:1,=,0,0,relative,dce; "
                                   "content:\"one\"; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_BYTETEST) {
        result = 0;
        goto end;
    }
    bd = (DetectBytetestData *)sm->ctx;
    if (bd->flags & DETECT_BYTETEST_LITTLE ||
        bd->flags & DETECT_BYTETEST_BIG ||
        bd->flags & DETECT_BYTETEST_STRING ||
        !(bd->flags & DETECT_BYTETEST_RELATIVE) ||
        !(bd->flags & DETECT_BYTETEST_DCE) ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    result &= (sm->next == NULL);

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest36(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectIsdataatData *isd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_iface:12345678-1234-1234-1234-123456789012; "
                                   "dce_opnum:10; dce_stub_data; "
                                   "isdataat:10,relative; "
                                   "content:\"one\"; within:4; distance:8; "
                                   "content:\"two\"; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_ISDATAAT) {
        result = 0;
        goto end;
    }
    isd = (DetectIsdataatData *)sm->ctx;
    if ( isd->flags & ISDATAAT_RAWBYTES ||
         !(isd->flags & ISDATAAT_RELATIVE)) {
        result = 0;
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest37(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectBytejumpData *bjd = NULL;
    DetectBytetestData *btd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_iface:12345678-1234-1234-1234-123456789012; "
                                   "dce_opnum:10; dce_stub_data; "
                                   "byte_jump:1,2,relative,align,dce; "
                                   "byte_test:1,=,2,0,relative,dce; "
                                   "content:\"one\"; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_BYTEJUMP) {
        result = 0;
        goto end;
    }
    bjd = (DetectBytejumpData *)sm->ctx;
    if (bjd->flags & DETECT_BYTEJUMP_BEGIN ||
        bjd->flags & DETECT_BYTEJUMP_LITTLE ||
        bjd->flags & DETECT_BYTEJUMP_BIG ||
        bjd->flags & DETECT_BYTEJUMP_STRING ||
        !(bjd->flags & DETECT_BYTEJUMP_RELATIVE) ||
        !(bjd->flags & DETECT_BYTEJUMP_ALIGN) ||
        !(bjd->flags & DETECT_BYTEJUMP_DCE) ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_BYTETEST) {
        result = 0;
        goto end;
    }
    btd = (DetectBytetestData *)sm->ctx;
    if (btd->flags & DETECT_BYTETEST_LITTLE ||
        btd->flags & DETECT_BYTETEST_BIG ||
        btd->flags & DETECT_BYTETEST_STRING ||
        !(btd->flags & DETECT_BYTETEST_RELATIVE) ||
        !(btd->flags & DETECT_BYTETEST_DCE) ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    result &= (sm->next == NULL);

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest38(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectPcreData *pd = NULL;
    DetectBytejumpData *bjd = NULL;
    DetectBytetestData *btd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_iface:12345678-1234-1234-1234-123456789012; "
                                   "dce_opnum:10; dce_stub_data; "
                                   "pcre:/boom/R; "
                                   "byte_jump:1,2,relative,align,dce; "
                                   "byte_test:1,=,2,0,relative,dce; "
                                   "content:\"one\"; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_PCRE) {
        result = 0;
        goto end;
    }
    pd = (DetectPcreData *)sm->ctx;
    if ( pd->flags & DETECT_PCRE_RAWBYTES ||
         !(pd->flags & DETECT_PCRE_RELATIVE) ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_BYTEJUMP) {
        result = 0;
        goto end;
    }
    bjd = (DetectBytejumpData *)sm->ctx;
    if (bjd->flags & DETECT_BYTEJUMP_BEGIN ||
        bjd->flags & DETECT_BYTEJUMP_LITTLE ||
        bjd->flags & DETECT_BYTEJUMP_BIG ||
        bjd->flags & DETECT_BYTEJUMP_STRING ||
        !(bjd->flags & DETECT_BYTEJUMP_RELATIVE) ||
        !(bjd->flags & DETECT_BYTEJUMP_ALIGN) ||
        !(bjd->flags & DETECT_BYTEJUMP_DCE) ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_BYTETEST) {
        result = 0;
        goto end;
    }
    btd = (DetectBytetestData *)sm->ctx;
    if (btd->flags & DETECT_BYTETEST_LITTLE ||
        btd->flags & DETECT_BYTETEST_BIG ||
        btd->flags & DETECT_BYTETEST_STRING ||
        !(btd->flags & DETECT_BYTETEST_RELATIVE) ||
        !(btd->flags & DETECT_BYTETEST_DCE) ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    result &= (sm->next == NULL);

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest39(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "content:\"one\"; "
                                   "dce_iface:12345678-1234-1234-1234-123456789012; "
                                   "dce_opnum:10; dce_stub_data; "
                                   "content:\"two\"; within:4; distance:8; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest40(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectBytetestData *btd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_iface:12345678-1234-1234-1234-123456789012; "
                                   "dce_opnum:10; dce_stub_data; "
                                   "content:\"one\"; within:10; "
                                   "content:\"two\"; distance:20; within:30; "
                                   "byte_test:1,=,2,0,relative,dce; "
                                   "content:\"three\"; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;


    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_BYTETEST) {
        result = 0;
        goto end;
    }
    btd = (DetectBytetestData *)sm->ctx;
    if (btd->flags & DETECT_BYTETEST_LITTLE ||
        btd->flags & DETECT_BYTETEST_BIG ||
        btd->flags & DETECT_BYTETEST_STRING ||
        !(btd->flags & DETECT_BYTETEST_RELATIVE) ||
        !(btd->flags & DETECT_BYTETEST_DCE) ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    result &= (sm->next == NULL);

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "three", 5) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest41(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectBytetestData *btd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_iface:12345678-1234-1234-1234-123456789012; "
                                   "dce_opnum:10; dce_stub_data; "
                                   "content:\"one\"; within:10; "
                                   "content:\"two\"; "
                                   "byte_test:1,=,2,0,relative,dce; "
                                   "content:\"three\"; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_BYTETEST) {
        result = 0;
        goto end;
    }
    btd = (DetectBytetestData *)sm->ctx;
    if (btd->flags & DETECT_BYTETEST_LITTLE ||
        btd->flags & DETECT_BYTETEST_BIG ||
        btd->flags & DETECT_BYTETEST_STRING ||
        !(btd->flags & DETECT_BYTETEST_RELATIVE) ||
        !(btd->flags & DETECT_BYTETEST_DCE) ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "three", 5) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test the working of consecutive relative matches with a negated content.
 */
int DcePayloadTest42(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x77, 0x65, 0x20, 0x6e, 0x65, 0x65, 0x64, 0x20, /* "we need " */
        0x74, 0x6f, 0x20, 0x66, 0x69, 0x78, 0x20, 0x74, /* "to fix t" */
        0x68, 0x69, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, /* "his and " */
        0x79, 0x65, 0x73, 0x20, 0x66, 0x69, 0x78, 0x20, /* "yes fix " */
        0x74, 0x68, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x77  /* "this now" */
    };
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(msg:\"testing dce consecutive relative matches\"; dce_stub_data; "
        "content:\"fix\"; distance:0; content:\"this\"; within:6; "
        "content:!\"and\"; distance:0; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if ((PacketAlertCheck(p, 1))) {
        printf("sid 1 matched but shouldn't have for packet: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test the working of consecutive relative pcres.
 */
int DcePayloadTest43(void)
{
    int result = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x68, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x74, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20,
        0x61, 0x20, 0x73, 0x75, 0x70, 0x65, 0x72, 0x20,
        0x64, 0x75, 0x70, 0x65, 0x72, 0x20, 0x6e, 0x6f,
        0x76, 0x61, 0x20, 0x69, 0x6e, 0x20, 0x73, 0x75,
        0x70, 0x65, 0x72, 0x20, 0x6e, 0x6f, 0x76, 0x61,
        0x20, 0x6e, 0x6f, 0x77
    };
    uint32_t request1_len = sizeof(request1);

    TcpSession ssn;
    Packet *p = NULL;
    ThreadVars tv;
    DetectEngineCtx *de_ctx = NULL;
    DetectEngineThreadCtx *det_ctx = NULL;
    Flow f;
    int r;

    char *sig1 = "alert tcp any any -> any any "
        "(msg:\"testing dce consecutive relative matches\"; dce_stub_data; "
        "pcre:/super/R; content:\"nova\"; within:7; sid:1;)";

    Signature *s;

    memset(&tv, 0, sizeof(ThreadVars));
    memset(&f, 0, sizeof(Flow));
    memset(&ssn, 0, sizeof(TcpSession));

    p = UTHBuildPacket(NULL, 0, IPPROTO_TCP);
    p->flow = &f;
    p->flags |= PKT_HAS_FLOW|PKT_STREAM_EST;
    p->flowflags |= FLOW_PKT_TOSERVER;
    p->flowflags |= FLOW_PKT_ESTABLISHED;

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;
    f.src.family = AF_INET;
    f.dst.family = AF_INET;
    f.alproto = ALPROTO_DCERPC;

    StreamTcpInitConfig(TRUE);
    FlowL7DataPtrInit(&f);

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;
    de_ctx->flags |= DE_QUIET;

    de_ctx->sig_list = SigInit(de_ctx, sig1);
    s = de_ctx->sig_list;
    if (s == NULL)
        goto end;

    SigGroupBuild(de_ctx);
    DetectEngineThreadCtxInit(&tv, (void *)de_ctx, (void *)&det_ctx);

    /* request 1 */
    r = AppLayerParse(&f, ALPROTO_DCERPC, STREAM_TOSERVER, request1, request1_len);
    if (r != 0) {
        printf("toserver chunk 1 returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    /* detection phase */
    SigMatchSignatures(&tv, de_ctx, det_ctx, p);
    if ( !(PacketAlertCheck(p, 1))) {
        printf("sid 1 didn't match but should have: ");
        goto end;
    }

    result = 1;

end:
    if (de_ctx != NULL) {
        SigGroupCleanup(de_ctx);
        SigCleanSignatures(de_ctx);

        DetectEngineThreadCtxDeinit(&tv, (void *)det_ctx);
        DetectEngineCtxFree(de_ctx);
    }

    FlowL7DataPtrFree(&f);
    StreamTcpFreeConfig(TRUE);

    UTHFreePackets(&p, 1);
    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest44(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectIsdataatData *isd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "content:\"one\"; "
                                   "dce_iface:12345678-1234-1234-1234-123456789012; "
                                   "dce_opnum:10; dce_stub_data; "
                                   "isdataat:10,relative; "
                                   "content:\"one\"; within:4; distance:8; "
                                   "content:\"two\"; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_ISDATAAT) {
        result = 0;
        goto end;
    }
    isd = (DetectIsdataatData *)sm->ctx;
    if ( isd->flags & ISDATAAT_RAWBYTES ||
         !(isd->flags & ISDATAAT_RELATIVE)) {
        result = 0;
        goto end;
    }

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        !(data->flags & DETECT_CONTENT_WITHIN) ||
        !(data->flags & DETECT_CONTENT_DISTANCE) ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_RELATIVE_NEXT ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        result = 0;
        printf("two failed\n");
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_RELATIVE_NEXT ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("three failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("two failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest45(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectBytejumpData *bjd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_iface:12345678-1234-1234-1234-123456789012; "
                                   "content:\"one\"; "
                                   "dce_opnum:10; dce_stub_data; "
                                   "byte_jump:1,2,relative,align,dce; "
                                   "content:\"two\"; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_BYTEJUMP) {
        result = 0;
        goto end;
    }
    bjd = (DetectBytejumpData *)sm->ctx;
    if (bjd->flags & DETECT_BYTEJUMP_BEGIN ||
        bjd->flags & DETECT_BYTEJUMP_LITTLE ||
        bjd->flags & DETECT_BYTEJUMP_BIG ||
        bjd->flags & DETECT_BYTEJUMP_STRING ||
        !(bjd->flags & DETECT_BYTEJUMP_RELATIVE) ||
        !(bjd->flags & DETECT_BYTEJUMP_ALIGN) ||
        !(bjd->flags & DETECT_BYTEJUMP_DCE) ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    result &= (sm->next == NULL);

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_RELATIVE_NEXT ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("one failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_RELATIVE_NEXT ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("two failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

/**
 * \test Test content for dce sig.
 */
int DcePayloadParseTest46(void)
{
    DetectEngineCtx *de_ctx = NULL;
    int result = 1;
    Signature *s = NULL;
    SigMatch *sm = NULL;
    DetectContentData *data = NULL;
    DetectBytetestData *btd = NULL;

    de_ctx = DetectEngineCtxInit();
    if (de_ctx == NULL)
        goto end;

    de_ctx->flags |= DE_QUIET;
    s = de_ctx->sig_list = SigInit(de_ctx, "alert tcp any any -> any any "
                                   "(msg:\"Testing bytejump_body\"; "
                                   "dce_iface:12345678-1234-1234-1234-123456789012; "
                                   "content:\"one\"; "
                                   "dce_opnum:10; dce_stub_data; "
                                   "byte_test:1,=,2,0,relative,dce; "
                                   "content:\"two\"; "
                                   "sid:1;)");
    if (de_ctx->sig_list == NULL) {
        result = 0;
        goto end;
    }

    if (s->sm_lists_tail[DETECT_SM_LIST_DMATCH] == NULL) {
        result = 0;
        goto end;
    }
    if (s->sm_lists_tail[DETECT_SM_LIST_PMATCH] == NULL) {
        result = 0;
        goto end;
    }

    sm = s->sm_lists[DETECT_SM_LIST_DMATCH];
    if (sm->type != DETECT_BYTETEST) {
        result = 0;
        goto end;
    }
    btd = (DetectBytetestData *)sm->ctx;
    if (btd->flags & DETECT_BYTETEST_LITTLE ||
        btd->flags & DETECT_BYTETEST_BIG ||
        btd->flags & DETECT_BYTETEST_STRING ||
        !(btd->flags & DETECT_BYTETEST_RELATIVE) ||
        !(btd->flags & DETECT_BYTETEST_DCE) ) {
        result = 0;
        printf("one failed\n");
        goto end;
    }

    result &= (sm->next == NULL);

    sm = s->sm_lists[DETECT_SM_LIST_PMATCH];
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_RELATIVE_NEXT ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("one failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "one", 3) == 0);
    if (result == 0)
        goto end;

    sm = sm->next;
    if (sm->type != DETECT_CONTENT) {
        result = 0;
        goto end;
    }
    data = (DetectContentData *)sm->ctx;
    if (data->flags & DETECT_CONTENT_RAWBYTES ||
        data->flags & DETECT_CONTENT_NOCASE ||
        data->flags & DETECT_CONTENT_WITHIN ||
        data->flags & DETECT_CONTENT_DISTANCE ||
        data->flags & DETECT_CONTENT_FAST_PATTERN ||
        data->flags & DETECT_CONTENT_RELATIVE_NEXT ||
        data->flags & DETECT_CONTENT_NEGATED ) {
        printf("two failed\n");
        result = 0;
        goto end;
    }
    result &= (strncmp((char *)data->content, "two", 3) == 0);
    if (result == 0)
        goto end;

    result &= (sm->next == NULL);

 end:
    SigGroupCleanup(de_ctx);
    SigCleanSignatures(de_ctx);
    DetectEngineCtxFree(de_ctx);

    return result;
}

#endif /* UNITTESTS */

void DcePayloadRegisterTests(void)
{

#ifdef UNITTESTS
    UtRegisterTest("DcePayloadTest01", DcePayloadTest01, 1);
    UtRegisterTest("DcePayloadTest02", DcePayloadTest02, 1);
    UtRegisterTest("DcePayloadTest03", DcePayloadTest03, 1);
    UtRegisterTest("DcePayloadTest04", DcePayloadTest04, 1);
    UtRegisterTest("DcePayloadTest05", DcePayloadTest05, 1);
    UtRegisterTest("DcePayloadTest06", DcePayloadTest06, 1);
    UtRegisterTest("DcePayloadTest07", DcePayloadTest07, 1);
    UtRegisterTest("DcePayloadTest08", DcePayloadTest08, 1);
    UtRegisterTest("DcePayloadTest09", DcePayloadTest09, 1);
    UtRegisterTest("DcePayloadTest10", DcePayloadTest10, 1);
    UtRegisterTest("DcePayloadTest11", DcePayloadTest11, 1);
    UtRegisterTest("DcePayloadTest12", DcePayloadTest12, 1);
    UtRegisterTest("DcePayloadTest13", DcePayloadTest13, 1);
    UtRegisterTest("DcePayloadTest14", DcePayloadTest14, 1);
    UtRegisterTest("DcePayloadTest15", DcePayloadTest15, 1);
    UtRegisterTest("DcePayloadTest16", DcePayloadTest16, 1);
    UtRegisterTest("DcePayloadTest17", DcePayloadTest17, 1);
    UtRegisterTest("DcePayloadTest18", DcePayloadTest18, 1);
    UtRegisterTest("DcePayloadTest19", DcePayloadTest19, 1);
    UtRegisterTest("DcePayloadTest20", DcePayloadTest20, 1);
    UtRegisterTest("DcePayloadTest21", DcePayloadTest21, 1);
    UtRegisterTest("DcePayloadTest22", DcePayloadTest22, 1);
    UtRegisterTest("DcePayloadTest23", DcePayloadTest23, 1);
    UtRegisterTest("DcePayloadTest24", DcePayloadTest24, 1);

    UtRegisterTest("DcePayloadParseTest25", DcePayloadParseTest25, 1);
    UtRegisterTest("DcePayloadParseTest26", DcePayloadParseTest26, 1);
    UtRegisterTest("DcePayloadParseTest27", DcePayloadParseTest27, 1);
    UtRegisterTest("DcePayloadParseTest28", DcePayloadParseTest28, 1);
    UtRegisterTest("DcePayloadParseTest29", DcePayloadParseTest29, 1);
    UtRegisterTest("DcePayloadParseTest30", DcePayloadParseTest30, 1);
    UtRegisterTest("DcePayloadParseTest31", DcePayloadParseTest31, 1);
    UtRegisterTest("DcePayloadParseTest32", DcePayloadParseTest32, 1);
    UtRegisterTest("DcePayloadParseTest33", DcePayloadParseTest33, 1);
    UtRegisterTest("DcePayloadParseTest34", DcePayloadParseTest34, 1);
    UtRegisterTest("DcePayloadParseTest35", DcePayloadParseTest35, 1);
    UtRegisterTest("DcePayloadParseTest36", DcePayloadParseTest36, 1);
    UtRegisterTest("DcePayloadParseTest37", DcePayloadParseTest37, 1);
    UtRegisterTest("DcePayloadParseTest38", DcePayloadParseTest38, 1);
    UtRegisterTest("DcePayloadParseTest39", DcePayloadParseTest39, 1);
    UtRegisterTest("DcePayloadParseTest40", DcePayloadParseTest40, 1);
    UtRegisterTest("DcePayloadParseTest41", DcePayloadParseTest41, 1);

    UtRegisterTest("DcePayloadTest42", DcePayloadTest42, 1);
    UtRegisterTest("DcePayloadTest43", DcePayloadTest43, 1);

    UtRegisterTest("DcePayloadParseTest44", DcePayloadParseTest44, 1);
    UtRegisterTest("DcePayloadParseTest45", DcePayloadParseTest45, 1);
    UtRegisterTest("DcePayloadParseTest46", DcePayloadParseTest46, 1);
#endif /* UNITTESTS */

    return;
}
