#===================================
#          prefs_dialog
#  Copyright 2010 - Nathan Osman
#
#   Provides a means by which a
# user may modify preferences for
#       the application.
#
#   StackApplet is released under
#        the MIT license
#===================================

# For translation support
from stackapplet import _

# we need this for the next line
import os

DESKTOP_FILE_NAME     = os.path.join(os.path.expanduser("~"),".config/autostart/stackapplet.desktop")
DESKTOP_FILE_CONTENTS = '''[Desktop Entry]
Type=Application
Exec=python /usr/share/stackapplet/stackapplet.py
Icon=/usr/share/pixmaps/stackapplet.png
Terminal=false
Comment=%s
Name=%s
Categories=Utility;''' % (_("a panel indicator for monitoring StackExchange sites"), _("StackApplet"))

import gobject
import gtk

# import our preference storage engine
import config_store

# import our API module
import stack_api

# and import the dialog that makes
# searching for users easier
import select_user

# Note that we are deriving from GObject
# this is necessary to allow us to send
# signals from this object
class prefs_dialog (gobject.GObject):
    
    # Constructor for the class
    # - initializes the widgets
    
    def __init__ (self):
        
        self.__gobject_init__()
        
        # Load the preferences dialog
        self.builder = gtk.Builder()
        
        # We need to tell our builder instance what we're
	# doing translation-wise
	self.builder.set_translation_domain("stackapplet")
        
        self.builder.add_from_file("/usr/share/stackapplet/stackapplet.glade")
        
        # Set up the signal handlers
        signal_dict = { "on_button_ok_clicked"     : self.OnOK,
                        "on_button_cancel_clicked" : self.OnCancel,
                        "on_button_add_clicked"    : self.OnAdd,
                        "on_button_remove_clicked" : self.OnRemove,
                        "on_button_find_clicked"   : self.OnFind }
        
        # Fill the combo box with the list of sites
        sites = stack_api.api.fetch_sites()
        
        if sites == None:
        	
        	# This means that there was an earlier
        	# error retrieving information from StackAuth
        	# and therefore the list of sites is not available.
        	raise Exception(_("There was an error retrieving information from StackAuth when StackApplet was initialized.\n\nThe preferences dialog cannot be displayed.\n\nPlease note that this is not an error in StackApplet - either a misconfigured Internet connection or StackAuth failure are likely the problem."))
        
        self.site_list = gtk.ListStore(gobject.TYPE_STRING,gobject.TYPE_STRING)
        self.entry_refresh = self.builder.get_object('entry_refresh')
        self.entry_user = self.builder.get_object("entry_user")
        
        self.checkbutton = self.builder.get_object("checkbutton")

        self.treeview = self.builder.get_object("treeview")

        self.combobox = self.builder.get_object('combobox')
        self.combobox.set_model(self.site_list)
        
        cell = gtk.CellRendererText()
        self.combobox.pack_start(cell, True)
        self.combobox.add_attribute(cell, 'text', 0)
        
        for site in sites["api_sites"]:
            
            self.site_list.append([site["name"],site["api_endpoint"]])
        
        self.combobox.set_active(0)
        
        # Set up the tree view
        self.account_model = gtk.ListStore(gobject.TYPE_STRING,gobject.TYPE_STRING)
        self.builder.get_object('treeview').set_model(self.account_model)
        
        # Set up the renderer
        column = gtk.TreeViewColumn()
        cell = gtk.CellRendererText()
        column.pack_start(cell)
        column.add_attribute(cell,'text',0)
        column.set_title("Accounts:")
        self.builder.get_object('treeview').append_column(column)
        
        # Load the accounts from the preferences
        self.load_accounts()
        
        #self.root.signal_autoconnect(signal_dict)
        self.builder.connect_signals(signal_dict)
        
        # Fetch the window
        self.window = self.builder.get_object("prefs_dialog")
        self.window.show()
    
    def load_accounts(self):
        
        # Begin by loading the preferences
        prefs = config_store.config_store()
        prefs.load_settings()
        accounts = prefs.get_val("accounts",{})
        
        for account in accounts:
            
            self.account_model.append([account["site"] + " - " + account["user_id"],account["site_name"]])
        
        # Now for the refresh rate...
        self.entry_refresh.set_text(str(prefs.get_val("refresh_rate",300000) / 1000))
        
        if prefs.get_val("theme","dark") == 'dark':
        	self.builder.get_object('dark_theme').set_active(True)
        else:
        	self.builder.get_object('light_theme').set_active(True)
        
        if prefs.get_val("auto_start",0) == 1:
        	self.builder.get_object('auto_start').set_active(True)
        else:
        	self.builder.get_object('auto_start').set_active(False)

    def OnOK(self,widget):
        
        self.window.hide()
        
        # Begin by enumerating the items in the list
        
        assoc_sites = []
        
        for account in self.account_model:
            
            (site_api,site_id) = account[0].split(' - ')
            
            site_name = account[1]
            
            assoc_sites.append({ "site"      : site_api,
                                 "site_name" : site_name,
                                 "user_id"   : site_id })
        
        # Now store the accounts
        prefs = config_store.config_store()
        prefs.set_val("accounts",assoc_sites)
        
        # Get the other settings
        prefs.set_val("refresh_rate",int(self.entry_refresh.get_text()) * 1000)
        
        # Get the desired theme
        if self.builder.get_object('light_theme').get_active():
        	prefs.set_val("theme","light")
        else:
        	prefs.set_val("theme","dark")
        
        global DESKTOP_FILE_NAME, DESKTOP_FILE_CONTENTS
        
        # Autostart
        if self.builder.get_object('auto_start').get_active():
        
        	# We have two things to do...
        	# first, set the preference
        	prefs.set_val("auto_start",1)
        	
        	# secondly, create the .desktop file
        	if not os.path.isfile(DESKTOP_FILE_NAME):
			f = open(DESKTOP_FILE_NAME,'w')
			f.write(DESKTOP_FILE_CONTENTS)
			f.close()
        else:
        	
        	# First make sure the preference is set
        	prefs.set_val("auto_start",0)
        	
        	# Now delete the auto start file if it exists
        	if os.path.isfile(DESKTOP_FILE_NAME):
        		os.remove(DESKTOP_FILE_NAME)
        
        prefs.save_settings()
        
        self.window.destroy()
        
        self.emit("new_settings")
        
    def OnCancel(self,widget):
        
        # Here we close the window
        # without making any changes
        
        self.window.destroy()
    
    def OnAdd(self,widget):
        
        # Add item to list
        
        # Make sure that the textbox holds a valid
        # value
        if self.entry_user.get_text() == '':
            
            return
        
        # Keep an eye out for errors
        
        try:
            
            # Get the user's GUID
            account_site = self.site_list[self.combobox.get_active()][1]
            account_site = account_site.replace("http://api.","")
            account_site = account_site.replace(".com","")
            
            # Check to see if the user wants to search
            # for associated accounts
            if self.checkbutton.get_active() == True:
                
                user = stack_api.api.fetch_user_profile(account_site,self.entry_user.get_text())
            
                # Now get the sites listed
                assoc_sites = stack_api.api.fetch_associated_accounts(user["users"][0]["association_id"])
                num_assoc = len(assoc_sites["associated_users"])
                
                assoc_msg = _("We found %d accounts associated with this user.\n\nWould you like to add them?") % (num_assoc)

                # Ask the user if they want to add the associated accounts
                ask = gtk.MessageDialog(flags = gtk.DIALOG_MODAL,
                                    type = gtk.MESSAGE_QUESTION,
                                    buttons = gtk.BUTTONS_YES_NO,
                                    message_format = assoc_msg)
                # Now get the response
                resp = ask.run()
            
                if resp == gtk.RESPONSE_YES:
                
                    ask.destroy()
                    
                    # We need to add the associated accounts
                    # to the list
                    
                    for site in assoc_sites["associated_users"]:
                    
                        # Get the proper part of the URL
                        
                        site_full_url = site["on_site"]["api_endpoint"]
                        site_full_url = site_full_url.replace("http://api.","")
                        site_full_url = site_full_url.replace(".com","")
                        
                        site_user_id = site["user_id"]
                        
                        self.account_model.append([site_full_url + " - " + str(site_user_id),site["on_site"]["name"]])
                else:
                    
                    ask.destroy()
                    
                    # Simply add this item to the list
                    self.account_model.append([account_site + " - " + self.entry_user.get_text(),
                                               self.site_list[self.combobox.get_active()][0]])
            
            else:
                
                # Simply add this item to the list
                self.account_model.append([account_site + " - " + self.entry_user.get_text(),
                                           self.site_list[self.combobox.get_active()][0]])
                        
        except Exception as e:
        
            pass

    def OnRemove (self,widget):
        
        try:
        
            # Remove the selected entry
            self.account_model.remove(self.treeview.get_selection().get_selected()[1])
        
        except Exception as e:
        
            pass
    
    def OnFind (self,widget):
        
        account_site = self.site_list[self.combobox.get_active()][1]
        account_site = account_site.replace("http://api.","")
        account_site = account_site.replace(".com","")
        
        find_me = select_user.select_user(account_site)
        find_me.run()
        
        if find_me.ret_val:
            
            # Now get the return value
            self.entry_user.set_text(str(find_me.ret_val))

# This signal gets sent everytime that
# the settings are changed
gobject.signal_new("new_settings", prefs_dialog,
                                   gobject.SIGNAL_RUN_FIRST,
                                   gobject.TYPE_NONE,
                                   ())
