-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

--  For sem*.adb, the source code uses the following patterns:
--
--  1. Subprograms declarations. For each node given as formal
--  parameter or return in the subprogram declaration, we must write
--  the type of the node in the pre/post/return annotation.
--
--  procedure Proc (Node1, Node2 : in     STree.SyntaxNode;
--                  Node3        : in out STree.SyntaxNode;
--                  Node4        :    out STree.SyntaxNode)
--  --# pre (Syntax_Node_Type (Node => Node1) = SP_Symbols.n1 or
--  --#        Node1 = STree.NullNode) and
--  --#   Syntax_Node_Type (Node => Node2) = SP_Symbols.n2 and
--  --#   (Syntax_Node_Type (Node => Node3) = SP_Symbols.n31 or
--  --#      Syntax_Node_Type (Node => Node3) = SP_Symbols.n32 or
--  --#      Syntax_Node_Type (Node => Node3) = SP_Symbols.n33);
--  --# post  (Syntax_Node_Type (Node => Node3) = SP_Symbols.n31 or
--  --#          Node3 = STree.NullNode) and
--  --#   Syntax_Node_Type (Node => Node4) = SP_Symbols.n4;

--  function Func (Node1, Node2 : STree.SyntaxNode) return STree.SyntaxNode;
--  --# pre (Syntax_Node_Type (Node => Node1) = SP_Symbols.n1 or
--  --#        Node1 = STree.NullNode) and
--  --#   Syntax_Node_Type (Node => Node2) = SP_Symbols.n2;
--  --# return Node => Syntax_Node_Type (Node => Node) = SP_Symbols.n;
--
--  2. Node navigation. This is in the body of a subprogram. It is
--  called "node navigation" all operations on a node that return
--  another node (sometime the same node). Those operations includes
--  STree.Child_Node, STree.Next_Sibling, STree.Last_Sibling_Of,
--  STree.Last_Child_Of, STree.Parent,...
--
--  There are 2 patterns: the optimized one and the normal one. For
--  efficiency reason it is always better to use the optimized one but
--  sometime, it is not possible.
--
--  2.1 Optimized version. it is called "optimized" because the check
--  of the node type are performed almost for free (no real extra
--  costs, there is actually one more condition to evaluate else ->
--  elsif).
--
--  is
--     Node1, Node2 : STree.SyntaxNode;
--  begin
--     Node2 := STree.Child_Node (Node => Node1);
--     -- ASSUME Node2 = n1 OR n2 OR NULL
--     if Node2 = STree.NullNode
--       or else Syntax_Node_Type (Node => Node2) = SP_Symbols.n1 then
--        -- ASSUME Node2 = n1 OR NULL
--        Do_Something_With_Node2;
--     elsif Syntax_Node_Type (Node => Node2) = SP_Symbols.n2 then
--        -- ASSUME Node2 = n2 OR NULL
--        Do_Something_Else_With_Node2;
--     else
--        SystemErrors.Fatal_Error
--          (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
--           Msg     => "Expect Node2 = n1 OR n2 OR NULL in Func");
--     end if;
--  end Func;
--
--  See the fact that "or else" is used instead of "or". This is more
--  efficient and this prevent to call Syntax_Node_Type (Node =>
--  Node2) on a null node.
--
--  2.2 Normal version. This version is ONLY used if the optimized
--  version can not be use.
--
--  is
--     Node1, Node2 : STree.SyntaxNode;
--  begin
--     Node2 := STree.Child_Node (Node => Node1);
--     -- ASSUME Node2 = n1 OR n2 OR NULL
--     SystemErrors.RT_Assert
--       (C       => Node2 = STree.NullNode
--          or else Syntax_Node_Type (Node => Node2) = SP_Symbols.n1
--          or else Syntax_Node_Type (Node => Node2) = SP_Symbols.n2,
--        Sys_Err => SystemErrors.Invalid_Syntax_Tree,
--        Msg     => "Expect Node2 = n1 OR n2 OR NULL in Func");
--     Do_Something_With_Node2;
--  end Func;
--
--  See the fact that "or else" is used instead of "or". This is more
--  efficient and this prevent to call Syntax_Node_Type (Node =>
--  Node2) on a null node.
--
--  Optimisation is quite important for these kind of checks because
--  they should NEVER EVER FAILED!!!

with CompleteCheck;
with ComponentManager;
with ContextManager.Ops;
with Dictionary;
with ErrorHandler;
with ExaminerConstants;
with E_Strings;
with FlowAnalyser;
with Heap;
with LexTokenManager;
with Lists;
with Maths;
with RefList;
with RelationAlgebra;
with SeqAlgebra;
with SimpleLists;
with SPARK_IO;
with SP_Symbols;
with SystemErrors;

use type CompleteCheck.TypCompleteState;
use type CompleteCheck.TypRangeState;
use type CompleteCheck.TypOverlapState;
use type Dictionary.Abstractions;
use type Dictionary.Contexts;
use type Dictionary.Iterator;
use type Dictionary.KindsOfOp;
use type Dictionary.Modes;
use type Dictionary.PrefixSort;
use type Dictionary.Scopes;
use type Dictionary.Symbol;
use type ErrorHandler.Error_Level;
use type ErrorHandler.Justification_Identifier;
use type ErrorHandler.Justification_Kinds;
use type LexTokenManager.Lex_String;
use type LexTokenManager.Str_Comp_Result;
use type Maths.ErrorCode;
use type Maths.Value;
use type SP_Symbols.SP_Symbol;

package body Sem
--# own State is Aggregate_Stack.State;
is

   -- Operator renames

   -- Long subprogram prefix renames
   function Child_Node (Current_Node : STree.SyntaxNode) return STree.SyntaxNode renames STree.Child_Node;

   function Next_Sibling (Current_Node : STree.SyntaxNode) return STree.SyntaxNode renames STree.Next_Sibling;

   function Parent_Node (Current_Node : STree.SyntaxNode) return STree.SyntaxNode renames STree.Parent_Node;

   function Find_First_Node
     (Node_Kind    : SP_Symbols.SP_Symbol;
      From_Root    : STree.SyntaxNode;
      In_Direction : STree.TraverseDirection)
     return         STree.Iterator renames STree.Find_First_Node;

   function Find_First_Branch_Node (From_Root    : STree.SyntaxNode;
                                    In_Direction : STree.TraverseDirection) return STree.Iterator
     renames STree.Find_First_Branch_Node;

   function Get_Node (It : STree.Iterator) return STree.SyntaxNode renames STree.Get_Node;

   function Syntax_Node_Type (Node : STree.SyntaxNode) return SP_Symbols.SP_Symbol renames STree.Syntax_Node_Type;

   function Node_Position (Node : STree.SyntaxNode) return LexTokenManager.Token_Position renames STree.Node_Position;

   function Node_Lex_String (Node : STree.SyntaxNode) return LexTokenManager.Lex_String renames STree.Node_Lex_String;

   -- Function returns the left most leaf node of the tree.
   function Last_Child_Of (Start_Node : STree.SyntaxNode) return STree.SyntaxNode renames STree.Last_Child_Of;

   function Last_Sibling_Of (Start_Node : STree.SyntaxNode) return STree.SyntaxNode renames STree.Last_Sibling_Of;

   ----------------------------------------------------------------------------

   type Exp_Record_Sort is (
                            Type_Result, -- should be this anywhere above primary
                            Is_Unknown,
                            Is_Parameter_Name, -- used in named association checks
                            Is_Package,
                            Is_Object,
                            Is_Function,
                            Is_Type_Mark);

   type Exp_Record is record
      Type_Symbol                       : Dictionary.Symbol;
      Other_Symbol                      : Dictionary.Symbol;
      Stream_Symbol                     : Dictionary.Symbol;
      Tagged_Parameter_Symbol           : Dictionary.Symbol;
      Variable_Symbol                   : Dictionary.Symbol;
      Param_Count                       : Natural;
      Param_List                        : Lists.List;
      Sort                              : Exp_Record_Sort;
      Arg_List_Found                    : Boolean;
      Is_AVariable                      : Boolean;
      Is_An_Entire_Variable             : Boolean;
      Errors_In_Expression              : Boolean;
      Has_Operators                     : Boolean;
      Is_Static, -- 3 flags meaningless unless Sort=Type_Result
        Is_Constant, Is_ARange          : Boolean;
      String_Value                      : LexTokenManager.Lex_String; -- if a String literal or constant
      Value                             : Maths.Value; -- value of scalar, or value of L if Is_ARange
      Range_RHS                         : Maths.Value; -- if Is_ARange (e.g. L .. R) then this is the value of R
   end record;

   Null_Exp_Record : constant Exp_Record :=
     Exp_Record'
     (Type_Symbol             => Dictionary.NullSymbol,
      Other_Symbol            => Dictionary.NullSymbol,
      Stream_Symbol           => Dictionary.NullSymbol,
      Tagged_Parameter_Symbol => Dictionary.NullSymbol,
      Variable_Symbol         => Dictionary.NullSymbol,
      Param_Count             => 0,
      Param_List              => Lists.Null_List,
      Sort                    => Is_Unknown,
      Arg_List_Found          => False,
      Is_AVariable            => False,
      Is_An_Entire_Variable   => False,
      Errors_In_Expression    => False,
      Has_Operators           => False,
      Is_Static               => False,
      Is_Constant             => False,
      Is_ARange               => False,
      String_Value            => LexTokenManager.Null_String,
      Value                   => Maths.NoValue,
      Range_RHS               => Maths.NoValue);

   type Typ_Case_Flags is record
      Check_Completeness  : Boolean;
      Signal_Out_Of_Range : Boolean;
      Out_Of_Range_Seen   : Boolean;
      Check_Overlap       : Boolean;
      Warn_No_Others      : Boolean;
      Others_Mandatory    : Boolean;
   end record;

   Null_Case_Flags : constant Typ_Case_Flags :=
     Typ_Case_Flags'
     (Check_Completeness  => False,
      Signal_Out_Of_Range => False,
      Out_Of_Range_Seen   => False,
      Check_Overlap       => False,
      Warn_No_Others      => False,
      Others_Mandatory    => False);

   type Typ_Type_Bound is record
      Is_Defined : Boolean;
      Value      : Integer;
   end record;

   Unknown_Type_Bound : constant Typ_Type_Bound := Typ_Type_Bound'(Is_Defined => False,
                                                                   Value      => 0);

   type Typ_Agg_Association_Type is (Aggregate_Is_Positional, Aggregate_Is_Named, Aggregate_Is_Lone_Others);

   type Typ_Agg_Flags is record
      Check_Completeness        : Boolean;
      Warn_No_Others            : Boolean;
      Check_Overlap             : Boolean;
      Signal_Out_Of_Range       : Boolean;
      Out_Of_Range_Seen         : Boolean;
      More_Entries_Than_Natural : Boolean;
      Has_Others_Part           : Boolean;
      Association_Type          : Typ_Agg_Association_Type;
   end record;

   Null_Typ_Agg_Flags : constant Typ_Agg_Flags :=
     Typ_Agg_Flags'
     (Check_Completeness        => False,
      Warn_No_Others            => False,
      Check_Overlap             => False,
      Signal_Out_Of_Range       => False,
      Out_Of_Range_Seen         => False,
      More_Entries_Than_Natural => False,
      Has_Others_Part           => False,
      Association_Type          => Typ_Agg_Association_Type'First);

   type Visibility_Error_Hint is (No_Hint, In_Global_List, In_Derives_Import_List, In_Derives_Export_List, In_Suspends_List);

   type Tilde_Context is (Code, Precondition, Postcondition, Function_Return);
   subtype Anno_Tilde_Context is Tilde_Context range Precondition .. Function_Return;

   -------------------------------------------------------

   --# inherit CompleteCheck,
   --#         Dictionary,
   --#         ExaminerConstants,
   --#         Sem,
   --#         SPARK_IO,
   --#         SystemErrors;
   package Aggregate_Stack
   --# own State : Stack_T;
   is

      --# type Stack_T is abstract;

      --# function Stack_Is_Valid (The_State : Stack_T) return Boolean;

      procedure Init;
      --# global out State;
      --# derives State from ;
      --# post Stack_Is_Valid (State);

      procedure Push
        (Type_Sym     : in Dictionary.Symbol;
         Lower_Bound  : in Sem.Typ_Type_Bound;
         Upper_Bound  : in Sem.Typ_Type_Bound;
         Agg_Flags    : in Sem.Typ_Agg_Flags;
         Counter      : in Natural;
         Complete_Rec : in CompleteCheck.T);
      --# global in out State;
      --# derives State from *,
      --#                    Agg_Flags,
      --#                    Complete_Rec,
      --#                    Counter,
      --#                    Lower_Bound,
      --#                    Type_Sym,
      --#                    Upper_Bound;
      --# pre Stack_Is_Valid (State) and
      --#   ((Lower_Bound.Is_Defined and Upper_Bound.Is_Defined) -> (Lower_Bound.Value <= Upper_Bound.Value)) and
      --#   (Complete_Rec.ActualUpperBound - Complete_Rec.LowerBound < ExaminerConstants.CompleteCheckSize);
      --# post Stack_Is_Valid (State);

      procedure Pop
        (Type_Sym     : out Dictionary.Symbol;
         Lower_Bound  : out Sem.Typ_Type_Bound;
         Upper_Bound  : out Sem.Typ_Type_Bound;
         Agg_Flags    : out Sem.Typ_Agg_Flags;
         Counter      : out Natural;
         Complete_Rec : out CompleteCheck.T);
      --# global in out State;
      --# derives Agg_Flags,
      --#         Complete_Rec,
      --#         Counter,
      --#         Lower_Bound,
      --#         State,
      --#         Type_Sym,
      --#         Upper_Bound  from State;
      --# pre Stack_Is_Valid (State);
      --# post Stack_Is_Valid (State) and
      --#   ((Lower_Bound.Is_Defined and Upper_Bound.Is_Defined) -> (Lower_Bound.Value <= Upper_Bound.Value)) and
      --#   (Complete_Rec.ActualUpperBound - Complete_Rec.LowerBound < ExaminerConstants.CompleteCheckSize);

      -- In the analysis of array aggregates, the expression walker
      -- needs to know the expected type of the array index for the
      -- current aggregate.  To make this work for multi-dimensional
      -- aggregates, this has to be stored in the Aggregate Stack, and
      -- is accessed with the following function.
      function Top_Type_Sym return  Dictionary.Symbol;
      --# global in State;

   end Aggregate_Stack;

   ---------------------------------------------------------------------

   --# inherit Aggregate_Stack,
   --#         CommandLineData,
   --#         CompleteCheck,
   --#         ComponentManager,
   --#         ContextManager.Ops,
   --#         Debug,
   --#         Dictionary,
   --#         ErrorHandler,
   --#         ExaminerConstants,
   --#         E_Strings,
   --#         Heap,
   --#         LexTokenManager,
   --#         Lists,
   --#         Maths,
   --#         Sem,
   --#         SeqAlgebra,
   --#         SLI,
   --#         SPARK_IO,
   --#         SP_Symbols,
   --#         Statistics,
   --#         STree,
   --#         SystemErrors;
   package Walk_Expression_P is

      procedure Walk_Expression
        (Exp_Node                : in     STree.SyntaxNode;
         Scope                   : in     Dictionary.Scopes;
         Type_Context            : in     Dictionary.Symbol;
         Context_Requires_Static : in     Boolean;
         Ref_Var                 : in     SeqAlgebra.Seq;
         Result                  :    out Sem.Exp_Record;
         Component_Data          : in out ComponentManager.ComponentData;
         The_Heap                : in out Heap.HeapRecord);
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.File_Heap;
      --#        in     ContextManager.Ops.Unit_Heap;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LexTokenManager.State;
      --#        in out SLI.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#           out Aggregate_Stack.State;
      --# derives Aggregate_Stack.State,
      --#         Component_Data,
      --#         Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         Result,
      --#         STree.Table,
      --#         The_Heap                   from CommandLineData.Content,
      --#                                         Component_Data,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         Exp_Node,
      --#                                         LexTokenManager.State,
      --#                                         Ref_Var,
      --#                                         Scope,
      --#                                         STree.Table,
      --#                                         The_Heap,
      --#                                         Type_Context &
      --#         ErrorHandler.Error_Context,
      --#         SLI.State,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Component_Data,
      --#                                         ContextManager.Ops.File_Heap,
      --#                                         ContextManager.Ops.Unit_Heap,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Context_Requires_Static,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Exp_Node,
      --#                                         LexTokenManager.State,
      --#                                         Ref_Var,
      --#                                         Scope,
      --#                                         SLI.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         The_Heap,
      --#                                         Type_Context &
      --#         Statistics.TableUsage      from *,
      --#                                         CommandLineData.Content,
      --#                                         Component_Data,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         Exp_Node,
      --#                                         LexTokenManager.State,
      --#                                         Ref_Var,
      --#                                         Scope,
      --#                                         STree.Table,
      --#                                         The_Heap,
      --#                                         Type_Context;
      --# pre STree.Syntax_Node_Type (Exp_Node, STree.Table) = SP_Symbols.name or
      --#   STree.Syntax_Node_Type (Exp_Node, STree.Table) = SP_Symbols.expression or
      --#   STree.Syntax_Node_Type (Exp_Node, STree.Table) = SP_Symbols.simple_expression or
      --#   STree.Syntax_Node_Type (Exp_Node, STree.Table) = SP_Symbols.range_constraint or
      --#   STree.Syntax_Node_Type (Exp_Node, STree.Table) = SP_Symbols.arange or
      --#   STree.Syntax_Node_Type (Exp_Node, STree.Table) = SP_Symbols.attribute;
      --# post STree.Table = STree.Table~;

      --------------------------------------------------------------------------------------

      procedure Walk_Annotation_Expression
        (Exp_Node       : in     STree.SyntaxNode;
         Scope          : in     Dictionary.Scopes;
         Type_Context   : in     Dictionary.Symbol;
         Context        : in     Sem.Anno_Tilde_Context;
         Result         :    out Sem.Exp_Record;
         Component_Data : in out ComponentManager.ComponentData;
         The_Heap       : in out Heap.HeapRecord);
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.File_Heap;
      --#        in     ContextManager.Ops.Unit_Heap;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LexTokenManager.State;
      --#        in out SLI.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#           out Aggregate_Stack.State;
      --# derives Aggregate_Stack.State,
      --#         Component_Data,
      --#         Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         Result,
      --#         STree.Table,
      --#         The_Heap                   from CommandLineData.Content,
      --#                                         Component_Data,
      --#                                         Context,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         Exp_Node,
      --#                                         LexTokenManager.State,
      --#                                         Scope,
      --#                                         STree.Table,
      --#                                         The_Heap,
      --#                                         Type_Context &
      --#         ErrorHandler.Error_Context,
      --#         SLI.State,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Component_Data,
      --#                                         Context,
      --#                                         ContextManager.Ops.File_Heap,
      --#                                         ContextManager.Ops.Unit_Heap,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Exp_Node,
      --#                                         LexTokenManager.State,
      --#                                         Scope,
      --#                                         SLI.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         The_Heap,
      --#                                         Type_Context &
      --#         Statistics.TableUsage      from *,
      --#                                         CommandLineData.Content,
      --#                                         Component_Data,
      --#                                         Context,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Dictionary.Dict,
      --#                                         Exp_Node,
      --#                                         LexTokenManager.State,
      --#                                         Scope,
      --#                                         STree.Table,
      --#                                         The_Heap,
      --#                                         Type_Context;
      --# pre STree.Syntax_Node_Type (Exp_Node, STree.Table) = SP_Symbols.annotation_expression;
      --# post STree.Table = STree.Table~;

   end Walk_Expression_P;

   ------------------------------------------------------------------------

   --# inherit Aggregate_Stack,
   --#         CommandLineData,
   --#         ComponentManager,
   --#         ContextManager.Ops,
   --#         Dictionary,
   --#         ErrorHandler,
   --#         Heap,
   --#         LexTokenManager,
   --#         Sem,
   --#         SLI,
   --#         SPARK_IO,
   --#         SP_Symbols,
   --#         Statistics,
   --#         STree,
   --#         SystemErrors;
   package Subprogram_Specification is

      procedure Wf_Subprogram_Specification_From_Body
        (Node          : in     STree.SyntaxNode;
         Hidden        : in     Boolean;
         Current_Scope : in out Dictionary.Scopes;
         Subprog_Sym   :    out Dictionary.Symbol;
         First_Seen    :    out Boolean);
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in     LexTokenManager.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --# derives Current_Scope,
      --#         Subprog_Sym                from ContextManager.Ops.Unit_Stack,
      --#                                         Current_Scope,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table &
      --#         Dictionary.Dict,
      --#         STree.Table                from CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Current_Scope,
      --#                                         Dictionary.Dict,
      --#                                         Hidden,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Current_Scope,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Hidden,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table &
      --#         First_Seen                 from Current_Scope,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table;
      --# pre STree.Syntax_Node_Type (Node, STree.Table) = SP_Symbols.function_specification or
      --#   STree.Syntax_Node_Type (Node, STree.Table) = SP_Symbols.procedure_specification;
      --# post STree.Table = STree.Table~;

      procedure Wf_Subprogram_Specification
        (Spec_Node, Anno_Node, Constraint_Node : in     STree.SyntaxNode;
         Current_Scope                         : in     Dictionary.Scopes;
         Current_Context                       : in     Dictionary.Contexts;
         Component_Data                        : in out ComponentManager.ComponentData;
         The_Heap                              : in out Heap.HeapRecord;
         Subprog_Sym                           :    out Dictionary.Symbol);
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.File_Heap;
      --#        in     ContextManager.Ops.Unit_Heap;
      --#        in     ContextManager.Ops.Unit_Stack;
      --#        in out Aggregate_Stack.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LexTokenManager.State;
      --#        in out SLI.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --# derives Aggregate_Stack.State,
      --#         Component_Data,
      --#         Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         Statistics.TableUsage,
      --#         STree.Table,
      --#         The_Heap                   from *,
      --#                                         Anno_Node,
      --#                                         CommandLineData.Content,
      --#                                         Component_Data,
      --#                                         Constraint_Node,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Current_Context,
      --#                                         Current_Scope,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Spec_Node,
      --#                                         STree.Table,
      --#                                         The_Heap &
      --#         ErrorHandler.Error_Context,
      --#         SLI.State,
      --#         SPARK_IO.File_Sys          from Anno_Node,
      --#                                         CommandLineData.Content,
      --#                                         Component_Data,
      --#                                         Constraint_Node,
      --#                                         ContextManager.Ops.File_Heap,
      --#                                         ContextManager.Ops.Unit_Heap,
      --#                                         ContextManager.Ops.Unit_Stack,
      --#                                         Current_Context,
      --#                                         Current_Scope,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         SLI.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         Spec_Node,
      --#                                         STree.Table,
      --#                                         The_Heap &
      --#         Subprog_Sym                from ContextManager.Ops.Unit_Stack,
      --#                                         Current_Scope,
      --#                                         Dictionary.Dict,
      --#                                         LexTokenManager.State,
      --#                                         Spec_Node,
      --#                                         STree.Table;
      --# pre (STree.Syntax_Node_Type (Spec_Node, STree.Table) = SP_Symbols.procedure_specification or
      --#        STree.Syntax_Node_Type (Spec_Node, STree.Table) = SP_Symbols.function_specification or
      --#        STree.Syntax_Node_Type (Spec_Node, STree.Table) = SP_Symbols.entry_specification) and
      --#   (STree.Syntax_Node_Type (Anno_Node, STree.Table) = SP_Symbols.procedure_annotation or
      --#      STree.Syntax_Node_Type (Anno_Node, STree.Table) = SP_Symbols.function_annotation or
      --#      Anno_Node = STree.NullNode) and
      --#   (STree.Syntax_Node_Type (Constraint_Node, STree.Table) = SP_Symbols.function_constraint or
      --#      STree.Syntax_Node_Type (Constraint_Node, STree.Table) = SP_Symbols.procedure_constraint);
      --# post STree.Table = STree.Table~;

   end Subprogram_Specification;

   -------------- Package bodies ------------------------------

   package body Aggregate_Stack is separate;

   ----------------------- Subprograms ---------------------------

   -- Function to determine whether a procedure, task or entry needs a synthesised
   -- 'each export from all imports' dependency.
   -- This is done when data flow analysis is to be used, ie in data flow mode, or in
   -- automatic flow analysis mode when there is no explicit derives annotation.
   -- This is not compatible with SPARK83. If data-flow mode is used in SPARK83 then
   -- the derives annotations are used to determine the modes of the global imports and
   -- exports, but that is done elsewhere.
   function Needs_Synthetic_Dependency (Proc_Task_Or_Entry : Dictionary.Symbol) return Boolean
   --# global Dictionary.Dict,
   --#        CommandLineData.Content;
   is
   begin

      return CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
          and then (CommandLineData.Content.Flow_Option = CommandLineData.Data_Flow or
                             (CommandLineData.Content.Flow_Option = CommandLineData.Auto_Flow and
                                not Dictionary.GetHasDerivesAnnotation (Proc_Task_Or_Entry)));

   end Needs_Synthetic_Dependency;

   -------------------------------------------------------

   function Unknown_Type_Record return Exp_Record
   -- returns a "safe" exp result designed to minimise consequential errs
   --# global in Dictionary.Dict;
      is separate;

   -------------------------------------------------------------------------

   procedure Get_Type_Bounds (Type_Symbol              : in     Dictionary.Symbol;
                              Lower_Bound, Upper_Bound :    out Typ_Type_Bound)
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --# derives Lower_Bound,
   --#         Upper_Bound from Dictionary.Dict,
   --#                          LexTokenManager.State,
   --#                          Type_Symbol;
   --# post (Lower_Bound.Is_Defined and Upper_Bound.Is_Defined) -> (Lower_Bound.Value <= Upper_Bound.Value);
      is separate;

   ---------------------------------------------------------------------

   procedure Check_Package_Prefix
     (Node_Pos : in     LexTokenManager.Token_Position;
      Pack_Sym : in     Dictionary.Symbol;
      Scope    : in     Dictionary.Scopes;
      OK       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node_Pos,
   --#                                         Pack_Sym,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys &
   --#         OK                         from Dictionary.Dict,
   --#                                         Pack_Sym,
   --#                                         Scope;
      is separate;

   ------------------------------------------------------------------

   function In_Package_Initialization (Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
      is separate;

   ----------------------------------------------------------------

   function Indexes_Match (Target, Source : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;

   -------------------------------------------------------------------

   function Is_External_Interface (Pragma_Node : STree.SyntaxNode) return Boolean
   --# global in CommandLineData.Content;
   --#        in LexTokenManager.State;
   --#        in STree.Table;
   --# pre Syntax_Node_Type (Pragma_Node, STree.Table) = SP_Symbols.apragma;
      is separate;

   -------------------------------------------------------------------

   function Is_Enclosing_Package (Outer_Pack : Dictionary.Symbol;
                                  Scope      : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
      is separate;

   ------------------------------------------------------------------------

   procedure Get_Literal_Value (Node : in     STree.SyntaxNode;
                                Val  :    out Maths.Value)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Val                        from LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.integer_number or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.real_number or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.based_integer or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.based_real;
      is separate;

   --------------------------------------------

   function Substitute_Protected_Type_Self_Reference (Sym, Prefix_Symbol : Dictionary.Symbol) return Dictionary.Symbol
   --# global in Dictionary.Dict;
      is separate;

   -----------------------------------------------------------------

   -- Checks Val against Typ'First .. Typ'Last.
   -- NewVal = Val if all OK or if arithmetic overflow raised a warning
   -- NewVal = Maths.NoValue if check performed and failed
   procedure Constraint_Check
     (Val           : in     Maths.Value;
      New_Val       :    out Maths.Value;
      Is_Annotation : in     Boolean;
      Typ           : in     Dictionary.Symbol;
      Position      : in     LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Is_Annotation,
   --#                                         LexTokenManager.State,
   --#                                         Position,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Typ,
   --#                                         Val &
   --#         New_Val                    from Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Typ,
   --#                                         Val;
      is separate;

   --------------------------------------------------------------------------

   function Illegal_Unconstrained (Left_Type, Right_Type : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;

   --------------------------------------------------------------------------

   procedure Assignment_Check
     (Position    : in     LexTokenManager.Token_Position;
      Scope       : in     Dictionary.Scopes;
      Target_Type : in     Dictionary.Symbol;
      Exp_Result  : in out Exp_Record)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Exp_Result,
   --#                                         LexTokenManager.State,
   --#                                         Position,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Target_Type &
   --#         Exp_Result                 from *,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         Target_Type &
   --#         LexTokenManager.State      from *,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Result,
   --#                                         Target_Type;
      is separate;

   --------------------------------------------------------------------------

   function Convert_Tagged_Actual (Actual, Tagged_Parameter_Sym : Dictionary.Symbol) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   --  Function is only called if we know substitution is required
      is separate;

   -----------------------------------------------------------------------------------

   -- Create_Implicit_Positive_Subtype extracted from Plant_Constraining_Type so that it can be
   -- called directly from wf_constant_declaration where it is used to created an implcit
   -- subtype for string constants.
   procedure Create_Implicit_Positive_Subtype
     (String_Length    : in     Maths.Value;
      Location         : in     Dictionary.Location;
      Index_Constraint :    out Dictionary.Symbol)
   --# global in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    String_Length &
   --#         Index_Constraint      from ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    String_Length &
   --#         LexTokenManager.State from *,
   --#                                    Dictionary.Dict,
   --#                                    String_Length;
      is separate;

   ---------------------------------------------------------------

   procedure Create_Interrupt_Stream_Variable
     (For_PO              : in Dictionary.Symbol;
      The_Handler         : in LexTokenManager.Lex_String;
      The_Stream_Variable : in LexTokenManager.Lex_String;
      Error_Node_Pos      : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict            from *,
   --#                                         For_PO,
   --#                                         LexTokenManager.State,
   --#                                         The_Handler,
   --#                                         The_Stream_Variable &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Node_Pos,
   --#                                         For_PO,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         The_Handler,
   --#                                         The_Stream_Variable;
      is separate;

   ---------------------------------------------------------------------

   procedure Check_Interrupt_Property_Consistency
     (Has_Interrupt_Property : in     Boolean;
      Sym                    : in     Dictionary.Symbol;
      Type_Sym               : in     Dictionary.Symbol;
      Error_Node_Pos         : in     LexTokenManager.Token_Position;
      Consistent             :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Consistent                 from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Has_Interrupt_Property,
   --#                                         Type_Sym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Node_Pos,
   --#                                         Has_Interrupt_Property,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Sym,
   --#                                         Type_Sym;
      is separate;

   ---------------------------------------------------------------------

   procedure Check_Suspendable_Property_Consistency
     (Sym                 : in     Dictionary.Symbol;
      Type_Sym            : in     Dictionary.Symbol;
      Is_In_Suspends_List : in     Boolean;
      Error_Node_Pos      : in     LexTokenManager.Token_Position;
      Consistent          :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Consistent                 from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Is_In_Suspends_List,
   --#                                         Sym,
   --#                                         Type_Sym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Node_Pos,
   --#                                         Is_In_Suspends_List,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Sym,
   --#                                         Type_Sym;
      is separate;

   ---------------------------------------------------------------------

   procedure Check_Priority_Property_Consistency
     (Sym                     : in     Dictionary.Symbol;
      Type_Sym                : in     Dictionary.Symbol;
      Priority_Property_Value : in     LexTokenManager.Lex_String;
      Error_Node_Pos          : in     LexTokenManager.Token_Position;
      Consistent              :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Consistent                 from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Priority_Property_Value,
   --#                                         Sym,
   --#                                         Type_Sym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Node_Pos,
   --#                                         LexTokenManager.State,
   --#                                         Priority_Property_Value,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Sym,
   --#                                         Type_Sym;
      is separate;

   ---------------------------------------------------------------------

   procedure Check_Priority_Range
     (Error_Sym   : in     Dictionary.Symbol;
      Scope       : in     Dictionary.Scopes;
      Pragma_Kind : in     Dictionary.RavenscarPragmas;
      Err_Pos     : in     LexTokenManager.Token_Position;
      Value       : in     Maths.Value;
      Value_Rep   :    out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Sym,
   --#                                         Err_Pos,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Kind,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Value &
   --#         LexTokenManager.State,
   --#         Value_Rep                  from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Kind,
   --#                                         Value;
      is separate;

   ---------------------------------------------------------------------

   procedure Add_Derives_Stream_Effects
     (Node_Pos    : in LexTokenManager.Token_Position;
      Subprog_Sym : in Dictionary.Symbol;
      Abstraction : in Dictionary.Abstractions)
   --# global in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict   from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                Node_Pos,
   --#                                Subprog_Sym &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Node_Pos,
   --#                                Subprog_Sym;
      is separate;

   ---------------------------------------------------------------------

   procedure Plant_Constraining_Type
     (Expression_Type : in Dictionary.Symbol;
      String_Length   : in Maths.Value;
      Actual_Node     : in STree.SyntaxNode)
   --# global in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         SPARK_IO.File_Sys,
   --#         STree.Table           from *,
   --#                                    Actual_Node,
   --#                                    ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    Expression_Type,
   --#                                    LexTokenManager.State,
   --#                                    STree.Table,
   --#                                    String_Length &
   --#         LexTokenManager.State from *,
   --#                                    Dictionary.Dict,
   --#                                    Expression_Type,
   --#                                    String_Length;
   --# pre Syntax_Node_Type (Actual_Node, STree.Table) = SP_Symbols.expression;
   --# post STree.Table = STree.Table~;
      is separate;

   -----------------------------------------------------------------------------------

   procedure Add_Record_Sub_Components
     (Record_Var_Sym  : in     Dictionary.Symbol;
      Record_Type_Sym : in     Dictionary.Symbol;
      Component_Data  : in out ComponentManager.ComponentData;
      The_Heap        : in out Heap.HeapRecord)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --# derives Component_Data,
   --#         Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         The_Heap              from *,
   --#                                    Component_Data,
   --#                                    ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    Record_Type_Sym,
   --#                                    Record_Var_Sym,
   --#                                    The_Heap &
   --#         null                  from CommandLineData.Content;
      is separate;

   ---------------------------------------------------------------

   procedure Get_Subprogram_Anno_Key_Nodes
     (Node            : in     STree.SyntaxNode;
      Global_Node     :    out STree.SyntaxNode;
      Dependency_Node :    out STree.SyntaxNode;
      Declare_Node    :    out STree.SyntaxNode;
      Constraint_Node :    out STree.SyntaxNode)
   --# global in STree.Table;
   --# derives Constraint_Node,
   --#         Declare_Node,
   --#         Dependency_Node,
   --#         Global_Node     from Node,
   --#                              STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.procedure_annotation or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.function_annotation;
   --# post (Syntax_Node_Type (Global_Node, STree.Table) = SP_Symbols.moded_global_definition or
   --#         Global_Node = STree.NullNode) and
   --#   (Syntax_Node_Type (Dependency_Node, STree.Table) = SP_Symbols.dependency_relation or
   --#      Dependency_Node = STree.NullNode) and
   --#   (Syntax_Node_Type (Declare_Node, STree.Table) = SP_Symbols.declare_annotation or
   --#      Declare_Node = STree.NullNode) and
   --#   (Syntax_Node_Type (Constraint_Node, STree.Table) = SP_Symbols.function_constraint or
   --#      Syntax_Node_Type (Constraint_Node, STree.Table) = SP_Symbols.procedure_constraint);
      is separate;

   ---------------------------------------------------------------

   procedure Check_Derives_Consistency
     (Subprog_Sym : in     Dictionary.Symbol;
      Position    : in     LexTokenManager.Token_Position;
      The_Heap    : in out Heap.HeapRecord)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Position,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Subprog_Sym,
   --#                                         The_Heap &
   --#         Statistics.TableUsage,
   --#         The_Heap                   from *,
   --#                                         Dictionary.Dict,
   --#                                         Subprog_Sym,
   --#                                         The_Heap;
      is separate;

   --------------------------------------------------------------------------

   procedure Wf_Type_Mark
     (Node          : in     STree.SyntaxNode;
      Current_Scope : in     Dictionary.Scopes;
      Context       : in     Dictionary.Contexts;
      Type_Sym      :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Context,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         STree.Table,
   --#         Type_Sym                   from CommandLineData.Content,
   --#                                         Context,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.type_mark;
   --# post STree.Table = STree.Table~;
      is separate;

   -----------------------------------------------------------------

   -- procedure for creating a "full" dependency relation

   procedure Create_Full_Dependency
     (Node_Pos                 : in LexTokenManager.Token_Position;
      Subprog_Sym              : in Dictionary.Symbol;
      Abstraction              : in Dictionary.Abstractions;
      Import_List, Export_List : in SeqAlgebra.Seq;
      The_Heap                 : in Heap.HeapRecord)
   --# global in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict   from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                Export_List,
   --#                                Import_List,
   --#                                Node_Pos,
   --#                                Subprog_Sym,
   --#                                The_Heap &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                Dictionary.Dict,
   --#                                Export_List,
   --#                                Import_List,
   --#                                LexTokenManager.State,
   --#                                Node_Pos,
   --#                                Subprog_Sym,
   --#                                The_Heap;
      is separate;

   -----------------------------------------------------------------------

   procedure Create_Full_Subprog_Dependency
     (Node_Pos    : in     LexTokenManager.Token_Position;
      Subprog_Sym : in     Dictionary.Symbol;
      Abstraction : in     Dictionary.Abstractions;
      The_Heap    : in out Heap.HeapRecord)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --# derives Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         The_Heap                   from *,
   --#                                         Abstraction,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Node_Pos,
   --#                                         Subprog_Sym,
   --#                                         The_Heap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node_Pos,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Subprog_Sym,
   --#                                         The_Heap;
      is separate;

   ---------------------------------------------------------------------

   procedure Wf_Formal_Part
     (Node             : in STree.SyntaxNode;
      Current_Scope    : in Dictionary.Scopes;
      Subprog_Sym      : in Dictionary.Symbol;
      First_Occurrence : in Boolean;
      Context          : in Dictionary.Contexts)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         First_Occurrence,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         Subprog_Sym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Occurrence,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Sym;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.formal_part;
   --# post STree.Table = STree.Table~;
      is separate;

   -----------------------------------------------------------------------------------------

   procedure Wf_Entire_Variable
     (Node       : in     STree.SyntaxNode;
      Scope      : in     Dictionary.Scopes;
      Error_Hint : in     Visibility_Error_Hint;
      Var_Sym    :    out Dictionary.Symbol;
      Dotted     :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dotted,
   --#         STree.Table,
   --#         Var_Sym                    from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Hint,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.entire_variable or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.annotation_primary;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------------------------------------------------------------------------

   procedure Wf_Global_Definition
     (Node          : in     STree.SyntaxNode;
      Scope         : in     Dictionary.Scopes;
      Subprog_Sym   : in     Dictionary.Symbol;
      First_Seen    : in     Boolean;
      Sem_Err_Found :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Sem_Err_Found,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         Subprog_Sym &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Sym &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Sym &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         STree.Table,
   --#                                         Subprog_Sym;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.moded_global_definition;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Dependency_Relation
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      Subprog_Sym  : in     Dictionary.Symbol;
      First_Seen   : in     Boolean;
      Glob_Def_Err : in     Boolean;
      The_Heap     : in out Heap.HeapRecord)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         The_Heap                   from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         Glob_Def_Err,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         Subprog_Sym,
   --#                                         The_Heap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         Glob_Def_Err,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Sym,
   --#                                         The_Heap &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Seen,
   --#                                         Glob_Def_Err,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Sym,
   --#                                         The_Heap &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Seen,
   --#                                         Glob_Def_Err,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         STree.Table,
   --#                                         Subprog_Sym,
   --#                                         The_Heap &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Subprog_Sym,
   --#                                         The_Heap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.dependency_relation;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Priority_Value
     (Node        : in     STree.SyntaxNode;
      Pragma_Kind : in     Dictionary.RavenscarPragmas;
      Error_Sym   : in     Dictionary.Symbol;
      Scope       : in     Dictionary.Scopes;
      The_Heap    : in out Heap.HeapRecord;
      Value_Rep   :    out LexTokenManager.Lex_String;
      Compatible  :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out Aggregate_Stack.State;
   --# derives Aggregate_Stack.State,
   --#         Compatible,
   --#         Dictionary.Dict,
   --#         STree.Table,
   --#         The_Heap                   from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         The_Heap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pragma_Kind,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         The_Heap &
   --#         LexTokenManager.State,
   --#         Value_Rep                  from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pragma_Kind,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         The_Heap &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         The_Heap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         The_Heap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.expression or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.annotation_expression;
   --# post STree.Table = STree.Table~;
      is separate;

   ---------------------------------------------------------------------

   procedure Wf_Property_List
     (Node          : in     STree.SyntaxNode;
      Type_Node_Pos : in     LexTokenManager.Token_Position;
      Scope         : in     Dictionary.Scopes;
      The_Owner     : in     Dictionary.Symbol;
      The_Heap      : in out Heap.HeapRecord)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         The_Heap                   from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         The_Heap,
   --#                                         The_Owner &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         The_Heap,
   --#                                         The_Owner,
   --#                                         Type_Node_Pos;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.property_list;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Declare_Annotation
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      Task_Or_Proc : in     Dictionary.Symbol;
      First_Seen   : in     Boolean;
      The_Heap     : in out Heap.HeapRecord)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         The_Heap                   from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         Task_Or_Proc,
   --#                                         The_Heap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Task_Or_Proc,
   --#                                         The_Heap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.declare_annotation;
   --# post STree.Table = STree.Table~;
      is separate;

   --------------------------------------------------------------------------------

   procedure Wf_Subprogram_Annotation
     (Node          : in     STree.SyntaxNode;
      Current_Scope : in     Dictionary.Scopes;
      Subprog_Sym   : in     Dictionary.Symbol;
      First_Seen    : in     Boolean;
      The_Heap      : in out Heap.HeapRecord)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         The_Heap                   from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         Subprog_Sym,
   --#                                         The_Heap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Sym,
   --#                                         The_Heap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.procedure_annotation or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.function_annotation;
   --# post STree.Table = STree.Table~;
      is separate;

   ----------------------------------------------------------------------

   package body Walk_Expression_P is separate;

   ---------------------------------------------------------------------

   procedure Wf_Predicate
     (Node           : in     STree.SyntaxNode;
      Scope          : in     Dictionary.Scopes;
      Context        : in     Anno_Tilde_Context;
      Component_Data : in out ComponentManager.ComponentData;
      The_Heap       : in out Heap.HeapRecord;
      Errors_Found   :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out Aggregate_Stack.State;
   --# derives Aggregate_Stack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         Errors_Found,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         The_Heap                   from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         The_Heap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         Context,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         The_Heap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         The_Heap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.predicate;
   --# post STree.Table = STree.Table~;
      is separate;

   --------------------------------------------------------------------

   procedure Wf_Subprogram_Constraint
     (Node           : in     STree.SyntaxNode;
      Scope          : in     Dictionary.Scopes;
      First_Seen     : in     Boolean;
      Component_Data : in out ComponentManager.ComponentData;
      The_Heap       : in out Heap.HeapRecord)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --# derives Aggregate_Stack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         The_Heap                   from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         The_Heap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         The_Heap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.procedure_constraint or
   --#   Syntax_Node_Type (Node, STree.Table) = SP_Symbols.function_constraint;
   --# post STree.Table = STree.Table~;
      is separate;

   -------------------------------------------------------------------------

   package body Subprogram_Specification is separate;

   -------------------------------------------------------------------------

   -- Exported subprogram
   procedure CompUnit (Top_Node : in STree.SyntaxNode;
                       Do_VCG   : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Declarations.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out STree.Table;
   --#        in out VCG.Invoked;
   --#           out Aggregate_Stack.State;
   --# derives Aggregate_Stack.State      from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Do_VCG,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Top_Node &
   --#         Declarations.State,
   --#         Dictionary.Dict,
   --#         ErrorHandler.Error_Context,
   --#         Graph.Table,
   --#         LexTokenManager.State,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys,
   --#         Statistics.TableUsage,
   --#         StmtStack.S,
   --#         STree.Table,
   --#         VCG.Invoked                from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Do_VCG,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Top_Node;
   --# post STree.Table = STree.Table~;
      is separate;

end Sem;
