<?php
/**
 * Vacation_Driver_sql:: implements the Vacation_Driver API for SQL servers.
 *
 * $Horde: vacation/lib/Driver/sql.php,v 1.34.2.5 2009/01/06 15:28:08 jan Exp $
 *
 * Copyright 2001-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Ilya Krel <mail@krel.org>
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @author  Jan Schneider <jan@horde.org>
 * @since   Vacation 2.1
 * @package Vacation
 */
class Vacation_Driver_sql extends Vacation_Driver {

    /**
     * SQL connection object.
     */
    var $_db;

    /**
     * Boolean which contains the state of the SQL connection.
     */
    var $_connected = false;

    /**
     * Sets up vacation notices for a user.
     *
     * @param string $password  The password for the user.
     * @param string $message   The text of the vacation notice.
     * @param string $subject   The subject of the vacation notice.
     * @param string $from      The From: address of the vacation notice.
     * @param string $alias     The alias email address.
     */
    function setVacation($password, $message, $subject, $from, $alias = '')
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to database.
        if (is_a($connected = $this->_connect(), 'PEAR_Error')) {
            return $connected;
        }

        // Build username.
        $myuser = $this->_buildUsername();

        // Encrypt password.
        $crypted_password = $this->_encryptPassword($password, $this->_getCurrentPassword($myuser));

        // Check if an entry already exists and create one otherwise
        $query = 'SELECT ' . $this->_params[$this->_realm]['vacation'] . ' AS vacation' .
                 ' FROM ' . $this->_params[$this->_realm]['table'] .
                 ' WHERE ' . $this->_params[$this->_realm]['user_col'] . ' = ?' .
                 ' AND ' . $this->_params[$this->_realm]['pass_col'] . ' = ?';
        $values = array($myuser, $crypted_password);
        $result = $this->_db->query($query, $values);
        if (!is_a($result, 'PEAR_Error')) {
            $query = 'INSERT INTO ' . $this->_params[$this->_realm]['table'] .
                     ' (' . $this->_params[$this->_realm]['vacation'] . ',' .
                     ' ' . $this->_params[$this->_realm]['user_col'] . ',' .
                     ' ' . $this->_params[$this->_realm]['pass_col'] . ')' .
                     ' VALUES (?, ?, ?)';
            $values = array('n', $myuser, $crypted_password);
            $result = $this->_db->query($query, $values);
        }

        // Build the SQL query.
        $query = 'UPDATE ' . $this->_params[$this->_realm]['table'] .
                 ' SET ' . $this->_params[$this->_realm]['vacation'] . ' = ?,' .
                 ' ' . $this->_params[$this->_realm]['message'] . ' = ?,' .
                 ' ' . $this->_params[$this->_realm]['subject'] . ' = ?' .
                 ' WHERE ' . $this->_params[$this->_realm]['user_col'] . ' = ?' .
                 ' AND ' . $this->_params[$this->_realm]['pass_col'] . ' = ?';
        $values = array('y', $message, $subject, $myuser, $crypted_password);

        $result = $this->_db->query($query, $values);

        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if ($this->_db->affectedRows() == 0) {
            return PEAR::raiseError(_("The vacation notice cannot be set. Check the password."));
        }
        if ($result !== DB_OK) {
            return PEAR::raiseError(_("An unknown error occured while enabling the vacation notice."));
        }
    }

    /**
     * Disables the vacation message for a user.
     *
     * @param string $password  The password of the user.
     */
    function unsetVacation($password)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to database.
        if (is_a($connected = $this->_connect(), 'PEAR_Error')) {
            return $connected;
        }

        // Build username.
        $myuser = $this->_buildUsername();

        // Build the SQL query.
        $query = 'UPDATE ' . $this->_params[$this->_realm]['table'] .
                 ' SET ' . $this->_params[$this->_realm]['vacation'] . ' = ?' .
                 ' WHERE ' . $this->_params[$this->_realm]['user_col'] . ' = ?' .
                 ' AND ' . $this->_params[$this->_realm]['pass_col'] . ' = ?';
        $values = array('n', $myuser,
                        $this->_encryptPassword($password, $this->_getCurrentPassword($myuser)));

        $result = $this->_db->query($query, $values);

        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if ($this->_db->affectedRows() == 0) {
            return PEAR::raiseError(_("The vacation notice cannot be set. Check the password."));
        }
        if ($result !== DB_OK) {
            return PEAR::raiseError(_("An unknown error occured while disabling the vacation notice."));
        }
    }

    /**
     * Retrieves the current vacation details for the user.
     *
     * @param string $password  The password for user.
     *
     * @return array  Vacation details or PEAR_Error.
     */
    function _getUserDetails($password)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to database.
        if (is_a($connected = $this->_connect(), 'PEAR_Error')) {
            return $connected;
        }

        // Build username.
        $myuser = $this->_buildUsername();

        // Build the SQL query.
        $query = 'SELECT ' . $this->_params[$this->_realm]['vacation'] . ' AS vacation,' .
                 ' ' . $this->_params[$this->_realm]['message'] . ' AS message,' .
                 ' ' . $this->_params[$this->_realm]['subject'] . ' AS subject' .
                 ' FROM ' . $this->_params[$this->_realm]['table'] .
                 ' WHERE ' . $this->_params[$this->_realm]['user_col'] . ' = ?' .
                 ' AND ' . $this->_params[$this->_realm]['pass_col'] . ' = ?';
        $values = array($myuser,
                        $this->_encryptPassword($password, $this->_getCurrentPassword($myuser)));

        $result = $this->_db->query($query, $values);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
        if (is_a($row, 'PEAR_Error')) {
            return $row;
        }
        if (is_array($row)) {
            return $row;
        }

        return array('vacation' => false);
    }

    /**
     * Retrieves the current password details for a user.
     *
     * @return string  Current password.
     */
    function _getCurrentPassword()
    {
        // Connect to database.
        if (is_a($connected = $this->_connect(), 'PEAR_Error')) {
            return $connected;
        }

        // Build the SQL query.
        $query = 'SELECT ' . $this->_params[$this->_realm]['pass_col'] . ' AS password'
            . ' FROM ' . $this->_params[$this->_realm]['table']
            . ' WHERE ' . $this->_params[$this->_realm]['user_col'] . ' = ?';
        $values = array($this->_user);

        $current_password = $this->_db->getOne($query, $values);
        if (is_a($current_password, 'PEAR_Error')) {
            Horde::logMessage($current_password, __FILE__, __LINE__, PEAR_LOG_ERR);
            return '';
        }

        return $current_password;
    }

    /**
     * Builds a username based on presence of a realm.
     *
     * @return string  user@realm or user.
     */
    function _buildUsername()
    {
        if ($this->_realm === 'default' ||
            $this->_realm === '') {
            return $this->_user;
        } else {
            return $this->_user . '@' . $this->_realm;
        }
    }

    /**
     * Checks if the realm has a specific configuration. If not, tries to fall
     * back on the default configuration. If still not a valid configuration
     * then exits with an error.
     */
    function _checkConfig()
    {
        // If no realm passed in, or no table config for the realm passed in,
        // then we fall back to the default realm
        if (empty($this->_params[$this->_realm]['table'])) {
            $this->_realm = 'default';
        }

        // If still no table,user_col,pass_col,message,subject,vacation then
        // we have a misconfigured module.
        if (empty($this->_params[$this->_realm]['table']) ||
            empty($this->_params[$this->_realm]['user_col']) ||
            empty($this->_params[$this->_realm]['pass_col']) ||
            empty($this->_params[$this->_realm]['message']) ||
            empty($this->_params[$this->_realm]['subject']) ||
            empty($this->_params[$this->_realm]['vacation']) ) {
            return PEAR::raiseError(_("The vacation application is not properly configured."));
        }
    }

    /**
     * Does an SQL connect and logs in as user with privilege to change
     * vacation.
     *
     * @return boolean  True or PEAR_Error based on success of connect.
     */
    function _connect()
    {
        if ($this->_connected) {
            return;
        }

        // Build the params array to pass to DB
        $args = array_merge($this->_params, $this->_params[$this->_realm]);

        Horde::assertDriverConfig($args, 'server', array('phptype', 'table'),
                                  'vacation authentication SQL');

        if (!isset($args['database'])) {
            $args['database'] = '';
        }
        if (!isset($args['username'])) {
            $args['username'] = '';
        }
        if (!isset($args['hostspec'])) {
            $args['hostspec'] = '';
        }

        // Connect to the SQL server using the supplied parameters.
        require_once 'DB.php';
        $this->_db = &DB::connect($args,
                                  array('persistent' => !empty($args['persistent'])));
        if (is_a($this->_db, 'PEAR_Error')) {
            return $this->_db;
        }

        // Set DB portability options.
        switch ($this->_db->phptype) {
            case 'mssql':
                $this->_db->setOption(
                    'portability',
                    DB_PORTABILITY_LOWERCASE | DB_PORTABILITY_ERRORS | DB_PORTABILITY_RTRIM);
                break;
            default:
                $this->_db->setOption(
                    'portability',
                    DB_PORTABILITY_LOWERCASE | DB_PORTABILITY_ERRORS);
        }

        $this->_connected = true;
    }

}
