<?php
/**
 * Vacation_Driver:: defines an API for implementing vacation backends for the
 * vacation module.
 *
 * $Horde: vacation/lib/Driver.php,v 1.35.2.6 2009/01/29 16:00:38 jan Exp $
 *
 * Copyright 2001-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @author  Eric Rostetter <eric.rostetter@physics.utexas.edu>
 * @author  Jan Schneider <jan@horde.org>
 * @package Vacation
 */
class Vacation_Driver {

    /**
     * The current user name.
     *
     * @var string
     */
    var $_user;

    /**
     * The user's realm.
     *
     * @var string
     */
    var $_realm;

    /**
     * Hash containing configuration data.
     *
     * @var array
     */
    var $_params;

    /**
     * The current vacation message.
     *
     * @var string
     */
    var $_message;

    /**
     * The current vacation subject.
     *
     * @var string
     */
    var $_subject;

    /**
     * The current vacation From: address.
     *
     * @var string
     */
    var $_from;

    /**
     * Constructor.
     *
     * @param string $user   A user name.
     * @param array $params  Configuration parameters for the backend.
     */
    function Vacation_Driver($user, $params = array())
    {
        $this->_params = $params;

        // Get the realm.
        @list(, $realm) = explode('@', $user, 2);
        if (empty($realm) || !isset($params[$realm])) {
            $realm = 'default';
        }
        $this->_realm = $realm;

        // Check if hordeauth is set to 'full'
        $hordeauth = $this->getParam('hordeauth');
        if ($hordeauth !== 'full') {
            @list($user,) = explode('@', $user, 2);
        }
        $this->_user = $user;
    }

    /**
     * Return a parameter value.
     *
     * @param string $param  The parameter to check in.
     *
     * @return mixed  The parameter value, or null if not found.
     */
    function getParam($param)
    {
        return isset($this->_params[$this->_realm][$param])
            ? $this->_params[$this->_realm][$param]
            : null;
    }

    /**
     * Returns the current user.
     *
     * @return string  The current user name.
     */
    function getUser()
    {
        return $this->_user;
    }

    /**
     * Sets up vacation notices for a user.
     *
     * @abstract
     *
     * @param string $password  The password for the user.
     * @param string $message   The text of the vacation notice.
     * @param string $subject   The subject of the vacation notice.
     * @param string $from      The From: address of the vacation notice.
     * @param string $alias     The alias email address.
     */
    function setVacation($password, $message, $subject, $from, $alias = '')
    {
        return PEAR::raiseError('Abstract method setVacation() not implemented');
    }

    /**
     * Disables vacation notices for a user.
     *
     * @abstract
     *
     * @param string $password  The password of the user.
     */
    function unsetVacation($password)
    {
        return PEAR::raiseError('Abstract method unsetVacation() not implemented');
    }

    /**
     * Retrieves status of vacation for a user.
     *
     * @param string $password  The password for the user.
     *
     * @return mixed  Returns 'Y' if vacation is enabled for the user, 'N' if
     *                vacation is currently disabled, false if the status
     *                cannot be determined, and PEAR_Error on error.
     */
    function isEnabled($password)
    {
        // Get current details.
        $current_details = $this->_getUserDetails($password);
        if (is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        // Check vacation flag.
        if ($current_details['vacation'] == 'y' ||
            $current_details['vacation'] == 'Y') {
            return 'Y';
        } elseif ($current_details['vacation'] == 'n' ||
                  $current_details['vacation'] == 'N') {
            return 'N';
        } else {
            return false;
        }
    }

    /**
     * Retrieves current vacation message.
     *
     * @param string $password  The password for user.
     *
     * @return string  The current vacation message, or false if none.
     */
    function currentMessage($password)
    {
        if (!isset($this->_message)) {
            $this->_processMessage($password);
        }
        return $this->_message;
    }

    /**
     * Retrieves current vacation subject.
     *
     * @param string $password  The password for user.
     *
     * @return string  The current vacation subject, or false if none.
     */
    function currentSubject($password)
    {
        if (!isset($this->_subject)) {
            $this->_processMessage($password);
        }
        return $this->_subject;
    }

    /**
     * Retrieves current vacation From: address.
     *
     * @param string $password  The password for user.
     *
     * @return string  The current vacation From: address, or false if none.
     */
    function currentFrom($password)
    {
        if (!isset($this->_from)) {
            $this->_processMessage($password);
        }
        return $this->_from;
    }

    /**
     * Builds a vacation message.
     *
     * @param string $message  The text of the vacation notice.
     * @param string $subject  The subject of the vacation notice.
     * @param string $from     The From: address of the vacation notice.
     *
     * @return string  The complete vacation message including all headers.
     */
    function _buildMessage($message, $subject, $from)
    {
        $vacationtxt = '';
        // Include the mail subject if the driver supports it.
        if ($GLOBALS['conf']['vacation']['subject']) {
            $vacationtxt .= 'Subject: '
                . MIME::encode($subject, NLS::getCharset()) . "\n";
        }
        // Include the mail sender if the driver supports it.
        if ($GLOBALS['conf']['vacation']['from']) {
            $vacationtxt .= 'From: '
                . MIME::encode($from, NLS::getCharset()) . "\n";
        }

        if (MIME::is8bit($message)) {
            $vacationtxt .= "Content-Transfer-Encoding: quoted-printable\n"
                . 'Content-Type: text/plain; charset=' . NLS::getCharset()
                . "\n" . MIME::quotedPrintableEncode($message, "\n");
        } else {
            $vacationtxt .= $message;
        }

        return $vacationtxt;
    }

    /**
     * Processes the current vacation message.
     *
     * @param string $password  The password for user.
     */
    function _processMessage($password)
    {
        $current_details = $this->_getUserDetails($password);
        if (is_a($current_details, 'PEAR_Error') ||
            $current_details === false) {
            return $current_details;
        }

        $this->_message = isset($current_details['message'])
            ? $current_details['message']
            : $GLOBALS['conf']['vacation']['default_message'];
        $this->_subject = isset($current_details['subject'])
            ? $current_details['subject']
            : $GLOBALS['conf']['vacation']['default_subject'];
        $this->_from    = isset($current_details['from'])
            ? $current_details['from']
            : $this->getFrom();
    }

    /**
     * Parses a vacation message.
     *
     * @param string $message   A vacation message.
     *
     * @return array  A hash with parsed results in the field 'message',
     *                'subject' and 'from'.
     */
    function _parseMessage($message)
    {
        // Split the vacation text in a subject and a message if the driver
        // supports it.
        $subject = '';
        if ($GLOBALS['conf']['vacation']['subject']) {
            if (preg_match('/^Subject: ([^\n]*)\n(.+)$/s',
                           $message, $matches)) {
                $subject = MIME::decode($matches[1], NLS::getCharset());
                $message = $matches[2];
            }
        }

        // Split the vacation text in a sender and a message if the driver
        // supports it.
        $from = '';
        if ($GLOBALS['conf']['vacation']['from']) {
            if (preg_match('/^From: ([^\n]*)\n(.+)$/s',
                           $message, $matches)) {
                $from = MIME::decode($matches[1], NLS::getCharset());
                $message = $matches[2];
            } else {
                $from = $this->getFrom();
            }
        }

        // Detect Content-Type and Content-Transfer-Encoding headers.
        if (preg_match('/^Content-Transfer-Encoding: ([^\n]+)\n(.+)$/s',
                       $message, $matches)) {
            $message = $matches[2];
            if ($matches[1] == 'quoted-printable') {
                $message = quoted_printable_decode($message);
            }
        }
        if (preg_match('/^Content-Type: ([^\n]+)\n(.+)$/s',
                       $message, $matches)) {
            $message = $matches[2];
            if (preg_match('/^text\/plain; charset=(.*)$/',
                           $matches[1], $matches)) {
                $message = String::convertCharset($message, $matches[1]);
            }
        }

        return array('message' => $message,
                     'subject' => $subject,
                     'from' => $from);
    }

    /**
     * Retrieves the current vacation details for the user.
     *
     * @abstract
     *
     * @param string $password  The password for user.
     *
     * @return array  Vacation details or PEAR_Error on failure.
     */
    function _getUserDetails($password)
    {
        return PEAR::raiseError('Abstract method _getUserDetails() not implemented');
    }

    /**
     * Returns the default From: address of the current user.
     *
     * @return string  The default From: address.
     */
    function getFrom()
    {
        require_once 'Horde/Identity.php';
        $identity = &Identity::singleton('none', $this->_user);
        // Default "From:" from identities, with name (name <address>)
        return $identity->getDefaultFromAddress(true);
    }

    /**
     * Formats a password using the current encryption.
     *
     * @param string $plaintext  The plaintext password to encrypt.
     * @param string $salt       The salt (or seed) to use.
     *
     * @return string  The crypted password.
     */
    function _encryptPassword($plaintext, $salt = '')
    {
        return Auth::getCryptedPassword($plaintext,
                                        $salt,
                                        $this->_params['encryption'],
                                        $this->_params['show_encryption']);
    }

    /**
     * Parses an email address list and returns it in a known standard form.
     *
     * This will attempt to add the domain (realm) to unqualified addresses if
     * the realm is non-blank and not 'default'.
     *
     * @param string $user   The email address.
     * @param string $realm  The domain/realm to add if none is present.
     *
     * @return string  The email address(es) on success, false on error.
     */
    function _makeEmailAddress($user)
    {
        $domain = $this->_realm != 'default' ? $this->_realm : '';
        if ($this->getParam('norealm')) {
            $domain = '';
        }
        $email = '';

        require_once 'Mail/RFC822.php';
        $parser = new Mail_RFC822();
        $parsed_email = $parser->parseAddressList($user, $domain, false, false);
        if (!is_array($parsed_email) || !count($parsed_email)) {
            return PEAR::raiseError(_("Cannot parse your email address"));
        }

        for ($i = 0; $i < count($parsed_email); $i++) {
            if (!empty($email)) {
                $email .= ',';
            }
            if (is_object($parsed_email[$i])) {
                $email .= $parsed_email[$i]->mailbox;
                if (!empty($parsed_email[$i]->host)) {
                    $email .=  '@' . $parsed_email[$i]->host;
                }
            } else {
                $email .= $parsed_email[$i];
            }
        }

        return $email;
    }

    /**
     * Attempts to return a concrete Vacation_Driver instance based on $driver.
     *
     * @param string $user    A user name.
     * @param string $driver  The type of concrete Vacation_Driver subclass to
     *                        return. The class name is based on the vacation
     *                        driver ($driver). The code is dynamically
     *                        included.
     * @param array $params   A hash containing any additional configuration or
     *                        connection parameters a subclass might need.
     *
     * @return mixed  The newly created concrete Vacation_Driver instance, or
     *                false on an error.
     */
    function factory($user, $driver = null, $params = null)
    {
        if (is_null($driver)) {
            $driver = $GLOBALS['conf']['server']['driver'];
        }
        $driver = basename($driver);

        if (is_null($params)) {
            $params = Horde::getDriverConfig('server', $driver);
        }

        require_once dirname(__FILE__) . '/Driver/' . $driver . '.php';
        $class = 'Vacation_Driver_' . $driver;
        if (class_exists($class)) {
            return new $class($user, $params);
        }

        return false;
    }

}
