/////////////////////////////////////////////////////////////////////////////
// File:        component.h
// Author:      Cesar Mauri Loba (cesar at crea-si dot com)
// Copyright:   (C) 2010 Cesar Mauri Loba - CREA Software Systems
//
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
/////////////////////////////////////////////////////////////////////////////
#ifndef SPCORE_COMPONENT_H
#define SPCORE_COMPONENT_H

#include "spcore/baseobj.h"
#include "spcore/coreruntime.h"
#include <vector>
using namespace std;
#include <string>
using namespace std;

#include "spcore/pin.h"
#include "spcore/iterator.h"
#include "spcore/configuration.h"
#include <algorithm>
#include <string.h>
#include <boost/type_traits.hpp>
#include <boost/static_assert.hpp>

// Forward declarations
namespace spcore { 
	template<class T> class IIterator; 
	class IInputPin; 
	class IOutputPin; 
}
// TODO: typedef as a neutral window type
class wxWindow;

namespace spcore {

///**
//Base class for any kind of component
//*/
class IComponent : public IBaseObject {
  protected:
    inline virtual ~IComponent();


  public:
    ///**
    //Return the name of the component
    //*/
    virtual const char* GetName() const = 0;

    virtual const char* GetTypeName() const = 0;

    ///**
    //Opens a window for this component if has one and returns a pointer to it or NULL.
    //The lifetime of this instance belongs to the component and should never be deleted.
    //*/
    virtual wxWindow* GetGUI(wxWindow * parent) = 0;

    // Return 0 when OK
    virtual int AddChild(SmartPtr<IComponent> component) = 0;

    virtual SmartPtr<IIterator<IComponent*> > QueryComponents() = 0;

    ///**
    //Returns an iterator which allows to enumerate the input pins of the component.
    //Returned pointer should be deleted explicitly or (better) using
    //a IteratorPtr object.
    //*/
    virtual SmartPtr<IIterator<IInputPin*> > GetInputPins() = 0;

    ///**
    //Returns an iterator which allows to enumerate the output pins of the component.
    //Returned pointer should be deleted explicitly or (better) using
    //a IteratorPtr object.
    //*/
    virtual SmartPtr<IIterator<IOutputPin*> > GetOutputPins() = 0;

    ///**
    //Returns whether this component provides a thread of execution so that
    //calls to Start and Stop are meaningful.
    //*/
    virtual bool ProvidesExecThread() = 0;

    ///**
    // Returns whether this component needs to be explicitly Initialized / Finished.
    //*/
    virtual bool NeedsInitialization() = 0;

    ///**
    //Starts the components. This calle only makes sense when the component provides
    //an execution thread
    //\return 0 if successfully started, -1 otherwise
    //\sa ProvidesExecThread
    //*/
    virtual int Start() = 0;

    ///**
    //Stops the component. Only makes sense for components that provide an execution thread.
    //*/
    virtual void Stop() = 0;

    ///**
    //Initializes the component.
    //\return 0 if successfully initialized, -1 otherwise
    //\sa NeedsInitialization
    //*/
    virtual int Initialize() = 0;

    ///**
    //Finishes the component.
    //*/
    virtual void Finish() = 0;

	/*!
		\brief Store internal settings
		\param cfg Configuration object		
	*/
	virtual void SaveSettings(IConfiguration& cfg)= 0;

	/*!
		\brief Store internal settings
		\param cfg Configuration object		
	*/
	virtual void LoadSettings(IConfiguration& cfg)= 0;

    ///**
    //  Search an input pin given its name
	//
	//	\return A pointer to the pin or NULL if not found
    //*/
    static inline IInputPin* FindInputPin(IComponent & component, const char * name);

	///**
    //  Search an output pin given its name
	//
	//	\return A pointer to the pin or NULL if not found
    //*/
    static inline IOutputPin* FindOutputPin(IComponent & component, const char * name);
};
inline IComponent::~IComponent() {
}

///**
//  Search a pin given its name
//*/
inline IInputPin* IComponent::FindInputPin(IComponent & component, const char * name)
{
  	assert (name);
  	if (!name) return NULL;
  	SmartPtr<IIterator<IInputPin*> > ipit= component.GetInputPins();
  	for (; !ipit->IsDone(); ipit->Next()) {
  		if (strcmp(ipit->CurrentItem()->GetName(), name)== 0) return ipit->CurrentItem();
  	}
  	return NULL;
}

inline IOutputPin* IComponent::FindOutputPin(IComponent & component, const char * name)
{
	assert (name);
  	if (!name) return NULL;
  	SmartPtr<IIterator<IOutputPin*> > opit= component.GetOutputPins();
  	for (; !opit->IsDone(); opit->Next()) {
  		if (strcmp(opit->CurrentItem()->GetName(), name)== 0) return opit->CurrentItem();
  	}
  	return NULL;
}

///**
//Adapter class to implement IComponent conforming classes.
//
//This class is intended to be derived to implement leaf type components
//(i.e. without child components). If you need to implement a component
//that is composed of children use CCompositeComponentAdapter instead.
//*/
class CComponentAdapter : public IComponent {
public:
	// Constructor expects, apart from the name, an optional pointer
	// (can be NULL) to an string contaning initialization arguments
    inline CComponentAdapter(const char * name, int argc, const char * argv[]);

    inline virtual ~CComponentAdapter();

    inline virtual const char* GetName() const;

    ///**
    //Opens a window for this component if has one and returns a pointer to it or NULL.
    //
    //*/
    inline virtual wxWindow* GetGUI(wxWindow * parent);

    ///**
    //Register a child component.
    //\return 0 when success, <0 otherwise
    //NOTE the adapter implementation always return -1
    //*/

    inline virtual int AddChild(SmartPtr<IComponent> component);

    inline virtual SmartPtr<IIterator<IComponent*> > QueryComponents();

    inline virtual SmartPtr<IIterator<IInputPin*> > GetInputPins();

    inline virtual SmartPtr<IIterator<IOutputPin*> > GetOutputPins();

    // Tells whether this component provides a thread of execution so that
    // call to Start and Stop are meaningful
    inline virtual bool ProvidesExecThread();

    // Tells whether this component needs to be explicitly Initialized / Finished
    inline virtual bool NeedsInitialization();

    inline virtual int Start();

    inline virtual void Stop();

    inline virtual int Initialize();

    inline virtual void Finish();

	virtual void SaveSettings(IConfiguration&) {}

	virtual void LoadSettings(IConfiguration&) {}

  protected:
    // Registers a new input pin. Returns -1 in case of error (pin already registered)
    inline int RegisterInputPin(IInputPin & pin);

    // Registers a new input pin. Returns -1 in case of error (pin already registered)
    inline int RegisterOutputPin(IOutputPin & pin);

	// TODO: temporary operation which allows derived classes to change the name
	// associated with this component. Update project to allow an additional parameter
	// with a full command-line.
	//void SetName (const char * name) { m_name= name; }

  private:
    vector<IInputPin *> m_inputPins;

    vector<IOutputPin *> m_outputPins;

    string m_name;

};
inline CComponentAdapter::CComponentAdapter(const char * name, int, const char * []) {
	m_name= name;
	// TODO: currently args is not used
}

inline CComponentAdapter::~CComponentAdapter() {
  	vector<IInputPin*>::iterator iti;
  	for (iti= m_inputPins.begin(); iti!= m_inputPins.end(); ++iti)
  		(*iti)->Release();
  	m_inputPins.clear();

  	vector<IOutputPin*>::iterator ito;
  	for (ito= m_outputPins.begin(); ito!= m_outputPins.end(); ++ito)
  		(*ito)->Release();
  	m_outputPins.clear();
}

inline const char* CComponentAdapter::GetName() const {
  	return m_name.c_str();
}

///**
//Opens a window for this component if has one and returns a pointer to it or NULL.
//
//*/
inline wxWindow* CComponentAdapter::GetGUI(wxWindow * ) {
  	return NULL;
}

///**
//Register a child component.
//\return 0 when success, <0 otherwise
//NOTE the adapter implementation always return -1
//*/

inline int CComponentAdapter::AddChild(SmartPtr<IComponent>) {
  	return -1;
}

inline SmartPtr<IIterator<IComponent*> > CComponentAdapter::QueryComponents() {
  	return SmartPtr<IIterator<IComponent*> >(NULL, false);
}

inline SmartPtr<IIterator<IInputPin*> > CComponentAdapter::GetInputPins() {
  	return SmartPtr<IIterator<IInputPin*> >(new CIteratorVector<IInputPin *>(m_inputPins), false);
}

inline SmartPtr<IIterator<IOutputPin*> > CComponentAdapter::GetOutputPins() {
  	return SmartPtr<IIterator<IOutputPin*> >(new CIteratorVector<IOutputPin *>(m_outputPins), false);
}

// Tells whether this component provides a thread of execution so that
// call to Start and Stop are meaningful
inline bool CComponentAdapter::ProvidesExecThread() {
  	return false;
}

// Tells whether this component needs to be explicitly Initialized / Finished
inline bool CComponentAdapter::NeedsInitialization() {
  	return false;
}

inline int CComponentAdapter::Start() {
  	return false;
}

inline void CComponentAdapter::Stop() {
}

inline int CComponentAdapter::Initialize() {
  	return false;
}

inline void CComponentAdapter::Finish() {
}

// Registers a new input pin. Returns -1 in case of error (pin already registered)
inline int CComponentAdapter::RegisterInputPin(IInputPin & pin) {
  	vector<IInputPin*>::iterator it= find (m_inputPins.begin(), m_inputPins.end(), &pin);
  	if (it!= m_inputPins.end())	return -1; // Already registered

  	pin.AddRef();
  	m_inputPins.push_back (&pin);

  	return 0;
}

// Registers a new input pin. Returns -1 in case of error (pin already registered)
inline int CComponentAdapter::RegisterOutputPin(IOutputPin & pin) {
  	vector<IOutputPin*>::iterator it= find (m_outputPins.begin(), m_outputPins.end(), &pin);
  	if (it!= m_outputPins.end()) return -1; // Already registered

  	pin.AddRef();
  	m_outputPins.push_back (&pin);

  	return 0;
}

class CCompositeComponentAdapter : public CComponentAdapter {
  public:
    inline virtual ~CCompositeComponentAdapter();

    inline CCompositeComponentAdapter(const char * name, int argc, const char * argv[]);

    ///**
    // Registers a child component.
    //
    // It checks whether the component was already registered by comparing
    // the pointer address.
    //
    // \return 0 when OK, -1 if the component was already registered
    //*/
    inline virtual int AddChild(SmartPtr<IComponent> component);

    inline virtual SmartPtr<IIterator<IComponent*> > QueryComponents();

	virtual bool ProvidesExecThread() { return true; }

    // Tells whether this component needs to be explicitly Initialized / Finished
    virtual bool NeedsInitialization()  { return true; }

    inline virtual int Start();

    inline virtual void Stop();


  protected:
    inline virtual int Initialize();

    inline virtual void Finish();


  private:
    vector<IComponent *> m_components;

};
inline CCompositeComponentAdapter::~CCompositeComponentAdapter(){
	Stop();
	Finish();
	// Delete components
	for (vector<IComponent*>::iterator it= m_components.begin(); it!= m_components.end(); ++it)
		(*it)->Release();
}

inline CCompositeComponentAdapter::CCompositeComponentAdapter(const char * name, int argc, const char * argv[])
: CComponentAdapter(name, argc, argv) { }

///**
// Registers a child component.
//
// It checks whether the component was already registered by comparing
// the pointer address.
//
// \return 0 when OK, -1 if the component was already registered
//*/
inline int CCompositeComponentAdapter::AddChild(SmartPtr<IComponent> component) {

	//vector<IComponent*>::iterator it= find(m_components.begin(), m_components.end(), component);
	vector<IComponent*>::iterator it= m_components.begin();

	for(; it!= m_components.end(); ++it) {
		// Check if the component (by pointer) has been already registered.
		// TODO: check that component is not registered (comparing pointers)
		// anywhere in this component composite
		if ((*it)== component.get()) break;

		// Check also by name
		if (strcmp((*it)->GetName(), component->GetName())== 0) break;
	}

	if (it!= m_components.end()) {
		// Component already registered
		return -1;
	}

	// Finaly add component to vector
	component->AddRef();
	m_components.push_back(component.get());

	return 0;
}

inline SmartPtr<IIterator<IComponent*> > CCompositeComponentAdapter::QueryComponents() {
  	return SmartPtr<IIterator<IComponent*> >(new CIteratorVector<IComponent*>(m_components), false);
}

inline int CCompositeComponentAdapter::Start() {
	int retval= Initialize();
	if (retval!= 0) return retval;

	vector<IComponent*>::iterator it;
	for (it= m_components.begin(); retval== 0 && it!= m_components.end(); ++it)
		if ((*it)->ProvidesExecThread()) retval= (*it)->Start();

	if (retval!= 0) Stop();

	return retval;
}

inline void CCompositeComponentAdapter::Stop() {
	vector<IComponent*>::iterator it;
	for (it= m_components.begin(); it!= m_components.end(); ++it)
		if ((*it)->ProvidesExecThread()) (*it)->Stop();
	//Finish();
}

inline int CCompositeComponentAdapter::Initialize() {
	int retval= 0;

	vector<IComponent*>::iterator it;
	for (it= m_components.begin(); retval== 0 && it!= m_components.end(); ++it)
		if ((*it)->NeedsInitialization()) retval= (*it)->Initialize();

	if (retval!= 0) Finish();

	return retval;
}

inline void CCompositeComponentAdapter::Finish() {
	Stop();
	vector<IComponent*>::iterator it;
	for (it= m_components.begin(); it!= m_components.end(); ++it)
		if ((*it)->NeedsInitialization()) (*it)->Finish();
}

class IComponentFactory : public IBaseObject {
  protected:
    inline virtual ~IComponentFactory();


  public:
    virtual const char* GetName() const = 0;

    virtual SmartPtr<IComponent> CreateInstance(const char * name, int argc, const char * argv[]) = 0;

};
inline IComponentFactory::~IComponentFactory() {
}


/**
	Connects an output pin of a component with an input of another component.
	It uses pin number to refer to a specific pin

	\return 0 when success
	\return -1 invalid component
	\return -2 trying to connect the component with itself
	\return -3 invalid pin number
	\return -4 pin type mismatch
*/
inline int Connect(IComponent * srcComponent, unsigned int srcPin, IComponent * dstComponent, unsigned int dstPin) {
  	if (srcComponent== NULL || dstComponent== NULL) return -1;	// Invalid component
  	if (srcComponent== dstComponent) return -2;	// Forbids connecting pins of the same component

  	unsigned int i;
	SmartPtr<IIterator<IOutputPin*> > itop= srcComponent->GetOutputPins();
  	itop->First();
  	i= 0;
  	while (i!= srcPin && !itop->IsDone()) {
  		itop->Next();
  		++i;
  	}
  	if (itop->IsDone()) return -3;	// Invalid pin number

  	SmartPtr<IIterator<IInputPin*> > itip= dstComponent->GetInputPins();
  	itip->First();
  	i= 0;
  	while (i!=dstPin && !itip->IsDone()) {
  		itip->Next();
  		++i;
  	}
  	if (itip->IsDone()) return -3; // Invalid pin number

  	if (itop->CurrentItem()->Connect (*(itip->CurrentItem()))!= 0) return -4; // Pin type mismatch

  	return 0;
}


/**
	Connects an output pin of a component with an input of another component.
	It uses pin name to refer to a specific pin

	\return 0 when success
	\return -1 invalid component
	\return -2 trying to connect the component with itself
	\return -3 invalid pin name
	\return -4 pin type mismatch
*/

inline int Connect(IComponent * srcComponent, const char * srcPin, IComponent * dstComponent, const char * dstPin) {
  	if (srcComponent== NULL || dstComponent== NULL) return -1;	// Invalid component
  	if (srcComponent== dstComponent) return -2;	// Forbids connecting pins of the same component
	if (srcPin== NULL || dstPin== NULL) return -3;	// Invalid pin name

  	SmartPtr<IIterator<IOutputPin*> > itop= srcComponent->GetOutputPins();
	for (itop->First(); !itop->IsDone(); itop->Next()) {
		if (strcmp(srcPin, itop->CurrentItem()->GetName())== 0) break;
	}
	if (itop->IsDone()) return -3;	// Invalid pin name

  	SmartPtr<IIterator<IInputPin*> > itip= dstComponent->GetInputPins();
  	for (itip->First(); !itip->IsDone(); itip->Next()) {
		if (strcmp(dstPin, itip->CurrentItem()->GetName())== 0) break;
	}
  	if (itip->IsDone()) return -3; // Invalid pin name

  	if (itop->CurrentItem()->Connect (*(itip->CurrentItem()))!= 0) return -4; // Pin type mismatch

  	return 0;
}

/**
	Helper template to create simple component factories
*/
template <typename COTYPE>
class ComponentFactory : public spcore::IComponentFactory
{
	BOOST_STATIC_ASSERT( (boost::is_base_of<IComponent,COTYPE>::value) );
public:
	virtual const char* GetName() const { return COTYPE::getTypeName(); }

	virtual SmartPtr<IComponent> CreateInstance(const char * name, int argc, const char * argv[]) {
		std::string exceptionMessage;
		try {
			return SmartPtr<IComponent>(new COTYPE(name, argc, argv), false);
		}
		catch(std::exception& e) {
			exceptionMessage= e.what();
		}
		catch(...) {
			exceptionMessage= "unexpected error creating component: " + std::string(name);
		}
		// If code reaches this point means that an exception has been raised
		// signal this error condition by adding a log entry and retuning a
		// null instance
		std::string msg("error creating instance:");
		msg.append(name);
		if (exceptionMessage.size()> 0) {
			msg.append (":");
			msg.append (exceptionMessage);
		}
		getSpCoreRuntime()->LogMessage(ICoreRuntime::LOG_ERROR, msg.c_str(), "spcore");
		return SmartPtr<IComponent>(NULL);
	}
};

/**
	Helper template to create simple singleton component factories
*/
template <typename COTYPE>
class SingletonComponentFactory : public spcore::IComponentFactory
{
	BOOST_STATIC_ASSERT( (boost::is_base_of<IComponent,COTYPE>::value) );
public:
	virtual const char* GetName() const { return COTYPE::getTypeName(); }

	virtual SmartPtr<IComponent> CreateInstance(const char * name, int argc, const char * argv[]) {
		if (m_instance.get()) return m_instance;

		// Crete instance for the first time
		std::string exceptionMessage;
		try {
			m_instance= SmartPtr<COTYPE>(new COTYPE(name, argc, argv), false);
			return m_instance;
		}
		catch(std::exception& e) {
			exceptionMessage= e.what();
		}
		catch(...) {
			exceptionMessage= "unexpected error creating component: " + std::string(name);
		}
		// If code reaches this point means that an exception has been raised
		// signal this error condition by adding a log entry and retuning a
		// null instance
		assert (m_instance.get()== NULL);
		std::string msg("error creating instance:");
		msg.append(name);
		if (exceptionMessage.size()> 0) {
			msg.append (":");
			msg.append (exceptionMessage);
		}
		getSpCoreRuntime()->LogMessage(ICoreRuntime::LOG_ERROR, msg.c_str(), "spcore");
		return SmartPtr<IComponent>(NULL);
	}
private:
	SmartPtr<COTYPE> m_instance;
};

} // namespace spcore
#endif
