/* deamoninit.c -- a way to run a desired process as a deamon.. 

   Copyright (C) Kenneth 'Redhead' Nielsen <kn@redhead.dk>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */


/****
 * $Id: deamoninit.c,v 1.3 2003/07/22 22:03:42 redhead Exp $ 
 * $Log: deamoninit.c,v $
 * Revision 1.3  2003/07/22 22:03:42  redhead
 * Cleaned up the code, which included rewriting every structure in the program.
 * Fixed some error checking, to make sure theres no undocumentet parts that might
 * produce an error.
 *
 * Revision 1.2  2002/10/18 18:25:23  redhead
 * Added the use of boxes to create a surrounding box around the signature
 * This introduced the --with-boxes flag to configure.
 * If that flag isn't used it will default be assumed boxes isn't installed
 * Else the configure script will search for boxes, if none should be found
 * in the search path, you can use --with-boxes=/where to specify it.
 *
 ****/


#include "deamoninit.h" /* include "config.h" */

#ifdef sigit


/*
 * Just a simple deamon making function, as suggested in 
 * Unix Network programming, second edition.
 * fork() and close() all file descriptors with stderr/stdout/stdin/etc, and 
 * finaly chroot your self to be running on your own.
 */

void deamon_init(const char* pname, int facility)
{
  int i;
  unsigned long pid;
  if((pid=fork())!=0)
    exit(0);
  setsid();
  signal(SIGHUP, SIG_IGN);
  if((pid=fork())!=0)
    exit(0);
  chdir("/");
  umask(0);
  for(i=0; i < MAXFD; i++)
    close(i);
}


/* 
 * A pretty simple signal handler, it will 
 * catch any signal(ignoring SIGPIPE) and terminate 
 * If I can get it to make an action the 
 * remove_pid_file() should be the final call in it.
 *
 */


void install_sig_handler(unsigned short int verbose)
{
  sigset_t s, s1, s2;
  
  if(verbose > 0)  
    printf("[install_sig_handler]\t: Starting signal handler.\n");
  
  /* Block everything first (in particular SIGPIPE) */
  sigfillset(&s);
  pthread_sigmask(SIG_BLOCK, &s, &s2);   
  sigemptyset(&s);                       
  sigaddset(&s, SIGUSR1);      /* For stopper() */
  sigaddset(&s, SIGINT);       /* For ^C */
  sigaddset(&s, SIGTERM);      /* For kill -15 */        
  sigaddset(&s, SIGSEGV);      /* For kill -11 */
  pthread_sigmask(SIG_UNBLOCK, &s, &s1);
  
  if(verbose > 0)
    printf("[install_sig_handler]\t:   DONE\n");
  
  /*** Well the things down here, is to add the same functionaleties in the sig_handler ***/
  /*** Just need to figure out how to make it useable ***/
 /*
  struct sigaction act;
 

  act.sa_handler = caught_signal;
  sigemptyset(&act.sa_mask);
  act.sa_flags = 0;
  sigaction(SIGINT, &act, 0);
  sigaction(SIGHUP, &act, 0);
  sigaction(SIGTERM, &act, 0);
  act.sa_handler = ignore_sigpipe;
  sigemptyset(&act.sa_mask);
  sigaction(SIGPIPE, &act, 0);

  act.sa_sigaction = caught_sigsegv;
  sigemptyset(&act.sa_mask);
  sigaction(SIGSEGV, &act, 0);
*/
}


/* 
 * In order to init the fifo, as a regular fifo.  
 * So we wont get _alot_ of problems with it not beeing correct.
 *
 * It should provide the needed blocking by it self, when it is created as a true fifo.
 */

int init_fifo(FIFO* fifo)
{
  if(fifo->verbose > 0)
    logging(fifo, "[init_fifo]\t\t: Creating [%s] as fifo file.\n", fifo->fifo_name);
  if (!access(fifo->fifo_name, F_OK)) 
    {
      /* FIFO already exists */
      if (unlink(fifo->fifo_name) == -1) 
	{
	  if(fifo->verbose > 0)
	    logging(fifo, "[init_fifo]\t\t: Couldn't unlink the old {%s} fifo.\n", fifo->fifo_name);
	  return FAILURE;
	}
    }
  if (mkfifo(fifo->fifo_name, 0644) == -1) 
    {
      if(fifo->verbose > 0)
	logging(fifo, "[init_fifo]\t\t: Couldn't create {%s} as fifo.\n", fifo->fifo_name);
      return FAILURE;
    }
  if(fifo->verbose > 0)
    logging(fifo, "[init_fifo]\t\t:    DONE\n");
  return SUCCESS;
}

/* 
 * Ironicaly, this function should be called on an interrupt.
 * but since I cant get the sigaction aswell as the right fifo
 * parsed to it, the function isn't called here.. 
 * Else it will see to, that the fifo is propperly removed.
 */

int unlink_fifo(FIFO* fifo)
{
  if(fifo->verbose > 0)
    logging(fifo,"[unlink_fifo]\t\t: Unlinking fifo file.\n");
  unlink(fifo->fifo_name);
  remove_pid_file();
  if(fifo->verbose > 0)
    logging(fifo, "[unlink_fifo]\t\t:    DONE\n");
  return SUCCESS;
}


#endif

