# Samizdat image renderer plugin
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'
require 'samizdat/plugins/content_file'
require 'fileutils'
require 'RMagick'

class ImagePlugin < ContentFilePlugin
  include Magick

  def match?(format)
    not IMAGE_FORMAT_PATTERN.match(format).nil?
  end

  def render(request, mode, content)
    title = CGI.escapeHTML(content.title.to_s)
    image_location = File.join(request.content_location, content.file.location)

    version = mode.to_s
    exists = version_exists?(content.file, version)

    link =
      if content.id.kind_of?(Integer) and 'short' == version
        content.id.to_s
      elsif 'full' == version and not exists
        nil
      else
        image_location
      end

    if exists
      image_location = version_path(content.file, image_location, version)
    end

    html = %{<img alt="#{title}" src="#{image_location}" />}

    if link
      html = %{<a class="image" href="#{link}">#{html}<div>} + _('Click to enlarge') + '</div></a>'
    end

    html
  end

  def new_file(file)
    return if versions.empty?   # nothing to generate

    original = ImageList.new(file.path)

    versions.each do |version|
      limit = @options[version].to_i
      next unless limit > 0 and
        (original.columns > limit or original.rows > limit)

      scale = limit.to_f / [ original.columns, original.rows ].max
      original.copy.resize(scale).write(version_path(file, file.path, version))
    end
  end

  def move_file(file, new_id)
    each_version(file) do |version|
      destination = version_path(file, file.path(new_id), version)

      File.exists?(File.dirname(destination)) or
        FileUtils.mkdir_p(File.dirname(destination))

      File.rename(version_path(file, file.path, version), destination)
    end
  end

  def delete_file(file)
    each_version(file) do |version|
      File.delete(version_path(file, file.path, version))
    end
  end

  private

  IMAGE_FORMAT_PATTERN = Regexp.new(%r{^image/}).freeze

  VERSION_NAME_PATTERN = Regexp.new(/\A[[:alnum:]_]+\z/).freeze

  def versions
    @options.kind_of?(Hash) ? @options.keys.grep(VERSION_NAME_PATTERN) : []
  end

  def version_path(file, path, version)
    ext = file.extension
    path.sub(%r{\.#{ext}\z}, '.' + version.to_s + '.' + ext)
  end

  def version_exists?(file, version)
    File.exists?(version_path(file, file.path, version))
  end

  def each_version(file)
    versions.each do |version|
      yield(version) if version_exists?(file, version)
    end
  end
end

PluginClasses.instance['image'] = ImagePlugin
