# -*- coding: utf-8 -*-

# Copyright (C) 2010-2011 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# Python X2go is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Python X2go is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""\
X2goTerminalSession class - core functions for handling your individual X2go sessions.

This backend handles X2go server implementations that respond with session infos 
via server-side STDOUT and use NX3 as graphical proxy.

"""
__NAME__ = 'x2goterminalsession-pylib'

# modules
import os
import sys
import types
import gevent
import threading
import signal
import cStringIO
import copy
import shutil

# Python X2go modules
import x2go.rforward as rforward
import x2go.sftpserver as sftpserver
import x2go.printqueue as printqueue
import x2go.mimebox as mimebox
import x2go.log as log
import x2go.defaults as defaults
import x2go.utils as utils
import x2go.x2go_exceptions as x2go_exceptions

from x2go.cleanup import x2go_cleanup

# we hide the default values from epydoc (that's why we transform them to _UNDERSCORE variables)
from x2go.defaults import X2GOCLIENT_OS as _X2GOCLIENT_OS
from x2go.defaults import LOCAL_HOME as _LOCAL_HOME
from x2go.defaults import CURRENT_LOCAL_USER as _CURRENT_LOCAL_USER
from x2go.defaults import X2GO_CLIENT_ROOTDIR as _X2GO_CLIENT_ROOTDIR
from x2go.defaults import X2GO_SESSIONS_ROOTDIR as _X2GO_SESSIONS_ROOTDIR
from x2go.defaults import X2GO_GENERIC_APPLICATIONS as _X2GO_GENERIC_APPLICATIONS

from x2go.backends.info import X2goServerSessionInfo as _X2goServerSessionInfo
from x2go.backends.info import X2goServerSessionList as _X2goServerSessionList
from x2go.backends.proxy import X2goProxy as _X2goProxy
from x2go.backends.printing import X2goClientPrinting as _X2goClientPrinting

_local_color_depth = utils.local_color_depth()

def _rewrite_cmd(cmd, params=None):

    # start with an empty string
    cmd = cmd or ''

    # find window manager commands
    if cmd in defaults.X2GO_DESKTOPSESSIONS.keys():
        cmd = defaults.X2GO_DESKTOPSESSIONS[cmd]

    if (cmd == 'RDP') and (type(params) == X2goSessionParams):
        if params.geometry == 'fullscreen':
            cmd = 'rdesktop -f -N %s %s -a %s' % (params.rdp_options, params.rdp_server, params.depth)
        else:
            cmd = 'rdesktop -g %s -N %s %s -a %s' % (params.geometry, params.rdp_options, params.rdp_server, params.depth)

    # place quot marks around cmd if not empty string
    if cmd:
        cmd = '"%s"' % cmd
    return cmd


def _rewrite_blanks(cmd):
    # X2go run command replace X2GO_SPACE_CHAR string with blanks
    if cmd:
        cmd = cmd.replace(" ", "X2GO_SPACE_CHAR")
    return cmd


class X2goSessionParams(object):
    """\
    The L{X2goSessionParams} class is used to store all parameters that
    C{X2goTerminalSession} backend objects are constructed with.

    """
    def rewrite_session_type(self):
        """\
        Rewrite the X2go session type, so that the X2go server
        can understand it (C{desktop} -> C{D}).

        Also if the object's C{command} property is a known window 
        manager, the session type will be set to 'D' 
        (i.e. desktop).

        @return: 'D' if session should probably a desktop session,
            'R' (for rootless) else
        @rtype: str

        """
        session_type = self.session_type
        cmd = self.cmd

        if session_type in ("D", "desktop"):
            self.session_type = 'D'
            return
        elif session_type in ("S", "shared", "shadow"):
            self.session_type = 'S'
            return
        elif cmd:
            if cmd == 'RDP':
                self.session_type = 'R'
                return
            elif cmd.startswith('rdesktop'):
                self.session_type = 'R'
                return
            elif cmd == 'XDMCP':
                self.session_type = 'D'
                return
            elif cmd in defaults.X2GO_DESKTOPSESSIONS.keys():
                self.session_type = 'D'
                return
            elif os.path.basename(cmd) in defaults.X2GO_DESKTOPSESSIONS.values():
                self.session_type = 'D'
                return
        self.session_type = 'R'

    def update(self, properties_to_be_updated={}):
        """\
        Update all properties in the object L{X2goSessionParams} object from
        the passed on dictionary.

        @param properties_to_be_updated: a dictionary with L{X2goSessionParams}
            property names as keys und their values to be update in 
            L{X2goSessionParams} object.
        @type properties_to_be_updated: dict

        """
        for key in properties_to_be_updated.keys():
            setattr(self, key, properties_to_be_updated[key] or '')
        self.rewrite_session_type()


class X2goTerminalSessionSTDOUT(object):
    """\
    Class for managing X2go sessions on a remote X2go server via Paramiko/SSH. 
    With the L{X2goTerminalSessionSTDOUT} class you can start new X2go sessions, resume suspended 
    sessions or suspend resp. terminate currently running sessions on a 
    connected X2go server.

    When suspending or terminating sessions there are two possible ways:

        1. Initialize an X2go session object, start a new session (or resume)
        and use the L{X2goTerminalSessionSTDOUT.suspend()} or L{X2goTerminalSessionSTDOUT.terminate()} method
        to suspend/terminate the current session object.
        2. Alternatively, you can pass a session name to L{X2goTerminalSessionSTDOUT.suspend()}
        or L{X2goTerminalSessionSTDOUT.terminate()}. If a session of this name exists on the
        X2go server the respective action will be performed on the session.

    An L{X2goTerminalSessionSTDOUT} object uses two main data structure classes: 

        - L{X2goSessionParams}: stores all parameters that have been passed to the 
        constructor method.

        - C{X2goServerSessionInfo} backend class: when starting or resuming a session, an object of this class 
        will be used to store all information retrieved from the X2go server.


    """
    def __init__(self, control_session, session_info=None,
                 geometry="800x600", depth=_local_color_depth, link="adsl", pack="16m-jpeg-9", 
                 cache_type="unix-kde", 
                 keyboard='', kblayout='null', kbtype='null/null',
                 session_type="application", snd_system='pulse', snd_port=4713, cmd=None,
                 rdp_server=None, rdp_options=None,
                 xdmcp_server=None,
                 convert_encoding=False, server_encoding='UTF-8', client_encoding='UTF-8',
                 rootdir=None,
                 profile_name='UNKNOWN', profile_id=utils._genSessionProfileId(),
                 print_action=None, print_action_args={},
                 info_backend=_X2goServerSessionInfo,
                 list_backend=_X2goServerSessionList,
                 proxy_backend=_X2goProxy, proxy_options={},
                 printing_backend=_X2goClientPrinting,
                 client_rootdir=os.path.join(_LOCAL_HOME, _X2GO_CLIENT_ROOTDIR),
                 sessions_rootdir=os.path.join(_LOCAL_HOME, _X2GO_SESSIONS_ROOTDIR),
                 session_instance=None,
                 logger=None, loglevel=log.loglevel_DEFAULT):
        """\
        Initialize an X2go session. With the L{X2goTerminalSessionSTDOUT} class you can start
        new X2go sessions, resume suspended sessions or suspend resp. terminate
        currently running sessions on a connected X2go server.

        @param geometry: screen geometry of the X2go session. Can be either C{<width>x<height>}
            or C{fullscreen}
        @type geometry: str
        @param depth: color depth in bits (common values: C{16}, C{24})
        @type depth: int
        @param link: network link quality (either one of C{modem}, C{isdn}, C{adsl}, C{wan} or C{lan})
        @type link: str
        @param pack: compression method for NX based session proxying
        @type pack: str
        @param cache_type: a dummy parameter that is passed to the L{X2goProxyBASE}. In NX Proxy 
            (class C{X2goProxyNX3}) this originally is the session name. With X2go it 
            defines the name of the NX cache directory. Best is to leave it untouched.
        @type cache_type: str
        @param kblayout: keyboard layout, e.g. C{us} (default), C{de}, C{fr}, ...
        @type kblayout: str
        @param kbtype: keyboard type, e.g. C{pc105/us} (default), C{pc105/de}, ...
        @type kbtype: str
        @param session_type: either C{desktop}, C{application} (rootless session) or C{shared}
        @type session_type: str
        @param snd_system: sound system to be used on server (C{none}, C{pulse} (default), 
            C{arts} (obsolete) or C{esd})
        @type snd_system: str
        @param cmd: command to be run on X2go server after session start (only used
            when L{X2goTerminalSessionSTDOUT.start()} is called, ignored on resume, suspend etc.
        @type cmd: str
        @param rootdir: X2go session directory, normally C{~/.x2go}
        @type rootdir: str
        @param info_backend: backend for handling storage of server session information
        @type info_backend: C{X2goServerSessionInfo*} instance
        @param list_backend: backend for handling storage of session list information
        @type list_backend: C{X2goServerSessionList*} instance
        @param proxy_backend: backend for handling the X-proxy connections
        @type proxy_backend: C{X2goProxy*} instance
        @param print_action: either a print action short name (PDFVIEW, PDFSAVE, PRINT, PRINTCMD) or the
            resp. C{X2goPrintActionXXX} class (where XXX equals one of the given short names)
        @type print_action: str or class
        @param print_action_args: optional arguments for a given print_action (for further info refer to
            L{X2goPrintActionPDFVIEW}, L{X2goPrintActionPDFSAVE}, L{X2goPrintActionPRINT} and L{X2goPrintActionPRINTCMD})
        @type print_action_args: dict
        @param proxy_options: a set of very C{X2goProxy*} backend specific options; any option that is not known
            to the C{X2goProxy*} backend will simply be ignored
        @type proxy_options: C{dict}
        @param logger: you can pass an L{X2goLogger} object to the
            L{X2goTerminalSessionSTDOUT} constructor
        @type logger: L{X2goLogger} instance
        @param loglevel: if no L{X2goLogger} object has been supplied a new one will be
            constructed with the given loglevel
        @type loglevel: int

        """
        self.proxy = None
        self.proxy_subprocess = None
        self.proxy_options = proxy_options

        self.active_threads = []
        self.reverse_tunnels = {}

        self.print_queue = None
        self.mimebox_queue = None

        if logger is None:
            self.logger = log.X2goLogger(loglevel=loglevel)
        else:
            self.logger = copy.deepcopy(logger)
        self.logger.tag = __NAME__

        self.control_session = control_session
        self.reverse_tunnels = self.control_session.get_transport().reverse_tunnels

        self.client_rootdir = client_rootdir
        self.sessions_rootdir = sessions_rootdir

        self.params = X2goSessionParams()

        self.params.geometry = str(geometry)
        self.params.link = str(link)
        self.params.pack = str(pack)
        self.params.cache_type = str(cache_type)
        self.params.session_type = str(session_type)
        self.params.keyboard = str(keyboard)
        self.params.kblayout = str(kblayout)
        self.params.kbtype = str(kbtype)
        self.params.snd_system = str(snd_system)
        self.params.cmd = str(cmd)
        self.params.depth = str(depth)

        self.params.rdp_server = str(rdp_server)
        self.params.rdp_options = str(rdp_options)
        self.params.xdmcp_server = str(xdmcp_server)

        self.params.convert_encoding = convert_encoding
        self.params.client_encoding = str(client_encoding)
        self.params.server_encoding = str(server_encoding)

        self.params.rootdir = (type(rootdir) is types.StringType) and rootdir or self.sessions_rootdir
        self.params.update()

        self.profile_name = profile_name
        self.proxy_backend = proxy_backend

        self.snd_port = snd_port
        self.print_action = print_action
        self.print_action_args = print_action_args
        self.printing_backend = printing_backend
        self.session_instance = session_instance
        if self.session_instance:
            self.client_instance = self.session_instance.client_instance
        else:
            self.client_instance = None

        self._mk_sessions_rootdir(self.params.rootdir)

        self.session_info = session_info
        if self.session_info is not None:
            if self.session_info.name:
                self.session_info.local_container = os.path.join(self.params.rootdir, 'S-%s' % self.session_info.name)
            else:
                raise X2goTerminalSessionException('no valid session info availble')
        else:
            self.session_info = info_backend()

        self._cleaned_up = False

    def __del__(self):
        self._x2go_tidy_up()

    def _x2go_tidy_up(self):

        self.release_proxy()

        try:
            if self.control_session.get_transport() is not None:
                try:
                    for _tunnel in [ _tun[1] for _tun in self.reverse_tunnels[self.session_info.name].values() ]:
                        if _tunnel is not None:
                            _tunnel.__del__()
                except KeyError:
                    pass

            if self.print_queue is not None:
                self.print_queue.__del__()

            if self.mimebox_queue is not None:
                self.mimebox_queue.__del__()

        except AttributeError:
            pass

        self.session_info.clear()

    def _mk_sessions_rootdir(self, d):

        try:
            os.mkdir(d)
        except OSError, e:
            if e.errno == 17:
                # file exists
                pass
            else:
                raise OSError, e

    def _rm_session_dirtree(self):

        if self.session_info.name:
            shutil.rmtree('%s/S-%s' % (self.params.rootdir, self.session_info), ignore_errors=True)

    def _rm_desktop_dirtree(self):

        if self.session_info.display:
            shutil.rmtree('%s/S-%s' % (self.params.rootdir, self.session_info.display), ignore_errors=True)

    def get_session_name(self):
        """\
        STILL UNDOCUMENTED

        """
        return self.session_info.name

    def start_sound(self):
        """\
        Initialize Paramiko/SSH reverse forwarding tunnel for X2go sound.

        Currently supported audio protocols:

            - PulseAudio
            - Esound 

        """
        _tunnel = None
        if self.reverse_tunnels[self.session_info.name]['snd'][1] is None:
            if self.params.snd_system == 'pulse':
                self.logger('initializing PulseAudio sound support in X2go session', loglevel=log.loglevel_INFO)
                ###
                ### PULSEAUDIO
                ###
                if os.path.exists(os.path.normpath('%s/.pulse-cookie' % _LOCAL_HOME)):
                    # setup pulse client config file on X2go server
                    cmd_line = "echo 'default-server=127.0.0.1:%s'>%s/.pulse-client.conf;" % (self.session_info.snd_port, self.session_info.remote_container) + \
                               "echo 'cookie-file=%s/.pulse-cookie'>>%s/.pulse-client.conf" % (self.session_info.remote_container, self.session_info.remote_container)
                    (stdin, stdout, stderr) = self.control_session._x2go_exec_command(cmd_line)

                    self.control_session._x2go_sftp_put(local_path='%s/.pulse-cookie' % _LOCAL_HOME, remote_path='%s/.pulse-cookie' % self.session_info.remote_container)

                    # start reverse SSH tunnel for pulse stream
                    _tunnel = rforward.X2goRevFwTunnel(server_port=self.session_info.snd_port, 
                                                       remote_host='127.0.0.1', 
                                                       remote_port=self.snd_port, 
                                                       ssh_transport=self.control_session.get_transport(),
                                                       session_instance=self.session_instance,
                                                       logger=self.logger
                                                      )
                else:
                    if self.client_instance:
                        self.client_instance.HOOK_on_sound_tunnel_failed(profile_name=self.profile_name, session_name=self.session_info.name)
            elif self.params.snd_system == 'arts':
                ###
                ### ARTSD AUDIO
                ###
                self.logger('the ArtsD sound server (as in KDE3) is obsolete and will not be supported by Python X2go...', loglevel=log.loglevel_WARNING)

            elif self.params.snd_system == 'esd':
                ###
                ### ESD AUDIO
                ###

                self.logger('initializing ESD sound support in X2go session', loglevel=log.loglevel_INFO)
                self.control_session._x2go_sftp_put(local_path='%s/.esd_auth' % _LOCAL_HOME, remote_path='%s/.esd_auth' % self.control_session._x2go_remote_home)

                # start reverse SSH tunnel for pulse stream
                _tunnel = rforward.X2goRevFwTunnel(server_port=self.session_info.snd_port, 
                                                   remote_host='127.0.0.1', 
                                                   remote_port=self.snd_port, 
                                                   ssh_transport=self.control_session.get_transport(),
                                                   session_instance=self.session_instance,
                                                   logger=self.logger
                                                  )


            if _tunnel is not None:
                self.reverse_tunnels[self.session_info.name]['snd'] = (self.session_info.snd_port, _tunnel)
                _tunnel.start()
                self.active_threads.append(_tunnel)

        else:
            # tunnel has already been started and might simply need a resume call
            self.reverse_tunnels[self.session_info.name]['snd'][1].resume()

    def start_sshfs(self):
        """\
        Initialize Paramiko/SSH reverse forwarding tunnel for X2go folder sharing.

        """
        if not self.control_session.is_folder_sharing_available():
            raise x2go_exceptions.X2goUserException('remote user %s is not member of X2go server group fuse' % self.session_info.username)

        # start reverse SSH tunnel for sshfs (folder sharing, printing)
        ssh_transport = self.control_session.get_transport()
        if self.reverse_tunnels[self.session_info.name]['sshfs'][1] is None:

            _tunnel = sftpserver.X2goRevFwTunnelToSFTP(server_port=self.session_info.sshfs_port,
                                                       ssh_transport=ssh_transport,
                                                       auth_key=self.control_session._x2go_session_auth_rsakey,
                                                       session_instance=self.session_instance,
                                                       logger=self.logger
                                                      )

            if _tunnel is not None:
                self.reverse_tunnels[self.session_info.name]['sshfs'] = (self.session_info.sshfs_port, _tunnel)
                _tunnel.start()
                self.active_threads.append(_tunnel)

        else:
            # tunnel has already been started and might simply need a resume call
            self.reverse_tunnels[self.session_info.name]['sshfs'][1].resume()

    def _x2go_pause_rev_fw_tunnel(self, name):
        # pause reverse SSH tunnel of name <name>
        ssh_transport = self.get_transport()
        _tunnel = self.reverse_tunnels[self.session_info.name][name][1]
        if _tunnel is not None:
            _tunnel.pause()

    def stop_sound(self):
        """\
        Shutdown (pause) Paramiko/SSH reverse forwarding tunnel for X2go sound.

        """
        self._x2go_pause_rev_fw_tunnel('snd')

    def stop_sshfs(self):
        """\
        Shutdown (pause) Paramiko/SSH reverse forwarding tunnel for X2go folder sharing.

        """
        self._x2go_pause_rev_fw_tunnel('sshfs')

    def start_printing(self):
        """\
        Initialize X2go print spooling.

        """
        if self.session_info.username not in self.control_session._x2go_remote_group('x2goprint'):
            raise x2go_exceptions.X2goUserException('remote user %s is not member of X2go server group x2goprint' % self.session_info.username)

        spool_dir = os.path.join(self.session_info.local_container, 'spool')
        if not os.path.exists(spool_dir):
            os.mkdir(spool_dir)
        self.share_local_folder(local_path=spool_dir, folder_type='spool')
        self.print_queue = printqueue.X2goPrintQueue(profile_name=self.profile_name,
                                                     session_name=self.session_info.name,
                                                     spool_dir=spool_dir,
                                                     print_action=self.print_action, 
                                                     print_action_args=self.print_action_args,
                                                     client_instance=self.client_instance,
                                                     printing_backend=self.printing_backend,
                                                     logger=self.logger,
                                                    )
        self.print_queue.start()
        self.active_threads.append(self.print_queue)

    def set_print_action(self, print_action, **kwargs):
        """\
        STILL UNDOCUMENTED

        """
        self.print_queue.set_print_action(print_action, logger=self.logger, **kwargs)

    def stop_printing(self):
        """\
        Shutdown (pause) the X2go Print Queue thread.

        """
        if self.print_queue is not None:
            self.print_queue.pause()

    def get_printing_spooldir(self):
        """\
        Return the server-side printing spooldir path.

        """
        return '%s/%s' % (self.session_info.remote_container, 'spool')

    def start_mimebox(self, mimebox_extensions=[], mimebox_action=None):
        """\
        Initialize X2go mimebox handling.

        """
        mimebox_dir = os.path.join(self.session_info.local_container, 'mimebox')
        if not os.path.exists(mimebox_dir):
            os.mkdir(mimebox_dir)
        self.share_local_folder(local_path=mimebox_dir, folder_type='mimebox')
        self.mimebox_queue = mimebox.X2goMIMEboxQueue(profile_name=self.profile_name,
                                                      session_name=self.session_info.name,
                                                      mimebox_dir=mimebox_dir,
                                                      mimebox_extensions=mimebox_extensions,
                                                      mimebox_action=mimebox_action,
                                                      client_instance=self.client_instance,
                                                      logger=self.logger,
                                                     )
        self.mimebox_queue.start()
        self.active_threads.append(self.mimebox_queue)

    def set_mimebox_action(self, mimebox_action, **kwargs):
        """\
        STILL UNDOCUMENTED

        """
        self.mimebox_queue.set_mimebox_action(mimebox_action, logger=self.logger, **kwargs)

    def stop_mimebox(self):
        """\
        Shutdown (pause) the X2go MIME box Queue thread.

        """
        if self.mimebox_queue is not None:
            self.mimebox_queue.pause()

    def get_mimebox_spooldir(self):
        """\
        Return the server-side mimebox spooldir path.

        """
        return '%s/%s' % (self.session_info.remote_container, 'mimebox')

    def share_local_folder(self, local_path=None, folder_type='disk'):
        """\
        Share a local folder with the X2go session.

        @param local_path: the full path to an existing folder on the local 
            file system
        @type local_path: str
        @param folder_type: one of 'disk' (a folder on your local hard drive), 'rm' (removeable device), 
            'cdrom' (CD/DVD Rom) or 'spool' (for X2go print spooling)
        @type folder_type: str

        @return: returns C{True} if the local folder has been successfully mounted within the X2go server session
        @rtype: bool

        """
        if not self.control_session.is_folder_sharing_available():
            raise x2go_exceptions.X2goUserException('remote user %s is not member of X2go server group fuse' % self.session_info.username)

        if local_path is None:
            self.logger('no folder name given...', log.loglevel_WARN)
            return False

        if type(local_path) not in (types.StringType, types.UnicodeType):
            self.logger('folder name needs to be of type StringType...', log.loglevel_WARN)
            return False

        if not os.path.exists(local_path):
            self.logger('local folder does not exist: %s' % local_path, log.loglevel_WARN)
            return False

        local_path = os.path.normpath(local_path)
        self.logger('sharing local folder: %s' % local_path, log.loglevel_INFO)

        _auth_rsakey = self.control_session._x2go_session_auth_rsakey
        _host_rsakey = defaults.RSAHostKey

        _tmp_io_object = cStringIO.StringIO()
        _auth_rsakey.write_private_key(_tmp_io_object)
        _tmp_io_object.write('----BEGIN RSA IDENTITY----')
        _tmp_io_object.write('%s %s' % (_host_rsakey.get_name(),_host_rsakey.get_base64(),))

        # _x2go_key_fname must be a UniX path
        _x2go_key_fname = '%s/%s/%s' % (os.path.dirname(self.session_info.remote_container), 'ssh', 'key.z%s' % self.session_info.agent_pid)
        _x2go_key_bundle = _tmp_io_object.getvalue()

        self.control_session._x2go_sftp_write(_x2go_key_fname, _x2go_key_bundle)

        _convert_encoding = self.params.convert_encoding
        _client_encoding = self.params.client_encoding
        _server_encoding = self.params.server_encoding

        if _X2GOCLIENT_OS == 'Windows':
            local_path = local_path.replace('\\', '/')
            local_path = local_path.replace(':', '')
            local_path = '/windrive/%s' % local_path
            _convert_encoding = True
            _client_encoding = 'WINDOWS-1252'

        if _convert_encoding:
            export_iconv_settings = 'export X2GO_ICONV=modules=iconv,from_code=%s,to_code=%s &&' % (_client_encoding, _server_encoding)
        else:
            export_iconv_settings = ''

        if folder_type == 'disk':

            cmd_line = [ '%s export HOSTNAME &&' % export_iconv_settings,
                         'x2gomountdirs', 
                         'dir',
                         str(self.session_info.name), 
                         '"%s"' % _CURRENT_LOCAL_USER,
                         _x2go_key_fname,
                         '%s__REVERSESSH_PORT__%s; ' % (local_path, self.session_info.sshfs_port),
                         'rm -f %s %s.ident' % (_x2go_key_fname, _x2go_key_fname), 
                       ]

        elif folder_type == 'spool':

            cmd_line = [ '%s export HOSTNAME &&' % export_iconv_settings,
                         'x2gomountdirs', 
                         'dir',
                         str(self.session_info.name), 
                         '"%s"' % _CURRENT_LOCAL_USER,
                         _x2go_key_fname,
                         '%s__PRINT_SPOOL___REVERSESSH_PORT__%s; ' % (local_path, self.session_info.sshfs_port),
                         'rm -f %s %s.ident' % (_x2go_key_fname, _x2go_key_fname), 
                       ]

        elif folder_type == 'mimebox':

            cmd_line = [ '%s export HOSTNAME &&' % export_iconv_settings,
                         'x2gomountdirs', 
                         'dir',
                         str(self.session_info.name), 
                         '"%s"' % _CURRENT_LOCAL_USER,
                         _x2go_key_fname,
                         '%s__MIMEBOX_SPOOL___REVERSESSH_PORT__%s; ' % (local_path, self.session_info.sshfs_port),
                         'rm -f %s %s.ident' % (_x2go_key_fname, _x2go_key_fname), 
                       ]

        (stdin, stdout, stderr) = self.control_session._x2go_exec_command(cmd_line)
        _stdout = stdout.read().split('\n')
        self.logger('x2gomountdirs output is : %s' % _stdout, log.loglevel_NOTICE)
        if _stdout[5].endswith('ok'):
            return True
        return False

    def unshare_all_local_folders(self):
        """\
        Unshare all local folders mount in the X2go session.

        @return: returns C{True} if all local folders could be successfully unmounted from the X2go server session
        @rtype: bool

        """
        self.logger('unsharing all local folders from session %s' % self.session_info, log.loglevel_INFO)

        cmd_line = [ 'export HOSTNAME &&',
                     'x2goumount-session', 
                     self.session_info.name,
                   ]

        (stdin, stdout, stderr) = self.control_session._x2go_exec_command(cmd_line)
        if not stderr.read():
            self.logger('x2goumount-session (all mounts) for session %s has been successful' % self.session_info, log.loglevel_NOTICE)
            return True
        else:
            self.logger('x2goumount-session (all mounts) for session %s failed' % self.session_info, log.loglevel_ERROR)
            return False

    def unshare_local_folder(self, local_path):
        """\
        Unshare local folder given as <local_path> from X2go session.

        @return: returns C{True} if the local folder <local_path> could be successfully unmounted from the X2go server session
        @rtype: bool

        """
        self.logger('unsharing local folder from session %s' % self.session_info, log.loglevel_INFO)

        cmd_line = [ 'export HOSTNAME &&',
                     'x2goumount-session', 
                     self.session_info.name,
                     local_path,
                   ]

        (stdin, stdout, stderr) = self.control_session._x2go_exec_command(cmd_line)
        if not stderr.read():
            self.logger('x2goumount-session (%s) for session %s has been successful' % (local_path, self.session_info, ), log.loglevel_NOTICE)
            return True
        else:
            self.logger('x2goumount-session (%s) for session %s failed' % (local_path, self.session_info, ), log.loglevel_ERROR)
            return False

    def color_depth(self):
        """\
        Retrieve the session's color depth.

        @return: the session's color depth
        @rtype: C{int}

        """
        return self.params.depth

    def has_command(self, cmd):
        """\
        Verify if the command <cmd> exists on the X2go server.

        """
        test_cmd = None;

        cmd = cmd.strip('"').strip('"')
        if cmd.find('RDP') != -1:
            cmd = 'rdesktop'

        if cmd in _X2GO_GENERIC_APPLICATIONS:
            return True
        elif 'XSHAD' in cmd:
            return True
        elif cmd and cmd.startswith('/'):
            # check if full path is correct _and_ if application is in server path
            test_cmd = 'test -x %s && which %s && echo OK' % (cmd, os.path.basename(cmd.split()[0]))
        elif cmd and '/' not in cmd.split()[0]:
            # check if application is in server path only
            test_cmd = 'which %s && echo OK' % os.path.basename(cmd.split()[0])

        if test_cmd:
            (stdin, stdout, stderr) = self.control_session._x2go_exec_command([test_cmd])
            _stdout = stdout.read()
            return _stdout.find('OK') != -1
        else:
            return False

    def run_command(self, cmd=None, env={}):
        """\
        Run a command in this session.

        After L{X2goTerminalSessionSTDOUT.start()} has been called 
        one or more commands can be executed with L{X2goTerminalSessionSTDOUT.run_command()}
        within the current X2go session.

        @param cmd: Command to be run
        @type cmd: str

        @return: stdout.read() and stderr.read() as returned by the run command
            on the X2go server
        @rtype: tuple of str

        """
        if not self.has_command(_rewrite_cmd(self.params.cmd)):
            if self.client_instance:
                self.client_instance.HOOK_no_such_command(profile_name=self.profile_name, session_name=self.session_info.name, cmd=self.params.cmd)
            return False

        if cmd in ("", None):
            if self.params.cmd is None:
                cmd = 'TERMINAL'
            else:
                cmd = self.params.cmd

        if cmd == 'XDMCP':
            # do not run command when in XDMCP mode...
            return None

        if 'XSHAD' in cmd:
            # do not run command when in DESKTOP SHARING mode...
            return None

        self.params.update({'cmd': cmd})

        # do not allow the execution of full path names
        if '/' in cmd:
            cmd = os.path.basename(cmd)

        cmd_line = [ "setsid x2goruncommand", 
                     str(self.session_info.display),
                     str(self.session_info.agent_pid),
                     str(self.session_info.name), 
                     str(self.session_info.snd_port),
                     _rewrite_blanks(_rewrite_cmd(cmd, params=self.params)),
                     str(self.params.snd_system),
                     str(self.params.session_type),
                     "&> /dev/null & exit",
                   ]

        if self.params.snd_system == 'pulse':
            cmd_line = [ 'PULSE_CLIENTCONFIG=%s/.pulse-client.conf' % self.session_info.remote_container ] + cmd_line

        if env:
            for env_var in env.keys():
                cmd_line = [ '%s=%s' % (env_var, env[env_var]) ] + cmd_line

        (stdin, stdout, stderr) = self.control_session._x2go_exec_command(cmd_line)

        return stdout.read(), stderr.read()

    def ok(self):
        """\
        Returns C{True} if this X2go session is up and running, 
        C{False} else

        @return: X2go session OK?
        @rtype: bool

        """
        return bool(self.session_info.name and self.proxy.ok())

    def is_running(self):
        """\
        Returns C{True} if this X2go session is in running state,
        C{False} else.

        @return: X2go session running?
        @rtype: bool

        """
        return self.session_info.is_running()

    def is_suspended(self):
        """\
        Returns C{True} if this X2go session is in suspended state,
        C{False} else.

        @return: X2go session suspended?
        @rtype: bool

        """
        return self.session_info.is_suspended()

    def is_connected(self):
        """\
        Returns C{True} if this X2go session's Paramiko/SSH transport is 
        connected/authenticated, C{False} else.

        @return: X2go session connected?
        @rtype: bool
        """
        return self.control_session.is_connected()

    def start(self):
        """\
        Start a new X2go session.

        The L{X2goTerminalSession.start()} method accepts any parameter
        that can be passed to the class constructor.

        """
        if not self.has_command(_rewrite_cmd(self.params.cmd)):
            if self.client_instance:
                self.client_instance.HOOK_no_such_command(profile_name=self.profile_name, session_name=self.session_info.name, cmd=self.params.cmd)
            return False

        setkbd = "0"
        if self.params.kblayout or self.params.kbtype:
            setkbd = "1"

        cmd = self.params.cmd
        if '/' in cmd:
            cmd = os.path.basename(cmd)

        cmd_line = [ "x2gostartagent",
                     str(self.params.geometry),
                     str(self.params.link),
                     str(self.params.pack),
                     str(self.params.cache_type+'-depth_'+self.params.depth),
                     str(self.params.kblayout),
                     str(self.params.kbtype),
                     str(setkbd),
                     str(self.params.session_type),
                     cmd,
                   ]

        if self.params.cmd == 'XDMCP' and self.params.xdmcp_server:
            cmd_line = ['X2GOXDMCP=%s' % self.params.xdmcp_server] + cmd_line

        (stdin, stdout, stderr) = self.control_session._x2go_exec_command(cmd_line)

        _stdout = stdout.read()
        _stderr = stderr.read()

        # if the first line of stdout is a "DEN(Y)" string then we will presume that
        # we tried to use X2go desktop sharing and the sharing was rejected
        if "ACCESS DENIED" in _stderr and "XSHAD" in _stderr:
            raise x2go_exceptions.X2goDesktopSharingException('X2go desktop sharing has been denied by the remote user')

        try:
            self.session_info.initialize(_stdout,
                                         username=self.control_session.remote_username(),
                                         hostname=self.control_session.get_transport().getpeername(),
                                        )
        except ValueError:
            raise X2goTerminalSessionException("failed to start X2go session")
        except IndexError:
            raise X2goTerminalSessionException("failed to start X2go session")

        # local path may be a Windows path, so we use the path separator of the local system
        self.session_info.local_container = os.path.join(self.params.rootdir, 'S-%s' % self.session_info.name)
        # remote path is always a UniX path...
        self.session_info.remote_container = '%s/.x2go/C-%s' % (self.control_session._x2go_remote_home,
                                                                self.session_info.name,
                                                               )

        # let the proxy backend know that we want to define a very special keymap
        if (self.params.kbtype.endswith('defkeymap') and self.params.kblayout == 'defkeymap'):
            self.proxy_options.update({'defkeymap': True, })

        # set up SSH tunnel for X11 graphical elements
        self.proxy = self.proxy_backend(session_info=self.session_info, 
                                        ssh_transport=self.control_session.get_transport(),
                                        sessions_rootdir=self.sessions_rootdir,
                                        session_instance=self.session_instance,
                                        proxy_options=self.proxy_options,
                                        logger=self.logger)
        self.proxy_subprocess = self.proxy.start_proxy()
        self.active_threads.append(self.proxy)

        return self.ok()

    def resume(self):
        """\
        Resume a running/suspended X2go session. 

        The L{X2goTerminalSessionSTDOUT.resume()} method accepts any parameter
        that can be passed to the class constructor.

        @return: True if the session could be successfully resumed
        @rtype: bool

        """
        setkbd = "0"
        if self.params.kblayout or self.params.kbtype:
            setkbd = "1"

        cmd_line = [ "x2goresume-session", self.session_info.name,
                     self.params.geometry,
                     self.params.link,
                     self.params.pack,
                     self.params.kblayout,
                     self.params.kbtype,
                     setkbd,
                   ]

        (stdin, stdout, stderr) = self.control_session._x2go_exec_command(cmd_line)

        self.proxy = self.proxy_backend(session_info=self.session_info, 
                                        ssh_transport=self.control_session.get_transport(), 
                                        sessions_rootdir=self.sessions_rootdir,
                                        session_instance=self.session_instance,
                                        logger=self.logger
                                       )
        self.proxy_subprocess = self.proxy.start_proxy()

        # local path may be a Windows path, so we use the path separator of the local system
        self.session_info.local_container = os.path.join(self.params.rootdir, 'S-%s' % self.session_info.name)
        # remote path is always a UniX path...
        self.session_info.remote_container = '%s/.x2go/C-%s' % (self.control_session._x2go_remote_home, 
                                                                self.session_info.name,
                                                               )
        self.params.depth = self.session_info.name.split('_')[2][2:]
        # on a session resume the user name comes in as a user ID. We have to translate this...
        self.session_info.username = self.control_session.remote_username()
        return self.ok()

    def suspend(self):
        """\
        Suspend this X2go session terminal.

        @return: True if the session terminal could be successfully suspended
        @rtype: bool

        """
        self.control_session.suspend(session_name=self.session_info.name)
        self.release_proxy()

        # TODO: check if session has really suspended
        _ret = True

        return _ret

    def terminate(self):
        """\
        Terminate this X2go session.

        @return: True if the session terminal could be successfully terminate
        @rtype: bool

        """
        self.control_session.terminate(session_name=self.session_info.name, destroy_terminals=False)
        self.release_proxy()
        self.post_terminate_cleanup()
        self.__del__()

        # TODO: check if session has really suspended
        _ret = True

        return _ret

    def release_proxy(self):
        """\
        STILL UNDOCUMENTED

        """
        if self.proxy is not None:
            self.proxy.__del__()

    def post_terminate_cleanup(self):
        """\
        STILL UNDOCUMENTED

        """
        # this method might be called twice (directly and from update_status in the session
        # registry instance. So we have to make sure, that this code will not fail
        # if called twice.
        if not self._cleaned_up and self.session_info.name:

            # otherwise we wipe the session files locally
            self.logger('cleaning up session %s after termination' % self.session_info, loglevel=log.loglevel_NOTICE)

            # if we run in debug mode, we keep local session directories
            if self.logger.get_loglevel() & log.loglevel_DEBUG != log.loglevel_DEBUG:

                self._rm_session_dirtree()
                self._rm_desktop_dirtree()

            self._cleaned_up = True
