//
//  PRGrayscaleFilter.m
//  PRICE
//
//  Created by Riccardo Mottola on Mon Dec 23 2002.
//  Copyright (c) 2002-2010 Carduus. All rights reserved.
//
// This application is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.
// This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.


#import "PRGrayscaleFilter.h"
#include <math.h>


@implementation PRGrayscaleFilter

- (PRImage *)filterImage:(PRImage *)image with:(NSArray *)parameters progressPanel:(PRCProgress *)progressPanel
{
    int method;
      
    /* interpret the parameters */
    method = [[parameters objectAtIndex:0] intValue];
    
    return [self filterImage:image :method];
}

- (NSString *)actionName
{
    return @"Make Grayscale";
}

- (PRImage *)filterImage:(PRImage *)srcImage :(int)method 
{
    NSBitmapImageRep *srcImageRep;
    PRImage          *destImage;
    NSBitmapImageRep *destImageRep;
    int              w, h;
    int              x, y;
    unsigned char    *srcData;
    unsigned char    *destData;
    unsigned char    *p1;
    int              srcSamplesPerPixel;
    int              srcBytesPerRow;
    int              destBytesPerRow;
    
    /* get source image representation and associated information */
    srcImageRep = [srcImage bitmapRep];
    
    w = [srcImageRep pixelsWide];
    h = [srcImageRep pixelsHigh];
    srcBytesPerRow = [srcImageRep bytesPerRow];
    srcSamplesPerPixel = [srcImageRep samplesPerPixel];
    
    /* if the image is already greyscale... */
    if ([srcImageRep hasAlpha])
    {
        if ([srcImageRep samplesPerPixel] == 2)
            return srcImage;
    }
    else
    {
        if ([srcImageRep samplesPerPixel] == 1)
            return srcImage;
    }
    
    /* allocate destination image and its representation */
    destImage = [[PRImage alloc] initWithSize:NSMakeSize(w, h)];
    destImageRep = [[NSBitmapImageRep alloc]
                    initWithBitmapDataPlanes:NULL
                    pixelsWide:w
                    pixelsHigh:h
                    bitsPerSample:8
                    samplesPerPixel:1
                    hasAlpha:NO
                    isPlanar:NO
                    colorSpaceName:NSCalibratedWhiteColorSpace
                    bytesPerRow:0
                    bitsPerPixel:0];
    
    srcData = [srcImageRep bitmapData];
    destData = [destImageRep bitmapData];
    destBytesPerRow = [destImageRep bytesPerRow];

    if (method == METHOD_AVERAGE)
    {
        /* execute the actual filtering (R+G+B)/3 */
        for (y = 0; y < h; y++)
           for (x = 0; x < w; x++)
           {
               p1 = srcData + srcBytesPerRow * y  + x * srcSamplesPerPixel;
               destData[y*destBytesPerRow + x] = (unsigned char)rint((p1[0] + p1[1] + p1[2]) / 3);
           }
    } else if (method == METHOD_LUMINANCE)
    {
        /* execute the actual filtering
         * JPEG-YCbCr (601) from "digital 8-bit R'G'B'  "
         * Y' =       + 0.299    * R'd + 0.587    * G'd + 0.114    * B'd
         */
        for (y = 0; y < h; y++)
            for (x = 0; x < w; x++)
            {
                p1 = srcData + srcBytesPerRow * y  + x * srcSamplesPerPixel;
                destData[y*destBytesPerRow + x] = (unsigned char)rintf(0.2990f*p1[0] + 0.5870f*p1[1] + 0.1140f*p1[2]);
            }
                
    }
    [destImage addRepresentation:destImageRep];
    [destImageRep release];
    [destImage autorelease];
    return destImage;
}

- (BOOL)displayProgress
{
    return NO;
}


@end
