/* ========================================================================
 * PlantUML : a free UML diagram generator
 * ========================================================================
 *
 * (C) Copyright 2009-2017, Arnaud Roques
 *
 * Project Info:  http://plantuml.com
 * 
 * This file is part of PlantUML.
 *
 * Licensed under The MIT License (Massachusetts Institute of Technology License)
 * 
 * See http://opensource.org/licenses/MIT
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
 * IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 *
 * Original Author:  Arnaud Roques
 */
package net.sourceforge.plantuml.png;

import java.awt.image.RenderedImage;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;

import javax.imageio.ImageIO;

import net.sourceforge.plantuml.Log;

public class PngIO {

	private static final String copyleft = "Generated by http://plantuml.com";

	public static void write(RenderedImage image, File file, int dpi) throws IOException {
		write(image, file, null, dpi);
	}

	public static void write(RenderedImage image, OutputStream os, int dpi) throws IOException {
		write(image, os, null, dpi);
	}

	public static void write(RenderedImage image, File file, String metadata, int dpi) throws IOException {
		OutputStream os = null;
		try {
			os = new BufferedOutputStream(new FileOutputStream(file));
			write(image, os, metadata, dpi);
		} finally {
			if (os != null) {
				os.close();
			}
		}
		Log.debug("File is " + file);
		Log.debug("File size " + file.length());
		if (file.length() == 0) {
			Log.error("File size is zero: " + file);
			ImageIO.write(image, "png", file);
		}
	}

	public static void write(RenderedImage image, OutputStream os, String metadata, int dpi) throws IOException {
		write(image, os, metadata, dpi, null);
	}

	public static void write(RenderedImage image, OutputStream os, String metadata, int dpi, String debugData)
			throws IOException {
		if (metadata != null && checkPNGMetadata()) {
			PngIOMetadata.writeWithMetadata(image, os, metadata, dpi, debugData);
		} else {
			ImageIO.write(image, "png", os);
		}

	}

	static boolean checkPNGMetadata() {
		try {
			final Class cl = Class.forName("com.sun.imageio.plugins.png.PNGMetadata");
			if (cl == null) {
				Log.info("Cannot load com.sun.imageio.plugins.png.PNGMetadata");
				return false;
			}
			Log.info("Ok for com.sun.imageio.plugins.png.PNGMetadata");
			return true;
		} catch (Exception e) {
			Log.info("Error loading com.sun.imageio.plugins.png.PNGMetadata " + e);
			return false;
		}
	}

}
