/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment unit test for pgmtext.c
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <tests/check/common/pgmcheck.h>
#include <pgm/pgm.h>
#include <string.h>

/* test default values */
PGM_START_TEST (test_text_default_values)
{
  PgmError ret;
  PgmText *text;
  gchar *label;
  gchar *font_family;
  gfloat font_height;
  PgmTextEllipsize ellipsize;
  gboolean justify;
  PgmTextAlignment alignment;
  PgmTextWrap wrap;
  PgmTextGravity gravity;
  PgmTextStretch stretch;
  PgmTextStyle style;
  PgmTextVariant variant;
  PgmTextWeight weight;
  gfloat line_spacing;
  guchar red;
  guchar green;
  guchar blue;
  guchar alpha;
  gfloat outline_width;

  pgm_init (NULL, NULL);

  text = PGM_TEXT (pgm_text_new ("bla fasel"));

  ret = pgm_text_get_label (text, &label);
  fail_if (ret != PGM_ERROR_OK, "text_get_label error");
  fail_if (strncmp (label, "bla fasel", 9), "bad default label");

  ret = pgm_text_get_font_family (text, &font_family);
  fail_if (ret != PGM_ERROR_OK, "text_get_font_family error");
  fail_if (strncmp (font_family, "Sans", 4), "bad default font-family");

  ret = pgm_text_get_font_height (text, &font_height);
  fail_if (ret != PGM_ERROR_OK, "text_get_font_height error");
  fail_if (font_height != 1.0f, "bad default height");

  ret = pgm_text_get_ellipsize (text, &ellipsize);
  fail_if (ret != PGM_ERROR_OK, "text_get_ellipsize error");
  fail_if (ellipsize != PGM_TEXT_ELLIPSIZE_NONE, "bad default ellipsize");

  ret = pgm_text_get_justify (text, &justify);
  fail_if (ret != PGM_ERROR_OK, "text_get_justify error");
  fail_if (justify != FALSE, "bad default justify");

  ret = pgm_text_get_alignment (text, &alignment);
  fail_if (ret != PGM_ERROR_OK, "text_get_alignment error");
  fail_if (alignment != PGM_TEXT_ALIGN_LEFT, "bad default alignment");

  ret = pgm_text_get_wrap (text, &wrap);
  fail_if (ret != PGM_ERROR_OK, "text_get_wrap error");
  fail_if (wrap != PGM_TEXT_WRAP_WORD, "bad default wrapping");

  ret = pgm_text_get_gravity (text, &gravity);
  fail_if (ret != PGM_ERROR_OK, "text_get_gravity error");
  fail_if (gravity != PGM_TEXT_GRAVITY_AUTO, "bad default gravity");

  ret = pgm_text_get_stretch (text, &stretch);
  fail_if (ret != PGM_ERROR_OK, "text_get_stretch error");
  fail_if (stretch != PGM_TEXT_STRETCH_NORMAL, "bad default stretch");

  ret = pgm_text_get_style (text, &style);
  fail_if (ret != PGM_ERROR_OK, "text_get_style error");
  fail_if (style != PGM_TEXT_STYLE_NORMAL, "bad default style");

  ret = pgm_text_get_variant (text, &variant);
  fail_if (ret != PGM_ERROR_OK, "text_get_variant error");
  fail_if (variant != PGM_TEXT_VARIANT_NORMAL, "bad default variant");

  ret = pgm_text_get_weight (text, &weight);
  fail_if (ret != PGM_ERROR_OK, "text_get_weight error");
  fail_if (weight != PGM_TEXT_WEIGHT_NORMAL, "bad default weight");

  ret = pgm_text_get_line_spacing (text, &line_spacing);
  fail_if (ret != PGM_ERROR_OK, "text_get_line_spacing error");
  fail_if (line_spacing != 0.0f, "bad default line-spacing");

  ret = pgm_text_get_outline_color (text, &red, &green, &blue, &alpha);
  fail_if (ret != PGM_ERROR_OK, "text_get_outline_color error");
  fail_if (red != 0, "bad default outline-color (red)");
  fail_if (green != 0, "bad default outline-color (green)");
  fail_if (blue != 0, "bad default outline-color (blue)");
  fail_if (alpha != 0, "bad default outline-color (alpha)");

  ret = pgm_text_get_outline_width (text, &outline_width);
  fail_if (ret != PGM_ERROR_OK, "text_get_outline_width error");
  fail_if (outline_width != 0.0f, "bad default outline-width");

  gst_object_unref (GST_OBJECT_CAST (text));

  pgm_deinit ();
}
PGM_END_TEST;

/* test set/get correctness */
PGM_START_TEST (test_text_set_get_correctness)
{
  PgmError ret;
  PgmText *text;
  const gchar *label = "unit test";
  const gchar *markup = "<b>unit</b> <i>test</i>";
  gchar *lbl = NULL;
  const gchar *font_family = "DejaVu Sans";
  gchar *family  = NULL;
  gfloat font_height;
  PgmTextEllipsize ellipsize;
  gboolean justify;
  PgmTextAlignment alignment;
  PgmTextWrap wrap;
  PgmTextGravity gravity;
  PgmTextStretch stretch;
  PgmTextStyle style;
  PgmTextVariant variant;
  PgmTextWeight weight;
  gfloat line_spacing;
  guchar red;
  guchar green;
  guchar blue;
  guchar alpha;
  gfloat width;

  pgm_init (NULL, NULL);
  text = PGM_TEXT (pgm_text_new ("unit test"));

  /* label */
  ret = pgm_text_set_label (text, label);
  fail_if (ret != PGM_ERROR_OK, "text_set_label error");
  ret = pgm_text_get_label (text, &lbl);
  fail_if (ret != PGM_ERROR_OK, "text_get_label error");
  fail_if (strcmp (lbl, label), "label was wrong");
  g_free (lbl);

  /* markup */
  ret = pgm_text_set_markup (text, markup);
  fail_if (ret != PGM_ERROR_OK, "text_set_markup error");
  ret = pgm_text_get_label (text, &lbl);
  fail_if (ret != PGM_ERROR_OK, "text_get_label error (markup)");
  fail_if (strcmp (lbl, markup), "label was wrong (markup)");
  g_free (lbl);

  /* font-family */
  ret = pgm_text_set_font_family (text, font_family);
  fail_if (ret != PGM_ERROR_OK, "text_set_font_family error");
  ret = pgm_text_get_font_family (text, &family);
  fail_if (ret != PGM_ERROR_OK, "text_get_font_family error");
  fail_if (strcmp (family, font_family), "font-family was wrong");
  g_free (family);

  /* font-height */
  ret = pgm_text_set_font_height (text, 10.0f);
  fail_if (ret != PGM_ERROR_OK, "text_set_font_height error");
  ret = pgm_text_get_font_height (text, &font_height);
  fail_if (ret != PGM_ERROR_OK, "text_get_font_height error");
  fail_if (font_height != 10.0f, "font-height was wrong");

  /* ellipsize */
  ret = pgm_text_set_ellipsize (text, PGM_TEXT_ELLIPSIZE_END);
  fail_if (ret != PGM_ERROR_OK, "text_set_ellipsize error");
  ret = pgm_text_get_ellipsize (text, &ellipsize);
  fail_if (ret != PGM_ERROR_OK, "text_get_ellipsize error");
  fail_if (ellipsize != PGM_TEXT_ELLIPSIZE_END, "ellipsize was wrong");

  /* justify */
  ret = pgm_text_set_justify (text, FALSE);
  fail_if (ret != PGM_ERROR_OK, "text_set_justify error");
  ret = pgm_text_get_justify (text, &justify);
  fail_if (ret != PGM_ERROR_OK, "text_get_justify error");
  fail_if (justify != FALSE, "justify was wrong");

  /* alignment */
  ret = pgm_text_set_alignment (text, PGM_TEXT_ALIGN_CENTER);
  fail_if (ret != PGM_ERROR_OK, "text_set_alignment error");
  ret = pgm_text_get_alignment (text, &alignment);
  fail_if (ret != PGM_ERROR_OK, "text_get_alignment error");
  fail_if (alignment != PGM_TEXT_ALIGN_CENTER, "alignment was wrong");

  /* wrapping */
  ret = pgm_text_set_wrap (text, PGM_TEXT_WRAP_WORD_CHAR);
  fail_if (ret != PGM_ERROR_OK, "text_set_wrap error");
  ret = pgm_text_get_wrap (text, &wrap);
  fail_if (ret != PGM_ERROR_OK, "text_get_wrap error");
  fail_if (wrap != PGM_TEXT_WRAP_WORD_CHAR, "wrap was wrong");

  /* gravity */
  ret = pgm_text_set_gravity (text, PGM_TEXT_GRAVITY_WEST);
  fail_if (ret != PGM_ERROR_OK, "text_set_gravity error");
  ret = pgm_text_get_gravity (text, &gravity);
  fail_if (ret != PGM_ERROR_OK, "text_get_gravity error");
  fail_if (gravity != PGM_TEXT_GRAVITY_WEST, "gravity was wrong");

  /* stretch */
  ret = pgm_text_set_stretch (text, PGM_TEXT_STRETCH_EXPANDED);
  fail_if (ret != PGM_ERROR_OK, "text_set_stretch error");
  ret = pgm_text_get_stretch (text, &stretch);
  fail_if (ret != PGM_ERROR_OK, "text_get_stretch error");
  fail_if (stretch != PGM_TEXT_STRETCH_EXPANDED, "stretch was wrong");

  /* style */
  ret = pgm_text_set_style (text, PGM_TEXT_STYLE_ITALIC);
  fail_if (ret != PGM_ERROR_OK, "text_set_style error");
  ret = pgm_text_get_style (text, &style);
  fail_if (ret != PGM_ERROR_OK, "text_get_style error");
  fail_if (style != PGM_TEXT_STYLE_ITALIC, "style was wrong");

  /* variant */
  ret = pgm_text_set_variant (text, PGM_TEXT_VARIANT_SMALL_CAPS);
  fail_if (ret != PGM_ERROR_OK, "text_set_variant error");
  ret = pgm_text_get_variant (text, &variant);
  fail_if (ret != PGM_ERROR_OK, "text_get_style error");
  fail_if (variant != PGM_TEXT_VARIANT_SMALL_CAPS, "variant was wrong");

  /* weight */
  ret = pgm_text_set_weight (text, PGM_TEXT_WEIGHT_BOLD);
  fail_if (ret != PGM_ERROR_OK, "text_set_weight error");
  ret = pgm_text_get_weight (text, &weight);
  fail_if (ret != PGM_ERROR_OK, "text_get_weight error");
  fail_if (weight != PGM_TEXT_WEIGHT_BOLD, "weight was wrong");

  /* line-spacing */
  ret = pgm_text_set_line_spacing  (text, 15.0f);
  fail_if (ret != PGM_ERROR_OK, "text_set_line_spacing error");
  ret = pgm_text_get_line_spacing  (text, &line_spacing);
  fail_if (ret != PGM_ERROR_OK, "text_get_line_spacing error");
  fail_if (line_spacing != 15.0f, "line-spacing was wrong");

  /* outline-color */
  ret = pgm_text_set_outline_color (text, 64, 128, 255, 128);
  fail_if (ret != PGM_ERROR_OK, "text_set_outline_color error");
  ret = pgm_text_get_outline_color (text, &red, &green, &blue, &alpha);
  fail_if (ret != PGM_ERROR_OK, "text_get_outline_color error");
  fail_if (red != 64, "outline-color (red) was wrong");
  fail_if (green != 128, "outline-color (green) was wrong");
  fail_if (blue != 255, "outline-color (blue) was wrong");
  fail_if (alpha != 128, "outline-color (alpha) was wrong");

  /* width */
  ret = pgm_text_set_outline_width (text, 3.45f);
  fail_if (ret != PGM_ERROR_OK, "text_set_outline_width error");
  ret = pgm_text_get_outline_width (text, &width);
  fail_if (ret != PGM_ERROR_OK, "text_get_outline_width error");
  fail_if (width != 3.45f, "outline-width was wrong");

  gst_object_unref (GST_OBJECT_CAST (text));
  pgm_deinit ();
}
PGM_END_TEST;

Suite *
pgm_text_suite (void)
{
  Suite *s = suite_create ("PgmText");
  TCase *tc_chain = tcase_create ("pgmtext tests");

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_text_default_values);
  tcase_add_test (tc_chain, test_text_set_get_correctness);

  return s;
}

GST_CHECK_MAIN (pgm_text);
