/*
 * SearchDialog.java
 *
 * Created on 10. August 2002, 12:23
 */

package pauker.program.gui.swing;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Toolkit;
import java.text.MessageFormat;
import javax.swing.JDialog;
import javax.swing.JPanel;
import pauker.program.Card;

/**
 * a dialog for controlling the string search in Pauker
 * @author Ronny.Standtke@gmx.net
 */
public class SearchDialog extends JDialog {

    private final PaukerFrame paukerFrame;
    private final JPanel searchArea;
    private static final int ADDITIONAL_DISTANCE = 10;
    private String oldSearchPattern = "";

    /**
     * Creates new form SearchDialog
     * @param paukerFrame the pauker frame
     * @param searchArea the area where search results are displayed
     */
    public SearchDialog(PaukerFrame paukerFrame, JPanel searchArea) {
        super(paukerFrame, false);
        this.paukerFrame = paukerFrame;
        this.searchArea = searchArea;
        initComponents();
        pack();
    }

    /**
     * shows/hides the dialog
     * @param visible if <CODE>true</CODE>, the dialog will be shown,
     * otherwise the dialog will be hidden
     */
    @Override
    public void setVisible(boolean visible) {

        if (visible) {
            setLocationRelativeTo(paukerFrame);

            // place dialog at a "perfect place" so that it is located next to
            // the searchArea
            // try order: left -> right -> upper -> lower -> fallback to left
            Point location = getLocation();
            Dimension size = getSize();
            Point searchAreaLocation = searchArea.getLocationOnScreen();

            if (searchAreaLocation.x > size.width) {
                // move left
                location.x = 
                        searchAreaLocation.x - size.width - ADDITIONAL_DISTANCE;
            } else {
                Dimension searchAreaDimension = searchArea.getSize();
                Dimension screenSize = 
                        Toolkit.getDefaultToolkit().getScreenSize();
                int rightSpace = screenSize.width - 
                        searchAreaLocation.x - searchAreaDimension.width;
                if (rightSpace > size.width) {
                    // move right
                    location.x = searchAreaLocation.x + 
                            searchAreaDimension.width + ADDITIONAL_DISTANCE;
                } else if (searchAreaLocation.y > size.height) {
                    // move up
                    location.y = searchAreaLocation.y - 
                            size.height - ADDITIONAL_DISTANCE;
                } else {
                    int lowerSpace = screenSize.height - 
                            searchAreaLocation.y - searchAreaDimension.height;
                    if (lowerSpace > size.height) {
                        // move down
                        location.y = searchAreaLocation.y + 
                                searchAreaDimension.height + 
                                ADDITIONAL_DISTANCE;
                    } else {
                        //move hard left
                        location.x = 0;
                    }
                }
            }
            setLocation(location);
        }

        super.setVisible(visible);
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        sideButtonGroup = new javax.swing.ButtonGroup();
        directionButtonGroup = new javax.swing.ButtonGroup();
        patternLabel = new javax.swing.JLabel();
        patternField = new javax.swing.JTextField();
        searchButton = new javax.swing.JButton();
        radioButtonPanel = new javax.swing.JPanel();
        bothRadioButton = new javax.swing.JRadioButton();
        forwardRadioButton = new javax.swing.JRadioButton();
        frontRadioButton = new javax.swing.JRadioButton();
        backwardRadioButton = new javax.swing.JRadioButton();
        backRadioButton = new javax.swing.JRadioButton();
        matchCaseCheckBox = new javax.swing.JCheckBox();
        cancelButton = new javax.swing.JButton();
        statusLabel = new javax.swing.JLabel();

        getContentPane().setLayout(new java.awt.GridBagLayout());

        setTitle(java.util.ResourceBundle.getBundle("pauker/Strings").getString("Search"));
        addComponentListener(new java.awt.event.ComponentAdapter() {
            public void componentShown(java.awt.event.ComponentEvent evt) {
                formComponentShown(evt);
            }
        });
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                closeDialog(evt);
            }
            public void windowActivated(java.awt.event.WindowEvent evt) {
                formWindowActivated(evt);
            }
        });

        patternLabel.setText(java.util.ResourceBundle.getBundle("pauker/Strings").getString("Pattern"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 0, 0);
        getContentPane().add(patternLabel, gridBagConstraints);

        patternField.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                patternFieldActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.RELATIVE;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 0, 0);
        getContentPane().add(patternField, gridBagConstraints);

        searchButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/pauker/icons/filefind.png")));
        searchButton.setText(java.util.ResourceBundle.getBundle("pauker/Strings").getString("Search"));
        searchButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                searchButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 0, 10);
        getContentPane().add(searchButton, gridBagConstraints);

        radioButtonPanel.setLayout(new java.awt.GridBagLayout());

        sideButtonGroup.add(bothRadioButton);
        bothRadioButton.setSelected(true);
        bothRadioButton.setText(java.util.ResourceBundle.getBundle("pauker/Strings").getString("Both_Sides"));
        bothRadioButton.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                bothRadioButtonItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.RELATIVE;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        radioButtonPanel.add(bothRadioButton, gridBagConstraints);

        directionButtonGroup.add(forwardRadioButton);
        forwardRadioButton.setSelected(true);
        forwardRadioButton.setText(java.util.ResourceBundle.getBundle("pauker/Strings").getString("Forward"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 20, 0, 0);
        radioButtonPanel.add(forwardRadioButton, gridBagConstraints);

        sideButtonGroup.add(frontRadioButton);
        frontRadioButton.setText(java.util.ResourceBundle.getBundle("pauker/Strings").getString("Frontside"));
        frontRadioButton.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                frontRadioButtonItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.RELATIVE;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        radioButtonPanel.add(frontRadioButton, gridBagConstraints);

        directionButtonGroup.add(backwardRadioButton);
        backwardRadioButton.setText(java.util.ResourceBundle.getBundle("pauker/Strings").getString("Backward"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 20, 0, 0);
        radioButtonPanel.add(backwardRadioButton, gridBagConstraints);

        sideButtonGroup.add(backRadioButton);
        backRadioButton.setText(java.util.ResourceBundle.getBundle("pauker/Strings").getString("Reverse_Side"));
        backRadioButton.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                backRadioButtonItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.RELATIVE;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        radioButtonPanel.add(backRadioButton, gridBagConstraints);

        matchCaseCheckBox.setText(java.util.ResourceBundle.getBundle("pauker/Strings").getString("Match_Case"));
        matchCaseCheckBox.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                matchCaseCheckBoxItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 20, 0, 0);
        radioButtonPanel.add(matchCaseCheckBox, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 0, 0);
        getContentPane().add(radioButtonPanel, gridBagConstraints);

        cancelButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/pauker/icons/stop.png")));
        cancelButton.setText(java.util.ResourceBundle.getBundle("pauker/Strings").getString("Cancel"));
        cancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.SOUTHEAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 10);
        getContentPane().add(cancelButton, gridBagConstraints);

        statusLabel.setText("  ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 10, 10);
        getContentPane().add(statusLabel, gridBagConstraints);

    }
    // </editor-fold>//GEN-END:initComponents
    private void formComponentShown(java.awt.event.ComponentEvent evt) {//GEN-FIRST:event_formComponentShown
        statusLabel.setText(" ");
        patternField.selectAll();
        pack();
    }//GEN-LAST:event_formComponentShown

    private void formWindowActivated(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowActivated
        patternField.requestFocusInWindow();
    }//GEN-LAST:event_formWindowActivated

    private void backRadioButtonItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_backRadioButtonItemStateChanged
        paukerFrame.setSearchCardSide(Card.Element.REVERSE_SIDE);
    }//GEN-LAST:event_backRadioButtonItemStateChanged

    private void bothRadioButtonItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_bothRadioButtonItemStateChanged
        paukerFrame.setSearchCardSide(Card.Element.BOTH_SIDES);
    }//GEN-LAST:event_bothRadioButtonItemStateChanged

    private void frontRadioButtonItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_frontRadioButtonItemStateChanged
        paukerFrame.setSearchCardSide(Card.Element.FRONT_SIDE);
    }//GEN-LAST:event_frontRadioButtonItemStateChanged

    private void matchCaseCheckBoxItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_matchCaseCheckBoxItemStateChanged
        paukerFrame.setMatchCase(matchCaseCheckBox.isSelected());
    }//GEN-LAST:event_matchCaseCheckBoxItemStateChanged

    private void patternFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_patternFieldActionPerformed
        search();
    }//GEN-LAST:event_patternFieldActionPerformed

    private void searchButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_searchButtonActionPerformed
        search();
    }//GEN-LAST:event_searchButtonActionPerformed

    private void search() {
        String searchPattern = patternField.getText();

        boolean stringFound = false;
        if (oldSearchPattern.equals(searchPattern)) {
            // continue search
            stringFound = paukerFrame.continueSearch(
                    forwardRadioButton.isSelected());

        } else {
            // new search
            Card.Element cardSide = Card.Element.REVERSE_SIDE;
            if (bothRadioButton.isSelected()) {
                cardSide = Card.Element.BOTH_SIDES;
            } else if (frontRadioButton.isSelected()) {
                cardSide = Card.Element.FRONT_SIDE;
            }

            stringFound = paukerFrame.search(searchPattern, cardSide,
                    matchCaseCheckBox.isSelected());
            oldSearchPattern = searchPattern;
        }

        if (stringFound) {
            statusLabel.setText(" ");
        } else {
            String statusMessage = MessageFormat.format(
                    PaukerFrame.STRINGS.getString("String_Not_Found"), 
                    searchPattern);
            statusLabel.setText(statusMessage);
            statusLabel.setForeground(Color.red);
        }

        patternField.requestFocusInWindow();
    }

    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed
        quit();
    }//GEN-LAST:event_cancelButtonActionPerformed

    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_closeDialog
        quit();
    }//GEN-LAST:event_closeDialog

    private void quit() {
        setVisible(false);
        dispose();
        oldSearchPattern = "";
        paukerFrame.stopSearching();
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JRadioButton backRadioButton;
    private javax.swing.JRadioButton backwardRadioButton;
    private javax.swing.JRadioButton bothRadioButton;
    private javax.swing.JButton cancelButton;
    private javax.swing.ButtonGroup directionButtonGroup;
    private javax.swing.JRadioButton forwardRadioButton;
    private javax.swing.JRadioButton frontRadioButton;
    private javax.swing.JCheckBox matchCaseCheckBox;
    private javax.swing.JTextField patternField;
    private javax.swing.JLabel patternLabel;
    private javax.swing.JPanel radioButtonPanel;
    private javax.swing.JButton searchButton;
    private javax.swing.ButtonGroup sideButtonGroup;
    private javax.swing.JLabel statusLabel;
    // End of variables declaration//GEN-END:variables
}
