/*
 * tcamsg.c - traffic control message parser
 * Copyright (C) 2014 Tetsumune KISO <t2mune@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "nield.h"
#include "rtnetlink.h"

/*
 * parse traffic control message
 */
int parse_tcamsg(struct nlmsghdr *nlh)
{
    struct tcamsg *tcam;
    int tcam_len;
    struct rtattr *tcaa[TCAA_MAX+1];
    char msg[MAX_MSG_SIZE] = "";
    char *mp = msg;
    int log_opts = get_log_opts();

    /* debug nlmsghdr */
    if(log_opts & L_DEBUG)
        debug_nlmsg(0, nlh);

    /* get tcamsg */
    tcam_len = NLMSG_PAYLOAD(nlh, 0);
    if(tcam_len < sizeof(*tcam)) {
        rec_log("error: %s: length too short", __func__);
        return(1);
    }
    tcam = (struct tcamsg *)NLMSG_DATA(nlh);

    /* parse traffic control action message attributes */
    parse_tca(tcaa, nlh);

    /* debug tcamsg */
    if(log_opts & L_DEBUG)
        debug_tcamsg(0, tcam, tcaa, tcam_len);

    /* kind of message */
    switch(nlh->nlmsg_type) {
        case RTM_NEWACTION:
            mp = add_log(msg, mp, "tc action added: ");
            break;
        case RTM_DELACTION:
            mp = add_log(msg, mp, "tc action deleted: ");
            break;
        default:
            return(1);
    }

    if(tcaa[TCA_ACT_TAB])
        if(parse_tca_acts(msg, mp, tcaa[TCA_ACT_TAB]))
            return(1);

    return(0);
}

/*
 * parse actions
 */
int parse_tca_acts(char *msg, char *mp, struct rtattr *tcaa)
{
    struct rtattr *acts[TCA_ACT_MAX_PRIO+1];

    parse_actions(acts, tcaa);

    /* logging for each action */
    int i;

    for(i = 0; i < TCA_ACT_MAX_PRIO; i++)
        if(acts[i] && parse_tca_act(msg, mp, acts[i]))
            return(1);

    return(0);
}

/*
 * parse attributes TCA_ACT_*
 */
int parse_tca_act(char *msg, char *mp, struct rtattr *acts)
{
    struct rtattr *act[__TCA_ACT_MAX];
    char kind[IFNAMSIZ] = "";

    mp = add_log(msg, mp, "order=%d ", acts->rta_type);

    parse_action(act, acts);

    if(act[TCA_ACT_KIND]) {
        if(RTA_PAYLOAD(act[TCA_ACT_KIND]) > sizeof(kind)) {
            rec_log("error: %s: payload too long", __func__);
            return(1);
        }
        strncpy(kind, (char *)RTA_DATA(act[TCA_ACT_KIND]), sizeof(kind));
    
        mp = add_log(msg, mp, "action=%s ", kind);
    }

    if(act[TCA_ACT_OPTIONS]) {
        if(!strncmp(kind, "police", sizeof(kind))) {
            if(parse_tca_act_options_police(msg, mp, act[TCA_ACT_OPTIONS]))
                return(1);
            return(0);
        } else if(!strncmp(kind, "gact", sizeof(kind))) {
            if(parse_tca_act_options_gact(msg, mp, act[TCA_ACT_OPTIONS]))
                return(1);
            return(0);
        } else if(!strncmp(kind, "pedit", sizeof(kind))) {
            if(parse_tca_act_options_pedit(msg, mp, act[TCA_ACT_OPTIONS]))
                return(1);
            return(0);
        } else if(!strncmp(kind, "mirred", sizeof(kind))) {
            if(parse_tca_act_options_mirred(msg, mp, act[TCA_ACT_OPTIONS]))
                return(1);
            return(0);
#ifdef HAVE_LINUX_TC_ACT_TC_NAT_H
        } else if(!strncmp(kind, "nat", sizeof(kind))) {
            if(parse_tca_act_options_nat(msg, mp, act[TCA_ACT_OPTIONS]))
                return(1);
            return(0);
#endif
#ifdef HAVE_LINUX_TC_ACT_TC_SKBEDIT_H
        } else if(!strncmp(kind, "skbedit", sizeof(kind))) {
            if(parse_tca_act_options_skbedit(msg, mp, act[TCA_ACT_OPTIONS]))
                return(1);
            return(0);
#endif
#ifdef HAVE_LINUX_TC_ACT_TC_CSUM_H
        } else if(!strncmp(kind, "csum", sizeof(kind))) {
            if(parse_tca_act_options_csum(msg, mp, act[TCA_ACT_OPTIONS]))
                return(1);
            return(0);
#endif
        }
    }

    rec_log("%s", msg);

    return(0);
}

/*
 * parse attribute TCA_POLICE_*
 */
int parse_tca_act_options_police(char *msg, char *mp, struct rtattr *act)
{
    struct rtattr *police[__TCA_POLICE_MAX];

    parse_police(police, act);

    if(police[TCA_POLICE_TBF]) {
        struct tc_police *tc_police;
        char rate[MAX_STR_SIZE];
        char burst[MAX_STR_SIZE];
        char peakrate[MAX_STR_SIZE];
        char mtu[MAX_STR_SIZE];
        double rate_latency = 0;
        double peakrate_latency = 0;
        char latency[MAX_STR_SIZE];

        if(RTA_PAYLOAD(police[TCA_POLICE_TBF]) < sizeof(*tc_police)) {
            rec_log("error: %s: payload too short", __func__);
            return(1);
        }
        tc_police = (struct tc_police *)RTA_DATA(police[TCA_POLICE_TBF]);
    
        mp = add_log(msg, mp, "index=%d ", tc_police->index);
    
        get_us2tick();
        convert_unit_rate(rate, sizeof(rate), tc_police->rate.rate);
        convert_unit_size(burst, sizeof(burst),
            get_burst_size(tc_police->rate.rate, tc_police->burst));
        rate_latency = get_latency(tc_police->rate.rate,
            tc_police->burst, tc_police->limit);
    
        mp = add_log(msg, mp, "rate=%s burst=%s ", rate, burst);
    
        if(tc_police->peakrate.rate) {
            convert_unit_rate(peakrate, sizeof(peakrate), tc_police->peakrate.rate);
            convert_unit_size(mtu, sizeof(mtu),
                get_burst_size(tc_police->peakrate.rate, tc_police->mtu));
            peakrate_latency =
                get_latency(tc_police->peakrate.rate,
                    tc_police->mtu, tc_police->limit);

            mp = add_log(msg, mp, "peakrate=%s minburst=%s ", peakrate, mtu);
        }
    
        if(rate_latency < peakrate_latency)
            convert_unit_usec(latency, sizeof(latency), peakrate_latency);
        else
            convert_unit_usec(latency, sizeof(latency), rate_latency);
    
        mp = add_log(msg, mp, "latency=%s exceed=%s ",
            latency, convert_tc_police_action(tc_police->action, 0));
    }

    rec_log("%s", msg);

    return(0);
}

/*
 * parse gact options(TCA_ACT_GACT: 5)
 */
int parse_tca_act_options_gact(char *msg, char *mp, struct rtattr *act)
{
    struct rtattr *gact[__TCA_GACT_MAX];

    parse_gact(gact, act);

    if(gact[TCA_GACT_PARMS]) {
        struct tc_gact *parms;

        if(RTA_PAYLOAD(gact[TCA_GACT_PARMS]) < sizeof(*parms)) {
            rec_log("error: %s: TCA_GACT_PARMS: payload too short", __func__);
            return(1);
        }
        parms = (struct tc_gact *)RTA_DATA(gact[TCA_GACT_PARMS]);

        mp = add_log(msg, mp, "index=%d kind=%s ",
            parms->index, convert_tc_action(parms->action, 0));
    }

    if(gact[TCA_GACT_PROB]) {
        struct tc_gact_p *prob;

        if(RTA_PAYLOAD(gact[TCA_GACT_PROB]) < sizeof(*prob)) {
            rec_log("error: %s: TCA_GACT_PROB: payload too short", __func__);
            return(1);
        }
        prob = (struct tc_gact_p *)RTA_DATA(gact);

        mp = add_log(msg, mp, "random(type/value/action)=%s/%d/%s ",
            convert_pgact(prob->ptype, 0), prob->pval,
            convert_tc_action(prob->paction, 0));
    }

    rec_log("%s", msg);

    return(0);
}

/*
 * parse pedit options(TCA_ACT_PEDIT: 7)
 */
int parse_tca_act_options_pedit(char *msg, char *mp, struct rtattr *act)
{
    struct rtattr *pedit[__TCA_PEDIT_MAX];
    char *mp_tmp;

    parse_pedit(pedit, act);

    if(pedit[TCA_PEDIT_PARMS]) {
        struct tc_pedit_sel *parms;
        struct tc_pedit_key *keys;
        int len, i;

        if(RTA_PAYLOAD(pedit[TCA_PEDIT_PARMS]) < sizeof(*parms)) {
            rec_log("error: %s: TCA_PEDIT_PARMS: payload too short", __func__);
            return(1);
        }
        parms = (struct tc_pedit_sel *)RTA_DATA(pedit[TCA_PEDIT_PARMS]);

        len = sizeof(*parms) + (sizeof(*keys) * parms->nkeys);
        if(RTA_PAYLOAD(pedit[TCA_PEDIT_PARMS]) < len) {
            rec_log("error: %s: TCA_PEDIT_PARMS: payload too short", __func__);
            return(1);
        }

        mp = add_log(msg, mp, "index=%d ", parms->index);
        mp_tmp = mp;
    
        keys = parms->keys;
        if(!keys) {
            rec_log("error: %s: no key", __func__);
            return(1);
        }
    
        for(i = 0; i < parms->nkeys; i++, keys++, mp = mp_tmp) {
            mp = add_log(msg, mp, "key=%d value=0x%08x/0x%08x offset=%d "
                "at=%d offmask=0x%08x shift=%d next=%s ",
                i+1, ntohl(keys->val), ntohl(keys->mask), keys->off,
                keys->at, ntohl(keys->offmask), keys->shift,
                convert_tc_action(parms->action, 0));
            rec_log("%s", msg);
        }

        return(0);
    }

    rec_log("%s", msg);

    return(0);
}

/*
 * parse mirred options(TCA_ACT_MIRRED: 8)
 */
int parse_tca_act_options_mirred(char *msg, char *mp, struct rtattr *act)
{
    struct rtattr *mirred[__TCA_MIRRED_MAX];

    parse_mirred(mirred, act);

    if(mirred[TCA_MIRRED_PARMS]) {
        struct tc_mirred *parms;
        char ifname[IFNAMSIZ] = "";

        if(RTA_PAYLOAD(mirred[TCA_MIRRED_PARMS]) < sizeof(*parms)) {
            rec_log("error: %s: payload too short", __func__);
            return(1);
        }
        parms = (struct tc_mirred *)RTA_DATA(mirred[TCA_MIRRED_PARMS]);
        if_indextoname_from_lists(parms->ifindex, ifname);
    
        mp = add_log(msg, mp, "index=%d %s to=%s next=%s ",
            parms->index, convert_tca_mirred_action(parms->eaction, 0),
            ifname, convert_tc_action(parms->action, 0));
    }

    rec_log("%s", msg);

    return(0);
}

#ifdef HAVE_LINUX_TC_ACT_TC_NAT_H
/*
 * parse nat options(TCA_ACT_NAT: 9)
 */
int parse_tca_act_options_nat(char *msg, char *mp, struct rtattr *act)
{
    struct rtattr *nat[__TCA_NAT_MAX];

    parse_mirred(nat, act);

    if(nat[TCA_NAT_PARMS]) {
        struct tc_nat *parms;
        char old[INET_ADDRSTRLEN+1] = "";
        char new[INET_ADDRSTRLEN+1] = "";
        int i, mask;

        if(RTA_PAYLOAD(nat[TCA_NAT_PARMS]) < sizeof(*parms)) {
            rec_log("error: %s: payload too short", __func__);
            return(1);
        }
        parms = (struct tc_nat *)RTA_DATA(nat[TCA_NAT_PARMS]);
    
        inet_ntop(AF_INET, &(parms->old_addr), old, sizeof(old));
        inet_ntop(AF_INET, &(parms->new_addr), new, sizeof(new));
    
        for(i = 0, mask = 0; i < 32; i++)
            if(parms->mask & (1 << i))
                ++mask;
    
        mp = add_log(msg, mp, "index=%d from=%s/%d to=%s direction=%s ",
            parms->index, old, mask, new,
            (parms->flags & TCA_NAT_FLAG_EGRESS) ? "egress" : "ingress");
    }

    rec_log("%s", msg);

    return(0);
}
#endif

#ifdef HAVE_LINUX_TC_ACT_TC_SKBEDIT_H
/*
 * parse skbedit options(TCA_ACT_SKBEDIT: 11)
 */
int parse_tca_act_options_skbedit(char *msg, char *mp, struct rtattr *act)
{
    struct rtattr *skb[__TCA_SKBEDIT_MAX];
    struct tc_skbedit *parms = NULL;

    parse_skbedit(skb, act);

    if(skb[TCA_SKBEDIT_PARMS]) {
        if(RTA_PAYLOAD(skb[TCA_SKBEDIT_PARMS]) < sizeof(*parms)) {
            rec_log("error: %s: TCA_SKBEDIT_PARMS: payload too short", __func__);
            return(1);
        }
        parms = (struct tc_skbedit *)RTA_DATA(skb[TCA_SKBEDIT_PARMS]);
        mp = add_log(msg, mp, "index=%d ", parms->index);
    }

    if(skb[TCA_SKBEDIT_PRIORITY]) {
        if(RTA_PAYLOAD(skb[TCA_SKBEDIT_PRIORITY]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_SKBEDIT_PRIORITY: payload too short", __func__);
            return(1);
        }
        mp = add_log(msg, mp, "priority=0x%x ",
            *(unsigned *)RTA_DATA(skb[TCA_SKBEDIT_PRIORITY]));
    }

    if(skb[TCA_SKBEDIT_QUEUE_MAPPING]) {
        if(RTA_PAYLOAD(skb[TCA_SKBEDIT_QUEUE_MAPPING]) < sizeof(unsigned short)) {
            rec_log("error: %s: TCA_SKBEDIT_QUEUE_MAPPING: payload too short", __func__);
            return(1);
        }
        mp = add_log(msg, mp, "queue-mapping=%u ",
            *(unsigned *)RTA_DATA(skb[TCA_SKBEDIT_QUEUE_MAPPING]));
    }

#if HAVE_DECL_TCA_SKBEDIT_MARK
    if(skb[TCA_SKBEDIT_MARK]) {
        if(RTA_PAYLOAD(skb[TCA_SKBEDIT_MARK]) < sizeof(unsigned)) {
            rec_log("error: %s: TCA_SKBEDIT_MARK: payload too short", __func__);
            return(1);
        }
        mp = add_log(msg, mp, "mark=%u ",
            *(unsigned *)RTA_DATA(skb[TCA_SKBEDIT_MARK]));
    }
#endif

    if(parms)
        mp = add_log(msg, mp, "next=%s ", convert_tc_action(parms->action, 0));

    rec_log("%s", msg);

    return(0);
}
#endif

#ifdef HAVE_LINUX_TC_ACT_TC_CSUM_H
/*
 * parse csum options(TCA_ACT_CSUM: 16)
 */
int parse_tca_act_options_csum(char *msg, char *mp, struct rtattr *act)
{
    struct rtattr *csum[__TCA_CSUM_MAX];

    parse_csum(csum, act);

    if(csum[TCA_CSUM_PARMS]) {
        struct tc_csum *parms;
        char flags_list[MAX_STR_SIZE] = "";

        if(RTA_PAYLOAD(csum[TCA_CSUM_PARMS]) < sizeof(*parms)) {
            rec_log("error: %s: payload too short", __func__);
            return(1);
        }
        parms = (struct tc_csum *)RTA_DATA(csum[TCA_CSUM_PARMS]);
        convert_tca_csum_update_flags(parms->update_flags,
            flags_list, sizeof(flags_list), 0);
    
        mp = add_log(msg, mp, "index=%d protocol=%s next=%s ",
            parms->index, flags_list,
            convert_tc_action(parms->action, 0));
    }

    rec_log("%s", msg);

    return(0);
}
#endif

/*
 * debug traffic control action message
 */
void debug_tcamsg(int lev, struct tcamsg *tcam, struct rtattr *tcaa[], int tcam_len)
{
    /* debug tcamsg */
    rec_dbg(lev, "*********************************************************************");

    rec_dbg(lev, "[ tcamsg(%d) ]",
        NLMSG_ALIGN(sizeof(*tcam)));
    rec_dbg(lev, "    tca_family(%d): 0x%02x(%s)",
        sizeof(tcam->tca_family), tcam->tca_family,
        convert_af_type(tcam->tca_family));
    rec_dbg(lev, "    tca__pad1(%d): 0x%02x",
        sizeof(tcam->tca__pad1), tcam->tca__pad1);
    rec_dbg(lev, "    tca__pad2(%d): 0x%04x",
        sizeof(tcam->tca__pad2), tcam->tca__pad2);

    /* debug traffic control action attributes */
    rec_dbg(lev,"*********************************************************************");
    rec_dbg(lev, "[ tcamsg attributes(%d) ]",
            NLMSG_ALIGN(tcam_len - NLMSG_ALIGN(sizeof(*tcam))));

    if(tcaa[TCA_ACT_TAB])
        debug_tca_act_tab(lev+1, tcaa[TCA_ACT_TAB]);

    rec_dbg(lev, "");
}

/*
 * debug attribute TCA_ACT_TAB
 */
void debug_tca_act_tab(int lev, struct rtattr *tcaa)
{
    rec_dbg(lev, "TCA_ACT_TAB(%hu):",
        RTA_ALIGN(tcaa->rta_len));

    debug_tca_acts(lev+1, tcaa);
}

/*
 * debug attributes of multiple actions
 */
void debug_tca_acts(int lev, struct rtattr *tcaa)
{
    struct rtattr *acts[TCA_ACT_MAX_PRIO+1];
    int i;

    parse_actions(acts, tcaa);

    for(i = 0; i < TCA_ACT_MAX_PRIO; i++)
        if(acts[i]) {
            rec_dbg(lev, "acts[%hu](%hu):", acts[i]->rta_type, acts[i]->rta_len);
            debug_tca_act(lev, acts[i]); /* not lev+1 */
        }
}

/*
 * debug attributes TCA_ACT_*
 */
void debug_tca_act(int lev, struct rtattr *acts)
{
    struct rtattr *act[__TCA_ACT_MAX];
    char kind[IFNAMSIZ];

    parse_action(act, acts);

    if(act[TCA_ACT_KIND])
        debug_tca_act_kind(lev+1, act[TCA_ACT_KIND], kind, sizeof(kind));

    if(act[TCA_ACT_OPTIONS])
        debug_tca_act_options(lev+1, act[TCA_ACT_OPTIONS], kind, sizeof(kind));

    if(act[TCA_ACT_INDEX])
        debug_tca_act_index(lev+1, act[TCA_ACT_INDEX]);

    if(act[TCA_ACT_STATS])
        debug_tca_act_stats(lev+1, act[TCA_ACT_STATS]);
}

/*
 * debug attribute TCA_ACT_KIND
 */
void debug_tca_act_kind(int lev, struct rtattr *act, char *kind, int len)
{
    if(RTA_PAYLOAD(act) > len) {
        rec_dbg(lev, "TCA_ACT_KIND(%hu): -- payload too short --",
            RTA_ALIGN(act->rta_len));
        return;
    }
    strncpy(kind, (char *)RTA_DATA(act), len);

    rec_dbg(lev, "TCA_ACT_KIND(%hu): %s", RTA_ALIGN(act->rta_len), kind);
}

/*
 * debug attribute TCA_ACT_OPTIONS
 */
void debug_tca_act_options(int lev, struct rtattr *act, char *kind, int len)
{
    rec_dbg(lev, "TCA_ACT_OPTIONS(%hu):", RTA_ALIGN(act->rta_len));

    if(!strncmp(kind, "police", len))
        debug_tca_act_options_police(lev, act);
    else if(!strncmp(kind, "gact", len))
        debug_tca_act_options_gact(lev, act);
    else if(!strncmp(kind, "pedit", len))
        debug_tca_act_options_pedit(lev, act);
    else if(!strncmp(kind, "mirred", len))
        debug_tca_act_options_mirred(lev, act);
#ifdef HAVE_LINUX_TC_ACT_TC_NAT_H
    else if(!strncmp(kind, "nat", len))
        debug_tca_act_options_nat(lev, act);
#endif
#ifdef HAVE_LINUX_TC_ACT_TC_SKBEDIT_H
    else if(!strncmp(kind, "skbedit", len))
        debug_tca_act_options_skbedit(lev, act);
#endif
#ifdef HAVE_LINUX_TC_ACT_TC_CSUM_H
    else if(!strncmp(kind, "csum", len))
        debug_tca_act_options_csum(lev, act);
#endif
    else
        rec_dbg(lev, "    -- unknown action %s --", kind);
}

/*
 * debug attribute TCA_ACT_INDEX
 */
void debug_tca_act_index(int lev, struct rtattr *act)
{
    if(RTA_PAYLOAD(act) < sizeof(int)) {
        rec_dbg(lev, "TCA_ACT_INDEX(%hu): -- payload too short --",
            RTA_ALIGN(act->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_ACT_INDEX(%hu): %d",
        RTA_ALIGN(act->rta_len), *(int *)RTA_DATA(act));
}

/*
 * debug attribute TCA_ACT_STATS
 */
void debug_tca_act_stats(int lev, struct rtattr *act)
{
    rec_dbg(lev, "TCA_ACT_STATS(%hu):", RTA_ALIGN(act->rta_len));

    debug_tca_stats2_attr(lev, act);
}

/*
 * debug attribute TCA_POLICE_*
 */
void debug_tca_act_options_police(int lev, struct rtattr *act)
{
    struct rtattr *police[__TCA_POLICE_MAX];

    parse_police(police, act);

    if(police[TCA_POLICE_TBF])
        debug_tca_police_tbf(lev+1, police[TCA_POLICE_TBF]);

    if(police[TCA_POLICE_RATE])
        debug_tca_police_rate(lev+1, police[TCA_POLICE_RATE]);

    if(police[TCA_POLICE_PEAKRATE])
        debug_tca_police_peakrate(lev+1, police[TCA_POLICE_PEAKRATE]);

    if(police[TCA_POLICE_AVRATE])
        debug_tca_police_avrate(lev+1, police[TCA_POLICE_AVRATE]);

    if(police[TCA_POLICE_RESULT])
        debug_tca_police_result(lev+1, police[TCA_POLICE_RESULT]);
}

/*
 * debug attribute TCA_POLICE_TBF
 */
void debug_tca_police_tbf(int lev, struct rtattr *police)
{
    struct tc_police *tc_police;
    struct tc_ratespec *rate;
    struct tc_ratespec *peak;

    if(RTA_PAYLOAD(police) < sizeof(*tc_police)) {
        rec_dbg(lev, "TCA_POLICE_TBF(%hu): -- payload too short --",
            RTA_ALIGN(police->rta_len));
        return;
    }
    tc_police = (struct tc_police *)RTA_DATA(police);
    rate = &(tc_police->rate);
    peak = &(tc_police->peakrate);

    rec_dbg(lev, "TCA_POLICE_TBF(%hu):", RTA_ALIGN(police->rta_len));
    rec_dbg(lev, "    [ tc_police(%d) ]", sizeof(*tc_police));
    rec_dbg(lev, "        index(%d): %u", sizeof(tc_police->index), tc_police->index);
    rec_dbg(lev, "        action(%d): %d(%s)",
        sizeof(tc_police->action), tc_police->action,
        convert_tc_police_action(tc_police->action, 1));
    rec_dbg(lev, "        limit(%d): %u", sizeof(tc_police->limit), tc_police->limit);
    rec_dbg(lev, "        burst(%d): %u", sizeof(tc_police->burst), tc_police->burst);
    rec_dbg(lev, "        mtu(%d): %u", sizeof(tc_police->mtu), tc_police->mtu);

    debug_tc_ratespec(lev+2, rate, "rate");
    debug_tc_ratespec(lev+2, peak, "peakrate");

    rec_dbg(lev, "        refcnt(%d): %d", sizeof(tc_police->refcnt), tc_police->refcnt);
    rec_dbg(lev, "        bindcnt(%d): %d", sizeof(tc_police->bindcnt), tc_police->bindcnt);
    rec_dbg(lev, "        capab(%d): %u", sizeof(tc_police->capab), tc_police->capab);
}

/*
 * debug attribute TCA_POLICE_RATE
 */
void debug_tca_police_rate(int lev, struct rtattr *police)
{
    rec_dbg(lev, "TCA_POLICE_RATE(%hu): -- ignored --", RTA_ALIGN(police->rta_len));
}

/*
 * debug attribute TCA_POLICE_PEAKRATE
 */
void debug_tca_police_peakrate(int lev, struct rtattr *police)
{
    rec_dbg(lev, "TCA_POLICE_PEAKRATE(%hu): -- ignored --", RTA_ALIGN(police->rta_len));
}

/*
 * debug attribute TCA_POLICE_AVRATE
 */
void debug_tca_police_avrate(int lev, struct rtattr *police)
{
    if(RTA_PAYLOAD(police) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_POLICE_AVRATE(%hu): -- payload too short --",
            RTA_ALIGN(police->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_POLICE_AVRATE(%hu): %u",
        RTA_ALIGN(police->rta_len), *(unsigned *)RTA_DATA(police));
}

/*
 * debug attribute TCA_POLICE_RESULT
 */
void debug_tca_police_result(int lev, struct rtattr *police)
{
    if(RTA_PAYLOAD(police) < sizeof(int)) {
        rec_dbg(lev, "TCA_POLICE_RESULT(%hu): -- payload too short --",
            RTA_ALIGN(police->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_POLICE_RESULT(%hu): %u(%s)",
        RTA_ALIGN(police->rta_len), *(unsigned *)RTA_DATA(police),
        convert_tc_police_action(*(int *)RTA_DATA(police), 1));
}

/*
 * debug gact options(TCA_ACT_GACT: 5)
 */
void debug_tca_act_options_gact(int lev, struct rtattr *act)
{
    struct rtattr *gact[__TCA_GACT_MAX];

    parse_gact(gact, act);

    if(gact[TCA_GACT_TM])
        debug_tca_gact_tm(lev+1, gact[TCA_GACT_TM]);

    if(gact[TCA_GACT_PARMS])
        debug_tca_gact_parms(lev+1, gact[TCA_GACT_PARMS]);

    if(gact[TCA_GACT_PROB])
        debug_tca_gact_prob(lev+1, gact[TCA_GACT_PROB]);
}

/*
 * debug attribute TCA_GACT_TM
 */
void debug_tca_gact_tm(int lev, struct rtattr *gact)
{
    struct tcf_t *tm;

    if(RTA_PAYLOAD(gact) < sizeof(*tm)) {
        rec_dbg(lev, "TCA_GACT_TM(%hu): -- payload too short --",
            RTA_ALIGN(gact->rta_len));
        return;
    }
    tm = (struct tcf_t *)RTA_DATA(gact);

    rec_dbg(lev, "TCA_GACT_TM(%hu):", RTA_ALIGN(gact->rta_len));
    debug_tcf_t(lev+1, tm);
}

/*
 * debug attribute TCA_GACT_PARMS
 */
void debug_tca_gact_parms(int lev, struct rtattr *gact)
{
    struct tc_gact *parms;

    if(RTA_PAYLOAD(gact) < sizeof(*parms)) {
        rec_dbg(lev, "TCA_GACT_PARMS(%hu): -- payload too short --",
            RTA_ALIGN(gact->rta_len));
        return;
    }
    parms = (struct tc_gact *)RTA_DATA(gact);

    rec_dbg(lev, "TCA_GACT_PARMS(%hu):", RTA_ALIGN(gact->rta_len));
    rec_dbg(lev, "    [ tc_gact(%d) ]", sizeof(*parms));
    rec_dbg(lev, "        index(%d): %u", sizeof(parms->index), parms->index);
    rec_dbg(lev, "        capab(%d): %u", sizeof(parms->capab), parms->capab);
    rec_dbg(lev, "        action(%d): %d(%s)",
        sizeof(parms->action), parms->action, convert_tc_action(parms->action, 1));
    rec_dbg(lev, "        refcnt(%d): %d", sizeof(parms->refcnt), parms->refcnt);
    rec_dbg(lev, "        bindcnt(%d): %d", sizeof(parms->bindcnt), parms->bindcnt);
}

/*
 * debug attribute TCA_GACT_PROB
 */
void debug_tca_gact_prob(int lev, struct rtattr *gact)
{
    struct tc_gact_p *prob;

    if(RTA_PAYLOAD(gact) < sizeof(*prob)) {
        rec_dbg(lev, "TCA_GACT_PROB(%hu): -- payload too short --",
            RTA_ALIGN(gact->rta_len));
        return;
    }
    prob = (struct tc_gact_p *)RTA_DATA(gact);

    rec_dbg(lev, "TCA_GACT_PROB(%hu):", RTA_ALIGN(gact->rta_len));
    rec_dbg(lev, "    [ tc_gact_p(%d) ]", sizeof(*prob));
    rec_dbg(lev, "        ptype(%d): %d(%s)",
        sizeof(prob->ptype), prob->ptype, convert_pgact(prob->ptype, 1));
    rec_dbg(lev, "        pval(%d): %d", sizeof(prob->pval), prob->pval);
    rec_dbg(lev, "        paction(%d): %d(%s)",
        sizeof(prob->paction), prob->paction, convert_tc_action(prob->paction, 0));
}

/*
 * debug pedit options(TCA_ACT_PEDIT: 7)
 */
void debug_tca_act_options_pedit(int lev, struct rtattr *act)
{
    struct rtattr *pedit[__TCA_PEDIT_MAX];

    parse_pedit(pedit, act);

    if(pedit[TCA_PEDIT_TM])
        debug_tca_pedit_tm(lev, pedit[TCA_PEDIT_TM]);

    if(pedit[TCA_PEDIT_PARMS])
        debug_tca_pedit_parms(lev, pedit[TCA_PEDIT_PARMS]);
}

/*
 * debug attribute TCA_PEDIT_TM
 */
void debug_tca_pedit_tm(int lev, struct rtattr *pedit)
{
    struct tcf_t *tm;

    if(RTA_PAYLOAD(pedit) < sizeof(*tm)) {
        rec_dbg(lev, "TCA_PEDIT_TM(%hu): -- payload too short --",
            RTA_ALIGN(pedit->rta_len));
        return;
    }
    tm = (struct tcf_t *)RTA_DATA(pedit);

    rec_dbg(lev, "TCA_PEDIT_TM(%hu):", RTA_ALIGN(pedit->rta_len));
    debug_tcf_t(lev+1, tm);
}

/*
 * debug attribute TCA_PEDIT_PARMS
 */
void debug_tca_pedit_parms(int lev, struct rtattr *pedit)
{
    struct tc_pedit_sel *parms;

    if(RTA_PAYLOAD(pedit) < sizeof(*parms)) {
        rec_dbg(lev, "TCA_PEDIT_PARMS(%hu): -- payload too short --",
            RTA_ALIGN(pedit->rta_len));
        return;
    }
    parms = (struct tc_pedit_sel *)RTA_DATA(pedit);

    rec_dbg(lev, "TCA_PEDIT_PARMS(%hu):", RTA_ALIGN(pedit->rta_len));
    rec_dbg(lev, "    [ tc_pedit_sel(%d) ]", sizeof(*parms));
    rec_dbg(lev, "        index(%d): 0x%08x", sizeof(parms->index), parms->index);
    rec_dbg(lev, "        capab(%d): 0x%08x", sizeof(parms->capab), parms->capab);
    rec_dbg(lev, "        action(%d): 0x%08x(%s)",
        sizeof(parms->action), parms->action, convert_tc_action(parms->action, 1));
    rec_dbg(lev, "        refcnt(%d): 0x%08x", sizeof(parms->refcnt), parms->refcnt);
    rec_dbg(lev, "        bindcnt(%d): 0x%08x", sizeof(parms->bindcnt), parms->bindcnt);
    rec_dbg(lev, "        nkeys(%d): 0x%02x", sizeof(parms->nkeys), parms->nkeys);
    rec_dbg(lev, "        flags(%d): 0x%02x", sizeof(parms->flags), parms->flags);

    int i, len;
    struct tc_pedit_key *keys = parms->keys;

    len = sizeof(*parms) + (sizeof(*keys) * parms->nkeys);
    if(RTA_PAYLOAD(pedit) < len) {
        rec_dbg(lev, "        nkeys[0](%d): -- payload too short --",
            RTA_PAYLOAD(pedit) - sizeof(*parms));
        return;
    }

    for(i = 0; i < parms->nkeys; i++, keys++) {
        rec_dbg(lev+2, "[ tc_pedit_key keys[%d](%d) ]", i, sizeof(*keys));
        rec_dbg(lev+3, "    mask(%d): 0x%08x(0x%08x)",
            sizeof(keys->mask), keys->mask, ntohl(keys->mask)); /* AND */
        rec_dbg(lev+3, "    val(%d): 0x%08x(0x%08x)",
            sizeof(keys->val), keys->val, ntohl(keys->val)); /* XOR */
        rec_dbg(lev+3, "    off(%d): %d", sizeof(keys->off), keys->off); /* Offset */
        rec_dbg(lev+3, "    at(%d): %d", sizeof(keys->at), keys->at);
        rec_dbg(lev+3, "    offmask(%d): 0x%08x(0x%08x)",
            sizeof(keys->offmask), keys->offmask, ntohl(keys->offmask));
        rec_dbg(lev+3, "    shift(%d): %d", sizeof(keys->shift), keys->shift);
    }
}

/*
 * debug mirred options(TCA_ACT_MIRRED: 8)
 */
void debug_tca_act_options_mirred(int lev, struct rtattr *act)
{
    struct rtattr *mirred[__TCA_MIRRED_MAX];

    parse_mirred(mirred, act);

    if(mirred[TCA_MIRRED_TM])
        debug_tca_mirred_tm(lev+1, mirred[TCA_MIRRED_TM]);

    if(mirred[TCA_MIRRED_PARMS])
        debug_tca_mirred_parms(lev+1, mirred[TCA_MIRRED_PARMS]);
}

/*
 * debug attribute TCA_MIRRED_TM
 */
void debug_tca_mirred_tm(int lev, struct rtattr *mirred)
{
    struct tcf_t *tm;

    if(RTA_PAYLOAD(mirred) < sizeof(*tm)) {
        rec_dbg(lev, "TCA_MIRRED_TM(%hu): -- payload too short --",
            RTA_ALIGN(mirred->rta_len));
        return;
    }
    tm = (struct tcf_t *)RTA_DATA(mirred);

    rec_dbg(lev, "TCA_MIRRED_TM(%hu):", RTA_ALIGN(mirred->rta_len));
    debug_tcf_t(lev+1, tm);
}

/*
 * debug attribute TCA_MIRRED_PARMS
 */
void debug_tca_mirred_parms(int lev, struct rtattr *mirred)
{
    struct tc_mirred *parms;
    char ifname[IFNAMSIZ] = "";

    if(RTA_PAYLOAD(mirred) < sizeof(*parms)) {
        rec_dbg(lev, "TCA_MIRRED_PARMS(%hu): -- payload too short --",
            RTA_ALIGN(mirred->rta_len));
        return;
    }
    parms = (struct tc_mirred *)RTA_DATA(mirred);
    if_indextoname_from_lists(parms->ifindex, ifname);

    rec_dbg(lev, "TCA_MIRRED_PARMS(%hu):", RTA_ALIGN(mirred->rta_len));
    rec_dbg(lev, "    [ tc_mirred(%d) ]", sizeof(*parms));
    rec_dbg(lev, "        index(%d): %u", sizeof(parms->index), parms->index);
    rec_dbg(lev, "        capab(%d): %u", sizeof(parms->capab), parms->capab);
    rec_dbg(lev, "        action(%d): %d(%s)",
        sizeof(parms->action), parms->action, convert_tc_action(parms->action, 1));
    rec_dbg(lev, "        refcnt(%d): %d", sizeof(parms->refcnt), parms->refcnt);
    rec_dbg(lev, "        bindcnt(%d): %d", sizeof(parms->bindcnt), parms->bindcnt);
    rec_dbg(lev, "        eaction(%d): %d(%s)",
        sizeof(parms->eaction), parms->eaction,
        convert_tca_mirred_action(parms->eaction, 1));
    rec_dbg(lev, "        ifindex(%d): %u(%s)", sizeof(parms->ifindex), parms->ifindex, ifname);
}

#ifdef HAVE_LINUX_TC_ACT_TC_NAT_H
/*
 * debug nat options(TCA_ACT_NAT: 9)
 */
void debug_tca_act_options_nat(int lev, struct rtattr *act)
{
    struct rtattr *nat[__TCA_NAT_MAX];

    parse_mirred(nat, act);

    if(nat[TCA_NAT_PARMS])
        debug_tca_nat_parms(lev+1, nat[TCA_NAT_PARMS]);

    if(nat[TCA_NAT_TM])
        debug_tca_nat_tm(lev+1, nat[TCA_NAT_TM]);
}

/*
 * debug attribute TCA_NAT_PARMS
 */
void debug_tca_nat_parms(int lev, struct rtattr *nat)
{
    struct tc_nat *parms;
    char old[INET_ADDRSTRLEN+1] = "";
    char new[INET_ADDRSTRLEN+1] = "";
    char mask[INET_ADDRSTRLEN+1] = "";

    if(RTA_PAYLOAD(nat) < sizeof(*parms)) {
        rec_dbg(lev, "TCA_NAT_PARMS(%hu): -- payload too short --",
            RTA_ALIGN(nat->rta_len));
        return;
    }
    parms = (struct tc_nat *)RTA_DATA(nat);

    inet_ntop(AF_INET, &(parms->old_addr), old, sizeof(old));
    inet_ntop(AF_INET, &(parms->new_addr), new, sizeof(new));
    inet_ntop(AF_INET, &(parms->mask), mask, sizeof(mask));

    rec_dbg(lev, "TCA_NAT_PARMS(%hu):", RTA_ALIGN(nat->rta_len));
    rec_dbg(lev, "    [ tc_nat(%d) ]", sizeof(*parms));
    rec_dbg(lev, "        index(%d): %u", sizeof(parms->index), parms->index);
    rec_dbg(lev, "        capab(%d): %u", sizeof(parms->capab), parms->capab);
    rec_dbg(lev, "        action(%d): %d(%s)",
        sizeof(parms->action), parms->action, convert_tc_action(parms->action, 1));
    rec_dbg(lev, "        refcnt(%d): %d", sizeof(parms->refcnt), parms->refcnt);
    rec_dbg(lev, "        bindcnt(%d): %d", sizeof(parms->bindcnt), parms->bindcnt);
    rec_dbg(lev, "        old_addr(%d): 0x%08x(%s)",
        sizeof(parms->old_addr), parms->old_addr, old);
    rec_dbg(lev, "        new_addr(%d): 0x%08x(%s)",
        sizeof(parms->new_addr), parms->new_addr, new);
    rec_dbg(lev, "        mask(%d): 0x%08x(%s)",
        sizeof(parms->mask), parms->mask, mask);
    rec_dbg(lev, "        flags(%d): 0x%08x(%s)",
        sizeof(parms->flags), parms->flags,
        (parms->flags & TCA_NAT_FLAG_EGRESS) ? "EGRESS" : "INGRESS");
}

/*
 * debug attribute TCA_NAT_TM
 */
void debug_tca_nat_tm(int lev, struct rtattr *nat)
{
    struct tcf_t *tm;

    if(RTA_PAYLOAD(nat) < sizeof(*tm)) {
        rec_dbg(lev, "TCA_NAT_TM(%hu): -- payload too short --",
            RTA_ALIGN(nat->rta_len));
        return;
    }
    tm = (struct tcf_t *)RTA_DATA(nat);

    rec_dbg(lev, "TCA_NAT_TM(%hu):", RTA_ALIGN(nat->rta_len));
    debug_tcf_t(lev+1, tm);
}
#endif

#ifdef HAVE_LINUX_TC_ACT_TC_SKBEDIT_H
/*
 * debug skbedit options(TCA_ACT_SKBEDIT: 11)
 */
void debug_tca_act_options_skbedit(int lev, struct rtattr *act)
{
    struct rtattr *skb[__TCA_SKBEDIT_MAX];

    parse_skbedit(skb, act);

    if(skb[TCA_SKBEDIT_TM])
        debug_tca_skbedit_tm(lev+1, skb[TCA_SKBEDIT_TM]);

    if(skb[TCA_SKBEDIT_PARMS])
        debug_tca_skbedit_parms(lev+1, skb[TCA_SKBEDIT_PARMS]);

    if(skb[TCA_SKBEDIT_PRIORITY])
        debug_tca_skbedit_priority(lev+1, skb[TCA_SKBEDIT_PRIORITY]);

    if(skb[TCA_SKBEDIT_QUEUE_MAPPING])
        debug_tca_skbedit_queue_mapping(lev+1, skb[TCA_SKBEDIT_QUEUE_MAPPING]);

#if HAVE_DECL_TCA_SKBEDIT_MARK
    if(skb[TCA_SKBEDIT_MARK])
        debug_tca_skbedit_mark(lev+1, skb[TCA_SKBEDIT_MARK]);
#endif
}

/*
 * debug attribute TCA_SKBEDIT_TM
 */
void debug_tca_skbedit_tm(int lev, struct rtattr *skb)
{
    struct tcf_t *tm;

    if(RTA_PAYLOAD(skb) < sizeof(*tm)) {
        rec_dbg(lev, "TCA_SKBEDIT_TM(%hu): -- payload too short --",
            RTA_ALIGN(skb->rta_len));
        return;
    }
    tm = (struct tcf_t *)RTA_DATA(skb);

    rec_dbg(lev, "TCA_SKBEDIT_TM(%hu):", RTA_ALIGN(skb->rta_len));
    debug_tcf_t(lev+1, tm);
}

/*
 * debug attribute TCA_SKBEDIT_PARMS
 */
void debug_tca_skbedit_parms(int lev, struct rtattr *skb)
{
    struct tc_skbedit *parms;

    if(RTA_PAYLOAD(skb) < sizeof(*parms)) {
        rec_dbg(lev, "TCA_SKBEDIT_PARMS(%hu): -- payload too short --",
            RTA_ALIGN(skb->rta_len));
        return;
    }
    parms = (struct tc_skbedit *)RTA_DATA(skb);

    rec_dbg(lev, "TCA_SKBEDIT_PARMS(%hu):", RTA_ALIGN(skb->rta_len));
    rec_dbg(lev, "    [ tc_skbedit(%d) ]", sizeof(*parms));
    rec_dbg(lev, "        index(%d): %u", sizeof(parms->index), parms->index);
    rec_dbg(lev, "        capab(%d): %u", sizeof(parms->capab), parms->capab);
    rec_dbg(lev, "        action(%d): %d(%s)",
        sizeof(parms->action), parms->action, convert_tc_action(parms->action, 0));
    rec_dbg(lev, "        refcnt(%d): %d", sizeof(parms->refcnt), parms->refcnt);
    rec_dbg(lev, "        bindcnt(%d): %d", sizeof(parms->bindcnt), parms->bindcnt);
}

/*
 * debug attribute TCA_SKBEDIT_PRIORITY
 */
void debug_tca_skbedit_priority(int lev, struct rtattr *skb)
{
    if(RTA_PAYLOAD(skb) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_SKBEDIT_PRIORITY(%hu): -- payload too short --",
            RTA_ALIGN(skb->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_SKBEDIT_PRIORITY(%hu): 0x%08x",
        RTA_ALIGN(skb->rta_len), *(unsigned *)RTA_DATA(skb));
}

/*
 * debug attribute TCA_SKBEDIT_QUEUE_MAPPING
 */
void debug_tca_skbedit_queue_mapping(int lev, struct rtattr *skb)
{
    if(RTA_PAYLOAD(skb) < sizeof(unsigned short)) {
        rec_dbg(lev, "TCA_SKBEDIT_QUEUE_MAPPING(%hu): -- payload too short --",
            RTA_ALIGN(skb->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_SKBEDIT_QUEUE_MAPPING(%hu): %u",
        RTA_ALIGN(skb->rta_len), *(unsigned *)RTA_DATA(skb));
}

#if HAVE_DECL_TCA_SKBEDIT_MARK
/*
 * debug attribute TCA_SKBEDIT_MARK
 */
void debug_tca_skbedit_mark(int lev, struct rtattr *skb)
{
    if(RTA_PAYLOAD(skb) < sizeof(unsigned)) {
        rec_dbg(lev, "TCA_SKBEDIT_MARK(%hu): -- payload too short --",
            RTA_ALIGN(skb->rta_len));
        return;
    }
    rec_dbg(lev, "TCA_SKBEDIT_MARK(%hu): %d",
        RTA_ALIGN(skb->rta_len), *(unsigned *)RTA_DATA(skb));
}
#endif
#endif

#ifdef HAVE_LINUX_TC_ACT_TC_CSUM_H
/*
 * debug csum options(TCA_ACT_CSUM: 16)
 */
void debug_tca_act_options_csum(int lev, struct rtattr *act)
{
    struct rtattr *csum[__TCA_CSUM_MAX];

    parse_csum(csum, act);

    if(csum[TCA_CSUM_TM])
        debug_tca_csum_tm(lev+1, csum[TCA_CSUM_TM]);

    if(csum[TCA_CSUM_PARMS])
        debug_tca_csum_parms(lev+1, csum[TCA_CSUM_PARMS]);
}

/*
 * debug attribute TCA_CSUM_TM
 */
void debug_tca_csum_tm(int lev, struct rtattr *csum)
{
    struct tcf_t *tm;

    if(RTA_PAYLOAD(csum) < sizeof(*tm)) {
        rec_dbg(lev, "TCA_CSUM_TM(%hu): -- payload too short --",
            RTA_ALIGN(csum->rta_len));
        return;
    }
    tm = (struct tcf_t *)RTA_DATA(csum);

    rec_dbg(lev, "TCA_CSUM_TM(%hu):", RTA_ALIGN(csum->rta_len));
    debug_tcf_t(lev+1, tm);
}

/*
 * debug attribute TCA_CSUM_PARMS
 */
void debug_tca_csum_parms(int lev, struct rtattr *csum)
{
    struct tc_csum *parms;
    char flags_list[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(csum) < sizeof(*parms)) {
        rec_dbg(lev, "TCA_CSUM_PARMS(%hu): -- payload too short --",
            RTA_ALIGN(csum->rta_len));
        return;
    }
    parms = (struct tc_csum *)RTA_DATA(csum);
    convert_tca_csum_update_flags(parms->update_flags, flags_list, sizeof(flags_list), 1);

    rec_dbg(lev, "TCA_CSUM_PARMS(%hu):",
        RTA_ALIGN(csum->rta_len));
    rec_dbg(lev, "    [ tc_csum(%d) ]", sizeof(*parms));
    rec_dbg(lev, "        index(%d): 0x%08x", sizeof(parms->index), parms->index);
    rec_dbg(lev, "        capab(%d): 0x%08x", sizeof(parms->capab), parms->capab);
    rec_dbg(lev, "        action(%d): 0x%08x(%s)",
        sizeof(parms->action), parms->action, convert_tc_action(parms->action, 1));
    rec_dbg(lev, "        refcnt(%d): 0x%08x", sizeof(parms->refcnt), parms->refcnt);
    rec_dbg(lev, "        bindcnt(%d): 0x%08x", sizeof(parms->bindcnt), parms->bindcnt);
    rec_dbg(lev, "        update_flags(%d): 0x%08x(%s)",
        sizeof(parms->update_flags), parms->update_flags, flags_list);
}
#endif

/*
 * debug tcf_t
 */
void debug_tcf_t(int lev, struct tcf_t *tm)
{ 
    rec_dbg(lev, "[ tcf_t(%d) ]", sizeof(*tm));
    rec_dbg(lev, "    install(%d): 0x%016x",
        sizeof(tm->install), tm->install);
    rec_dbg(lev, "    lastuse(%d): 0x%016x",
        sizeof(tm->lastuse), tm->lastuse);
    rec_dbg(lev, "    expires(%d): 0x%016x",
        sizeof(tm->expires), tm->expires);
}

/*
 * convert TC_ACT_* action for debug
 */
const char *convert_tc_action(int action, int debug)
{
#define _TC_ACTION(s1, s2) \
    if(action == TC_ACT_##s1) \
        return(debug ? #s1 : #s2);
    _TC_ACTION(UNSPEC, continue);
    _TC_ACTION(OK, ok);
    _TC_ACTION(RECLASSIFY, reclassify);
    _TC_ACTION(SHOT, drop);
    _TC_ACTION(PIPE, pipe);
    _TC_ACTION(STOLEN, stolen);
    _TC_ACTION(QUEUED, queued);
    _TC_ACTION(REPEAT, repeat);
    _TC_ACTION(JUMP, jump);
#undef _TC_ACTION
    return(debug ? "UNKNOWN" : "unkonwn");
}

/*
 * convert TC_POLICE_* action from number to string
 */
const char *convert_tc_police_action(int action, int debug)
{
#define _TC_POLICE_ACTION(s1, s2) \
    if(action == TC_POLICE_##s1) \
        return(debug ? #s1 : #s2);
    _TC_POLICE_ACTION(UNSPEC, continue);
    _TC_POLICE_ACTION(OK, ok);
    _TC_POLICE_ACTION(RECLASSIFY, reclassify);
    _TC_POLICE_ACTION(SHOT, drop);
    _TC_POLICE_ACTION(PIPE, pipe);
#undef _TC_POLICE_ACTION
    return(debug ? "UNKNOWN" : "unkonown");
}

/*
 * convert PGACT_* from number to string
 */
const char *convert_pgact(int pgact, int debug)
{
#define _PGACT(s1, s2) \
    if(pgact == PGACT_##s1) \
        return(debug ? #s1 : #s2);
    _PGACT(NONE, none);
    _PGACT(NETRAND, netrand);
    _PGACT(DETERM, determ);
#undef _PGACT
    return(debug ? "UNKNOWN" : "unkonwn");
}

/*
 * convert TCA_EGRESS/INGRESS_REDIR/MIRROR from number to string
 */
const char *convert_tca_mirred_action(int action, int debug)
{
#define _TCA_MIRRED_ACTION(s1, s2) \
    if(action == TCA_##s1) \
        return(debug ? #s1 : #s2);
    _TCA_MIRRED_ACTION(EGRESS_REDIR, mode=redirect direction=egress);
    _TCA_MIRRED_ACTION(EGRESS_MIRROR, mode=mirror direction=egress);
    _TCA_MIRRED_ACTION(INGRESS_REDIR, mode=redirect direction=ingress);
    _TCA_MIRRED_ACTION(INGRESS_MIRROR, mode=mirror direction=ingress);
#undef _TCA_MIRRED_ACTION
    return(debug ? "UNKNOWN" : "mode=unknown direction=unknown");
}

#ifdef HAVE_LINUX_TC_ACT_TC_CSUM_H
/*
 * convert TCA_CSUM_UPDATE_FLAG_* flags for parse
 */
void convert_tca_csum_update_flags(int flags, char *flags_list, int len, int debug)
{
    if(!flags) {
        strncpy(flags_list, debug ? "NONE" : "none", len);
        return;
    }
#define _TCA_CSUM_UPDATE_FLAGS(s1, s2) \
    if((flags & TCA_CSUM_UPDATE_FLAG_##s1) && (len - strlen(flags_list) - 1 > 0)) \
        (flags &= ~TCA_CSUM_UPDATE_FLAG_##s1) ? \
            strncat(flags_list, #s2 ",", len - strlen(flags_list) - 1) : \
            strncat(flags_list, #s2, len - strlen(flags_list) - 1);
    _TCA_CSUM_UPDATE_FLAGS(IPV4HDR, ipv4hdr);
    _TCA_CSUM_UPDATE_FLAGS(ICMP, icmp);
    _TCA_CSUM_UPDATE_FLAGS(IGMP, igmp);
    _TCA_CSUM_UPDATE_FLAGS(TCP, tcp);
    _TCA_CSUM_UPDATE_FLAGS(UDP, udp);
    _TCA_CSUM_UPDATE_FLAGS(UDPLITE, udplite);
#undef _TCA_CSUM_UPDATE_FLAGS
    if(!strlen(flags_list))
        strncpy(flags_list, debug ? "UNKNOWN" : "unkonwn", len);
}
#endif
