/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 * Minolta Raw file plugin
 *
 * Copyright (C) 2002-2004 Bradley Broom
 *
 * Based on the Gimp PostScript plugin.
 * Copyright (C) 1997-98 Peter Kirchgessner
 * (email: peter@kirchgessner.net, WWW: http://www.kirchgessner.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

/* Major event history:
 * V 0.1.0, BB, 2004-03-29; Convert to Gimp-2.0
 * V 0.0.10, BB, 2003-03-01; Make preview generation non-blocking
 * V 0.0.9, BB, 2002-??-??; Add libeog derived preview window
 * V 0.0.8, BB, 2002-11-18; Initialize search paths to be same as mrwtoppm
 * V 0.0.6a, BB, 2002-03-21: Creation.
 */
static char dversio[] = "v0.0.6a  22-Mar-2002";
static char ident[] = "@(#) GIMP Minolta Raw file-plugin ";

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <unistd.h>

#include "../../config.h"

#include <glib/gthread.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#ifdef G_OS_WIN32
#include <process.h>		/* For _getpid() */

#define USE_REAL_OUTPUTFILE
#endif

#include "lcms.h"

#include <MRI.h>
#include <MRW_Loader.h>
#include "../FindFile.h"
#include "../options.h"

#include "PersistentVals.h"
#include "UI_RunVals.h"
#include "presets.h"
#include "preview.h"
#include "histogram.h"
#include "interp_methods.h"
#include "tonecurves.h"

#include "mrwplugin-intl.h"

char *progname = "Gimp Raw Plugin";

typedef struct
{
  GtkWidget *redBalance_entry;
  GtkWidget *redAvg_entry;
  GtkWidget *greenBalance_entry;
  GtkWidget *greenAvg_entry;
  GtkWidget *blueBalance_entry;
  GtkWidget *blueAvg_entry;
  GtkWidget *balanceMenu_entry;
  GtkWidget *balance2Menu_entry;
  GtkWidget *mixture_entry;
  GtkWidget *luminance_entry;
  GtkWidget *crop_size_entry;
  GtkWidget *seln_size_entry;
  GtkWidget *menu_bar;
  GtkWidget *fixedLength_entry;
  GtkWidget *longSize_entry;
} UI_WidgetTable;

typedef struct
{
    gint32 image_ID;
    gint32 layer_ID;
    GimpDrawable *drawable;
    GimpPixelRgn pixel_rgn;
} NewImageInfo;

extern PersistentVals plvals;

static UI_WidgetTable ui_widgets;
static UI_RunVals ui_runvals;

static MRI_Region cropRegion;

/* Declare some local functions.
 */
static void   init       (void);
static void   query      (void);
static void   run        (const gchar   *name,
                          gint     nparams,
                          const GimpParam  *param,
                          gint    *nreturn_vals,
                          GimpParam **return_vals);

static gint32 load_image (UI_RunVals *runVals, gchar   *filename);

static void   create_new_image (guint       width,
				guint       height,
				GimpImageBaseType  type,
				NewImageInfo *info);

static void   check_load_vals  (void);

static MRI *  load_mri      (gchar *filename);
static gint32 load_raw      (MRI *mri,
			     gchar *filename,
			     PersistentVals *loadopt);

static void   mrw_calculate_balance (MRI_balance *, MRI *, PersistentVals *, int xyz);
static void   update_autoBalance (MRI_balance *);
static void   update_setGainButtons (MRI_balance *);

/* Dialog-handling */

static gint   do_user_dialog               (UI_RunVals *);

GimpPlugInInfo PLUG_IN_INFO =
{
  init,  /* init_proc  */
  NULL,  /* quit_proc  */
  query, /* query_proc */
  run,   /* run_proc   */
};

void
set_current_crop_size (gdouble cropwidth, gdouble cropheight)
{
    guchar *temp;
    double width, height;

    if (cropheight > 0 && cropwidth > 0) {
	if ((cropwidth < cropheight) == plvals.fixShortSide) {
	    width = plvals.fixedSideLength;
	    height = cropheight * (width/cropwidth);
	}
	else {
	    height = plvals.fixedSideLength;
	    width = cropwidth * (height/cropheight);
	}
	temp = g_strdup_printf (_("%g by %g"), width, height);
    }
    else
	temp = g_strdup_printf ("error: empty crop - how can that be???");
    gtk_entry_set_text (GTK_ENTRY (ui_widgets.crop_size_entry), temp);
    g_free (temp);
}

void
set_current_selection_size (gdouble selnwidth, gdouble selnheight)
{
    guchar *temp;
    double width, height;

    if (selnheight > 0 && selnwidth > 0) {
	if ((selnwidth < selnheight) == plvals.fixShortSide) {
	    width = plvals.fixedSideLength;
	    height = selnheight * (width/selnwidth);
	}
	else {
	    height = plvals.fixedSideLength;
	    width = selnwidth * (height/selnheight);
	}
	temp = g_strdup_printf (_("%g by %g"), width, height);
    }
    else
	temp = g_strdup_printf (_("No region selected"));
    gtk_entry_set_text (GTK_ENTRY (ui_widgets.seln_size_entry), temp);
    g_free (temp);
}

void
do_luminance ()
{
    guchar *temp;

    temp = g_strdup_printf ("%2.1f -- %2.1f", ui_runvals.info.Lmin, ui_runvals.info.Lmax);
    gtk_entry_set_text (GTK_ENTRY (ui_widgets.luminance_entry), temp);
    g_free (temp);
}

static void
UpdateUI (UI_RunVals *runVals)
{
  if (runVals->doPreview) {
    do_preview (runVals);
  }
}

static void
UpdateUI2 (UI_RunVals *runVals, int doPreview)
{
  runVals->doPreview = doPreview;
  if (runVals->doPreview) {
    do_preview (runVals);
  }
}

struct balanceTable_t {
	char	*baseName;
	char	*intlName;
	char	*shortName;
} balanceTable[] = {
	{ N_("Camera Auto"), NULL, "camera" },
	{ N_("Daylight"), NULL, "daylight" },
	{ N_("Cloudy"), NULL, "cloudy" },
	{ N_("Flash"), NULL, "flash" },
	{ N_("Fluorescent"), NULL, "fluorescent" },
	{ N_("Preset6"), NULL, "preset6" },
	{ N_("Tungsten"), NULL, "tungsten" },

	{ N_("Gray World"), NULL, "grayworld" },
	{ N_("Binned Gray World"), NULL, "bingrayworld" },
	{ N_("Median Gray World"), NULL, "medgrayworld" },
	{ N_("Average Estimate"), NULL, "avgestimate" },

	{ N_("User"), NULL, "user" },
};

int userBalance;

void
InitBalanceTable (MRI *mri)
{
  unsigned idx;

  for (idx = 0; idx < sizeof(balanceTable)/sizeof(balanceTable[0]); idx++) {
    balanceTable[idx].intlName = _(balanceTable[idx].baseName);
    if (strcmp (balanceTable[idx].shortName, "user") == 0) {
      userBalance = idx;
    }
  }
}

int grayWorldRegionSet = 0;
MRI_Region grayWorldRegion;

gboolean
LookupBalanceIdx (MRI_balance *bal, MRI *mri, int idx)
{
  if (idx < 0 || idx >= sizeof(balanceTable)/sizeof(balanceTable[0])) {
    fprintf (stderr, "Error: White balance index (%d) out of range\n", idx);
    return FALSE;
  }
  else if (strcmp (balanceTable[idx].shortName, "user") == 0) {
    *bal = plvals.userBalance;
  }
  else if (!MRI_GetBalanceRegion (bal, mri, balanceTable[idx].shortName, grayWorldRegionSet ? &grayWorldRegion : NULL))
    FindBalanceSpec (bal, balanceTable[idx].shortName, colorMapPath);
   return TRUE;
}

int
FindBalanceIndex (char *str)
{
  int idx;

  for (idx = 0; idx < sizeof(balanceTable)/sizeof(balanceTable[0]); idx++)
    if (strcmp (balanceTable[idx].shortName, str) == 0) {
      return idx;
    }
  fprintf (stderr, "Error: Unknown white balance (%s)\n", str);
  return -1;
}

static MRI_balance oldBalance;

static void
mrw_balance_update_begin ()
{
  mrw_calculate_balance (&oldBalance, ui_runvals.mri, &plvals, 1);
}

static void
mrw_balance_update_end ()
{
  MRI_balance newBalance;
  mrw_calculate_balance (&newBalance, ui_runvals.mri, &plvals, 2);
  if (oldBalance.rgain != newBalance.rgain ||
      oldBalance.ggain != newBalance.ggain ||
      oldBalance.bgain != newBalance.bgain)
  {
#if 0
    fprintf (stderr, "Updating balance from %d %d %d -> %d %d %d\n",
		      oldBalance.rgain, oldBalance.ggain, oldBalance.bgain,
		      newBalance.rgain, newBalance.ggain, newBalance.bgain);
#endif
    update_setGainButtons (&newBalance);
    update_autoBalance(&newBalance);
  }
}

void
mrw_set_gray_world_region (MRI *mri, MRI_Region *gwr)
{
    mrw_balance_update_begin ();
    grayWorldRegion = *gwr;
    grayWorldRegionSet = gwr->width > 0 && gwr->height > 0;

    MRI_ClearBalance (mri, "grayworld");
    MRI_ClearBalance (mri, "bingrayworld");
    MRI_ClearBalance (mri, "medgrayworld");
    MRI_ClearBalance (mri, "avgestimate");
    mrw_balance_update_end ();
    UpdateUI (&ui_runvals);
}

void
mrw_set_crop_region (MRI *mri, MRI_Region *cr)
{
    MRI_Region selnRegion;
    cropRegion = *cr;
    set_current_crop_size (cropRegion.width, cropRegion.height);
    if (get_preview_selected_rectangle (&selnRegion))
        set_current_selection_size (selnRegion.width, selnRegion.height);
    else
        set_current_selection_size (0.0, 0.0);
    UpdateUI (&ui_runvals);
}

static void
InitOptionsAndDefaults (UI_RunVals *runVals, PersistentVals *pVals)
{
	char	buffer[2048];

        InitOptions (); /* Initialize option paths etc. */

	/* Load options. */
	snprintf (buffer, sizeof(buffer), "%s/mrwtoppm.rc", DATAPREFIX);
	ProcessOptionFile (buffer, "gimp");
	if (getenv ("HOME")) {
		snprintf (buffer, sizeof(buffer), "%s/.mrwtoppm/mrwtoppm.rc", getenv("HOME"));
		ProcessOptionFile (buffer, "gimp");
	}

	/* If this is the first time, copy persisent options into plvals. */
	if (pVals->initialized == 0) {
	    InitializePersistentVals (pVals);
	}

	runVals->pVals = pVals;
	runVals->doPreview = FALSE;
}

void
InitImageOptions (UI_RunVals *runVals)
{
        InitBalanceTable (runVals->mri);
	/* Initially process entire image. */
	cropRegion.x = cropRegion.y = 0;
	cropRegion.width = MRI_GetWidth (runVals->mri);
	cropRegion.height = MRI_GetHeight (runVals->mri);

        if (cameraProfileName && cameraProfileName[0] != '\0')
	  runVals->mri->colorSpace = cameraProfileName;
}

MAIN ()

/* The run mode */
static GimpRunMode l_run_mode;

static void
init (void)
{
}

static void
query (void)
{
  static GimpParamDef load_args[] =
  {
    { GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive" },
    { GIMP_PDB_STRING, "filename", "The name of the file to load" },
    { GIMP_PDB_STRING, "raw_filename", "The name of the file to load" }
  };
  static GimpParamDef load_return_vals[] =
  {
    { GIMP_PDB_IMAGE, "image", "Output image" },
  };
  static gint nload_args = sizeof (load_args) / sizeof (load_args[0]);
  static gint nload_return_vals = (sizeof (load_return_vals) /
				   sizeof (load_return_vals[0]));

  static GimpParamDef set_load_args[] =
  {
    { GIMP_PDB_INT32, "resolution", "Resolution to interprete image (dpi)" },
    { GIMP_PDB_INT32, "width", "Desired width" },
    { GIMP_PDB_INT32, "height", "Desired height" },
    { GIMP_PDB_INT32, "check_bbox", "0: Use width/height, 1: Use BoundingBox" },
    { GIMP_PDB_INT32, "iMethodNum", "Interpolation Method" },
    { GIMP_PDB_INT32, "coloring", "4: b/w, 5: gray, 6: colour image, 7: automatic" },
    { GIMP_PDB_INT32, "TextAlphaBits", "1, 2, or 4" },
    { GIMP_PDB_INT32, "GraphicsAlphaBits", "1, 2, or 4" }
  };
  static gint nset_load_args = (sizeof (set_load_args) /
				sizeof (set_load_args[0]));

  gimp_install_procedure ("file_raw_load",
                          "load file of Minolta RAW file format",
                          "load file of Minolta RAW file format",
                          "Bradley Broom",
                          "Bradley Broom",
                          dversio,
                          "<Load>/Minolta RAW",
                          NULL,
                          GIMP_PLUGIN,
                          nload_args, nload_return_vals,
                          load_args, load_return_vals);

  gimp_install_procedure ("file_raw_load_setargs",
                          "set additional parameters for procedure file_raw_load",
                          "set additional parameters for procedure file_raw_load",
                          "Bradley Broom",
                          "Bradley Broom",
                          dversio,
                          NULL,
                          NULL,
                          GIMP_PLUGIN,
                          nset_load_args, 0,
                          set_load_args, NULL);

  gimp_register_magic_load_handler ("file_raw_load",
				    "mrw",
				    "",
                                    "1,string,MRM");
}

static void
run (const gchar   *name,
     gint     nparams,
     const GimpParam  *param,
     gint    *nreturn_vals,
     GimpParam **return_vals)
{
  static GimpParam values[2];
  GimpRunMode  run_mode;
  GimpPDBStatusType   status = GIMP_PDB_SUCCESS;
  gint32        image_ID = -1;

  init();
  l_run_mode = run_mode = param[0].data.d_int32;

  *nreturn_vals = 1;
  *return_vals  = values;
  values[0].type          = GIMP_PDB_STATUS;
  values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;

  if (strcmp (name, "file_raw_load") == 0)
    {
      bindtextdomain (GETTEXT_PACKAGE, MRWPLUGINLOCALEDIR);
      bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
      textdomain (GETTEXT_PACKAGE);

      plvals.initialized = 0;
      gimp_get_data ("file_raw_load", &plvals);
      InitOptionsAndDefaults (&ui_runvals, &plvals);

      ui_runvals.mri = load_mri (param[1].data.d_string);
      if (ui_runvals.mri == (MRI *)0)
	status = GIMP_PDB_EXECUTION_ERROR;
      else {
        InitImageOptions (&ui_runvals);

        switch (run_mode)
	  {
	  case GIMP_RUN_INTERACTIVE:
	    gimp_ui_init ("mrw", FALSE);
	    if (! do_user_dialog (&ui_runvals))
	      status = GIMP_PDB_CANCEL;
	    break;

	  case GIMP_RUN_NONINTERACTIVE:
	    fprintf (stderr, "gimp run noninteractive\n");
	    /*  Make sure all the arguments are there!  */
	    if (nparams != 3)
	      status = GIMP_PDB_CALLING_ERROR;
	    else {  /* Get additional interpretation arguments */
	    }
	    break;

	  case GIMP_RUN_WITH_LAST_VALS:
	    fprintf (stderr, "gimp run with last vals\n");
	    /* Possibly retrieve data */
	    break;

	  default:
	    fprintf (stderr, "gimp run default\n");
	    break;
	  }
	}

      if (status == GIMP_PDB_SUCCESS)
	{
	  check_load_vals ();
	  image_ID = load_image (&ui_runvals, param[1].data.d_string);

	  if (image_ID != -1)
	    {
	      *nreturn_vals = 2;
	      values[1].type         = GIMP_PDB_IMAGE;
	      values[1].data.d_image = image_ID;
	    }
	  else
	    {
	      status = GIMP_PDB_EXECUTION_ERROR;
	    }
	}

      /*  Store plvals data  */
      if (status == GIMP_PDB_SUCCESS) {
	gimp_set_data ("file_raw_load", &plvals, sizeof (PersistentVals));
      }

      if (ui_runvals.mri)
      	MRI_Free (ui_runvals.mri);
    }
  else if (strcmp (name, "file_raw_load_setargs") == 0)
    {
      /*  Make sure all the arguments are there!  */
      if (nparams != 8)
	{
	  status = GIMP_PDB_CALLING_ERROR;
	}
      else
	{
	  plvals.rotate         = param[0].data.d_int32;
	  plvals.iSharpen       = param[1].data.d_int32;
	  plvals.cSharpen       = param[2].data.d_int32;
	  plvals.useLab         = param[3].data.d_int32;
	  plvals.iMethodNum     = param[4].data.d_int32;
	  plvals.applyVMF       = param[5].data.d_int32;
	  plvals.applyMedianFilter = param[6].data.d_int32;
	  plvals.toneCurveNum = param[7].data.d_int32;
	  check_load_vals ();
	  gimp_set_data ("file_raw_load", &plvals, sizeof (PersistentVals));
	}
    }
  else
    {
      status = GIMP_PDB_CALLING_ERROR;
    }

  values[0].data.d_status = status;
}

static gint32
load_image (UI_RunVals *runVals, gchar *filename)
{
  gint32 image_ID;
  char *temp;
  MRI *mri = runVals->mri;

#undef RAW_PLUGIN_DEBUG
#ifdef RAW_PLUGIN_DEBUG
  g_print ("load_image:\n rotate = %d\n", plvals.rotate);
  g_print (" useLab: %d\n", plvals.useLab);
  g_print (" Interpolation Method: %s\n", get_interpolation_method_shortname(plvals.iMethodNum));
  g_print (" Apply VMF: %d\n", plvals.applyVMF);
  g_print (" Apply Median Filter: %d\n", plvals.applyMedianFilter);
  g_print (" Radius: %d\n", plvals.radius);
  g_print (" Blur: %g\n", plvals.blur);
  g_print (" cSharpen: %g\n", plvals.cSharpen);
  g_print (" iSharpen: %g\n", plvals.iSharpen);
#endif

  if (l_run_mode != GIMP_RUN_NONINTERACTIVE)
    {
      temp = g_strdup_printf (_("Interpreting and Loading %s:"), filename);
      gimp_progress_init (temp);
      g_free (temp);
    }

  image_ID = load_raw (mri, filename, &plvals);
  if (image_ID == -1)
      g_message (_("Gimp Raw Plugin: can't interprete file"));
  return (image_ID);
}

/* Check (and correct) the load values plvals */
static void
check_load_vals (void)
{
  if (plvals.rotate != 0 && plvals.rotate != 90 && plvals.rotate != 180
      && plvals.rotate != 270)
    plvals.rotate = 0;

  if (plvals.finalShrink < 1)
    plvals.finalShrink = 1;
  else if (plvals.finalShrink != 1 && (plvals.finalShrink & 1))
    plvals.finalShrink++;
  if (plvals.finalShrink > 10)
    plvals.finalShrink = 10;

  plvals.useLab = (plvals.useLab != 0);
}

/* Create an image. Sets all fields in *info. */
static void
create_new_image (guint        width,
                  guint        height,
                  GimpImageBaseType   type,
		  NewImageInfo *info)
{
  GimpImageType gdtype;

  if (type == GIMP_GRAY) gdtype = GIMP_GRAY_IMAGE;
  else if (type == GIMP_INDEXED) gdtype = GIMP_INDEXED_IMAGE;
  else gdtype = GIMP_RGB_IMAGE;

  info->image_ID = gimp_image_new (width, height, type);

  info->layer_ID = gimp_layer_new (info->image_ID, "Background", width, height,
			           gdtype, 100, GIMP_NORMAL_MODE);
  gimp_image_add_layer (info->image_ID, info->layer_ID, 0);

  info->drawable = gimp_drawable_get (info->layer_ID);
  gimp_pixel_rgn_init (&info->pixel_rgn, info->drawable, 0, 0, info->drawable->width,
		       info->drawable->height, TRUE, FALSE);
}

struct GimpLoaderData {
  int	        width, height;
  int	        freedata;
  int	        tile_height;
  NewImageInfo  *info;
  int	        reversed;
  guchar       *data, *dest;
  gint          bpp;
  gint          scan_lines, total_scan_lines;
};

void
GimpLoaderStart (void *private, int width, int height, int freedata)
{
  struct GimpLoaderData *wd = private;

#ifdef DEBUG
  g_print ("load_image:\n Got a start (%dx%d) ... \n", width, height);
#endif
  wd->width = width;
  wd->height = height;
  wd->freedata = freedata;
  wd->dest = wd->data = g_malloc (wd->tile_height * width * wd->bpp);
#ifdef DEBUG
  g_print ("load_image:\n Finished starting ... \n");
#endif
  create_new_image (width, height, GIMP_RGB, wd->info);
}

void
GimpLoaderRow (void *private, void *data)
{
  struct GimpLoaderData *wd = private;
  struct MRI_ScanLine *sl = data;
  int i;

  if (sl->scanLineType != LINETYPE_SHORT) {
	fprintf (stderr, "Error: GimpLoaderRow: unexpected line type %d\n", sl->scanLineType);
	_exit (1);
  }
  else {
	  unsigned short *R = (unsigned short *)sl->R;
	  unsigned short *G = (unsigned short *)sl->G;
	  unsigned short *B = (unsigned short *)sl->B;

	  for (i = 0; i < wd->width; i++) {
		*wd->dest++ = R[i] >> 8;
		*wd->dest++ = G[i] >> 8;
		*wd->dest++ = B[i] >> 8;
	  }
  }
  wd->scan_lines++;
  wd->total_scan_lines++;

  if ((l_run_mode != GIMP_RUN_NONINTERACTIVE) && ((wd->total_scan_lines % 20) == 0))
    gimp_progress_update ((double)(wd->total_scan_lines+1) / (double)wd->height);

  if ((wd->scan_lines == wd->tile_height) ||
      ((wd->total_scan_lines) == wd->height))
    {
      gimp_pixel_rgn_set_rect (&wd->info->pixel_rgn, wd->data,
                               0, wd->total_scan_lines-wd->scan_lines,
			       wd->width, wd->scan_lines);
      wd->scan_lines = 0;
      wd->dest = wd->data;
    }
}

void
GimpLoaderClose (void *private)
{
  struct GimpLoaderData *wd = private;

  g_free (wd->data);
}

MRI_Link *
GenGimpLoader (NewImageInfo *info, int reversed)
{
  struct GimpLoaderData *wd = malloc(sizeof (struct GimpLoaderData));
  MRI_Link *ep = malloc (sizeof (*ep));
  if (wd == (struct GimpLoaderData *)0 || ep == (MRI_Link *)0) {
	fprintf (stderr, "Error: out of memory\n");
	_exit (1);
  }
#ifdef DEBUG
  g_print ("load_image:\n Generating GimpLoader ... \n");
#endif
  ep->start = GimpLoaderStart;
  ep->row = GimpLoaderRow;
  ep->close = GimpLoaderClose;
  ep->private = wd;

  wd->info = info;
  wd->tile_height = gimp_tile_height ();
  wd->bpp = 3;
  wd->total_scan_lines = wd->scan_lines = 0;
#ifdef DEBUG
  g_print ("load_image:\n GimpLoader Generated ... \n");
#endif

  return ep;
}

/* Load PNM image generated from RAW file */
static MRI *
load_mri (gchar *filename)
{
         FILE  *ifp;

	MRI *mri;
	char	*errmsg;


	if ((ifp = fopen (filename, "r")) == NULL)
		return (MRI *)0;
	mri = MRW_Loader (ifp, &errmsg);
	fclose (ifp);

	if (mri) {

	    /* Apply camera details adjustment (if requested). */
	    if (cameraDetails && strcmp (cameraDetails, "none") != 0) {
		char *cameraDetailsFile;
		if ((cameraDetailsFile = FindFile (cameraDetails, cameraDetailsPath, ".mrw")) != (char *)0) {
			MRI_ApplyNoiseMap (mri, cameraDetailsFile);
		}
	    }
	}

	return mri;
}

static gint32
load_raw (MRI *mri,
	  gchar *filename,
	  PersistentVals *loadopt)
{
  NewImageInfo info;
  MRI_Link *head;
  char *tmp;

  head = GenGimpLoader (&info, FALSE);
  head = MakeRenderRaw (mri, head, 0, NULL, plvals.finalShrink, loadopt, NULL);

  MRI_ProcessImageRegion (head, mri, loadopt->rotate, &cropRegion);
  MRI_FlushPipeline (head);

  tmp = strdup (filename);
  gimp_image_set_filename (info.image_ID, tmp);
  g_free (tmp);

  gimp_drawable_flush (info.drawable);

  return info.image_ID;
}

MRI_Link *
MakeRenderRaw (MRI *mri,
	   MRI_Link *head,
	   int N,
	   int *counts,
	   int shrink,
	   PersistentVals *loadopt,
	   FILE *tracefd)
{
	MRI_balance balanceSpec;
	double lumascale;

	mrw_calculate_balance (&balanceSpec, mri, loadopt, 3);
	MRI_AdjustBalanceLuminance (mri, &balanceSpec, &lumascale);


  /* Create output filter sequence starting from the rear.
   * The last link in the pipeline outputs scanlines to the output device.
   */

	/* At this point, the entire image has been input.  We now construct the processing pipeline
 	 * to convert the input image into the output. The process starts from the last stage, the
	 * output writer, and builds towards the front of the processing pipeline.
	 */

	/* Before generating the last link in that pipeline, we need to know the order of pixels in the scanlines
	 * coming down the pipeline. The rotate option means that the scanlines can begin with either
	 * a RG..RG scanline (if working from the top of the image), or with a GB...GB scanline (if working from
	 * the bottom). If we also need to reverse the order of pixels in each scanline, we defer doing so until
	 * the scanline is output. (It considerably simplifies the earlier code, which need not consider GR...GR
	 * scanlines as well as the normal order, with only a trivial increase in complexity of the output stage.
	 */
  if (Reversed(loadopt->rotate))
	head = MRI_GenReverser (head);


	{ char *cameraProfileFile;
	  char *outputProfileFile;

	  if (strcmp (mri->colorSpace, MRI_NATIVE_PROFILE) == 0)
	      cameraProfileFile = strdup (MRI_NATIVE_PROFILE);
	  else {
	      cameraProfileFile = FindFile (mri->colorSpace, profilePath, NULL);
	      if (cameraProfileFile == (char *)0) {
		  fprintf (stderr, "%s: Warning: cannot find camera color space profile: '%s'. Falling back to builtin profile.\n", progname, mri->colorSpace);
		  cameraProfileFile = strdup (MRI_NATIVE_PROFILE);
	      }
	  }

	  if (strcmp (outputProfileName, "__sRGB__") == 0)
		outputProfileFile = strdup ("__sRGB__");
	  else {
		outputProfileFile = FindFile (outputProfileName, profilePath, NULL);
		if (outputProfileFile == (char *)0)
			fprintf (stderr, "%s: Error: cannot find output color space profile: %s\n", progname, outputProfileName);
	  }
	  InitColorProfiles (mri, loadopt->useLab, cameraProfileFile, outputProfileFile);
	}

	/* Apply additional gamma correction and "contrast" stretching.
	 */
        /* head = GenMRIEnhance (autolevel, gammaVal, head); */

	/* Convert from the internal color space to the output color space.
	 */
	head = GenOutputProfileConverter (mri, head);

	/* Convert from scanlines of fpixels into scanlines of unsigned shorts.
	 */
	head = MRI_GenFtSConverter (head, mri, loadopt->useLab);
	if (counts)
		head = MRI_GenHistogramCounter (N, counts, NULL, NULL, head);

	{ MRI_ImproverOptions io;
	  io.radius = loadopt->radius;
	  io.useLab = loadopt->useLab;
	  io.blur = loadopt->blur;
	  io.iSharpen = loadopt->iSharpen;
	  io.cSharpen = loadopt->cSharpen;
	  io.applyMedianFilter = loadopt->applyMedianFilter;
	  io.applyVMF = loadopt->applyVMF;
	  io.filterColorOnly = loadopt->filterColorOnly;
	  io.vMedianTolerance = loadopt->vmedian_tolerance;
	  io.toneCurveFile = (char *)0;
	  if (loadopt->toneCurveNum != 0) {
		char *toneCurve = get_tone_curve_shortname (loadopt->toneCurveNum);
		if ((io.toneCurveFile = FindFile (toneCurve, toneCurvePath, ".tc")) != (char *)0) {
#ifdef DEBUG
			fprintf (stderr, "Info: Applying tone curve %s\n", toneCurve);
#endif
		}
		else {
			fprintf (stderr, "Warning: Could not find tone curve '%s'. Output not adjusted.\n", toneCurve);
		}
	  }
#ifdef DEBUG
	  fprintf (stderr, "  Radius=%d, Blur=%g, iSharpen=%g, cSharpen=%g\n", io.radius, io.blur, io.iSharpen, io.cSharpen);
#endif
	  head = MRI_GenImageImprover (head, &io, &ui_runvals.info);
	}

	if (loadopt->useLab) {
#ifdef DEBUG
		fprintf (stderr, "Darkness=%g, contrast=%g, saturation=%g\n", loadopt->darkness, loadopt->contrast, loadopt->saturation);
#endif
		head = MRI_GenContrastAdjuster (loadopt->darkness, lumascale, loadopt->contrast, loadopt->shadows, loadopt->saturation, head);
	}

	{ MRI_InterpolatorOptions io;
	  io.useLab = loadopt->useLab;
	  io.rotate = loadopt->rotate;
	  io.imethod = get_interpolation_method_shortname(loadopt->iMethodNum);
	  io.vmedian_tolerance = loadopt->vmedian_tolerance;
	  io.doInterpolate = shrink == 1;
	  head = MRI_GenBayerInterpolator (mri, head, &io, tracefd);
	}

	FreeProfileFiles ();

	/* head = MRI_GenColorBalancer (balanceSpec, head); */

	if (shrink > 1)
	    head = MRI_GenSubsampler (shrink, shrink, head);

	/* if (tracefd) head = MRI_GenPPMWriterTee (head, 8, FALSE, tracefd); */
	head = MRI_GenLimitCorrecter (balanceSpec, head, MRI_RGFirst(mri, loadopt->rotate));
	head = MRI_GenLimitMasker (head);
	return head;
}

static void
mrw_color_toggle_callback (GtkWidget *widget, gpointer data)
{
  gimp_toggle_button_update (widget, data);
  UpdateUI (&ui_runvals);
}

static void
mrw_double_adjustment_update (GtkAdjustment *widget, gpointer data)
{
  gimp_double_adjustment_update (widget, data);
  UpdateUI (&ui_runvals);
}

static void
mrw_mixing_adjustment_update (GtkAdjustment *widget, gpointer data)
{
  mrw_balance_update_begin ();
  gimp_double_adjustment_update (widget, data);
  mrw_balance_update_end ();
}

static void
mrw_radius_adjustment_update (GtkAdjustment *widget, gpointer data)
{
  gimp_int_adjustment_update (widget, data);
  UpdateUI (&ui_runvals);
}

static void
mrw_red_gain_adjustment_update (GtkWidget *widget, gpointer data)
{
  guint rgain;
  guint doPreviewSave;
  MRI_balance oldBalance;

  doPreviewSave = ui_runvals.doPreview;
  ui_runvals.doPreview = FALSE;
  gimp_double_adjustment_update (GTK_ADJUSTMENT(widget), (double *)data);
  plvals.redBalance = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(ui_widgets.redBalance_entry));
  if (doPreviewSave) {
      mrw_calculate_balance (&oldBalance, ui_runvals.mri, &plvals, 4);
      update_autoBalance (&oldBalance);
      UpdateUI2 (&ui_runvals, TRUE);
  }
}

static void
mrw_green_gain_adjustment_update (GtkWidget *widget, gpointer data)
{
  guint ggain;
  guint doPreviewSave;
  MRI_balance oldBalance;

  doPreviewSave = ui_runvals.doPreview;
  ui_runvals.doPreview = FALSE;
  gimp_double_adjustment_update (GTK_ADJUSTMENT(widget), (double *)data);
  plvals.greenBalance = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(ui_widgets.greenBalance_entry));
  mrw_calculate_balance (&oldBalance, ui_runvals.mri, &plvals, 5);
  update_autoBalance (&oldBalance);
  UpdateUI2 (&ui_runvals, doPreviewSave);
}

static void
mrw_blue_gain_adjustment_update (GtkWidget *widget, gpointer data)
{
  guint bgain;
  guint doPreviewSave;
  MRI_balance oldBalance;

  doPreviewSave = ui_runvals.doPreview;
  ui_runvals.doPreview = FALSE;
  gimp_double_adjustment_update (GTK_ADJUSTMENT(widget), (double *)data);
  plvals.blueBalance = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(ui_widgets.blueBalance_entry));
  mrw_calculate_balance (&oldBalance, ui_runvals.mri, &plvals, 6);
  update_autoBalance (&oldBalance);
  UpdateUI2 (&ui_runvals, doPreviewSave);
}

static void
mrw_fixed_length_adjustment_update (GtkWidget *widget, gpointer data)
{
  guint doPreviewSave;
  MRI_Region selnRegion;

  doPreviewSave = ui_runvals.doPreview;
  ui_runvals.doPreview = FALSE;
  gimp_double_adjustment_update (GTK_ADJUSTMENT(widget), (double *)data);
  set_current_crop_size (cropRegion.width, cropRegion.height);
  if (get_preview_selected_rectangle (&selnRegion))
      set_current_selection_size (selnRegion.width, selnRegion.height);
  else
      set_current_selection_size (0.0, 0.0);
  ui_runvals.doPreview = doPreviewSave;
}

GtkWidget *tone_menu;

static int
mrw_tone_curve_menu_item_update (GtkWidget *widget, gpointer data)
{
  gimp_menu_item_update (widget, data);
  plvals.toneCurveNum = gtk_option_menu_get_history (GTK_OPTION_MENU(tone_menu));
  UpdateUI (&ui_runvals);
  return TRUE;
}

GtkWidget *
GenToneCurveOptionMenu (int initial)
{
  GtkWidget *menu;
  GtkWidget *mi;
  GtkWidget *label;
  unsigned i;

  tone_menu = gtk_option_menu_new ();
  menu = gtk_menu_new ();

  for (i = 0; i < num_tone_curves(); i++) {
    mi = gtk_menu_item_new_with_label (get_tone_curve_intlname(i));
    gtk_signal_connect(GTK_OBJECT(mi), "activate", GTK_SIGNAL_FUNC (mrw_tone_curve_menu_item_update), get_tone_curve_handle(i));
    gtk_widget_show (mi);
    gtk_menu_shell_append (GTK_MENU_SHELL(menu), mi);
  }

  gtk_widget_show (menu);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (tone_menu), menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (tone_menu), initial);
  gtk_widget_show (tone_menu);

  return tone_menu;
}

static void
newbal_item_update (GtkWidget *widget, gpointer data)
{
  mrw_balance_update_begin ();
  gimp_menu_item_update (widget, data);
  mrw_balance_update_end ();
}

void       
my_table_attach_aligned (GtkTable *table,
                         gint column,
                         gint row,
                         gchar *label_text,
                         gfloat xalign,
                         gfloat yalign,
                         GtkWidget *widget,
                         gint colspan,
                         gboolean left_align,
			 GtkAttachOptions a_options)
{
  GtkWidget *label = gtk_label_new (label_text);

  gtk_widget_show (label);
  gtk_widget_show (widget);
  gtk_table_attach (table, label, 0, 1, row, row+1,
                    0, 0, 2, 2);
  gtk_table_attach (table, widget, 1, 2, row, row+1,
		    a_options, 0, 2, 2);
}

void
AddInfoEntry (GtkWidget *table, int *rowp, const char *label, const char *format, ...)
{
  GtkWidget *entry;
  char *buffer;
  va_list ap;

  va_start (ap, format);
  buffer = g_strdup_vprintf (format, ap);
  va_end (ap);

  entry = gtk_entry_new ();
  gtk_widget_set_usize (entry, 150, 0);
  gtk_entry_set_text (GTK_ENTRY (entry), buffer);
  gtk_entry_set_editable (GTK_ENTRY (entry), FALSE);
  my_table_attach_aligned (GTK_TABLE (table), 0, (*rowp)++,
			     _(label), 1.0, 0.5,
			     entry, 1, TRUE, 0);
}

void
rotate_clockwise (GtkWidget *widget, gpointer data)
{
  plvals.rotate = plvals.rotate == 270 ? 0 : plvals.rotate+90;
  rotate_preview (90);
}

void
rotate_180 (GtkWidget *widget, gpointer data)
{
  switch (plvals.rotate) {
    case   0: plvals.rotate = 180; break;
    case  90: plvals.rotate = 270; break;
    case 180: plvals.rotate =   0; break;
    case 270: plvals.rotate =  90; break;
  }
  rotate_preview (180);
}

void
rotate_anticlockwise (GtkWidget *widget, gpointer data)
{
  plvals.rotate = plvals.rotate == 0 ? 270 : plvals.rotate-90;
  rotate_preview (270);
}

#if 0
GtkWidget *
RotateMenuItem ()
{
  GtkWidget *rotate_menu;
  GtkWidget *mi;
  GtkWidget *label;

  label = gtk_menu_item_new_with_label (_("Rotate"));
  gtk_widget_show (label);

  rotate_menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM(label), rotate_menu);

  mi = gtk_menu_item_new_with_label (_("Clockwise"));
  gtk_widget_show (mi);
  gtk_menu_shell_append (GTK_MENU_SHELL(rotate_menu), mi);
  gtk_signal_connect(GTK_OBJECT(mi), "activate",
      GTK_SIGNAL_FUNC (rotate_clockwise), NULL);
  mi = gtk_menu_item_new_with_label ("180");
  gtk_widget_show (mi);
  gtk_menu_shell_append (GTK_MENU_SHELL(rotate_menu), mi);
  gtk_signal_connect(GTK_OBJECT(mi), "activate",
      GTK_SIGNAL_FUNC (rotate_180), NULL);
  mi = gtk_menu_item_new_with_label (_("Anti-clockwise"));
  gtk_widget_show (mi);
  gtk_menu_shell_append (GTK_MENU_SHELL(rotate_menu), mi);
  gtk_signal_connect(GTK_OBJECT(mi), "activate",
      GTK_SIGNAL_FUNC (rotate_anticlockwise), NULL);

  return label;
}
#endif

char *scales[] = {
   N_("1"),
   N_("1/2"),
   N_("1/4"),
   N_("1/6"),
   N_("1/8"),
   N_("1/10")
};

int
strtoshrink (const char *str)
{
  int shrink;

  if (strcmp (str, "1") == 0)
    shrink = 1;
  else
    sscanf (str, "1/%d", &shrink);
  return shrink;
}

static int
set_final_shrink (GtkWidget *widget, gpointer data)
{
  plvals.finalShrink = strtoshrink ((char *)data);
  return TRUE;
}

static int
set_preview_shrink (GtkWidget *widget, gpointer data)
{
  char *shrink = (char *)data;
  int oldShrink = plvals.previewShrink;
  int newShrink = strtoshrink (shrink);

  if (newShrink != oldShrink) {
    if (!get_preview_selected_rectangle (&ui_runvals.selection))
        ui_runvals.selection.width = 0;
    plvals.previewShrink = newShrink;
    UpdateUI (&ui_runvals);
  }
  return TRUE;
}

unsigned
find_scale_index (int scale)
{
  unsigned i;
  char buffer[40];

  if (scale == 1)
      strcpy (buffer, "1");
  else
      sprintf (buffer, "1/%d", scale);
  for (i = 0; i < sizeof(scales)/sizeof(scales[0]); i++)
    if (strcmp (scales[i], buffer) == 0)
        return i;
  return 0;
}

GtkWidget *preview_shrink_scale_widget;

update_preview_shrink_scale (int scale)
{
  gtk_option_menu_set_history (GTK_OPTION_MENU (preview_shrink_scale_widget),
                               find_scale_index(scale));
}


GtkWidget *
GenScaleMenu (GtkSignalFunc func, int init_scale)
{
  GtkWidget *optmenu;
  GtkWidget *scale_menu;
  GtkWidget *mi;
  GtkWidget *label;
  unsigned i;

  optmenu = gtk_option_menu_new ();
  scale_menu = gtk_menu_new ();

  for (i = 0; i < sizeof(scales)/sizeof(scales[0]); i++) {
    mi = gtk_menu_item_new_with_label (scales[i]);
    gtk_signal_connect(GTK_OBJECT(mi), "activate", func, scales[i]);
    gtk_widget_show (mi);
    gtk_menu_shell_append (GTK_MENU_SHELL(scale_menu), mi);
  }

  gtk_widget_show (scale_menu);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (optmenu), scale_menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (optmenu),
                               find_scale_index(init_scale));
  gtk_widget_show (optmenu);
  return optmenu;
}

GtkWidget *
ScaleMenuItem (char *name, GtkSignalFunc func, int init_scale)
{
  GtkWidget *scale_menu;
  GtkWidget *mi;
  GtkWidget *label;
  unsigned i;
  char buffer[40];

  label = gtk_menu_item_new_with_label (name);
  gtk_widget_show (label);

  scale_menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM(label), scale_menu);

  sprintf (buffer, "1/%d", init_scale);
  for (i = 0; i < sizeof(scales)/sizeof(scales[0]); i++) {
    mi = gtk_menu_item_new_with_label (scales[i]);
    gtk_signal_connect(GTK_OBJECT(mi), "activate", func, scales[i]);
    if (strcmp (scales[i], buffer) == 0)
      gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mi), TRUE);
    gtk_widget_show (mi);
    gtk_menu_shell_append (GTK_MENU_SHELL(scale_menu), mi);
  }

  return label;
}


#if 0
GtkWidget *
OptionsMenuItem ()
{
  GtkWidget *options_menu;
  GtkWidget *label;

  label = gtk_menu_item_new_with_label (_("Options"));
  gtk_widget_show (label);

  options_menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM(label), options_menu);

  gtk_menu_shell_append (GTK_MENU_SHELL(options_menu), ScaleMenuItem("Image Scale", GTK_SIGNAL_FUNC(set_final_shrink), 1));
  gtk_menu_shell_append (GTK_MENU_SHELL(options_menu), ScaleMenuItem("Preview Scale", GTK_SIGNAL_FUNC(set_preview_shrink), 8));
  gtk_menu_shell_append (GTK_MENU_SHELL(options_menu), InterpolationMenuItem(plvals.iMethodNum));
  gtk_menu_shell_append (GTK_MENU_SHELL(options_menu), PreviewInterpolationMenuItem());

  return label;
}
#endif


#if 0
GtkWidget *
MainMenuBar ()
{
  GtkWidget *main_menu;

  main_menu = gtk_menu_bar_new ();

  /* gtk_menu_bar_append (GTK_MENU_BAR(main_menu), RotateMenuItem()); */

  gtk_menu_bar_append (GTK_MENU_BAR(main_menu), OptionsMenuItem());

  return main_menu;
}
#endif

static void
mrw_calculate_balance (MRI_balance *bal, MRI *mri, PersistentVals *pv, int xyz)
{
  MRI_balance nb1, nb2;

  switch (pv->balanceMode) {
  default:
    fprintf (stderr, "mrw plugin: Illegal balanceMode %d, trying manual\n",
    	     pv->balanceMode);
    /* Fall through */
  case BALANCE_MODE_MANUAL:
#if 0
    fprintf (stderr, "mrw_calculate_balance %d manual %g,%g,%g\n", xyz,
     		     pv->redBalance,
     		     pv->greenBalance,
     		     pv->blueBalance);
#endif
    bal->rgain = pv->redBalance;
    bal->ggain = pv->greenBalance;
    bal->bgain = pv->blueBalance;
    break;
  case BALANCE_MODE_PRESET:
    if (!LookupBalanceIdx (&nb1, mri, pv->newbal1)) {
        nb1.rgain = pv->redBalance;
        nb1.ggain = pv->greenBalance;
        nb1.bgain = pv->blueBalance;
    }
    if (!LookupBalanceIdx (&nb2, mri, pv->newbal2)) {
        nb2.rgain = pv->redBalance;
        nb2.ggain = pv->greenBalance;
        nb2.bgain = pv->blueBalance;
    }
#if 0
    fprintf (stderr, "mrw_calculate_balance %d preset", xyz);
    fprintf (stderr, " %g of %d,%d,%d",
     		     (100.0-pv->mix)/100.0, nb1.rgain, nb1.ggain, nb1.bgain);
    fprintf (stderr, " + %g of %d,%d,%d",
		    (pv->mix)/100.0, nb2.rgain, nb2.ggain, nb2.bgain);
#endif
    bal->rgain = (nb1.rgain * (100.0-pv->mix) + nb2.rgain * pv->mix) / 100.0;
    bal->ggain = (nb1.ggain * (100.0-pv->mix) + nb2.ggain * pv->mix) / 100.0;
    bal->bgain = (nb1.bgain * (100.0-pv->mix) + nb2.bgain * pv->mix) / 100.0;
#if 0
    fprintf (stderr, "= %d,%d,%d\n",
     		     bal->rgain, bal->ggain, bal->bgain);
#endif
    break;
  }
}

static void
mrw_set_balance ()
{
  MRI_balance nb;

  mrw_calculate_balance (&nb, ui_runvals.mri, &plvals, 7);
  if (oldBalance.rgain != nb.rgain ||
      oldBalance.ggain != nb.ggain ||
      oldBalance.bgain != nb.bgain)
  {
    update_setGainButtons (&nb);
    update_autoBalance(&nb);
  }
}


static int
mrw_set_balance1 (GtkWidget *widget, gpointer data)
{
  plvals.newbal1 = (struct balanceTable_t *)data - balanceTable;
  mrw_set_balance ();
  return TRUE;
}

static int
mrw_set_balance2 (GtkWidget *widget, gpointer data)
{
  plvals.newbal2 = (struct balanceTable_t *)data - balanceTable;
  mrw_set_balance ();
  return TRUE;
}

GtkWidget *
mrw_ui_balance_menu (GtkSignalFunc sig_fun, int init_balance)
{
  GtkWidget *optmenu, *menu;
  GtkWidget *mi;
  GtkWidget *label;
  int	i, active;

  optmenu = gtk_option_menu_new ();
  menu = gtk_menu_new ();

  active = 0;
  for (i = 0; i < sizeof(balanceTable)/sizeof(balanceTable[0]); i++) {
    mi = gtk_menu_item_new_with_label (balanceTable[i].intlName);
    if (i == init_balance)
      active = i;
    gtk_widget_show (mi);
    gtk_menu_shell_append (GTK_MENU_SHELL(menu), mi);
    gtk_signal_connect(GTK_OBJECT(mi), "activate", sig_fun, (gpointer)&balanceTable[i]);
  }

  gtk_widget_show (menu);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (optmenu), menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (optmenu), active);
  gtk_widget_show (optmenu);
  return optmenu;
}

void
mrw_color_mode_select (GtkNotebook *notebook,
		       GtkNotebookPage *page,
		       gint page_num,
		       gpointer user_data)
{
  guint *balanceMode = user_data;

  /* fprintf (stderr, "notebook page %d selected\n", page_num); */
  if (*balanceMode != page_num) {
      *balanceMode = page_num;
      UpdateUI (&ui_runvals);
  }
}

GtkWidget *
CreateColorControl (GtkWidget *vbox2)
{
  GtkWidget *frame;
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *spinbutton;
  GtkObject *adj;
  GtkWidget *newbal_menu;
  GtkWidget *notebook;
  GtkWidget *slider;
  GtkWidget *new_preset_button;
  int row;

  /* Color Control */
#if 0
  frame = gtk_frame_new (_("Color Control"));
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  /* gtk_box_pack_start (GTK_BOX (vbox2), frame, FALSE, FALSE, 0); */
#endif

  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 4);
#if 0
  gtk_container_add (GTK_CONTAINER (frame), vbox);
#endif

  notebook = gtk_notebook_new();

  /* Resolution/Width/Height/Pages labels */
  table = gtk_table_new (4, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  row = 0;

  ui_widgets.balanceMenu_entry = newbal_menu = 
		mrw_ui_balance_menu(GTK_SIGNAL_FUNC(mrw_set_balance1), plvals.newbal1);

  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Balance 1:"), 1.0, 0.5,
			     newbal_menu, 1, TRUE, GTK_FILL);
  ui_widgets.balance2Menu_entry = mrw_ui_balance_menu(GTK_SIGNAL_FUNC(mrw_set_balance2), plvals.newbal2);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Balance 2:"), 1.0, 0.5,
			     ui_widgets.balance2Menu_entry, 1, TRUE, GTK_FILL);
  adj = gtk_adjustment_new( /*value*/plvals.mix,
  			    /*lower*/0.0,
		            /*upper*/100.0,
		            /*step_increment*/1.0,
		            /*page_increment*/10.0,
		            /*page_size*/0.0 );
  slider = gtk_hscale_new( GTK_ADJUSTMENT (adj) );
  gtk_widget_show (slider);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_mixing_adjustment_update),
		      &plvals.mix);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("% Balance 2:"), 1.0, 0.5,
			     slider, 1, TRUE,
                             GTK_EXPAND | GTK_SHRINK | GTK_FILL);
  gtk_widget_show (table);

  gtk_notebook_append_page( GTK_NOTEBOOK (notebook),
                            table,
			    gtk_label_new (_("Presets")) );

  table = gtk_table_new (4, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  row = 0;

  ui_widgets.redBalance_entry = gimp_spin_button_new (&adj, plvals.redBalance,
				     0, 4096, 1, 10, 0, 1, 0);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Red Gain:"), 1.0, 0.5,
			     GTK_WIDGET (ui_widgets.redBalance_entry), 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_red_gain_adjustment_update),
		      &plvals.redBalance);

  ui_widgets.greenBalance_entry = gimp_spin_button_new (&adj, plvals.greenBalance,
				     0, 4096, 1, 10, 0, 1, 0);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Green Gain:"), 1.0, 0.5,
			     GTK_WIDGET (ui_widgets.greenBalance_entry), 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_green_gain_adjustment_update),
		      &plvals.greenBalance);

  ui_widgets.blueBalance_entry = gimp_spin_button_new (&adj, plvals.blueBalance,
				     0, 4096, 1, 10, 0, 1, 0);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Blue Gain:"), 1.0, 0.5,
			     GTK_WIDGET (ui_widgets.blueBalance_entry), 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_blue_gain_adjustment_update),
		      &plvals.blueBalance);
  gtk_widget_show (table);
  gtk_notebook_append_page( GTK_NOTEBOOK (notebook),
                            table,
			    gtk_label_new (_("Manual")) );

  gtk_notebook_set_current_page (GTK_NOTEBOOK(notebook), plvals.balanceMode);
  gtk_signal_connect (GTK_OBJECT (notebook), "switch_page",
		      GTK_SIGNAL_FUNC (mrw_color_mode_select),
		      &plvals.balanceMode);
  gtk_widget_show (notebook);
  gtk_box_pack_start (GTK_BOX (vbox), notebook, TRUE, TRUE, 0);

  new_preset_button = gtk_button_new_with_label(_("Set user preset"));
  gtk_signal_connect (GTK_OBJECT (new_preset_button), "clicked",
                      GTK_SIGNAL_FUNC (ui_set_preset), (gpointer)&plvals);
  gtk_widget_show (new_preset_button);
  gtk_box_pack_start (GTK_BOX (vbox), new_preset_button, FALSE, FALSE, 0);

  /* Start Average entries. */
  table = gtk_table_new (4, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  row = 0;

  ui_widgets.redAvg_entry = gtk_entry_new ();
  gtk_widget_set_usize (ui_widgets.redAvg_entry, 80, 0);
  gtk_entry_set_text (GTK_ENTRY (ui_widgets.redAvg_entry), _("UNKNOWN"));
  gtk_entry_set_editable (GTK_ENTRY (ui_widgets.redAvg_entry), FALSE);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Average red level:"), 1.0, 0.5,
			     ui_widgets.redAvg_entry, 1, TRUE, 0);

  ui_widgets.greenAvg_entry = gtk_entry_new ();
  gtk_widget_set_usize (ui_widgets.greenAvg_entry, 80, 0);
  gtk_entry_set_text (GTK_ENTRY (ui_widgets.greenAvg_entry), _("UNKNOWN"));
  gtk_entry_set_editable (GTK_ENTRY (ui_widgets.greenAvg_entry), FALSE);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Average green level:"), 1.0, 0.5,
			     ui_widgets.greenAvg_entry, 1, TRUE, 0);

  ui_widgets.blueAvg_entry = gtk_entry_new ();
  gtk_widget_set_usize (ui_widgets.blueAvg_entry, 80, 0);
  gtk_entry_set_text (GTK_ENTRY (ui_widgets.blueAvg_entry), _("UNKNOWN"));
  gtk_entry_set_editable (GTK_ENTRY (ui_widgets.blueAvg_entry), FALSE);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Average blue level:"), 1.0, 0.5,
			     ui_widgets.blueAvg_entry, 1, TRUE, 0);

  spinbutton = gimp_spin_button_new (&adj, plvals.saturation,
				     -5, 10, 0.1, 1, 0, 1, 1);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Color Saturation:"), 1.0, 0.5,
			     spinbutton, 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_double_adjustment_update),
		      &plvals.saturation);

  gtk_widget_show (table);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_widget_show (vbox);
#if 0
  gtk_widget_show (frame);

  return frame;
#else
  return vbox;
#endif
}

GtkWidget *
CreateExposureControl (GtkWidget *vbox2)
{
  GtkWidget *frame, *vbox;
  GtkWidget *histogram_frame, *histogram_vbox;
  GtkWidget *tone_menu;
  GtkWidget *spinbutton;
  GtkWidget *table;
  GtkObject *adj;
  int row;

  /* Exposure Control */
  frame = gtk_frame_new (_("Exposure Control"));
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  /* gtk_box_pack_start (GTK_BOX (vbox2), frame, FALSE, FALSE, 0); */

  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 4);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  /* Resolution/Width/Height/Pages labels */
  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 2);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);
  row = 0;

  spinbutton = gimp_spin_button_new (&adj, plvals.darkness,
				     0, 10, 0.1, 1, 0, 1, 1);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Darkness:"), 1.0, 0.5,
			     spinbutton, 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_double_adjustment_update),
		      &plvals.darkness);

  spinbutton = gimp_spin_button_new (&adj, plvals.contrast,
				     -10, 100, 0.1, 1, 0, 1, 1);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Contrast:"), 1.0, 0.5,
			     spinbutton, 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_double_adjustment_update),
		      &plvals.contrast);

  spinbutton = gimp_spin_button_new (&adj, plvals.shadows,
				     -10, 10, 0.1, 1, 0, 1, 1);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Shadows:"), 1.0, 0.5,
			     spinbutton, 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_double_adjustment_update),
		      &plvals.shadows);

  #if 0
  tone_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (tone_hbox);
  gtk_box_pack_start (GTK_BOX (vbox), tone_hbox, FALSE, FALSE, 0);

  tone_label = gtk_label_new (_("Tone Curve"));
  gtk_widget_show (tone_label);
  gtk_box_pack_start (GTK_BOX (tone_hbox), tone_label, FALSE, FALSE, 0);
  gtk_label_set_justify (GTK_LABEL (tone_label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_padding (GTK_MISC (tone_label), 20, 0);
  #endif

  tone_menu = GenToneCurveOptionMenu (plvals.toneCurveNum);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Tone Curve:"), 1.0, 0.5,
			     tone_menu, 1, TRUE, 0);

  ui_widgets.luminance_entry = gtk_entry_new ();
  gtk_widget_set_usize (ui_widgets.luminance_entry, 120, 0);
  gtk_entry_set_text (GTK_ENTRY (ui_widgets.luminance_entry), _("UNKNOWN"));
  gtk_entry_set_editable (GTK_ENTRY (ui_widgets.luminance_entry), FALSE);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Luminance Range:"), 1.0, 0.5,
			     ui_widgets.luminance_entry, 1, TRUE, 0);

  /* Histogram */
  histogram_frame = gtk_frame_new (_("Histogram"));
  gtk_widget_show (histogram_frame);
  gtk_box_pack_start (GTK_BOX (vbox), histogram_frame, FALSE, FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (histogram_frame), 6);

  histogram_vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (histogram_vbox);
  gtk_container_add (GTK_CONTAINER (histogram_frame), histogram_vbox);
  gtk_container_set_border_width (GTK_CONTAINER (histogram_vbox), 6);

  init_histogram();
  histogram_new (histogram_vbox, plvals.histWidth, plvals.histHeight);

  gtk_widget_show (vbox);
  gtk_widget_show (frame);
  return frame;
}

GtkWidget *
CreateSharpeningControl (GtkWidget *vbox2)
{
  GtkWidget *frame, *vbox;
  GtkWidget *spinbutton;
  GtkWidget *table;
  GtkObject *adj;
  int row;

  /* Sharpening/Bluring */
  frame = gtk_frame_new (_("Sharpening"));
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  /* gtk_box_pack_start (GTK_BOX (vbox2), frame, FALSE, FALSE, 0); */

  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 4);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  /* Resolution/Width/Height/Pages labels */
  table = gtk_table_new (3, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 3);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);
  row = 0;

  spinbutton = gimp_spin_button_new (&adj, plvals.radius,
				     1, 5, 1, 1, 0, 1, 1);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Radius:"), 1.0, 0.5,
			     spinbutton, 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_radius_adjustment_update),
		      &plvals.radius);

  spinbutton = gimp_spin_button_new (&adj, plvals.blur,
				     1, 5, 0.1, 1, 0, 1, 1);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Blur:"), 1.0, 0.5,
			     spinbutton, 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_double_adjustment_update),
		      &plvals.blur);

  spinbutton = gimp_spin_button_new (&adj, plvals.iSharpen,
				     -3, 3, 0.1, 1, 0, 1, 1);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Lightness:"), 1.0, 0.5,
			     spinbutton, 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_double_adjustment_update),
		      &plvals.iSharpen);

  spinbutton = gimp_spin_button_new (&adj, plvals.cSharpen,
				     -3, 3, 0.1, 1, 0, 1, 1);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Color:"), 1.0, 0.5,
			     spinbutton, 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_double_adjustment_update),
		      &plvals.cSharpen);

#if 0
  toggle = gtk_check_button_new_with_label (_("Use Lab color space"));
  gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
                      GTK_SIGNAL_FUNC (mrw_color_toggle_callback),
                      &plvals.useLab);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggle), plvals.useLab);
  gtk_widget_show (toggle);
#endif
  gtk_widget_show (vbox);
  gtk_widget_show (frame);

  return frame;
}

GtkWidget *
CreateFilteringControl (GtkWidget *vbox2)
{
  GtkWidget *frame, *vbox;
  GtkWidget *toggle;

  /* Filtering */
  frame = gtk_frame_new (_("Filtering"));
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  /* gtk_box_pack_start (GTK_BOX (vbox2), frame, TRUE, TRUE, 0); */

  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 4);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  toggle = gtk_check_button_new_with_label (_("Median"));
  gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
                      GTK_SIGNAL_FUNC (mrw_color_toggle_callback),
                      &plvals.applyMedianFilter);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggle), plvals.applyMedianFilter);
  gtk_widget_show (toggle);

  toggle = gtk_check_button_new_with_label (_("Color Only"));
  gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
                      GTK_SIGNAL_FUNC (mrw_color_toggle_callback),
                      &plvals.filterColorOnly);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggle), plvals.filterColorOnly);
  gtk_widget_show (toggle);

  toggle = gtk_check_button_new_with_label (_("Vector Median"));
  gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
                      GTK_SIGNAL_FUNC (mrw_color_toggle_callback),
                      &plvals.applyVMF);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggle), plvals.applyVMF);
  gtk_widget_show (toggle);

  gtk_widget_show (vbox);
  gtk_widget_show (frame);

  return frame;
}

static int
interpolation_menu_callback (GtkWidget *widget, gpointer data)
{
    int oldMethod = plvals.iMethodNum;
    plvals.iMethodNum = get_interpolation_method_index (data);
    if (plvals.previewShrink == 1 && plvals.iMethodNum != oldMethod)
        UpdateUI (&ui_runvals);
    return TRUE;
}

GtkWidget *
GenInterpolationMenu (int init_method)
{
  GtkWidget *optmenu, *imethod_menu;
  GtkWidget *mi;
  GtkWidget *label;
  int	i, active;

  optmenu = gtk_option_menu_new ();
  imethod_menu = gtk_menu_new ();

  active = 0;
  for (i = 0; i < num_interpolation_methods(); i++) {
      mi = gtk_menu_item_new_with_label (get_interpolation_method_intlname(i));
      if (i == init_method)
          active = i;
      gtk_widget_show (mi);
      gtk_menu_shell_append (GTK_MENU_SHELL(imethod_menu), mi);
      gtk_signal_connect(GTK_OBJECT(mi), "activate",
                         GTK_SIGNAL_FUNC (interpolation_menu_callback),
	                 get_interpolation_method_handle(i));
  }

  gtk_widget_show (imethod_menu);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (optmenu), imethod_menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (optmenu), active);
  gtk_widget_show (optmenu);
  return optmenu;
}

static void
mrw_debug_toggle_callback (GtkWidget *widget, gpointer data)
{
  gimp_toggle_button_update (widget, data);
  UpdateUI (&ui_runvals);
}

GtkWidget *
CreateOptionsControl (void)
{
  GtkWidget *table;
  GtkWidget *toggle;
  GtkWidget *menu, *optmenu;
  GtkWidget *item;
  GtkWidget *clist;
  gchar	    *titles[2];
  gint	    row;

  /* Resolution/Width/Height/Pages labels */
  table = gtk_table_new (5, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  row = 0;

  optmenu = GenScaleMenu(GTK_SIGNAL_FUNC(set_final_shrink), plvals.finalShrink);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Image Scale:"), 1.0, 0.5,
			     optmenu, 1, TRUE, 0);

  optmenu = GenScaleMenu(GTK_SIGNAL_FUNC(set_preview_shrink), plvals.previewShrink);
  preview_shrink_scale_widget = optmenu;
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Preview Scale:"), 1.0, 0.5,
			     optmenu, 1, TRUE, 0);

  optmenu = GenInterpolationMenu(plvals.iMethodNum),
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Interpolation Method:"), 1.0, 0.5,
			     optmenu, 1, TRUE, 0);

  optmenu = GenPreviewInterpolationMenu(),
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Preview Interpolation:"), 1.0, 0.5,
			     optmenu, 1, TRUE, 0);

#if 0
  toggle = gtk_toggle_button_new ();
  gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
                      GTK_SIGNAL_FUNC (mrw_debug_toggle_callback),
                      &debugFlag);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggle), debugFlag);
  gtk_widget_show (toggle);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     ("Debug flag:"), 1.0, 0.5,
			     toggle, 1, TRUE, 0);
#endif

  gtk_widget_show (table);

  return table;
}

GtkWidget *side_menu;

static int
mrw_side_menu_item_update (GtkWidget *widget, gpointer data)
{
    MRI_Region selnRegion;
    plvals.fixShortSide = gtk_option_menu_get_history (GTK_OPTION_MENU(side_menu));
    set_current_crop_size (cropRegion.width, cropRegion.height);
    if (get_preview_selected_rectangle (&selnRegion))
        set_current_selection_size (selnRegion.width, selnRegion.height);
    else
        set_current_selection_size (0.0, 0.0);
    return TRUE;
}

GtkWidget *
GenSideOptionMenu (int initial)
{
  GtkWidget *menu;
  GtkWidget *mi;
  GtkWidget *label;
  unsigned i;

  side_menu = gtk_option_menu_new ();
  menu = gtk_menu_new ();

  for (i = 0; i < 2; i++) {
    mi = gtk_menu_item_new_with_label (i == 0 ? _("Long side") : _("Short side"));
    gtk_signal_connect(GTK_OBJECT(mi), "activate", GTK_SIGNAL_FUNC (mrw_side_menu_item_update), mi);
    gtk_widget_show (mi);
    gtk_menu_shell_append (GTK_MENU_SHELL(menu), (gpointer)mi);
  }

  gtk_widget_show (menu);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (side_menu), menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (side_menu), initial);
  gtk_widget_show (side_menu);

  return side_menu;
}

GtkWidget *
CreateImageInfoControl (GtkWidget *vbox2, MRI *mri)
{
  GtkWidget *vbox, *table;
  int row;
  GtkObject *adj;

  /* Image Data */
  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 4);

  /* Resolution/Width/Height/Pages labels */
  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 2);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);
  row = 0;

  AddInfoEntry (table, &row, _("Timestamp:"), "%s", MRI_GetTimestamp (mri));
  AddInfoEntry (table, &row, _("Aperture:"), "%1.1f", MRI_GetAperture (mri));
  { double s = MRI_GetShutter (mri);
  if (s < (1.0-1.0e-5))
	  AddInfoEntry (table, &row, _("Shutter:"), "1/%.1f", 1.0/s);
  else
	  AddInfoEntry (table, &row, _("Shutter:"), "%.1f s", s);
  }
  AddInfoEntry (table, &row, _("ISO:"), "%g", MRI_GetISO (mri));
  AddInfoEntry (table, &row, _("Zoom:"), "%.0f mm", MRI_GetFocalLen (mri));
  { double focusLen = MRI_GetFocusLen (mri)/1000.0;
    char *focusMode = MRI_GetFocusMode (mri) ? _("MF") : _("AF");
    if (focusLen < 0.0)
	  AddInfoEntry (table, &row, _("Focus:"), _("%s Unknown"), focusMode);
    else if (focusLen == 0.0)
	  AddInfoEntry (table, &row, _("Focus:"), _("%s Infinity"), focusMode);
    else
	  AddInfoEntry (table, &row, _("Focus:"), "%s %g m", focusMode, focusLen);
  }

  /* Resolution/Width/Height/Pages labels */
  table = gtk_table_new (4, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_widget_show (table);

  ui_widgets.fixedLength_entry = gimp_spin_button_new (&adj, plvals.fixedSideLength,
				     0, 4096, 0.1, 1, 0, 1, 1);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Fixed length:"), 1.0, 0.5,
			     GTK_WIDGET (ui_widgets.fixedLength_entry), 1, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		      GTK_SIGNAL_FUNC (mrw_fixed_length_adjustment_update),
		      &plvals.fixedSideLength);
  side_menu = GenSideOptionMenu (plvals.fixShortSide);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Length applies to:"), 1.0, 0.5,
			     GTK_WIDGET (side_menu), 1, TRUE, 0);


  ui_widgets.crop_size_entry = gtk_entry_new ();
  gtk_widget_set_usize (ui_widgets.crop_size_entry, 120, 0);
  gtk_entry_set_editable (GTK_ENTRY (ui_widgets.crop_size_entry), FALSE);
  set_current_crop_size (cropRegion.width, cropRegion.height);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Aspect ratio of view:"), 1.0, 0.5,
			     ui_widgets.crop_size_entry, 1, TRUE, 0);

  ui_widgets.seln_size_entry = gtk_entry_new ();
  gtk_widget_set_usize (ui_widgets.seln_size_entry, 120, 0);
  gtk_entry_set_editable (GTK_ENTRY (ui_widgets.seln_size_entry), FALSE);
  set_current_selection_size (0.0, 0.0);
  my_table_attach_aligned (GTK_TABLE (table), 0, row++,
			     _("Aspect ratio of selection:"), 1.0, 0.5,
			     ui_widgets.seln_size_entry, 1, TRUE, 0);

  gtk_widget_show (vbox);

  return vbox;
}

GtkWidget *
CreateDialog (UI_RunVals *runVals)
{
  GtkWidget *dialog;
  GtkWidget *main_vbox;
  GtkWidget *hbox;
  GtkWidget *frame;
  GtkWidget *vbox, *vbox2;
  GtkWidget *table;
  GtkWidget *preview_frame;
  GtkWidget *menu_bar;
  GtkWidget *spinbutton;
  GtkWidget *notebook;
  int row;
  MRI *mri = runVals->mri;

  dialog = gimp_dialog_new (_("Load RAW"), "mrw",
			    NULL, 0,
			    gimp_standard_help_func, "filters/mrw.html",
			    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			    GTK_STOCK_OK,     GTK_RESPONSE_OK,

			    NULL);

  main_vbox = gtk_vbox_new (FALSE, 6);
  gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 6);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), main_vbox,
                      TRUE, TRUE, 0);

#if 0
  menu_bar = MainMenuBar ();
  gtk_box_pack_start (GTK_BOX (main_vbox), menu_bar, FALSE, FALSE, 0);
  gtk_widget_show (menu_bar);
#endif

  hbox = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, TRUE, TRUE, 0);

  notebook = gtk_notebook_new();
  gtk_notebook_append_page( GTK_NOTEBOOK (notebook),
                            CreateColorControl ( NULL ),
			    gtk_label_new (_("Color")) );
  gtk_notebook_append_page( GTK_NOTEBOOK (notebook),
                            CreateExposureControl ( NULL ),
			    gtk_label_new (_("Exposure")) );
  gtk_notebook_append_page( GTK_NOTEBOOK (notebook),
                            CreateSharpeningControl ( NULL ),
			    gtk_label_new (_("Sharpening")) );
  gtk_notebook_append_page( GTK_NOTEBOOK (notebook),
                            CreateFilteringControl ( NULL ),
			    gtk_label_new (_("Filters")) );
  gtk_notebook_append_page( GTK_NOTEBOOK (notebook),
                            CreateOptionsControl (),
			    gtk_label_new (_("Options")) );
  gtk_notebook_append_page( GTK_NOTEBOOK (notebook),
                            CreateImageInfoControl ( NULL , mri),
			    gtk_label_new (_("Info")) );
  gtk_widget_show (notebook);
  gtk_box_pack_start (GTK_BOX (hbox), notebook, TRUE, TRUE, 0);

  /* Preview */
  preview_frame = CreatePreviewPane(runVals);
  gtk_box_pack_start (GTK_BOX (hbox), preview_frame, TRUE, TRUE, 0);

  gtk_widget_show (hbox);
  gtk_widget_show (main_vbox);

  {
  MRI_balance balanceSpec;
  mrw_calculate_balance (&balanceSpec, mri, runVals->pVals, 8);
  update_setGainButtons (&balanceSpec);
  update_autoBalance(&balanceSpec);
  }

  gtk_widget_show (dialog);

  return dialog;
}

static gint
do_user_dialog (UI_RunVals *runVals)
{
  guint doPreview;
  GtkWidget *dialog;

  if (!g_thread_supported ()) {
    g_thread_init (NULL);
  }
  dialog = CreateDialog (runVals);
  UpdateUI2 (&ui_runvals, TRUE);

  return gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK;
}

static void
update_setGainButtons (MRI_balance *balanceSpec)
{
  guint  doPreviewSave;

  doPreviewSave = ui_runvals.doPreview;
  ui_runvals.doPreview = FALSE;
  plvals.redBalance = balanceSpec->rgain;
  plvals.greenBalance = balanceSpec->ggain;
  plvals.blueBalance = balanceSpec->bgain;
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(ui_widgets.redBalance_entry), plvals.redBalance);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(ui_widgets.greenBalance_entry), plvals.greenBalance);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(ui_widgets.blueBalance_entry), plvals.blueBalance);
  ui_runvals.doPreview = doPreviewSave;
}

static void
update_autoBalance (MRI_balance *balanceSpec)
{
  double rAvg, gAvg, bAvg, maxAvg;
  char buffer[20];
  guint  doPreviewSave;

  doPreviewSave = ui_runvals.doPreview;
  ui_runvals.doPreview = FALSE;
  maxAvg = rAvg = ui_runvals.mri->Ravg * balanceSpec->rgain;
  gAvg = ui_runvals.mri->Gavg * balanceSpec->ggain;
  if (gAvg > maxAvg) maxAvg = gAvg;
  bAvg = ui_runvals.mri->Bavg * balanceSpec->bgain;
  if (bAvg > maxAvg) maxAvg = bAvg;
  if (maxAvg < 1.0)
  	strcpy (buffer, "1.0");
  else
        snprintf (buffer, sizeof(buffer), "%1.5f", rAvg/maxAvg);
  gtk_entry_set_text (GTK_ENTRY (ui_widgets.redAvg_entry), buffer);
  if (maxAvg < 1.0)
  	strcpy (buffer, "1.0");
  else
        snprintf (buffer, sizeof(buffer), "%1.5f", gAvg/maxAvg);
  gtk_entry_set_text (GTK_ENTRY (ui_widgets.greenAvg_entry), buffer);
  if (maxAvg < 1.0)
  	strcpy (buffer, "1.0");
  else
        snprintf (buffer, sizeof(buffer), "%1.5f", bAvg/maxAvg);
  gtk_entry_set_text (GTK_ENTRY (ui_widgets.blueAvg_entry), buffer);
  UpdateUI2 (&ui_runvals, doPreviewSave);
}

/* Function to open a dialog box displaying the message provided.
*/
void quick_message(gchar *message)
{
    GtkWidget *dialog, *label, *okay_button;
  
   /* Create the widgets */
  
   dialog = gtk_dialog_new();
   gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
   label = gtk_label_new (message);
   okay_button = gtk_button_new_with_label(_("Okay"));
  
   /* Ensure that the dialog box is destroyed when the user clicks ok. */
  
   gtk_signal_connect_object (GTK_OBJECT (okay_button), "clicked",
                              GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer)dialog);
   gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
                      okay_button);

   /* Add the label, and show everything we've added to the dialog. */
   gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
                      label);
   gtk_widget_show_all (dialog);
}
