/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/connector_factory.h>
#include <messaging/dictionary.h>
#include <messaging/enumerator.h>
#include <messaging/parameter.h>
#include <messaging/runner.h>
#include <messaging/variant.h>

#include "mock_connector.h"

#include <core/posix/fork.h>
#include <core/posix/wait.h>

#include "blocking_connector.h"
#include "did_finish.h"
#include "typed_connector_factory.h"

#include <core/dbus/fixture.h>

#include <gmock/gmock.h>
#include <gtest/gtest.h>

#include <condition_variable>
#include <thread>

using namespace ::testing;

namespace
{

struct NullConnectorFactory : public messaging::ConnectorFactory
{
    Generator create_instance_for_name(const std::string&) const override
    {
        return [](const messaging::Dictionary<std::string, std::string>&)
        {
            return std::shared_ptr<messaging::Connector>{};
        };
    }
};

struct MockConnectorFactory : public messaging::ConnectorFactory
{
    MOCK_CONST_METHOD1(create_instance_for_name, Generator(const std::string&));
};

struct MockQtRuntime : public messaging::qt::Runtime
{
    MockQtRuntime()
        : messaging::qt::Runtime(QDBusConnection::SessionBus)
    {
    }

    MockQtRuntime(const QString& dbus_address)
        : messaging::qt::Runtime(dbus_address)
    {
    }

    MOCK_METHOD0(run, int());
    MOCK_METHOD1(stop, void(int));
};

/// @brief the_dummy_runner returns a functor executing a runner with a dummy configuration,
/// pointing it to the session bus given in session_bus_address.
std::function<core::posix::exit::Status()> the_dummy_runner(const std::string& session_bus_address)
{
    typedef testing::TypedConnectorFactory<testing::BlockingConnector> BlockingConnectorFactory;

    return [session_bus_address]()
    {
        const char* argv[]{"ThisIsAnExecutableNameForTestingPurposes",
                           "--bus",
                           "with_address",
                           "--bus-address",
                           session_bus_address.c_str(),
                           "--connection-manager-name",
                           "telepathy_test",
                           "--protocol-name",
                           "protocol_test",
                           "--connector",
                           "JustADummyConnector"};

        return static_cast<core::posix::exit::Status>(messaging::Runner::main(
            messaging::Runner::Configuration::from_command_line_args(11, argv, BlockingConnectorFactory())));
    };
}
}

TEST(RunnerConfiguration, parses_connection_manager_name_correctly)
{
    const char* argv[] = {"ThisIsAnExecutableNameForTestingPurposes",
                          "--connection-manager-name",
                          "telepathy_test",
                          "--protocol-name",
                          "protocol_test",
                          "--connector",
                          "JustADummyConnector"};

    NullConnectorFactory factory;

    auto config = messaging::Runner::Configuration::from_command_line_args(7, argv, factory);
    EXPECT_EQ("telepathy_test", config.connection_manager_name);
}

TEST(RunnerConfiguration, parses_protocol_name_correctly)
{
    const char* argv[] = {"ThisIsAnExecutableNameForTestingPurposes",
                          "--connection-manager-name",
                          "telepathy_test",
                          "--protocol-name",
                          "protocol_test",
                          "--connector",
                          "JustADummyConnector"};

    NullConnectorFactory factory;

    auto config = messaging::Runner::Configuration::from_command_line_args(7, argv, factory);
    EXPECT_EQ("protocol_test", config.protocol_name);
}

TEST(RunnerConfiguration, parses_connector_name_and_queries_the_provided_connector_factory)
{
    using namespace ::testing;

    static const char* connector_name{"JustADummyConnector"};

    const char* argv[]{"ThisIsAnExecutableNameForTestingPurposes",
                       "--connection-manager-name",
                       "telepathy_test",
                       "--protocol-name",
                       "protocol_test",
                       "--connector",
                       "JustADummyConnector"};

    MockConnectorFactory factory;
    auto gen = [](const messaging::Dictionary<std::string, std::string>&)
    {
        return std::shared_ptr<messaging::Connector>();
    };
    EXPECT_CALL(factory, create_instance_for_name(connector_name)).Times(1).WillRepeatedly(Return(gen));

    auto config = messaging::Runner::Configuration::from_command_line_args(7, argv, factory);
    EXPECT_EQ(nullptr, config.connector);
}

TEST(RunnerConfiguration, throws_help_required_if_help_flag_is_set)
{
    const char* argv[]{"ThisIsAnExecutableNameForTestingPurposes", "--help"};
    EXPECT_THROW(messaging::Runner::Configuration::from_command_line_args(2, argv, NullConnectorFactory()),
                 messaging::Runner::HelpRequired);
}

TEST(RunnerConfiguration, throws_help_required_if_connector_missing)
{
    const char* argv[]{
        "ThisIsAnExecutableNameForTestingPurposes", "--connection-manager-name", "42", "--protocol-name", "42"};
    EXPECT_THROW(messaging::Runner::Configuration::from_command_line_args(5, argv, NullConnectorFactory()),
                 messaging::Runner::HelpRequired);
}

TEST(RunnerConfiguration, throws_help_required_if_connection_manager_name_missing)
{
    const char* argv[]{"ThisIsAnExecutableNameForTestingPurposes", "--connector", "42", "--protocol-name", "42"};
    EXPECT_THROW(messaging::Runner::Configuration::from_command_line_args(5, argv, NullConnectorFactory()),
                 messaging::Runner::HelpRequired);
}

TEST(RunnerConfiguration, throws_help_required_if_protocol_name_missing)
{
    const char* argv[]{
        "ThisIsAnExecutableNameForTestingPurposes", "--connector", "42", "--connection-manager-name", "42"};
    EXPECT_THROW(messaging::Runner::Configuration::from_command_line_args(5, argv, NullConnectorFactory()),
                 messaging::Runner::HelpRequired);
}

TEST(Runner, DISABLED_calls_run_stop_on_connector)
{
    core::dbus::Fixture private_bus_fixture{core::dbus::Fixture::default_session_bus_config_file(),
                                            core::dbus::Fixture::default_system_bus_config_file()};

    auto connector = std::make_shared<NiceMock<MockConnector>>();

    EXPECT_CALL(*connector, run()).Times(1);
    EXPECT_CALL(*connector, stop()).Times(1);

    messaging::Runner::Configuration config{
        0,
        nullptr,
        "DoesNotMatter",
        "DoesNotMatter",
        connector,
        std::make_shared<MockQtRuntime>(QString::fromStdString(private_bus_fixture.session_bus_address()))};

    EXPECT_EQ(EXIT_SUCCESS, messaging::Runner::main(config));
}

namespace
{
struct RunnerSignalHandling : public ::testing::TestWithParam<core::posix::Signal>
{
};
}

// We cleanly shutdown on SIGTERM and SIGQUIT. Here, cleanly means:
//   * The messaging::Connector is correctly stopped.
//   * The overall runtime is cleaned up.
TEST_P(RunnerSignalHandling, DISABLED_shuts_down_cleanly_on_signal)
{
    core::dbus::Fixture private_bus_fixture{core::dbus::Fixture::default_session_bus_config_file(),
                                            core::dbus::Fixture::default_system_bus_config_file()};

    auto child = core::posix::fork(the_dummy_runner(private_bus_fixture.session_bus_address()),
                                   core::posix::StandardStream::empty);

    // Kinda ugly, but we have to make sure that messaging::Runner::main has got some time
    // to setup all required signal handlers.
    std::this_thread::sleep_for(std::chrono::milliseconds(100));

    child.send_signal_or_throw(GetParam());
    EXPECT_TRUE(testing::did_finish_successfully(child.wait_for(core::posix::wait::Flags::untraced)));
}

// INSTANTIATE_TEST_CASE_P(sig_term_sig_quit,
//                        RunnerSignalHandling,
//                        ::testing::Values(core::posix::Signal::sig_term, core::posix::Signal::sig_quit));
