/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/plugin_connector.h>
#include <messaging/dictionary.h>
#include <messaging/enumerator.h>
#include <messaging/named_symbol_loader.h>
#include <messaging/parameter.h>
#include <messaging/variant.h>

#include "mock_connector.h"

#include <gmock/gmock.h>
#include <gtest/gtest.h>

namespace
{
struct NullConnector : public messaging::Connector
{
    messaging::Enumerator<messaging::Parameter>& parameters() const override
    {
        throw std::logic_error{"Not implemented."};
    }

    std::shared_ptr<messaging::Connection> request_connection(
        const std::shared_ptr<messaging::Connection::Observer>&,
        const std::shared_ptr<messaging::Messenger::Observer>&,
        const std::shared_ptr<messaging::PresenceManager::Observer>&,
        const messaging::Dictionary<std::string, messaging::Variant>&) override
    {
        throw std::logic_error{"Not implemented."};
    }

    void run() override
    {
    }

    void stop() override
    {
    }
};

class Creator
{
public:
    static messaging::Connector* create()
    {
        return Creator::instance().do_create();
    }

    static Creator& instance()
    {
        static ::testing::NiceMock<Creator> creator;
        return creator;
    }

    virtual ~Creator() = default;

    MOCK_METHOD0(do_create, messaging::Connector*());

private:
    template <typename>
    friend class testing::NiceMock;
    Creator() = default;
};

class Destroyer
{
public:
    static void destroy(messaging::Connector* c)
    {
        Destroyer::instance().do_destroy(c);
    }

    static Destroyer& instance()
    {
        static ::testing::NiceMock<Destroyer> destroyer;
        return destroyer;
    }

    virtual ~Destroyer() = default;

    MOCK_METHOD1(do_destroy, void(messaging::Connector*));

private:
    template <typename>
    friend class testing::NiceMock;
    Destroyer() = default;
};

messaging::Connector* create()
{
    return nullptr;
}

void destroy(messaging::Connector*)
{
}

struct MockNamedSymbolLoader : public messaging::NamedSymbolLoader
{
    MOCK_CONST_METHOD1(load_symbol_for_name, void*(const std::string&));
};
}

TEST(PluginConnector, tries_to_resolve_create_destroy_symbols_on_construction)
{
    using namespace ::testing;

    auto symbol_loader = std::make_shared<MockNamedSymbolLoader>();

    EXPECT_CALL(*symbol_loader, load_symbol_for_name(messaging::PluginConnector::Create::name)).Times(1).WillRepeatedly(
        Return(reinterpret_cast<void*>(&create)));
    EXPECT_CALL(*symbol_loader, load_symbol_for_name(messaging::PluginConnector::Destroy::name))
        .Times(1)
        .WillRepeatedly(Return(reinterpret_cast<void*>(&destroy)));

    messaging::PluginConnector pc(symbol_loader);
}

TEST(PluginConnector, propagates_exceptions_thrown_by_named_symbol_loader)
{
    using namespace ::testing;

    auto symbol_loader = std::make_shared<NiceMock<MockNamedSymbolLoader>>();

    ON_CALL(*symbol_loader, load_symbol_for_name(messaging::PluginConnector::Create::name))
        .WillByDefault(Throw(std::runtime_error{""}));
    ON_CALL(*symbol_loader, load_symbol_for_name(messaging::PluginConnector::Destroy::name))
        .WillByDefault(Return(reinterpret_cast<void*>(&destroy)));

    EXPECT_THROW(messaging::PluginConnector pc(symbol_loader), std::runtime_error);

    ON_CALL(*symbol_loader, load_symbol_for_name(messaging::PluginConnector::Create::name))
        .WillByDefault(Return(reinterpret_cast<void*>(&create)));
    ON_CALL(*symbol_loader, load_symbol_for_name(messaging::PluginConnector::Destroy::name))
        .WillByDefault(Throw(std::runtime_error{""}));

    EXPECT_THROW(messaging::PluginConnector pc(symbol_loader), std::runtime_error);
}

TEST(PluginConnector, creates_and_destroys_connector_instance_via_loaded_symbols)
{
    using namespace ::testing;

    auto symbol_loader = std::make_shared<NiceMock<MockNamedSymbolLoader>>();

    ON_CALL(*symbol_loader, load_symbol_for_name(messaging::PluginConnector::Create::name))
        .WillByDefault(Return(reinterpret_cast<void*>(&Creator::create)));
    ON_CALL(*symbol_loader, load_symbol_for_name(messaging::PluginConnector::Destroy::name))
        .WillByDefault(Return(reinterpret_cast<void*>(&Destroyer::destroy)));

    EXPECT_CALL(Creator::instance(), do_create()).Times(1).WillRepeatedly(Return(new NullConnector()));
    EXPECT_CALL(Destroyer::instance(), do_destroy(_)).Times(1);

    {
        messaging::PluginConnector pc(symbol_loader);
    }

    ::testing::Mock::VerifyAndClear(&Creator::instance());
    ::testing::Mock::VerifyAndClear(&Destroyer::instance());
}

namespace
{
std::shared_ptr<messaging::NamedSymbolLoader> the_default_named_symbol_loader()
{
    using namespace ::testing;

    auto symbol_loader = std::make_shared<NiceMock<MockNamedSymbolLoader>>();

    ON_CALL(*symbol_loader, load_symbol_for_name(messaging::PluginConnector::Create::name))
        .WillByDefault(Return(reinterpret_cast<void*>(&Creator::create)));
    ON_CALL(*symbol_loader, load_symbol_for_name(messaging::PluginConnector::Destroy::name))
        .WillByDefault(Return(reinterpret_cast<void*>(&Destroyer::destroy)));

    return symbol_loader;
}
}

TEST(PluginConnector, forwards_parameters_call_to_implementation)
{
    using namespace ::testing;

    messaging::StdVectorEnumerator<messaging::Parameter> params;

    MockConnector connector;
    EXPECT_CALL(connector, parameters()).Times(1).WillRepeatedly(ReturnRef(params));

    ON_CALL(Creator::instance(), do_create()).WillByDefault(Return(&connector));

    messaging::PluginConnector pc(the_default_named_symbol_loader());
    pc.parameters();
}

TEST(PluginConnector, forwards_run_call_to_implementation)
{
    using namespace ::testing;

    MockConnector connector;
    EXPECT_CALL(connector, run()).Times(1);

    ON_CALL(Creator::instance(), do_create()).WillByDefault(Return(&connector));

    messaging::PluginConnector pc(the_default_named_symbol_loader());
    pc.run();
}

TEST(PluginConnector, forwards_stop_call_to_implementation)
{
    using namespace ::testing;

    MockConnector connector;
    EXPECT_CALL(connector, stop()).Times(1);

    ON_CALL(Creator::instance(), do_create()).WillByDefault(Return(&connector));

    messaging::PluginConnector pc(the_default_named_symbol_loader());
    pc.stop();
}
