/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "did_finish.h"

#include <iostream>

namespace ex = core::posix::exit;
namespace wa = core::posix::wait;

std::ostream& core::posix::operator<<(std::ostream& out, Signal signal)
{
    switch (signal)
    {
        case core::posix::Signal::unknown:
            return out << "unknown";
        case core::posix::Signal::sig_hup:
            return out << "sig_hup";
        case core::posix::Signal::sig_int:
            return out << "sig_int";
        case core::posix::Signal::sig_quit:
            return out << "sig_quit";
        case core::posix::Signal::sig_ill:
            return out << "sig_ill";
        case core::posix::Signal::sig_abrt:
            return out << "sig_abrt";
        case core::posix::Signal::sig_fpe:
            return out << "sig_fpe";
        case core::posix::Signal::sig_kill:
            return out << "sig_kill";
        case core::posix::Signal::sig_segv:
            return out << "sig_segv";
        case core::posix::Signal::sig_pipe:
            return out << "sig_pipe";
        case core::posix::Signal::sig_alrm:
            return out << "sig_alrm";
        case core::posix::Signal::sig_term:
            return out << "sig_term";
        case core::posix::Signal::sig_usr1:
            return out << "sig_usr1";
        case core::posix::Signal::sig_usr2:
            return out << "sig_usr2";
        case core::posix::Signal::sig_chld:
            return out << "sig_chld";
        case core::posix::Signal::sig_cont:
            return out << "sig_cont";
        case core::posix::Signal::sig_stop:
            return out << "sig_stop";
        case core::posix::Signal::sig_tstp:
            return out << "sig_tstp";
        case core::posix::Signal::sig_ttin:
            return out << "sig_ttin";
        case core::posix::Signal::sig_ttou:
            return out << "sig_ttou";
    }

    return out;
}

std::ostream& wa::operator<<(std::ostream& out, wa::Result::Status status)
{
    switch (status)
    {
        case Result::Status::undefined:
            return out << "undefined";
        case Result::Status::no_state_change:
            return out << "no_state_change";
        case Result::Status::exited:
            return out << "exited";
        case Result::Status::signaled:
            return out << "signaled";
        case Result::Status::stopped:
            return out << "stopped";
        case Result::Status::continued:
            return out << "continued";
    }

    return out;
}

::testing::AssertionResult testing::did_finish_successfully(const wa::Result& result)
{
    if (result.status != wa::Result::Status::exited)
    {
        if (result.status != wa::Result::Status::signaled)
        {
            return ::testing::AssertionFailure() << "Process did not exit, but: " << result.status;
        }
        else
        {
            return ::testing::AssertionFailure() << "Process was signaled with: " << result.detail.if_signaled.signal;
        }
    }

    if (result.detail.if_exited.status != ex::Status::success)
    {
        return ::testing::AssertionFailure() << "Process did exit with failure.";
    }

    return ::testing::AssertionSuccess();
}

::testing::AssertionResult testing::did_abort(const wa::Result& result)
{
    if (result.status != wa::Result::Status::signaled)
    {
        return ::testing::AssertionFailure() << "Process did not get signaled, but: " << result.status;
    }

    if (result.detail.if_signaled.signal != core::posix::Signal::sig_abrt)
    {
        return ::testing::AssertionFailure() << "Process was signaled with " << result.detail.if_signaled.signal
                                             << " instead of " << core::posix::Signal::sig_abrt;
    }

    return ::testing::AssertionSuccess();
}
