/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/connector_factory.h>

#include <messaging/associative_connector_factory.h>
#include <messaging/associative_dictionary.h>

#include <gtest/gtest.h>

namespace
{
messaging::ConnectorFactory::Generator the_null_generator()
{
    return [](const messaging::Dictionary<std::string, std::string>&)
    {
        return std::shared_ptr<messaging::Connector>{};
    };
}
}

TEST(AssociativeConnectorFactory, construction_and_deconstruction_work)
{
    messaging::AssociativeConnectorFactory acf;
}

TEST(AssociativeConnectorFactory, adding_a_generator_works)
{
    messaging::AssociativeConnectorFactory acf;

    EXPECT_FALSE(acf.has_generator_for_name("42"));
    acf.add_generator_for_name("42", the_null_generator());
    EXPECT_TRUE(acf.has_generator_for_name("42"));
}

TEST(AssociativeConnectorFactory, removing_a_generator_works)
{
    messaging::AssociativeConnectorFactory acf;

    EXPECT_FALSE(acf.has_generator_for_name("42"));
    acf.add_generator_for_name("42", the_null_generator());
    EXPECT_TRUE(acf.has_generator_for_name("42"));
    acf.remove_generator_for_name("42");
    EXPECT_FALSE(acf.has_generator_for_name("42"));
}

TEST(AssociativeConnectorFactory, requesting_generation_for_known_generator_works)
{
    messaging::AssociativeConnectorFactory acf;
    messaging::AssociativeDictionary<std::string, std::string> dict;
    acf.add_generator_for_name("42", the_null_generator());
    EXPECT_NO_THROW(acf.create_instance_for_name("42")(dict));
}

TEST(AssociativeConnectorFactory, requesting_generation_for_unknown_generator_throws)
{
    messaging::AssociativeConnectorFactory acf;
    messaging::AssociativeDictionary<std::string, std::string> dict;
    EXPECT_FALSE(acf.has_generator_for_name("42"));
    EXPECT_THROW(acf.create_instance_for_name("42")(dict), std::out_of_range);
}
